## -*-Tcl-*-
 # ###################################################################
 #  AlphaTk - the ultimate editor
 # 
 #  FILE: "alpha_windows.tcl"
 #                                    created: 04/12/98 {22:45:38 PM} 
 #                                last update: 1999-09-06T17:31:15Z 
 #  Author: Vince Darley
 #  E-mail: vince@santafe.edu
 #    mail: 317 Paseo de Peralta, Santa Fe, NM 87501
 #     www: http://www.santafe.edu/~vince
 #  
 # Copyright (c) 1998-1999  Vince Darley
 # 
 # See the file "license.terms" for information on use and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 #  Description: 
 # 
 #  History
 # 
 #  modified by  rev reason
 #  -------- --- --- -----------
 #  04/12/98 VMD 1.0 original
 # ###################################################################
 ##


namespace eval win {}

# This allows some global things to happen in a slightly simpler
# fashion even when there are no windows around.  I don't think
# it has any adverse side-effects.  It associates the Tk window '.'
# with the Alpha window "" (i.e. no window).
set win::tk() .

# used to over-ride the text widget
# they are selectively over-ridden with alpha-specific functionality later.
namespace eval tw {
    proc bbox {w args} { uplevel 1 ::tw::$w bbox $args } 
    proc cget {w args} { uplevel 1 ::tw::$w cget $args } 
    proc compare {w args} { uplevel 1 ::tw::$w compare $args } 
    proc configure {w args} { uplevel 1 ::tw::$w configure $args } 
    proc debug {w args} { uplevel 1 ::tw::$w debug $args } 
    proc delete {w args} { uplevel 1 ::tw::$w delete $args } 
    proc dlineinfo {w args} { uplevel 1 ::tw::$w dlineinfo $args } 
    proc get {w args} { uplevel 1 ::tw::$w get $args } 
    proc image {w args} { uplevel 1 ::tw::$w image $args } 
    proc index {w args} { uplevel 1 ::tw::$w index $args } 
    proc insert {w args} { uplevel 1 ::tw::$w insert $args } 
    proc mark {w args} { uplevel 1 ::tw::$w mark $args } 
    proc scan {w args} { uplevel 1 ::tw::$w scan $args } 
    proc search {w args} { uplevel 1 ::tw::$w search $args } 
    proc see {w args} { uplevel 1 ::tw::$w see $args } 
    proc tag {w args} { uplevel 1 ::tw::$w tag $args } 
    proc window {w args} { uplevel 1 ::tw::$w window $args } 
    proc xview {w args} { uplevel 1 ::tw::$w xview $args } 
    proc yview {w args} { uplevel 1 ::tw::$w yview $args }
    proc silent_insert {w args} {}
    proc silent_delete {w args} {}
    
    variable lmatch [list "(" "\{" "\[" "\""]
    variable rmatch [list ")" "\}" "\]" "\""]
    proc split {w} {}
    proc unsplit {w} {}
    
    variable split
}

proc tw::base_window {w} {
    variable split
    return [expr {[info exists split($w)] ? $split($w) : $w}]
}

proc tw::silent_insert {w args} { 
    variable split
    if {[info exists split($w)]} {
	variable splitter
	foreach ww $splitter([set split($w)]) {
	    uplevel 1 ::tw::$ww insert $args 
	}
    } else {
	uplevel 1 ::tw::$w insert $args 
    }
}

proc tw::silent_delete {w args} { 
    variable split
    if {[info exists split($w)]} {
	variable splitter
	foreach ww $splitter([set split($w)]) {
	    uplevel 1 ::tw::$ww delete $args 
	}
    } else {
	uplevel 1 ::tw::$w delete $args 
    }
}

proc tw::windowCleanup {w} {
    variable split
    variable splitter
    if {[info exists split($w)]} {
	set original $split($w)
	foreach ww $splitter($original) {
	    unset split($ww)
	    if {$ww != $original} {
		rename ::$ww {}
	    }
	}
	unset splitter($original)
    }
}

proc tw::split {w} {
    variable split
    if {[info exists split($w)]} {
	return
    }
    duplicate $w [winfo parent $w]
}

proc tw::duplicate {w intoFrame} {
    variable split
    # find a unique name
    set i 2
    while {[winfo exists $intoFrame.text$i]} {
	incr i
    }
    set ww $intoFrame.text$i
    text $ww -height 0
    set W [winfo parent $ww]
    $ww configure -relief flat -bd 2 -yscrollcommand "$W.scroll$i set" \
      -bg white 
    scrollbar $W.scroll$i -command "$ww yview"
    set sash [frame $W.split$i]
    bind $sash <Button-1> "tw::_startGrip $sash %y $w $ww"
    bind $sash <B1-Motion> "tw::_handleGrip $sash %Y $w $ww"
    bind $sash <B1-ButtonRelease-1> "tw::_endGrip $sash %y $w $ww"
    grid $W.split$i -sticky ew -columnspan 2 -row 4
    grid $ww -sticky news -column 0 -row 5
    grid $W.scroll$i -sticky nse -column 1 -row 5
    grid rowconfigure $W 5 -weight 1
    $W.split$i configure -height 6 -bd 3 -relief raised
    ::alpha::textwidget $ww
    $ww configure -font [$W.text cget -font] -tabs [$W.text cget -tabs]
    $ww insert 1.0 [$W.text get 1.0 "end -1c"]
    rename $ww ::tw::$ww
    uplevel [list proc ::$ww {cmd args} "namespace eval tw \"\$cmd $ww \$args\""]
    variable splitter
    if {[info exists split($w)]} {
	set original $split($w)
	lappend splitter($original) $ww
    } else {
	set original $w
	set split($w) $w
	lappend splitter($original) $w $ww
    }
    set split($ww) $original

}

proc tw::_startGrip {sash y w ww} {
    $sash configure -relief sunken
    grab $sash
}

# not an ideal solution, but kind of works.
proc tw::_handleGrip {sash y w ww} {
    set tophalf [expr {[winfo height $w] + ($y-[winfo rooty $sash])}]
    set height [expr {[winfo height $w] + [winfo height $ww]}]
    if {[set ww [expr {100*$tophalf/$height}]] < 0} {
	set ww 0
    }
    grid rowconfigure [winfo toplevel $w] 2 -weight $ww
    if {[set ww [expr {100-100*$tophalf/$height}]] < 0} {
	set ww 0
    }
    grid rowconfigure [winfo toplevel $w] 5 -weight $ww
    update
}

proc tw::_endGrip {sash y w ww} {
    $sash configure -relief raised
    grab release $sash
}


proc tw::toggleSplit {w} {
    variable split
    if {[info exists split($w)]} {
	unsplit $w
    } else {
	split $w
    }
}

proc tw::unsplit {w} {
    variable split
    variable splitter
    set original $split($w)
    set W [winfo parent $original]
    foreach ww $splitter($original) {
	unset split($ww)
	if {$ww != $original} {
	    rename ::$ww {}
	    set W [winfo parent $ww]
	    regexp {[0-9]+$} $ww num
	    # remove any possible bindings which may trigger
	    # side-effects (esp. for destroy)
	    bindtags $W.text$num $W.text$num
	    destroy $W.text$num
	    destroy $W.scroll$num
	    destroy $W.split$num
	}
    }
    grid rowconfigure $W 5 -weight 0
    grid rowconfigure $W 4 -weight 0
    grid rowconfigure $W 2 -weight 1
    grid rowconfigure $W 1 -weight 0
    unset splitter($original)
}


proc splitter {w} {
    $w configure -relief sunken
    tw::toggleSplit [winfo parent $w].text 
}

proc win::kill {w} {
    variable tktitle
    if {[info exists tktitle($w)]} {
	killWindow $win::tktitle($w) 1
    }
}

proc alpha::createWin {n {text ""}} {
    global defWidth defHeight defTop defLeft showFullPathsInWindowTitles
    variable colours
    set nn 0
    while {[winfo exists .al$nn]} {incr nn}
    toplevel [set w .al$nn]
    wm withdraw $w
    wm geometry $w ${defWidth}x${defHeight}+${defLeft}+${defTop}
    if {$showFullPathsInWindowTitles} {
	wm title $w $n
    } else {
	wm title $w [file tail $n]
    }
    #::bind $w <Destroy> [list killWindow $n]
    text $w.text -relief flat -bd 2 -yscrollcommand "$w.scroll set" \
      -bg white -font "Monaco 9" -height 0
    scrollbar $w.scroll -command "$w.text yview"
    frame $w.rt -highlightcolor red -height 48
    frame $w.splitter -height 8 -relief raised -borderwidth 3 \
      -highlightthickness 1 -width [winfo reqwidth $w.scroll]
    bindtags $w.splitter Splitter
    grid $w.text -sticky news -column 0 -rowspan 3
    grid $w.rt -sticky nse -column 1 -row 0
    grid $w.splitter -sticky we -column 1 -row 1
    grid $w.scroll -sticky nse -column 1 -row 2
    global horScrollBar
    if {$horScrollBar} {
	scrollbar $w.hscroll -command "$w.text xview" -orient horizontal
	grid $w.hscroll -sticky sew -column 0 -row 3
	$w.text configure -xscrollcommand "$w.hscroll set"
    }
    
    #label $w.rt.dirty -text ""
    label $w.rt.dirty -image clean
    bind $w.rt.dirty <[lindex $alpha::modifier_keys 0]-Button-1> "tw::lockClick $w.text"
    # Note: 'width 1' is too narrow, but 'width 2' is a little larger
    # than is necessary.  Hence we use width 1 here, but then use
    # grid -sticky ew to let them expand to the available space.  This
    # allows the right margin to size itself to the width of the scrollbar,
    # which can vary depending on the current appearance/colour-scheme.
    menubutton $w.rt.marks -text "M" -padx 0 -pady 0 -width 1 -relief ridge \
      -activebackground $colours(activebackground) -menu $w.rt.marks.menu \
      -activeforeground $colours(activeforeground)
    menubutton $w.rt.func -text "\{ \}" -padx 0 -pady 0 -width 1 -relief ridge \
      -activebackground $colours(activebackground) -menu $w.rt.func.menu \
      -activeforeground $colours(activeforeground)
    menubutton $w.rt.files -text "f" -padx 0 -pady 0 -width 1 -relief ridge \
      -activebackground $colours(activebackground) -menu $w.rt.files.menu \
      -activeforeground $colours(activeforeground)
    ::menu $w.rt.marks.menu -tearoff 0 \
      -postcommand [list alpha::markFile $w.rt.marks.menu]
    $w.rt.marks.menu add command -label "Mark File" \
      -command "alpha::markMenuClear $w.rt.marks.menu ; markFile"
    $w.rt.marks.menu add separator
    ::menu $w.rt.func.menu -tearoff 0 \
      -postcommand [list alpha::parseFuncs $w.rt.func.menu]
    ::menu $w.rt.files.menu -tearoff 0 \
      -postcommand [list alpha::filesMenu $w.rt.files.menu]
    #bind $w.splitter <Button-1> "tw::toggleSplit $w.text"
    grid $w.rt.dirty -row 0 -sticky ew
    grid $w.rt.marks -row 1 -sticky ew
    grid $w.rt.func -row 2 -sticky ew
    grid $w.rt.files -row 3 -sticky ew
    #update idletasks
    alpha::setIcon $w
    grid columnconfigure $w 1 -weight 0 -minsize [winfo reqwidth $w.scroll]
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 2 -weight 1
    global win::tk alpha_winfo win::tktitle
    set win::tk($n) $w.text
    set win::tktitle($w.text) $n
    global ::tw::$w.text tabSize
    array set ::tw::$w.text [list dirty 0 read-only 0 tabsize $tabSize]
    textwidget $w.text
    $w.text insert 1.0 $text
    $w.text mark set insert 1.0
    rename $w.text ::tw::$w.text
    uplevel [list proc ::$w.text {cmd args} "namespace eval tw \"\$cmd $w.text \$args\""]
    wm protocol $w WM_DELETE_WINDOW [list killWindow $n]
    global useGlobalMenuBarOnly
    if {![info exists useGlobalMenuBarOnly] || !$useGlobalMenuBarOnly} {
	$w configure -menu .menubar
    }

    focus $w.text
    return $w
}

proc alpha::textwidget {w {n ""}} {
    global defaultFont fontSize blockCursor tabSize
    bindtags $w [concat AlphaStyle Alpha [bindtags $w]]
    #bindtags $w [concat AlphaStyle Alpha AlphaMenu [bindtags $w]]
    $w configure -wrap none -font "$defaultFont $fontSize"
    tw::setTabSize $w $tabSize
    if {$blockCursor} {
	$w configure -insertwidth $fontSize -insertbackground grey25
    }
    $w tag configure blink -background black -foreground white
    $w tag configure backsel -background darkgray
    $w tag configure color1 -foreground blue
    $w tag configure color2 -foreground cyan
    $w tag configure color3 -foreground green
    $w tag configure color4 -foreground magenta
    $w tag configure color5 -foreground red
    $w tag configure color6 -foreground white
    $w tag configure color7 -foreground yellow
    $w tag configure color8 -foreground blue
    $w tag configure color9 -foreground blue
    $w tag configure color10 -foreground blue
    $w tag configure color11 -foreground blue
    $w tag configure color13 -underline 1
    $w tag configure color14 -underline 1
    $w tag configure color15 -underline 1 -foreground green
    $w tag bind color15 <ButtonPress> "text_cmd hyper activate %x %y ; break"
    $w tag bind color15 <Enter> "text_cmd hyper enter %x %y ; break"
    $w tag bind color15 <Leave> "text_cmd hyper leave %x %y ; break"
    
    if {$n != ""} {
	global ::file::config
	if {[info exists file::config($n)]} {
	    foreach opt [set file::config($n)] {
		catch {eval [list setWinInfo -w $n] $opt}
	    }
	    unset file::config($n)
	}
    }
}


proc tw::addHyper {w from to hyper} {
    global alphaPriv
    set alphaPriv(hyper:$w:$from:$to) $hyper
}

proc tw::hyper {w what x y} {
    switch $what {
	"leave" {
	    message ""
	    $w config -cursor xterm
	    return
	}
    }
    set pos [tkTextClosestGap $w $x $y]
    set range [$w tag prevrange color15 $pos]
    if {[lindex $range 1] < $pos} {
	echo "Weird, I thought I had a hyper"
	return
    }
    global alphaPriv
    set from [lindex $range 0]
    set to [lindex $range 1]
    switch -- $what {
	"activate" {
	    eval ::select $range
	    update
	    $w tag delete sel
	    update
	    uplevel \#0 $alphaPriv(hyper:$w:$from:$to)
	}
	"enter" {
	    message "---> '$alphaPriv(hyper:$w:$from:$to)'"
	    $w config -cursor arrow
	}
    }
}


proc flash {char} {
    set pos [pos::math [getPos] - 1]
    if {[lookAt $pos] == "\\"} {
	# it's a literal character
	return
    }
    if {[catch {matchIt $char $pos} matched]} {
	beep
	message "No matching '$char'!"
	return
    } else {
	blink $matched
    }
}

proc tw::match {w char pos {limit ""}} {
    #echo "tw::match $w $char $pos $limit"
    variable lmatch
    variable rmatch
    if {[set i [lsearch -exact $lmatch $char]] != -1} {
	lappend looking [lindex $rmatch $i]
	if {$limit == ""} { 
	    set limit end 
	} else {
	    set limit "$pos + ${limit}c"
	}
	return [matchForward $w $pos $looking $limit]
    } elseif {[set i [lsearch -exact $rmatch $char]] != -1} {
	lappend looking [lindex $lmatch $i]
	if {$limit == "0" || $limit == ""} { 
	    set limit 1.0 
	} else {
	    set limit "$pos - ${limit}c"
	}
	return [matchBack $w $pos $looking $limit]
    } else {
	error "Char '$char' unrecognised by match"
    }
}


if {[info tclversion] < 8.1} {
    proc tw::matchQuoteBack {w pos limit} {
	set look "(^|\[^\\\])\""
	set pos [$w search -backwards -regexp -- $look "$pos" $limit]
	if {$pos == ""} { error "No match" }
	return $pos
    }
    
    proc tw::matchQuoteForward {w pos limit} {
	set look "(^|\[^\\\])\""
	set pos [$w search -forwards -regexp -count count -- $look "$pos" $limit]
	if {$pos == ""} { error "No match" }
	return "$pos +[expr {$count -1}]c"
    }
    proc tw::matchForward {w pos looking limit} {
	variable rmatch
	variable lmatch
	set pos "$pos -1c"
	while 1 {
	    set look "(^|\[^\\\])\[[lindex ${looking} end]\"\[\(\{\]"
	    set pos [$w search -forwards -regexp -count count -- $look $pos $limit]
	    if {$pos == ""} { error "No match" }
	    set pos "$pos +[expr {$count -1}]c"
	    set char [$w get $pos]
	    if {$char == "\""} {
		set pos [matchQuoteForward $w $pos $limit]
		continue
	    }
	    if {[lindex $looking end] == $char} {
		set looking [lreplace $looking end end]
		if {$looking == ""} {
		    return $pos
		}
	    } else {
		lappend looking [lindex $rmatch [lsearch -exact $lmatch $char]]
	    }
	}
    }

    proc tw::matchBack {w pos looking limit} {
	variable rmatch
	variable lmatch
	while 1 {
	    set look "(^|\[^\\\])\[\]\"\)\}[lindex ${looking} end]\]"
	    set pos [$w search -backwards -regexp -count count -- $look "$pos" $limit]
	    if {$pos == ""} { error "No match" }
	    set char [$w get "$pos +[expr {$count -1}]c"]
	    if {$char == "\""} {
		set pos [matchQuoteBack $w $pos $limit]
		continue
	    }
	    if {[lindex $looking end] == $char} {
		set looking [lreplace $looking end end]
		if {$looking == ""} {
		    return "$pos +1c"
		}
	    } else {
		lappend looking [lindex $lmatch [lsearch -exact $rmatch $char]]
	    }
	}
    }

} else {
    proc tw::matchError {w dir look pos} {
	set start [lindex [::split [set pos [index $w $pos]] .] 0]
	error "No match $dir for '$look' from $pos; possible unmatched\
	  delimiter at: [get $w ${start}.0 [expr {$start+1}].0]"
    }
    
    proc tw::matchQuoteBack {w pos limit} {
	set look "(^|\[^\\\\])\""
	set pos1 [$w search -backwards -regexp -- $look "$pos" $limit]
	if {$pos1 == ""} { matchError $w back \" $pos }
	return $pos1
    }
    
    proc tw::matchQuoteForward {w pos limit} {
	set look "(^|\[^\\\\])\""
	set pos1 [$w search -forwards -regexp -count count -- $look "$pos" $limit]
	if {$pos1 == ""} { matchError $w forward \" $pos }
	return "$pos1 +[expr {$count -1}]c"
    }

    proc tw::matchForward {w pos looking limit} {
	variable rmatch
	variable lmatch
	set pos "$pos -1c"
	while 1 {
	    #echo "$pos $looking $limit"
	    set look "(^|\[^\\\\])\[[lindex ${looking} end]\"\[\(\{\]"
	    set pos1 [$w search -forwards -regexp -count count -- $look $pos $limit]
	    if {$pos1 == ""} { matchError $w forward [lindex ${looking} end] $pos }
	    set pos $pos1
	    set pos "$pos +[expr {$count -1}]c"
	    set char [$w get $pos]
	    if {$char == "\""} {
		set pos [matchQuoteForward $w $pos $limit]
		continue
	    }
	    if {[lindex $looking end] == $char} {
		set looking [lreplace $looking end end]
		if {$looking == ""} {
		    return $pos
		}
	    } else {
		lappend looking [lindex $rmatch [lsearch -exact $lmatch $char]]
	    }
	}
    }

    proc tw::matchBack {w pos looking limit} {
	variable rmatch
	variable lmatch
	while 1 {
	    set look "(^|\[^\\\\])\[\]\"\)\}[lindex ${looking} end]\]"
	    set pos1 [$w search -backwards -regexp -count count -- $look "$pos" $limit]
	    if {$pos1 == ""} { matchError $w back [lindex ${looking} end] $pos }
	    set pos $pos1
	    set char [$w get "$pos +[expr {$count -1}]c"]
	    if {$char == "\""} {
		set pos [matchQuoteBack $w $pos $limit]
		continue
	    }
	    if {[lindex $looking end] == $char} {
		set looking [lreplace $looking end end]
		if {$looking == ""} {
		    return "$pos +1c"
		}
	    } else {
		lappend looking [lindex $lmatch [lsearch -exact $rmatch $char]]
	    }
	}
    }


}



proc tw::select {w from to} {
    $w tag add sel $from $to
}

proc tw::balance {w} {
    variable lmatch ; variable rmatch
    set f [$w search -forwards -regexp -- "\[\]\[\{\(\\)\}\]" insert]
    set b [$w search -backwards -regexp -- "\[\]\[\{\(\\)\}\]" insert]
    if {[set i [lsearch -exact $rmatch [$w get $f]]] != -1} {
	# we found a backwards looking element while looking forwards
	# We need to find its partner
	select $w [match $w [$w get $f] $f] "$f +1c"
    } elseif {[set i [lsearch -exact $lmatch [$w get $b]]] != -1} {
	# we found the opposite, which is also ok
	select $w $b "[match $w [$w get $b] $b] +1c"
    } else {
	# hmm, this is harder
	echo "balance: harder case not yet implemented"
    }
}

proc tw::undo {w {allcontiguous 1}} {
    variable $w
    if {![info exists ${w}(undo)]} {
	return
    }
    set first 1
    while 1 {
	set action [lindex [set ${w}(undo)] end]
	if {$action == ""} {break}
	switch -- [lindex $action 0] {
	    "insert" {
		set len [string length [join [lrange $action 2 end] ""]]
		set where [lindex $action 1]
		if {!$first} {
		    if {[compare $w $where != $new_pos]} {
			break
		    }	    
		}
	    }
	    "delete" {
		set len [string length [join [lrange $action 3 end] ""]]
		set where [lindex $action 1]
		if {!$first} {
		    if {[compare $w $where != "$new_pos + ${len}c"]} {
			break
		    }	    
		}
	    }
	}
	# perform action
	# adjust the undo/redo lists
	lappend ${w}(redo) $action
	set ${w}(undo) [lrange [set ${w}(undo)] 0 [expr {[llength [set ${w}(undo)]] -2}]]
	
	switch -- [lindex $action 0] {
	    "insert" {
		if {$len > 1} {
		    uplevel 1 [list ::tw::silent_delete $w $where [list $where +${len}c]]
		} else {
		    uplevel 1 ::tw::silent_delete $w $where
		}
		set new_pos [index $w "$where - ${len}c"]
	    }
	    "delete" {
		uplevel 1 [list ::tw::silent_insert $w $where [lindex $action 3]]
		set new_pos $where
	    }
	}
	if {!$allcontiguous} {
	    break
	}
	set first 0
    }
    
    # undirty if necessary
    if {[llength [set ${w}(undo)]] == 0} { 
	unset ${w}(undo)
	dirty $w 0
    }
}

proc tw::redo {w {allcontiguous 1}} {
    variable $w
    if {![info exists ${w}(redo)]} {
	return
    }
    if {![info exists ${w}(undo)]} {
	dirty $w 1
    }
    set first 1
    while 1 {
	set action [lindex [set ${w}(redo)] end]
	if {$action == ""} {break}
	switch -- [lindex $action 0] {
	    "insert" {
		set len [string length [join [lrange $action 2 end] ""]]
		set where [lindex $action 1]
		if {!$first} {
		    if {[compare $w $where != $new_pos]} {
			break
		    }	    
		}
	    }
	    "delete" {
		set len [string length [join [lrange $action 3 end] ""]]
		set where [lindex $action 1]
		if {!$first} {
		    if {[compare $w "$where +${len}c" != $new_pos]} {
			break
		    }	    
		}
	    }
	}
	# perform action
	# adjust the undo/redo lists
	lappend ${w}(undo) $action
	set ${w}(redo) [lrange [set ${w}(redo)] 0 [expr {[llength [set ${w}(redo)]] -2}]]
	
	switch -- [lindex $action 0] {
	    "delete" {
		if {$len > 1} {
		    uplevel 1 [list ::tw::silent_delete $w $where [list $where +${len}c]]
		} else {
		    uplevel 1 ::tw::silent_delete $w $where
		}
		set new_pos $where
	    }
	    "insert" {
		uplevel 1 [list ::tw::silent_insert $w $where [lindex $action 2]]
		set new_pos [index $w "$where + ${len}c"]
	    }
	}
	if {!$allcontiguous} {
	    break
	}
	set first 0
    }
    # undirty if necessary
    if {[llength [set ${w}(redo)]] == 0} { 
	unset ${w}(redo)
    }
}

proc tw::lockClick {w} {
    variable $w
    if {[info exists ${w}(shell)]} {
	message "Clicking doesn't affect shell windows."
	return
    }
    if {[set ${w}(dirty)]} {
	message "Clicking only affects locked or clean windows."
	return
    }
    if {[$w cget -state] == "disabled"} {
	tw::read_only $w 0
	global ::win::tktitle
	set f $::win::tktitle($w)
	if {[file exists $f]} {
	    file attributes $f -readonly 0
	}
    } else {
	tw::read_only $w 1
    }
}

proc tw::read_only {w {d 1}} {
    if {$d} {
	tw::dirty $w 0
	$w configure -state disabled
	[winfo toplevel $w].rt.dirty configure -image lock
    } else {
	$w configure -state normal
	[winfo toplevel $w].rt.dirty configure -image clean
    }
    variable $w
    set ${w}(read-only) $d
}

proc tw::dirty {w {d 1}} {
    global ::win::tktitle
    set w [base_window $w]
    variable $w
    if {[info exists ${w}(shell)]} {
	return
    }
    if {[set ${w}(dirty)] != $d} {
	set ${w}(dirty) $d
	::dirtyHook $::win::tktitle($w) $d
    }
    if {$d == 0} {
	if {[info exists ${w}(undo)]} {
	    unset ${w}(undo)
	}
	if {[info exists ${w}(redo)]} {
	    unset ${w}(redo)
	}
    }
    
    if {[$w cget -state] == "disabled"} {
	# it's a read-only window
	dialog::yesno -c "Modified a read-only window!  Hit cancel to stack dump"
    }
    
    [winfo toplevel $w].rt.dirty configure -image [expr {$d ? "dirty" : "clean"}]
    
}

proc tw::save {w} {
    global ::win::tk
    variable undo
    variable redo
    set tkw $::win::tk($w)
    variable $tkw
    if {[set ${tkw}(dirty)] != 0} {
	set ${tkw}(dirty) 0
	::dirtyHook $w 0
    }
    [winfo toplevel $tkw].rt.dirty configure -image clean
    if {[info exists ${tkw}(undo)]} {
	unset ${tkw}(undo)
    }
    if {[info exists ${tkw}(redo)]} {
	unset ${tkw}(redo)
    }
}

proc tw::insert {w where args} {
    variable split
    global wordWrap fillColumn
    # make sure we use the 'where' which corresponds to the correct pane if a
    # window has been split into pieces.  To do this we must turn it into a 
    # canonical line.col form
    set where [index $w $where]
    if {[info exists split($w)]} {
	set w $split($w)
    }
    variable $w
    # Basic wrap technique is as follows:
    # (i) Only bother with any wrapping if it's a single character being
    # inserted.
    # (ii) Given it's a single char, any whitespace just moves to the next
    # line
    # (iii) If not whitespace, check if we're in the middle of a word or
    # starting a new word.  The former shifts the whole word to the next
    # line, the latter just moves the word we're typing to the next line.
    if {[info exists wordWrap] && $wordWrap} {
	# only wrap if it's a single character
	if {[string length [set char [join $args ""]]] == 1} {
	    set where [index $w $where]
	    if {[lindex [::split $where .] 1] >= $fillColumn} {
		switch -- $char {
		    "\r" - "\n" {}
		    " " {
			set args [list "\n"]
			set addReturn 1
		    }
		    "\t" {
			set args [concat [list "\n" ""] $args]
			set addReturn 1
		    }
		    default {
			if {[regexp "\[ \t\r\n\]" [get $w "$where -1c"]]} {
			    set args [concat [list "\n" ""] $args]
			    set addReturn 1
			} else {
			    # we've got a word going back
			    set p [search $w -backward -regexp -- "\[ \t\r\n\]" $where]
			    mark $w set tmp-wordwrap $where
			    if {![compare $w $p <= "$where linestart"]} {
				delete $w $p
				insert $w $p "\n"
			    }
			    set where [index $w tmp-wordwrap]
			    mark $w unset tmp-wordwrap
			    set addReturn 1
			}
		    }
		}
	    }
	}
    }
    
    if {![info exists ${w}(shell)]} {
	if {![info exists ${w}(undo)]} {
	    dirty $w
	}
	set where [index $w $where]
	lappend ${w}(undo) "insert $where $args"
	if {[info exists ${w}(redo)]} {
	    unset ${w}(redo)
	}
    }
    if {[info exists split($w)]} {
	variable splitter
	foreach ww $splitter([set split($w)]) {
	    uplevel 1 ::tw::$ww insert $where $args 
	}
    } else {
	uplevel 1 ::tw::$w insert $where $args
    }
    if {[info exists addReturn]} {
	global indentOnReturn
	if {$indentOnReturn} {
	    mark $w set tmp-wordwrap insert
	    bind::IndentLine
	    mark $w set insert tmp-wordwrap
	    mark $w unset tmp-wordwrap
	}
    }
    win::colour_line $w
}

proc tw::delete {w where1 {where2 ""}} {
    # make sure we use the 'where' which corresponds to the correct pane if a
    # window has been split into pieces.  To do this we must turn it into a 
    # canonical line.col form
    set where1 [index $w $where1]
    if {$where2 != ""} {
	set where2 [index $w $where2]
    }
    
    variable split
    if {[info exists split($w)]} {
	set w $split($w)
    }
    variable $w
    if {![info exists ${w}(shell)]} {
	if {![info exists ${w}(undo)]} {
	    dirty $w
	}
	if {$where2 != ""} { 
	    lappend ${w}(undo) [list delete $where1 $where2 [::tw::$w get $where1 $where2]]
	} else {
	    lappend ${w}(undo) [list delete $where1 $where2 [::tw::$w get $where1]]
	}
	if {[info exists ${w}(redo)]} {
	    unset ${w}(redo)
	}
    }
    if {[info exists split($w)]} {
	variable splitter
	foreach ww $splitter([set split($w)]) {
	    uplevel 1 ::tw::$ww delete $where1 $where2
	}
    } else {
	uplevel 1 ::tw::$w delete $where1 $where2
    }
    
}

if {[info tclversion] < 8.1} {
    proc tw::backward_word {w pos} {
	global wordBreak wordBreakPreface
	regsub -all "\\\\w" "(^|$wordBreakPreface)$wordBreak" "a-zA-Z0-9_" reg
	set to [search $w -backwards -regexp -- $reg "$pos -1c"]
	if {$to == ""} {
	    return end
	} else {
	    regsub -all "\\\\w" "($wordBreakPreface|^)" "a-zA-Z0-9_" reg
	    if {[regexp -- $reg [get $w $to] match]} {
		if {[string length $match]} {
		    return "$to +1c"
		} else {
		    return $to
		}
	    } else {
		return "$to +1c"
	    }
	}
    }
    proc tw::forward_word {w pos} {
	global wordBreakPreface
	regsub -all "\\\\w" "($wordBreakPreface|\$)" "a-zA-Z0-9_" reg
	set to [search $w -forwards -regexp -- $reg "$pos +1c"]
	if {$to == ""} {
	    return end
	} else {
	    return $to
	}
    }
} else {
    proc tw::backward_word {w pos} {
	global wordBreak wordBreakPreface
	set reg "(^|$wordBreakPreface)$wordBreak"
	set to [search $w -backwards -regexp -- $reg "$pos -1c"]
	if {$to == ""} {
	    return 1.0
	} else {
	    set reg "($wordBreakPreface|^)"
	    if {[regexp -- $reg [get $w $to] match]} {
		if {[string length $match]} {
		    return "$to +1c"
		} else {
		    return $to
		}
	    } else {
		return "$to +1c"
	    }
	}
    }
    proc tw::forward_word {w pos} {
	global wordBreakPreface
	set to [search $w -forwards -regexp -- "($wordBreakPreface|\$)" "$pos +1c"]
	if {$to == ""} {
	    return end
	} else {
	    return $to
	}
    }
}

proc tw::double_click {w x y} {
    set cur [tkTextClosestGap $w $x $y]
    select $w [backward_word $w "$cur +1c"] [forward_word $w "$cur -1c"]
}

proc tw::binding_capture {w bt} {
    variable $w
    if {[info exists ${w}(bindtags)]} {
	error "Already got binding capture!"
    } else {
	set ${w}(bindtags) [bindtags $w]
	bindtags $w "BindReset $bt BindNoMatch"
	bind BindReset <Key> [list tw::binding_reset $w]
    }
}

bind BindNoMatch <Key> [list message "No matching prefixed binding."]

proc tw::binding_reset {w} {
    variable $w
    message ""
    if {[info exists ${w}(bindtags)]} {
	bindtags $w [set ${w}(bindtags)]
	unset ${w}(bindtags)
    } else {
	error "No previous binding capture!"
    }
}

proc text_cmd {cmd args} {
    if {[set w [focus]] == "" || ![string match ".al*.text" $w]} {
	global win::tk win::Active
	set w $win::tk([lindex $win::Active 0])
    }
    uplevel 1 tw::$cmd $w $args
}

proc tw::setTabSize {w {v ""}} {
    variable $w
    if {$v == ""} {
	set v [set ${w}(tabsize)]
    } else {
	set ${w}(tabsize) $v
    }
    # If we have a patched text widget which can tab properly
    if {![catch {$w cget -fixedtabs}]} {
	$w configure -tabs ""
	$w configure -fixedtabs $v
	return
    }
    set charWidth [font measure [$w cget -font] " "]
    $w configure -tabs [expr {$v * $charWidth}]
    return
    # The following code is only more efficient for mono-spaced fonts,
    # and doesn't really work for proportional fonts.
    if {$v == 8} {
	global ::win::tk
	$w configure -tabs ""
    } else {
	set charWidth [expr {[font measure [$w cget -font] "Abc Def"]/7}]
	$w configure -tabs [expr {$v * $charWidth}]
    }
}

proc tw::setFontsTabs {w} {
    variable $w
    set fnt [$w cget -font]
    regexp {([^0-9]+)([0-9]+)} $fnt "" fnt size
    foreach {f s t} [chooseFontTab [string trim $fnt] $size [set ${w}(tabsize)]] {}
    $w configure -font "$f $s"
    setTabSize $w $t
}

proc text_cmds {args} {
    if {[set w [focus]] == "" || ![string match ".al*.text" $w]} {
	global win::tk win::Active
	set w $win::tk([lindex $win::Active 0])
    }
    foreach cmd $args {
	set rest [lrange $cmd 1 end]
	set cmd [lindex $cmd 0]
	uplevel 1 tw::$cmd $w $rest
    }
}
proc text_wcmd {ww cmd args} {
    global win::tk 
    set w $win::tk($ww)
    uplevel 1 tw::$cmd $w $args
}
proc text_wcmds {ww args} {
    global win::tk
    set w $win::tk($ww)
    foreach cmd $args {
	set rest [lrange $cmd 1 end]
	set cmd [lindex $cmd 0]
	uplevel 1 tw::$cmd $w $rest
    }
}


proc tw::yview {w args} {
    set ret [uplevel $w yview $args]
    # now recolour as appropriate
    if {$args != ""} {
	colour $w
    }
    return $ret
}

proc tw::colour {w} {
    global coloring
    if {!$coloring} {return}
    variable colouring
    if {![info exists colouring($w)]} {
	set colouring($w) 1
	#event generate $w <<Colourise>> -when tail
	after idle event generate $w <<Colourise>>
    }
}

proc win::colourise {w} {
    set el [lindex [split [$w index end] .] 0]
    foreach i [$w yview] {
	lappend id "[expr {int($i * $el)}].0"
    }
    foreach {st end} $id {}
    
    while {$st <= $end} {
	colour_line $w $st
	set st [expr {$st + 1.0}]
    }
    global ::tw::colouring
    if {[info exists ::tw::colouring($w)]} {
	unset ::tw::colouring($w)
    }
}


namespace eval alpha_winfo {}

proc tw::activateHook {n} {
    set range [$n tag ranges backsel]
    if {$range != ""} {
	eval $n tag remove backsel $range
	eval $n tag add sel $range
    }
    global ::win::tktitle
    ::activateHook $win::tktitle([base_window $n])
}

proc tw::deactivateHook {n} {
    variable $n
    if {[info exists ${n}(bindtags)]} {
	tw::binding_reset $n
    }
    set range [$n tag ranges sel]
    if {$range != ""} {
	eval $n tag remove sel $range
	eval $n tag add backsel $range
    }
    global ::win::tktitle
    ::deactivateHook $win::tktitle([base_window $n])
}

proc win::openHook {name} {
    variable Modes 
    variable tk
    set mode $Modes($name)
    global ${mode}::keywords
    set w $tk($name)
    bindtags $w [concat ${mode}AlphaStyle [bindtags $w]]
    #bindtags $w [concat ${mode}AlphaStyle ${mode}AlphaMenuStyle [bindtags $w]]
    
    $w tag configure kw -foreground #0000AA
    $w tag configure quote -foreground #00AA00
    $w tag configure comment -foreground #CC0000
    $w tag configure special -foreground purple

    ::tw::colour $w
    
}

proc win::colour_range {w from to} {
    while {[$w compare $from < $to]} {
	# Word 1 - after the current point
	set ws1 [$w index [tw::backward_word $w $from]]
	set we1 [$w index [tw::forward_word $w $from]]
	
	colour_word $w $ws1 $we1
	set from $we1
    }
}

proc win::colour_line {w {li ""}} {
    global mode
    global ${mode}::lineRegexp ${mode}::lineVars ${mode}::specialChars
    if {![info exists ${mode}::lineRegexp]} { return}
    if {$li == ""} { set li [$w index "insert linestart"] }
    set txt [$w get $li [set lend [$w index "${li} lineend"]]]
    while {[eval regexp [list [set ${mode}::lineRegexp] $txt ""] \
      [set ${mode}::lineVars]]} {
	#echo "-$space-$comment-$quote-$txt-"
	set offset [string length $space] ; set space ""
	set li [$w index "$li + ${offset}c"]
	if {[info exists comment] && ($comment != "")} {
	    $w tag add comment $li "$li lineend"
	    break
	}
	if {[info exists quote] && ($quote != "")} {
	    set le [$w index "$li +[string length $quote]c"]
	    $w tag add quote $li $le
	    set li $le
	    set quote ""
	} else {
	    if {$txt == ""} {break}
	    if {[string first [string index $txt 0] [set ${mode}::specialChars]] != -1} {
		$w tag add special $li "$li +1c"
		set li [$w index "$li +1c"]
		set txt [string range $txt 1 end]
	    } else {
		set we1 [$w index [tw::forward_word $w ${li}]]
		if {[tw::compare $w $we1 > $lend]} {
		    set offplus [colour_word $w $li $lend]
		} else {
		    set offplus [colour_word $w $li $we1]
		}
		set li $we1
		set txt [string range $txt $offplus end]
	    }
	}
    }
}
      
proc win::colour_word {w i1 i2} {
    global tags mode

    set word [$w get $i1 $i2]
    if {$word == {}} {return 0}
    $w tag remove green $i1 $i2
    global ${mode}::keywords ${mode}::magicPrefix
    if {[info exists ${mode}::keywords($word)]} {
	$w tag add [set ${mode}::keywords($word)] $i1 $i2
    } elseif {[info exists ${mode}::magicPrefix] && \
      ([regexp "^[set ${mode}::magicPrefix]" $word])} {
	$w tag add kw $i1 $i2
    }
    return [string length $word]
}

proc win::colour_2words {w} {
    global tags
    
    # Word 1 - after the current point
    set ws1 [$w index {insert wordstart}]
    set we1 [$w index {insert wordend}]

    colour_word $w $ws1 $we1
    
    # Word 2 - before the current point
    set ws2 [$w index {insert -1 char wordstart}]
    set we2 [$w index {insert -1 char wordend}]

    if {$ws2 != $ws1} {
	colour_word $w $ws2 $we2
    }
}


switch -- $tcl_platform(platform) {
    "unix" -
    "windows" {
	bind AlphaStyle <Control-Double-Button-1> {cmdDoubleClick}
    }
    "macintosh" {
	bind AlphaStyle <Cmd-Double-Button-1> {cmdDoubleClick}
    }
}
#bind AlphaStyle <Key> {win::colour_2words %W}
#bind AlphaStyle <Key> {win::colour_line %W}
bind AlphaStyle <<Colourise>> {win::colourise %W}
bind AlphaStyle <<Paste>> {paste ; break}
bind AlphaStyle <<Cut>> {cut ; break}
bind AlphaStyle <<Copy>> {copy ; break}
bind AlphaStyle <<Clear>> {clear ; break}
bind Toplevel <Destroy> [list win::kill %W.text]
bind Splitter <Enter> "%W configure -highlightbackground $alpha::colours(activebackground)"
bind Splitter <Leave> "%W configure -highlightbackground $alpha::colours(background)"
bind Splitter <Button-1> "splitter %W"
bind Splitter <ButtonRelease-1> "%W configure -relief raised"
bind AlphaStyle <FocusIn> {::tw::activateHook %W}
bind AlphaStyle <FocusOut> {::tw::deactivateHook %W}
bind AlphaStyle <Key-bracketright> [list flash "\]"]
bind AlphaStyle <Key-parenright> [list flash "\)"]
bind Text <Double-Button-1> "::tw::double_click %W %x %y"
