#!/usr/bin/env texlua
-- Package make4ht. Author Michal Hoftich <michal.h21@gmail.com>
-- This package is subject of LPPL license, version 1.3 
kpse.set_program_name("luatex")

-- logging should be globally available
logging = require "make4ht-logging"
if os.type == "windows" then logging.use_colors = false end
local log = logging.new("make4ht")
local make4ht = require("make4ht-lib")
local lapp    = require("lapp-mk4")
local mkutils = require("mkutils")
local mkparams = require("mkparams")
local mk_config = require("make4ht-config")
-- args string is here just as sample, we dont pass it it to 
-- mkparams.get_args() so default args string is used
local args    =  [[
make4ht - build system for TeX4ht
Usage:
make4ht [options] filename ["tex4ht.sty op." "tex4ht op." "t4ht op" "latex op"]
-c,--config (default xhtml) Custom config file
-d,--output-dir (default nil)  Output directory
-l,--lua  Use lualatex for document compilation
-s,--shell-escape Enables running external programs from LaTeX
-u,--utf8  For output documents in utf8 encoding
-x,--xetex Use xelatex for document compilation
<filename> (string) Input file name
]]

-- set version number. the template should be replaced by the
-- actual version number by the build script
local version = "v0.3g"
mkparams.version_number = version

local args = mkparams.get_args()

local parameters = mkparams.process_args(args) 

log:status("Conversion started")
log:status("Input file: " .. parameters.tex_file)

local mode = parameters.mode
local build_file = parameters.build_file 

-- handle output formats
local allowed_output_formats = {xhtml = true, html5=true, odt = true, docbook=true, tei=true, jats=true}
-- formatter is Lua library which must provide at least prepare_parameters
-- and process_build_sequence functions
local formatter
local output_format = parameters.output_format
if allowed_output_formats[ output_format ] then
  formatter = mkutils.load_output_format(output_format)
else
  -- load html5 as default output format
  if output_format then 
    log:warning("Cannot load output format: ".. output_format)
  end
  formatter = mkutils.load_output_format("html5")
end
-- find make4ht configuration file
local configname = "make4ht"
local conffile = mk_config.find_config(configname) or mk_config.find_xdg_config(configname)
if conffile then
  log:info("Using configuration file: " .. conffile)
  mkutils.load_config(parameters, conffile)
end
local extensions = formatter.prepare_extensions(parameters.extensions)
extensions = mkutils.load_extensions(extensions, output_format)



-- run extensions with prepare_parameters function
parameters = formatter.prepare_parameters(parameters,extensions)
local make = mkutils.load_config(parameters, build_file)["Make"]
make.params = parameters
if make:length() < 1 then
	if mode == "draft" then
		make:htlatex()
	else
		make:htlatex()
		make:htlatex()
		make:htlatex()
	end
end


if not args["no-tex4ht"] then
  make:tex4ht()
end

local ext = args.xetex and "xdv" or "dvi"
if #make.image_patterns > 0 then
  make.params.t4ht_par = make.params.t4ht_par .. " -p"
end
make:t4ht {ext = ext}
-- run extensions which modify the build sequence
if #extensions > 0 then
  make = mkutils.extensions_modify_build(extensions, make)
end

-- allow output formats to modify the build process at the end
make = formatter.modify_build(make) or make

make:match("tmp$", function(filename,params) 
  -- remove the temporary tex file created when the input comes from the standard input
  if params.is_tmp_file then
    log:info("removing temp file", params.tex_file)
    os.remove(params.tex_file)
  end
  -- prevent copying of the temporary file to the outdir
  return false,"tmp file" end
)

make:match(".*",function(filename,par)
	local outdir =  '' --par["outdir"] and par["outdir"] .."/" or ''
	if par['outdir'] ~= "" then outdir = par['outdir'] .. '/' end
	log:info("outdir: "..outdir)
	local outfilename = outdir .. filename
	mkutils.copy(filename,outfilename)
	return true
end)

make:run()

log:status("Conversion finished")
logging.exit_status()

