#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include "lite.h"

#define	LIB_MAG	0x1234
#define	LIB_VER	1

#define	INF_FUNC	1
#define INF_PARAM	2
#define	INF_CODE	3
#define	INF_LIT		4


typedef	struct	_lhdr {
	int	magic,
		version;
} lhdr_t;

typedef struct _inf {
	char	type;
	u_short	len;
} inf_t;


extern	funct_t	*functions;
extern	sym_t	*litHead;
extern	int	curLitIdx,
		curLabel;
extern	code_t  *codeHead, *codeTail;
extern	sym_t   *paramHead, *paramTail;

int	litBase,
	labelBase;

static 	char	buf[1024],
		tmpBuf[128];

char	*libNames[20];
int	curLib = 0;


funct_t	*curFunct;


#define HD_LEN	(1 + sizeof(cur->line))
static int setupCode(cur)
	code_t	*cur;
{
	char	*arg,
		*cp;
	sym_t	*sym;

	*buf = cur->op;
	cp = buf+1;
	bcopy(&cur->line,cp,sizeof(cur->line));
	cp += sizeof(cur->line);
	switch(cur->op)
	{
		case OP_PUSH:
		case OP_STORE:
		case OP_A_STORE:
		case OP_DEREF:
		case OP_ECHO:
		case OP_CALL:
			if (cur->arg)
			{
				strcpy(cp,cur->arg);
				*(cp + strlen(cur->arg) + 1)  = 0;
				return(strlen(cur->arg) + 1 + HD_LEN);
			}
			else
			{
				return(HD_LEN);
			}

		case OP_CAST:
		case OP_ALU:
		case OP_CMP:
			bcopy(&cur->type,cp,sizeof(cur->type));
			return(HD_LEN+sizeof(cur->type));
	
		case OP_LABEL:
		case OP_JMP:
		case OP_JMP_BACK:
		case OP_JMP_FALSE:
			bcopy(&cur->label,cp,sizeof(cur->label));
			return(HD_LEN+sizeof(cur->label));
	}
	return(HD_LEN);
}


extern char *opCodes[];

static code_t *decodeCode(buf)
	char	*buf;
{
	code_t	*new;
	char	*arg,
		*cp;
	sym_t	*sym;

	new = (code_t *)malloc(sizeof(code_t));
	new->file = curLib;
	new->op = *buf;
	bcopy(buf+1, &new->line, sizeof(new->line));
	cp = buf + 1 + sizeof(new->line);

#ifdef DEBUG
	printf("\tcode '%s'\t",opCodes[new->op]);
#endif
	switch(new->op)
	{
		case OP_PUSH:
		case OP_STORE:
		case OP_A_STORE:
		case OP_DEREF:
		case OP_ECHO:
			if (isdigit(*(cp)))
			{
				sprintf(tmpBuf,"%d",atoi(cp)+litBase);
				new->arg = (char *)strdup(tmpBuf);
			}
			else
				new->arg = (char *)strdup(cp);
#ifdef DEBUG
			printf("%s",new->arg);
#endif
			break;

		case OP_CALL:
			new->arg = (char *)strdup(cp);
#ifdef DEBUG
			printf("%s",new->arg);
#endif
			break;


		case OP_CAST:
		case OP_ALU:
		case OP_CMP:
			bcopy(cp, &new->type,sizeof(new->type));
			break;
	
		case OP_LABEL:
		case OP_JMP:
		case OP_JMP_BACK:
		case OP_JMP_FALSE:
			bcopy(cp,&new->label,sizeof(new->label));
			new->label += labelBase;
			curLabel++;
			break;
	}
#ifdef DEBUG
	printf("\n");
#endif
	return(new);
}


int libWriteLibrary(libfile)
	char	*libfile;
{
	lhdr_t	hdr;
	int	fd;
	inf_t	inf;
	funct_t	*curFunct;
	sym_t	*curSym;
	code_t	*curCode;


	/*
	** A library cannot contain a main function so scan the
	** function list and bail if it's there
	*/
	curFunct = functions;
	while(curFunct)
	{
		if (strcmp(curFunct->name,"main") == 0)
		{
			if (curFunct->code)
			{
				runError(
				   "Library files can't include 'main' code!");
				return(-1);
			}
			break;
		}
		curFunct = curFunct->next;
	}

	/*
	** Create the library file
	*/
	fd = open(libfile,O_RDWR|O_CREAT|O_TRUNC, 0744);
	if (fd < 0)
		return(-1);
	hdr.magic = LIB_MAG;
	hdr.version = LIB_VER;
	write(fd,&hdr,sizeof(hdr));



	/*
	** Write the literal values
	*/
	curSym = litHead;
	while(curSym)
	{
		inf.type = INF_LIT;
		inf.len = strlen(curSym->name) + curSym->length +
			sizeof(curSym->type) + 1;
		write(fd,&inf,sizeof(inf));
		write(fd,curSym->name, strlen(curSym->name)+1);
		write(fd,&curSym->type,sizeof(curSym->type));
		write(fd,curSym->val, curSym->length);
		curSym = curSym->next;
	}

	/*
	** Write the function info
	*/
	curFunct = functions;
	while(curFunct)
	{
		/*
		** Write the function header
		*/
		if (strcmp(curFunct->name,"main") == 0)
		{
			curFunct = curFunct->next;
			continue;
		}
		inf.type = INF_FUNC;
		inf.len = strlen(curFunct->name) + 1;
		write(fd,&inf,sizeof(inf));
		write(fd,curFunct->name,inf.len);

		/*
		** Write the params
		*/
		curSym = curFunct->params;
		while(curSym)
		{
			inf.type = INF_PARAM;
			inf.len = strlen(curSym->name)+1+sizeof(curSym->type);
			write(fd,&inf,sizeof(inf));
			write(fd,curSym->name,strlen(curSym->name)+1);
			write(fd,&curSym->type,sizeof(curSym->type));
			curSym = curSym->next;
		}


		/*
		** Write the code
		*/
		curCode = curFunct->code;
		while(curCode)
		{
			inf.type = INF_CODE;
			inf.len = setupCode(curCode);
			write(fd,&inf,sizeof(inf));
			write(fd,buf,inf.len);
			curCode = curCode->next;
		}

		curFunct = curFunct->next;
	}
	close(fd);
}


static int libSetupFunction(name)
	char	*name;
{
	funct_t	*cur;

	cur = (funct_t *)malloc(sizeof(funct_t));
	cur->name = (char *)strdup(name);
	cur->next = functions;
	functions = cur;
	curFunct = cur;
	codeHead = codeTail = NULL;
	paramHead = paramTail = NULL;
}




int libLoadLibrary(libfile)
	char	*libfile;
{
	lhdr_t	hdr;
	int	fd,
		numBytes;
	inf_t	inf;
	sym_t	*curSym,
		tmpSym;
	code_t	*curCode;
	char	*cp,
		*filename;

	if (*libfile == '"')
	{
		filename = libfile+1;
		libfile[strlen(libfile)-1]=0;
	}
	else
		filename = libfile;
#ifdef DEBUG
	printf("Loading library %s\n",filename);
#endif
	curLib++;
	libNames[curLib] = (char *)filename;

	fd = open(filename,O_RDONLY,0);
	if (fd < 0)
	{
		return(-1);
	}
	numBytes = read(fd,&hdr,sizeof(hdr));
	if (hdr.magic != LIB_MAG || hdr.version != LIB_VER)
	{
		parseError("Bad library format!");
		return(-1);
	}

	litBase = curLitIdx;
	labelBase = curLabel;
	numBytes = read(fd,&inf,sizeof(inf));
	while(numBytes)
	{
		switch(inf.type)
		{
			case INF_LIT:
				read(fd,buf,inf.len);
				sprintf(tmpBuf,"%d",atoi(buf)+litBase);
				curLitIdx++;
				curSym = (sym_t*)malloc(sizeof(sym_t));
				curSym->name = (char *)strdup(tmpBuf);
				curSym->array = SCALAR;
				cp = (char *)index(buf,0);
				bcopy(cp+1,&(curSym->type),
					sizeof(curSym->type));
				curSym->length = inf.len - 1 - 
					sizeof(curSym->type);
				curSym->val = (char *)malloc(curSym->length);
				bcopy(cp+1+sizeof(curSym->type),curSym->val,
					curSym->length);
#ifdef DEBUG
				printf("Literal value '%s' = '%s'\n",
					curSym->name, symUnpackSymbol(curSym));
#endif
				curSym->next = litHead;
				litHead = curSym;
				break;

			case INF_FUNC:
				read(fd,buf,inf.len);
#ifdef DEBUG
				printf("Lib function '%s'\n",buf);
#endif
				libSetupFunction(buf);
				break;

			case INF_PARAM:
				read(fd,buf,inf.len);
#ifdef DEBUG
				printf("\tparam '%s'\n",buf);
#endif
				curSym = (sym_t *)malloc(sizeof(sym_t));
				curSym->name = (char *)strdup(buf);
				cp = (char *)index(buf,0);
				bcopy(cp+1,&curSym->type,sizeof(curSym->type));
				curSym->length = 0;
				if (!paramHead)
				{
					curFunct->params = paramHead = 	
						paramTail = curSym;
					curSym->next = NULL;
				}
				else
				{
					paramTail->next = curSym;
					paramTail = curSym;
					paramTail->next = NULL;
				}
				break;

			case INF_CODE:
				read(fd,buf,inf.len);
				curCode = decodeCode(buf);
				if (!codeHead)
				{
					curFunct->code = codeHead = 	
						codeTail = curCode;
					curCode->next = NULL;
				}
				else
				{
					curCode->prev = codeTail;
					codeTail->next = curCode;
					codeTail = curCode;
					codeTail->next = NULL;
				}
				break;
		}
		numBytes = read(fd,&inf,sizeof(inf));
	}
#ifdef DEBUG
	printf("\n");
#endif
	close(fd);
}
