package Amazon::API::Signature4;

use strict;
use warnings;

use parent qw( Exporter AWS::Signature4 );

our $VERSION = '2.1.8';

our @EXPORT_OK = qw( parse_service_url );

our @SERVICE_URL_REGEXP = (
  qr/(s3)[.]amazonaws[.]com\z/xsm,                   qr/(s3)[.]([^.]+)[.]amazonaws[.]com\z/xsm,
  qr/(s3)[.][^.]+[.]([^.]+)[.]amazonaws[.]com\z/xsm, qr/(s3)[-][^.]+.+[.]([^.]+)[.]amazonaws[.]com\z/xsm,
  qr/^([[:alpha:]-]+)[.]amazonaws[.]com\z/xsm,       qr/^([[:alpha:]-]+)[.]([^.]*)[.]amazonaws[.]com/xsm,
);

########################################################################
sub new {
########################################################################
  my ( $class, @options ) = @_;
  $class = ref $class || $class;

  my %args = ref $options[0] ? $options[0] : @options;

  my $self = $class->SUPER::new(%args);

  my @extra = grep { !/access_key|secret_key|security_token/xsm } keys %args;

  @{$self}{@extra} = @args{@extra};

  return $self;
}

########################################################################
sub parse_service_url {
########################################################################
  my (%args) = @_;

  my ( $host,   $service )        = @args{qw{host service}};
  my ( $region, $default_region ) = @args{qw{region default_region}};

  if ( !$service || !$region ) {
    foreach my $regexp (@SERVICE_URL_REGEXP) {
      if ( $host =~ $regexp ) {
        $service = $1;
        $region  = $2 || $region || $default_region;
        last;
      }
    }
  }

  $region ||= $default_region;

  return ( $host, $service, $region );
}

###############################################################
sub scope {
###############################################################
  goto &_scope;
}

###############################################################
sub _scope {
###############################################################
  my ( $self, $request, $region ) = @_;

  my $host     = $request->uri->host;
  my $datetime = $self->_datetime($request);
  my ($date)   = $datetime =~ /^(\d+)T/xsm;

  my $service = $self->{service};

  ( $host, $service, $region ) = parse_service_url(
    host           => $host,
    service        => $service,
    region         => $region,
    default_region => 'us-east-1'
  );

  return "$date/$region/$service/aws4_request";
}

1;

__END__
