# -*- mode: perl; -*-

use strict;
use warnings;

use Test::More tests => 1548;

my $class;

BEGIN {
    $class = 'Math::BigInt';
    use_ok($class);
}

can_ok($class, 'bhyperop', 'hyperop');

while (<DATA>) {
    s/#.*$//;                   # remove comments
    s/\s+$//;                   # remove trailing whitespace
    next unless length;         # skip empty lines

    my ($a, $n, $b, $expected) = split /:/;

    # bhyperop() modifies the invocand.

    {
        my ($x, $y);
        my $test = qq|\$x = $class->new("$a"); \$y = \$x->bhyperop("$n", "$b");|;

        note("\n$test\n\n");

        subtest $test,
          sub {
              plan tests => 4;

              eval $test;
              is($@, "", "'$test' gives emtpy \$\@");

              is(ref($y), $class,
                 "'$test' output arg is a $class");

              is($y -> bstr(), $expected,
                 "'$test' output arg has the right value");

              is($x -> bstr(), $expected,
                 "'$test' invocand has the right value");
          };
    }

    # hyperop() does not modify the invocand.

    {
        my ($x, $y);
        my $test = qq|\$x = $class->new("$a"); \$y = \$x->hyperop("$n", "$b");|;

        note("\n$test\n\n");

        subtest $test,
          sub {
              plan tests => 4;

              eval $test;
              is($@, "", "'$test' gives emtpy \$\@");

              is(ref($y), $class,
                 "'$test' output arg is a $class");

              is($y -> bstr(), $expected,
                 "'$test' output arg has the right value");

              is($x -> bstr(), $a,
                 "'$test' invocand has the right value");
          };
    }
}

__DATA__

# n = 0 (succession)

0:0:0:1
0:0:1:2
0:0:2:3
0:0:3:4
0:0:4:5
0:0:5:6
0:0:6:7
0:0:7:8
0:0:8:9
0:0:9:10
1:0:0:1
1:0:1:2
1:0:2:3
1:0:3:4
1:0:4:5
1:0:5:6
1:0:6:7
1:0:7:8
1:0:8:9
1:0:9:10
2:0:0:1
2:0:1:2
2:0:2:3
2:0:3:4
2:0:4:5
2:0:5:6
2:0:6:7
2:0:7:8
2:0:8:9
2:0:9:10
3:0:0:1
3:0:1:2
3:0:2:3
3:0:3:4
3:0:4:5
3:0:5:6
3:0:6:7
3:0:7:8
3:0:8:9
3:0:9:10
4:0:0:1
4:0:1:2
4:0:2:3
4:0:3:4
4:0:4:5
4:0:5:6
4:0:6:7
4:0:7:8
4:0:8:9
4:0:9:10
5:0:0:1
5:0:1:2
5:0:2:3
5:0:3:4
5:0:4:5
5:0:5:6
5:0:6:7
5:0:7:8
5:0:8:9
5:0:9:10
6:0:0:1
6:0:1:2
6:0:2:3
6:0:3:4
6:0:4:5
6:0:5:6
6:0:6:7
6:0:7:8
6:0:8:9
6:0:9:10
7:0:0:1
7:0:1:2
7:0:2:3
7:0:3:4
7:0:4:5
7:0:5:6
7:0:6:7
7:0:7:8
7:0:8:9
7:0:9:10
8:0:0:1
8:0:1:2
8:0:2:3
8:0:3:4
8:0:4:5
8:0:5:6
8:0:6:7
8:0:7:8
8:0:8:9
8:0:9:10
9:0:0:1
9:0:1:2
9:0:2:3
9:0:3:4
9:0:4:5
9:0:5:6
9:0:6:7
9:0:7:8
9:0:8:9
9:0:9:10

# n = 1 (addition)

0:1:0:0
0:1:1:1
0:1:2:2
0:1:3:3
0:1:4:4
0:1:5:5
0:1:6:6
0:1:7:7
0:1:8:8
0:1:9:9
1:1:0:1
1:1:1:2
1:1:2:3
1:1:3:4
1:1:4:5
1:1:5:6
1:1:6:7
1:1:7:8
1:1:8:9
1:1:9:10
2:1:0:2
2:1:1:3
2:1:2:4
2:1:3:5
2:1:4:6
2:1:5:7
2:1:6:8
2:1:7:9
2:1:8:10
2:1:9:11
3:1:0:3
3:1:1:4
3:1:2:5
3:1:3:6
3:1:4:7
3:1:5:8
3:1:6:9
3:1:7:10
3:1:8:11
3:1:9:12
4:1:0:4
4:1:1:5
4:1:2:6
4:1:3:7
4:1:4:8
4:1:5:9
4:1:6:10
4:1:7:11
4:1:8:12
4:1:9:13
5:1:0:5
5:1:1:6
5:1:2:7
5:1:3:8
5:1:4:9
5:1:5:10
5:1:6:11
5:1:7:12
5:1:8:13
5:1:9:14
6:1:0:6
6:1:1:7
6:1:2:8
6:1:3:9
6:1:4:10
6:1:5:11
6:1:6:12
6:1:7:13
6:1:8:14
6:1:9:15
7:1:0:7
7:1:1:8
7:1:2:9
7:1:3:10
7:1:4:11
7:1:5:12
7:1:6:13
7:1:7:14
7:1:8:15
7:1:9:16
8:1:0:8
8:1:1:9
8:1:2:10
8:1:3:11
8:1:4:12
8:1:5:13
8:1:6:14
8:1:7:15
8:1:8:16
8:1:9:17
9:1:0:9
9:1:1:10
9:1:2:11
9:1:3:12
9:1:4:13
9:1:5:14
9:1:6:15
9:1:7:16
9:1:8:17
9:1:9:18

# n = 2 (multiplication)

0:2:0:0
0:2:1:0
0:2:2:0
0:2:3:0
0:2:4:0
0:2:5:0
0:2:6:0
0:2:7:0
0:2:8:0
0:2:9:0
1:2:0:0
1:2:1:1
1:2:2:2
1:2:3:3
1:2:4:4
1:2:5:5
1:2:6:6
1:2:7:7
1:2:8:8
1:2:9:9
2:2:0:0
2:2:1:2
2:2:2:4
2:2:3:6
2:2:4:8
2:2:5:10
2:2:6:12
2:2:7:14
2:2:8:16
2:2:9:18
3:2:0:0
3:2:1:3
3:2:2:6
3:2:3:9
3:2:4:12
3:2:5:15
3:2:6:18
3:2:7:21
3:2:8:24
3:2:9:27
4:2:0:0
4:2:1:4
4:2:2:8
4:2:3:12
4:2:4:16
4:2:5:20
4:2:6:24
4:2:7:28
4:2:8:32
4:2:9:36
5:2:0:0
5:2:1:5
5:2:2:10
5:2:3:15
5:2:4:20
5:2:5:25
5:2:6:30
5:2:7:35
5:2:8:40
5:2:9:45
6:2:0:0
6:2:1:6
6:2:2:12
6:2:3:18
6:2:4:24
6:2:5:30
6:2:6:36
6:2:7:42
6:2:8:48
6:2:9:54
7:2:0:0
7:2:1:7
7:2:2:14
7:2:3:21
7:2:4:28
7:2:5:35
7:2:6:42
7:2:7:49
7:2:8:56
7:2:9:63
8:2:0:0
8:2:1:8
8:2:2:16
8:2:3:24
8:2:4:32
8:2:5:40
8:2:6:48
8:2:7:56
8:2:8:64
8:2:9:72
9:2:0:0
9:2:1:9
9:2:2:18
9:2:3:27
9:2:4:36
9:2:5:45
9:2:6:54
9:2:7:63
9:2:8:72
9:2:9:81

# n = 3 (exponentiation)

0:3:0:1
0:3:1:0
0:3:2:0
0:3:3:0
0:3:4:0
0:3:5:0
0:3:6:0
0:3:7:0
0:3:8:0
0:3:9:0
1:3:0:1
1:3:1:1
1:3:2:1
1:3:3:1
1:3:4:1
1:3:5:1
1:3:6:1
1:3:7:1
1:3:8:1
1:3:9:1
2:3:0:1
2:3:1:2
2:3:2:4
2:3:3:8
2:3:4:16
2:3:5:32
2:3:6:64
2:3:7:128
2:3:8:256
2:3:9:512
3:3:0:1
3:3:1:3
3:3:2:9
3:3:3:27
3:3:4:81
3:3:5:243
3:3:6:729
3:3:7:2187
3:3:8:6561
3:3:9:19683
4:3:0:1
4:3:1:4
4:3:2:16
4:3:3:64
4:3:4:256
4:3:5:1024
4:3:6:4096
4:3:7:16384
4:3:8:65536
4:3:9:262144
5:3:0:1
5:3:1:5
5:3:2:25
5:3:3:125
5:3:4:625
5:3:5:3125
5:3:6:15625
5:3:7:78125
5:3:8:390625
5:3:9:1953125
6:3:0:1
6:3:1:6
6:3:2:36
6:3:3:216
6:3:4:1296
6:3:5:7776
6:3:6:46656
6:3:7:279936
6:3:8:1679616
6:3:9:10077696
7:3:0:1
7:3:1:7
7:3:2:49
7:3:3:343
7:3:4:2401
7:3:5:16807
7:3:6:117649
7:3:7:823543
7:3:8:5764801
7:3:9:40353607
8:3:0:1
8:3:1:8
8:3:2:64
8:3:3:512
8:3:4:4096
8:3:5:32768
8:3:6:262144
8:3:7:2097152
8:3:8:16777216
8:3:9:134217728
9:3:0:1
9:3:1:9
9:3:2:81
9:3:3:729
9:3:4:6561
9:3:5:59049
9:3:6:531441
9:3:7:4782969
9:3:8:43046721
9:3:9:387420489

# n = 4 (tetration)

0:4:0:1
0:4:1:0
0:4:2:1
0:4:3:0
0:4:4:1
0:4:5:0
0:4:6:1
0:4:7:0
0:4:8:1
0:4:9:0

1:4:0:1
1:4:1:1
1:4:2:1
1:4:3:1
1:4:4:1
1:4:5:1
1:4:6:1
1:4:7:1
1:4:8:1
1:4:9:1

2:4:0:1
2:4:1:2
2:4:2:4
2:4:3:16
2:4:4:65536
2:4:5:2003529930406846464979072351560255750447825475569751419265016973710894059556311453089506130880933348101038234342907263181822949382118812668869506364761547029165041871916351587966347219442930927982084309104855990570159318959639524863372367203002916969592156108764948889254090805911457037675208500206671563702366126359747144807111774815880914135742720967190151836282560618091458852699826141425030123391108273603843767876449043205960379124490905707560314035076162562476031863793126484703743782954975613770981604614413308692118102485959152380195331030292162800160568670105651646750568038741529463842244845292537361442533614373729088303794601274724958414864915930647252015155693922628180691650796381064132275307267143998158508811292628901134237782705567421080070065283963322155077831214288551675554073345107213112427399562982719769150054883905223804357045848197956393157853510018992000024141963706813559840464039472194016069517690156119726982337890017641517190051133466306898140219383481435426387306539552969691388024158161859561100640362119796101859534802787167200122604642492385111393400464351623867567078745259464670903886547743483217897012764455529409092021959585751622973333576159552394885297579954028471943529913543763705986928913757153740001986394332464890052543106629669165243419174691389632476560289415199775477703138064781342309596190960654591300890188887588084733625956065444888501447335706058817090162108499714529568344061979690565469813631162053579369791403236328496233046421066136200220175787851857409162050489711781820400187282939943446186224328009837323764931814789848119452713007440220765680910376203999203492023906626264491909167985461515778839060397720759279378852241294301017458086862263369284725851403039615558564330385450688652213114813638408384778263790459607186876728509763471271988890680478243230394718650525660978150729861141430305816927924971409161059417185352275887504477592218301158780701975535722241400019548102005661773589781499532325208589753463547007786690406429016763808161740550405117670093673202804549339027992491867306539931640720492238474815280619166900933805732120816350707634351669869625020969023162859350071874190579161241536897514808261904847946571736601005892476655445840838334790544144817684255327207315586349347605137419779525190365032198020108764738368682531025183377533908861426184800374008082238104076468878471647552945326947661700424461063311238021134588694532200116564076327023074292426051582811070387018345324567635625951430032037432740780879056283663406965030844225855967039271869461158513793386475699748568670079823960604393478850861649260304945061743412365828352144806726676841807083754862211408236579802961200027441324438432402331257403545019352428776430880232850855886089962774458164680857875115807014743763867976955049991643998284357290415378143438847303484261903388841494031366139854257635577105335580206622185577060082551288893332226436281984838613239570676191409638533832374343758830859233722284644287996245605476932428998432652677378373173288063210753211238680604674708428051166488709084770291208161104912555598322366244868556651402684641209694982590565519216188104341226838996283071654868525536914850299539675503954938371853405900096187489473992880432496373165753803673586710175783994818471798498246948060532081996066183434012476096639519778021441199752546704080608499344178256285092726523709898651539462193004607364507926212975917698293892367015170992091531567814439791248475706237804600009918293321306880570046591458387208088016887445835557926258465124763087148566313528934166117490617526671492672176128330845273936469244582892571388877839056300482483799839692029222215486145902373478222682521639957440801727144146179559226175083889020074169926238300282286249284182671243405751424188569994272331606998712986882771820617214453142574944015066139463169197629181506579745526236191224848063890033669074365989226349564114665503062965960199720636202603521917776740668777463549375318899587866282125469797102065747232721372918144666659421872003474508942830911535189271114287108376159222380276605327823351661555149369375778466670145717971901227117812780450240026384758788339396817962950690798817121690686929538248529830023476068454114178139110648560236549754227497231007615131870024053910510913817843721791422528587432098524957878034683703337818421444017138688124249984418618129271198533315382567321870421530631197748535214670955334626336610864667332292409879849256691109516143618601548909740241913509623043612196128165950518666022030715613684732364660868905014263913906515063908199378852318365059897299125404479443425166774299659811849233151555272883274028352688442408752811283289980625912673699546247341543333500147231430612750390307397135252069338173843322950701049061867539433130784798015655130384758155685236218010419650255596181934986315913233036096461905990236112681196023441843363334594927631946101716652913823717182394299216272538461776065694542297877071383198817036964588689811863210976900355735884624464835706291453052757101278872027965364479724025405448132748391794128826423835171949197209797145936887537198729130831738033911016128547415377377715951728084111627597186384924222802373441925469991983672192131287035585307966942713416391033882754318613643490100943197409047331014476299861725424423355612237435715825933382804986243892498222780715951762757847109475119033482241412025182688713728193104253478196128440176479531505057110722974314569915223451643121848657575786528197564843508958384722923534559464521215831657751471298708225909292655638836651120681943836904116252668710044560243704200663709001941185557160472044643696932850060046928140507119069261393993902735534545567470314903886022024639948260501762431969305640666366626090207048887438898907498152865444381862917382901051820869936382661868303915273264581286782806601337500096593364625146091723180312930347877421234679118454791311109897794648216922505629399956793483801699157439700537542134485874586856047286751065423341893839099110586465595113646061055156838541217459801807133163612573079611168343863767667307354583494789788316330129240800836356825939157113130978030516441716682518346573675934198084958947940983292500086389778563494693212473426103062713745077286156922596628573857905533240641849018451328284632709269753830867308409142247659474439973348130810986399417379789657010687026734161967196591599588537834822988270125605842365589539690306474965584147981310997157542043256395776070485100881578291408250777738559790129129407309462785944505859412273194812753225152324801503466519048228961406646890305102510916237770448486230229488966711380555607956620732449373374027836767300203011615227008921843515652121379215748206859356920790214502277133099987729459596952817044582181956080965811702798062669891205061560742325686842271306295009864421853470810407128917646906550836129916694778023822502789667843489199409657361704586786242554006942516693979292624714524945408858422726153755260071904336329196375777502176005195800693847635789586878489536872122898557806826518192703632099480155874455575175312736471421295536494084385586615208012115079075068553344489258693283859653013272046970694571546959353658571788894862333292465202735853188533370948455403336565356988172582528918056635488363743793348411845580168331827676834646291995605513470039147876808640322629616641560667508153710646723108461964247537490553744805318226002710216400980584497526023035640038083472053149941172965736785066421400842696497103241919182121213206939769143923368374709228267738708132236680086924703491586840991153098315412063566123187504305467536983230827966457417620806593177265685841681837966106144963432544111706941700222657817358351259821080769101961052229263879745049019254311900620561906577452416191913187533984049343976823310298465893318373015809592522829206820862230332585280119266496314441316442773003237792274712330696417149945532261035475145631290668854345426869788447742981777493710117614651624183616680254815296335308490849943006763654806102940094693750609845588558043970485914449584445079978497045583550685408745163316464118083123079704389849190506587586425810738422420591191941674182490452700288263983057950057341711487031187142834184499153456702915280104485145176055306971441761368582384102787659324662689978418319620312262421177391477208004883578333569204533935953254564897028558589735505751235129536540502842081022785248776603574246366673148680279486052445782673626230852978265057114624846595914210278122788941448163994973881884622768244851622051817076722169863265701654316919742651230041757329904473537672536845792754365412826553581858046840069367718605020070547247548400805530424951854495267247261347318174742180078574693465447136036975884118029408039616746946288540679172138601225419503819704538417268006398820656328792839582708510919958839448297775647152026132871089526163417707151642899487953564854553553148754978134009964854498635824847690590033116961303766127923464323129706628411307427046202032013368350385425360313636763575212604707425311209233402837482949453104727418969287275572027615272268283376741393425652653283068469997597097750005560889932685025049212884068274139881631540456490350775871680074055685724021758685439053228133770707415830756269628316955687424060527726485853050611356384851965918968649596335568216975437621430778665934730450164822432964891270709898076676625671517269062058815549666382573829274182082278960684488222983394816670984039024283514306813767253460126007269262969468672750794346190439996618979611928750519442356402644303271737341591281496056168353988188569484045342311424613559925272330064881627466723523751234311893442118885085079358163848994487544756331689213869675574302737953785262542329024881047181939037220666894702204258836895840939998453560948869946833852579675161882159410981624918741813364726965123980677561947912557957446471427868624053750576104204267149366084980238274680575982591331006919941904651906531171908926077949119217946407355129633864523035673345588033313197080365457184791550432654899559705862888286866606618021882248602144999973122164138170653480175510438406624412822803616648904257377640956326482825258407669045608439490325290526337532316509087681336614242398309530806549661879381949120033919489494065132398816642080088395554942237096734840072642705701165089075196155370186264797456381187856175457113400473810762763014953309735174180655479112660938034311378532532883533352024934365979129341284854970946826329075830193072665337782559314331110963848053940859283988907796210479847919686876539987477095912788727475874439806779824968278272200926449944559380414608770641941810440758269805688038949654616587983904660587645341810289907194293021774519976104495043196841503455514044820928933378657363052830619990077748726922998608279053171691876578860908941817057993404890218441559791092676862796597583952483926734883634745651687016166240642424241228961118010615682342539392180052483454723779219911228595914191877491793823340010078128326506710281781396029120914720100947878752551263372884222353869490067927664511634758101193875319657242121476038284774774571704578610417385747911301908583877890152334343013005282797038580359815182929600305682612091950943737325454171056383887047528950563961029843641360935641632589408137981511693338619797339821670761004607980096016024823096943043806956620123213650140549586250615282588033022908385812478469315720323233601899469437647726721879376826431828382603564520699468630216048874528424363593558622333506235945002890558581611275341783750455936126130852640828051213873177490200249552738734585956405160830583053770732533971552620444705429573538361113677523169972740292941674204423248113875075631319078272188864053374694213842169928862940479635305150560788126366206497231257579019598873041195626227343728900516561111094111745277965482790471250581999077498063821559376885546498822938985408291325129076478386322494781016753491693489288104203015610283386143827378160946341335383578340765314321417150655877547820252454780657301342277470616744241968952613164274104695474621483756288299771804186785084546965619150908695874251184435837306590951460980451247409411373899927822492983367796011015387096129749705566301637307202750734759922943792393824427421186158236161317886392553095117188421298508307238259729144142251579403883011359083331651858234967221259621812507058113759495525022747274674369887131926670769299199084467161228738858457584622726573330753735572823951616964175198675012681745429323738294143824814377139861906716657572945807804820559511881687188075212971832636442155336787751274766940790117057509819575084563565217389544179875074523854455200133572033332379895074393905312918212255259833790909463630202185353848854825062897715616963860712382771725621313460549401770413581731931763370136332252819127547191443450920711848838366818174263342949611870091503049165339464763717766439120798347494627397822171502090670190302469762151278521956142070806461631373236517853976292092025500288962012970141379640038055734949269073535145961208674796547733692958773628635660143767964038430796864138563447801328261284589184898528048048844180821639423974014362903481665458114454366460032490618763039502356402044530748210241366895196644221339200757479128683805175150634662569391937740283512075666260829890491877287833852178522792045771846965855278790447562192663992008409302075673925363735628390829817577902153202106409617373283598494066652141198183810884515459772895164572131897797907491941013148368544639616904607030107596818933741217575988165127000761262789169510406315857637534787420070222051070891257612361658026806815858499852631465878086616800733264676830206391697203064894405628195406190685242003053463156621891327309069687353181641094514288036605995220248248886711554429104721929134248346438705368508648749099178812670565665387191049721820042371492740164460943459845392536706132210616533085662021188968234005752675486101476993688738209584552211571923479686888160853631615862880150395949418529489227074410828207169303387818084936204018255222271010985653444817207470756019245915599431072949578197878590578940052540122867517142511184356437184053563024181225473266093302710397968091064939272722683035410467632591355279683837705019855234621222858410557119921731717969804339317707750755627056047831779844447637560254637033369247114220815519973691371975163241302748712199863404548248524570118553342675264715978310731245663429805221455494156252724028915333354349341217862037007260315279870771872491234494477147909520734761385425485311552773301030342476835865496093722324007154518129732692081058424090557725645803681462234493189708138897143299831347617799679712453782310703739151473878692119187566700319321281896803322696594459286210607438827416919465162267632540665070881071030394178860564893769816734159025925194611823642945652669372203155504700213598846292758012527715422016629954863130324912311029627923723899766416803497141226527931907636326136814145516376656559839788489381733082668779901962886932296597379951931621187215455287394170243669885593888793316744533363119541518404088283815193421234122820030950313341050704760159987985472529190665222479319715440331794836837373220821885773341623856441380700541913530245943913502554531886454796252260251762928374330465102361057583514550739443339610216229675461415781127197001738611494279501411253280621254775810512972088465263158094806633687670147310733540717710876615935856814098212967730759197382973441445256688770855324570888958320993823432102718224114763732791357568615421252849657903335093152776925505845644010552192644505312073756287744998163646332835816140330175813967359427327690448920361880386754955751806890058532927201493923500525845146706982628548257883267398735220457228239290207144822219885587102896991935873074277815159757620764023951243860202032596596250212578349957710085626386118233813318509014686577064010676278617583772772895892746039403930337271873850536912957126715066896688493880885142943609962012966759079225082275313812849851526902931700263136328942095797577959327635531162066753488651317323872438748063513314512644889967589828812925480076425186586490241111127301357197181381602583178506932244007998656635371544088454866393181708395735780799059730839094881804060935959190907473960904410150516321749681412100765719177483767355751000733616922386537429079457803200042337452807566153042929014495780629634138383551783599764708851349004856973697965238695845994595592090709058956891451141412684505462117945026611750166928260250950770778211950432617383223562437601776799362796099368975191394965033358507155418436456852616674243688920371037495328425927131610537834980740739158633817967658425258036737206469351248652238481341663808061505704829059890696451936440018597120425723007316410009916987524260377362177763430621616744884930810929901009517974541564251204822086714586849255132444266777127863728211331536224301091824391243380214046242223349153559516890816288487989988273630445372432174280215755777967021666317047969728172483392841015642274507271779269399929740308072770395013581545142494049026536105825409373114653104943382484379718606937214444600826798002471229489405761853892203425608302697052876621377373594394224114707074072902725461307358541745691419446487624357682397065703184168467540733466346293673983620004041400714054277632480132742202685393698869787607009590048684650626771363070979821006557285101306601010780633743344773073478653881742681230743766066643312775356466578603715192922768440458273283243808212841218776132042460464900801054731426749260826922155637405486241717031027919996942645620955619816454547662045022411449404749349832206807191352767986747813458203859570413466177937228534940031631599544093684089572533438702986717829770373332806801764639502090023941931499115009105276821119510999063166150311585582835582607179410052528583611369961303442790173811787412061288182062023263849861515656451230047792967563618345768105043341769543067538041113928553792529241347339481050532025708728186307291158911335942014761872664291564036371927602306283840650425441742335464549987055318726887926424102147363698625463747159744354943443899730051742525110877357886390946812096673428152585919924857640488055071329814299359911463239919113959926752576359007446572810191805841807342227734721397723218231771716916400108826112549093361186780575722391018186168549108500885272274374212086524852372456248697662245384819298671129452945515497030585919307198497105414181636968976131126744027009648667545934567059936995464500558921628047976365686133316563907395703272034389175415267500915011198856872708848195531676931681272892143031376818016445477367518353497857924276463354162433601125960252109501612264110346083465648235597934274056868849224458745493776752120324703803035491157544831295275891939893680876327685438769557694881422844311998595700727521393176837831770339130423060958999137314684569010422095161967070506420256733873446115655276175992727151877660010238944760539789516945708802728736225121076224091810066700883474737605156285533943565843756271241244457651663064085939507947550920463932245202535463634444791755661725962187199279186575490857852950012840229035061514937310107009446151011613712423761426722541732055959202782129325725947146417224977321316381845326555279604270541871496236585252458648933254145062642337885651464670604298564781968461593663288954299780722542264790400616019751975007460545150060291806638271497016110987951336633771378434416194053121445291855180136575558667615019373029691932076120009255065081583275508499340768797252369987023567931026804136745718956641431852679054717169962990363015545645090044802789055701968328313630718997699153166679208958768572290600915472919636381673596673959975710326015571920237348580521128117458610065152598883843114511894880552129145775699146577530041384717124577965048175856395072895337539755822087777506072339445587895905719156736

3:4:0:1
3:4:1:3
3:4:2:27
3:4:3:7625597484987
# 3:4:4:... (a 3638334640025 digit number)

4:4:0:1
4:4:1:4
4:4:2:256
4:4:3:13407807929942597099574024998205846127479365820592393377723561443721764030073546976801874298166903427690031858186486050853753882811946569946433649006084096
# 4:4:4:... (a 8072304726028210248596771043321139499346627442954296810188546463927511841459889094223339600264516627999927712174459405837378703581264109741141923867978290 digit number)

# n = 5 (pentation)

5:4:0:1
5:4:1:5
5:4:2:3125
5:4:3:1911012597945477520356404559703964599198081048990094337139512789246520530242615803012059386519739850265586440155794462235359212788673806972288410146915986602087961896757195701839281660338047611225975533626101001482651123413147768252411493094447176965282756285196737514395357542479093219206641883011787169122552421070050709064674382870851449950256586194461543183511379849133691779928127433840431549236855526783596374102105331546031353725325748636909159778690328266459182983815230286936572873691422648131291743762136325730321645282979486862576245362218017673224940567642819360078720713837072355305446356153946401185348493792719514594505508232749221605848912910945189959948686199543147666938013037176163592594479746164220050885079469804487133205133160739134230540198872570038329801246050197013467397175909027389493923817315786996845899794781068042822436093783946335265422815704302832442385515082316490967285712171708123232790481817268327510112746782317410985888683708522000711733492253913322300756147180429007527677793352306200618286012455254243061006894805446584704820650982664319360960388736258510747074340636286976576702699258649953557976318173902550891331223294743930343956161328334072831663498258145226862004307799084688103804187368324800903873596212919633602583120781673673742533322879296907205490595621406888825991244581842379597863476484315673760923625090371511798941424262270220066286486867868710182980872802560693101949280830825044198424796792058908817112327192301455582916746795197430548026404646854002733993860798594465961501752586965811447568510041568687730903712482535343839285397598749458497050038225012489284001826590056251286187629938044407340142347062055785305325034918189589707199305662188512963187501743535960282201038211616048545121039313312256332260766436236688296850208839496142830484739113991669622649948563685234712873294796680884509405893951104650944137909502276545653133018670633521323028460519434381399810561400652595300731790772711065783494174642684720956134647327748584238274899668755052504394218232191357223054066715373374248543645663782045701654593218154053548393614250664498585403307466468541890148134347714650315037954175778622811776585876941680908203125

6:4:0:1
6:4:1:6
6:4:2:46656
6:4:3:265911977215322677968248940438791859490534220026992430066043278949707355987388290912134229290617558303244068282650672342560163577559027938964261261109302039893034777446061389442537960087466214788422902213385381919290542791575075927495293510931902036227198983057853932880763319683450709063994613113899946027767197828941253221232925203296051182048791364008389549044365792095626712629192228922460944103484957826646121969087967503992005139138817452525944319386504163034800032329572923169272580812038862895645133020319941418621365459381247897039074948526861497196424842856278264872081854849393729902569313271916554406043106913602901064055895953421212374607966076698330673506053248292555308212118869609719907808556324604601321610265543147645240039660236473266424625276737459325658213465426520922004237014324045664647970387863402847679951630130232058962063799462389599996237653062587414899860913010693318793720947688749687131107213871394369489435728806106155770846190228969206308973600818603623114117110987742542396241094684938897236922298685698434294496424701408029483172932386618629065996040198226842823646538422586113704883857989007702229553937496834838684871215875596667442178077474619334121028488932766083813638995310139077457475830177551325338144732131509087834262021410745896573377350678674850098385414388602041091113739704476471248961781346576074509649789560993834854196961759483379704895730837108476073243820752302407048567636674324794945261956856304309751928127990186556725142283682860915312621646073453158008803215388643682616411219867012372209679528053501516815952615060246385651031976562807101388014614012727625982704958770707410574826165853317280489979664766579951201612185779455431473280794446393767971020284196670959037332691306485094879711657607592738752654869501755331152983075423102199223873789151170394698211199449920074131308554214681213100553681303428933484193987068952218348302678831827921107401559352065829092083220235853420468576169753257004117360848240587083162510683198727366804395473509269321561136071328831038756345050322110155370605205773322174682858545420089112607776943756235718685791083329489150388849193534368560214077259275017330081801022785506555813374097953179402013552081883260678201008163982491139468985286586839525924516399022060889167034026307259495537031036865710606177193275628159515876151135875206383384565803345010001967990900297897721047625041503686189053938223646710134213940404719809893782173642263134508441550849287889447186844367723211381684596533430978485492183483286280866059762937533235790152346913482052907574065476985428074987463200630523320192960729488033220100471728132419843750933418994651827766195280398535457879210590938725541731207476215251178380018544287270261438480751698862350923128502235087740792752699447916376634508067462510897466136541982989588498885556218913512465902304156653376336871566688387251967036152900780064068602888069931935207999310689140776424351948418730004965515832221710968997431459333784941295229440309655511855867028445035198762119848288829295380202219936661078120427828843001276230841951275304807279299943098104127984653113848966391547080166051152516695839546951857411342509029205658037023718240037774832603354543093511500118479701534297400745136744846115854618011592988204941297108616076522379205004659217817374565514468243397003282844487472709177905919919935899275425573964493827579365446000027476477191798414513254581650275367333019821278843172709465750692752171808594901217509501207417421617586944497161282705033574579740947969818330006129341292987484417693746915156267819694419069070551740834127011011653057480954762302376223721506011117278656836781687467642565660245054195922383015063540983302976564585032232212348328492684438517686135813543777864727386998363224693825363151801851106716453198841449354389812191334113521409843330236524538792076198600969222577026478681492227501540566298351707639566416336862868467950597297476836923209472900922293702286186940890332878669151652070026861441427046793265655884082960714097320267268049408177667445088496534818017853672361822688485733913385400255456567934282631484871977114047875708058685436192476663298724761448156342842830402602505021574204296564158930366078365481437442997463536675091000411699963360673064732160583745460680016965456758566096290515270920164587535961300182886653830784412200942190677050444291047244003389456102541814425739429044704330880008696148653836354976484820864386333590772821957611006776947872616515008395331769033371947624815253698065061836502368055443145909315203440556148456886065074331669131896824759958256508800018017148943114323386161400405917446760035859282703814428874771624962132036197765252454933467861555487197397619695431510692383342974202667545414955466938030160673169706724462625855137085164539132257730434810596984801783923003483105541564030100015743183630882716196703166554265650963071038792205396803159204779044332369316517777859190655421620788215534975404890121605354486244302325493234970592691383707828560134937722533674320536537516987734918377779951255422227597271611802930324522204021758057752412497127455218252920942278428479471698539697813452696767907179228508491839775153628228243052306992659549440199892772061036616473355839050070498310473487182064267545001577079206261540203265717301064221019125068350266791249399179609291090367076356849258440718452955522072597481584063311817403113771618731801958998128677729597378367094064062033675539166479397859978760723803175210934594321121324518477664328229171706732939059061137733423725445425333376420568500011937009911220804854369888817399953540372960084640042151028030142531122831961619571524522422266809290751616197961883530242609434051902857830543763397061153243582494210503092681521241839983612388292362168558481080453724926008984522786772923739257464771832909220710235406334318389309363173179239077529730921381408865699869199075222777463146266117543342519940204770331943339078678783474761260772613750616583256430398445294250039146099139399860652378000419745042078241181875123263807183683939638150929032237564582040219925725544066194365465456072775046043019953993748322983683489516208150738424224196778813811196748091317343045579356718550325307737095111433942009322636455898150323810716497918542945385958749141645826113403821601639049146905546228789873935871271101782517043060004493333790352302613291297240328564529779500158901992057657168980489272863072158461749965174823389901373567049766197477456695811834009402652112653589115553592994292565814973047995369272297261017111119344511199618772851626644393751712810887048244077398755032272241691310649437516764253759232572706022781934855221122174485770810461351760722786628516143158455550642611470543692693710220729528959120340445210529788692903692277110929238083192013051552699177390638979085275991136415947758803248065658708433324533317375936823619422963019751271123958949266745154287327508514137369673627508864714146824134776235394625172665199539967765285667592328910514509788672574445865951869081323812569736008479758269334749025290058433197876943284080580745193611989429990468280533178199550083230163227944716082271056280731820677005355961868362024509891022139820125623841787690585309928612267303595625824508436018544241421864042750273384716167606981280931250329649599434752052971462973363672222035938392416581549006207453756828721700346944745735294757240987973604996890528648783086812325888847405838796103442374669651862064044434887259551215107382960629850522578340713349255321774853789028601004245977335322044392190426364996651123770623213960923374849645936962093842457558760552195368235329700552845151069077200413182755865827314444969724970410721026079430403668206368143382579398573856744555443907200016627368907470972273592217094439747656651587759921449935310041011278838427588582684058312705649787648119492683806365037319800437060302992518823221372565574622697024120942134211743868144784037217825939334570966080175296107167014761063573878764298769872073630597860069568435695489708455471417296025446297551851567364487849328103367222701989883802494786171436302954969918207348486513230712677788751869942315794571999554857157496720173838789205258104548081079810451185601751030835725087918474499339432087503990193076470750547948775977855358485409413012204167161008344913898947567902878642168826811118376878528543846712653149082404118060490652755864988462101460370601811780023726130518568384806414483578827026111605055591597332741401022568196421828157231909693624098089055609103548620026370881684170231338454875054885411126510070161725747437966591249357866528886086706489298876392642744160118970606956928665435440332410837187184771741684096117675991110268740224339044107121286235825797872641933884252895920171694272374852330986381424704405554720432213808590250910052515845368806455661289841979888898496284938747083462857147394174768624657638140589204233692069874340266435896620604197215859497775734519355596485093355986281552555553876637144264112107050854226245879960823641485704807633869148606767226023183261796956540334262005758333261045868501266693308281023681427721918822056924812008821204534830044654602066446687314813657144604524271412597676737460939657352003890495077862128279292449414208247189079648329501330142509407727347591066686369332364802156261024141032358374502783755634343861798500282951879660646660629481466668133405040220522200280171108546969716343594555462039079692851957417002424888105893065048277613075277549575313245632715147314669715412024040329823708192959224423808246692757409437349315576119847840425210718798954425192388505562440936682434151237375513226295724498075849529979937732877694105147208871848111189812916480877473371785496167602764408871411247754224503353946883319174406712144432837791420728008577200624913032381852272196096837422198435262420299836299729271608009378269710389890410171747674715111904373088567457435014594879091368705600688007101215820026528663115527646246147728140985412548401456814934852708715538761594295805551267452651180429333684268516031849931062195799522678787157577600563347342341331836322988591894412992525937987173318134193990210689518078163555907235238853815379773691546246995995608375329977513075931839086826932047597565683018282299362258470622934630351415117782501732398783298437154204578260937966327569667172864420534855709163219251858795348798337006617691315353399699461860343190483637203756593186066419144028800227652609457125666376633516790109663620347636975698740142905735080137176798745760015315507027808438442428818313304866101754151500082701256096763943176106018568210388507516985167245380074711202913270160244940132970964370790189988317746584257548400289775525880140679745537412020982395367711258261144075551696849205873413439885660637900372629399072607756167012840505370788404636652118089611175070254411191967276530749834800340315078515624047044443843800548133076292757931322031245477272208271569139366156660369445314106758948526343231972951616250248160130237019508749245330726407137979944932556407418997416340099124064330450253506970063255261524844335867914461321021406288026937068429350126536671931757868016862812597156713876878172616168625524292260940347541700734738562142047647305723053881410229273825705072603260789109530288429356116848550927749075462006726604534197266327419654040613692101581093498295056301997299527017704165195710476563419145822684533889701667936555956484762791914445656139676481352345551216386449858109988937265671471973844681311429975749166840093462727879587818457712428704255344559061591532375113973314282008450900403092885758378292759533393050237320140953727613312702323636738951262974123205418288741397543451587603552646624247785552013314031192733246010912730484816415238192195349735466749770728067928503549656651656073233219678107283595565211925536596476285568332871576455097190154739727932800047918608913057369146767557746792528779299399728183694415823499687330127087758737047474994989156500107530548160976123883318437794772844111307075048213457629500443809360274691131620176241365042370710483640300663273050001492189318116979891143885697328268812532306505266364178977237563209358836554989475411347896892051678879301807258515307322923197206927569654412473870860647597082046284096970211213086853213077079583376130225517380758198285144994545668777118410072441336848088288912710026485935090558379495040979631540762442690074319825071132295476718724755365322307056654838102078522716992464692185740996789940963759426635815082609874311642814465663278310925100433649714232243396943774667447600732323274092995514794127098681498564314483494785776699171605823087465819773805874903515931812982154302270951498194208323195734298452691177766317870774297896878337595050008916366692834531015019438079697216037584094989943337553566464573425551343857891020619203587260757327436584898400993791510054147654356650858983129505994893865621299442621781153092790652176034693439864930622511013686730943869682925636443692558679546898863603665855630583172805499503349850642014514121193862834030570451253380424533982166942882325081614482406954734609046200523990247481209299536211933236590300256653077831592975673453496541090621701855413666597584167024844986010971736908181470245722975460872681422559593960315974723533197673148835514287190868362599834270400499224298567901025841345491864396262256475645201871059473180234586620834044589586160820199442461923914785856105411868402806753113119614849401467281772004893587038668657846417413716863007212681900774342602539212792220982908691560673641270058922493307705207693953852412366850882359948312420885917410290539202108431364158028831478952581466061057898564248286789006302089681470922690768111466742981273588223554788036400897829393832820314005422341091232015012900850944803712813177904595244116656449724186006465154847998942775267146316247645761883742804889848246791047994217728623038776307185127119085603589168365310026686699717767682202672035099875544905437455762474952669266826185925494760109140188350857814663423722054270499500299780865313880819240215932197686815613601058506815284724120507808761695627491647723994169983628082403721237061521250736485898629687166636110705477603528516177144778256890343648794041841228026365658117440073990089619025538739489191448585986816990161467401163679675742143222965321708414428129093810694714794064755941973350198612586814344308856251415549285762605619198901973252095999562808987773675654102973681561875892313268432764268630178621453552636520965667196316764034017722091666434789744950852473244657028634169270005001819431542040030172084599795191051798000372557785415768631806304331979721857525124326614036667339830688282444658238941288478669002310876132729503261262054301680693611889765562642391666129911836236611052659658675015666836246339735878483580736126251721492564510017824333924256759757104426644702004525230397427721324746167652910676129219635786478600427070678272625615428041849395626325111576358427602533598962529090699274873940416160695412097599267113126195867660840416457036419601438801767026373993563567374041380716659079947109713880435283320310340418840400006748983425147322869892866456106784243368166276491001043027738157129824773590467857765440706061341139892694918287329393435501095272854345385425759862664207901348095621343861624169032267733079271215106361980031739837397443252721253861444927543937398843985100588467186098922679135834654204236540579567720424333553573026139793867213711726734594997346716921068511887113468577183206220984939442957319753319701649775406168208888735292783748091505628859253662062402814748990091517499775398994412131990711116701806279801762213809809742790235685284108795453665970190523832117054042610767320725588540527996267650384289783805201679611190735044814267455083346488538176439282296344699381075190427267955101076921278189266206795852459943997547981950831527958646685773915067665412058659535964094172358493522912265025350044444517073944582065840479233816563753250588358010388823033147262606260903572966866661513824618140389522937363208322485410406357066138237538169911849381134193030378943781947634127904084745365618684528148004877911534280611852531652130002074329762748614955388918330020720697669159867040909571161103275938008718814332363439287947278005283066831078950153622593636076264069522747373357670031079500890891180431992065781991891921742367825343055866099321871970209934401863036961327661239319991637468025733947258443019537105699601636862210867982110963614287526566134345592092209734615961572329973184429215030829361179224300085964204734328464692710684304672186911785670568896173325697007134133808580335315007145509287686224644510112656026142171246348439697874978450468431519035002262396924289167744597623621006121042262278601469416979637060489934748965688080386494347933043119779331388649543253873533920471939791980828036522414144542989399184229888734954941080387791374729264602243046275162453298013463979814323763208568814908551670330583136102351395754428351841674160433800986699064733712674710338921125140638398786016364200272312026231970962873045513931700196813555606961732228887872506911609937164230689479726518689273237627495626505371179030038296496157201468169424133928191808115243786209845843036387479229838995429956820298177524928653119079262956383497038961417196832641047310616397286588471055182298284174546890188264991312551407389236987459407711433787616010960449520735466638516190877517628715763573713499273526194467203324805988213337736974381162024332452909250396288994124148315509243791182380425590106447395053094679039474272936701768064847072107163292619276334960808681518904078807377598977202909225473012661178239349904871648397166160349391422007998555807238335163595744012726486003972757115734101966460743821532022859918387077331253175572094767473034990147061835884693194220103610195359462069701879060370058816128132397917633322661832076420105665300421196345882636122310107049448090349486531407780885952826080138331536295852904972315103219470562658391019463008301093325544676476745220513240843273988894397363754748826640317547540708429269194630233756057054027655982526115175006345274369721659327892990894276623610814215972772580534039036115566745947585879869710178470141069683014158293470466674015404641151895596292241104271989383730209567357939768599634076515647021503455694360168312464713764497386906314124282347673152838368276565889307665549538023182548189885740728443857260979036520440451722088724577389277741443766464029498915747411102613663322984251421937287887095410172694013765016605302418431168106150791075738627806964887786906139961007250586656591133559643715442286592858586236142538017781731820527486015709802133501974028722035090693451450306497245245215528304909056469902922121779701079055424156392458682629256744994346072147630703817055226383332507726259384615824707840577783629783466008871371167317746879749418084368553113139786077545998529102211148634248576771908391839750864525017059934870026110151139293831284247885292358444680220558420653075015786922413126862003392343108515624920339701061554340826024789206010806116365712658771877097703881170908531182205477324806097504257022224177029472211284503267700113702305252751341039289809014030966135250358397752970870522090287477767349465854312033304994361963835823181413600126244550799595224717138817269690354088331878145260812534962536394578542427927349325605221444612219970160602813480868607014195544269170142079701984468319409573188521205369735140508849965452010589074620574734722211621181056839968271681386086158143341056227474435761755114553134720912996350742991987455275716806187335192882806387477755095565118299464970462913403262924040062737335315869032230805280720986765101692725258804052810013344193659820284083349726000668974775769977416428569330903730831808424720315529943948118591415585703667948439960382465339321107685686342063187160959886399917413955596599866887830868180570306328331158195513701521495944030282495670450166585528664632018435343089871895767687435158019062939508962976787026744453442447348178603455305571256558224353437999976346113240615045577906938914648762713106879932434523718967649656006933427411784596621660858217413405789595804803951105739961515750763022664559388807757433977383507362278120932963306010369271922899485352297012779458770876329621720815200962887581209479769466748019564370865483485048606490745173049067330388573472552036559269484892805536744136041286999099573093891799335305566445799572190205268207850077513887970443364276364878847242295710383378583695690058361306317352505574780146387079894593195689746353431572593163491913406577524442326656960190772216978449626440188897907450525773517566686053118883345509958845235103412740510255697130927637557179261574308099488342096889819805159974999184591751320856671292211104691417981830712254968392083688622671687360862070940314289779784493036141339337953514387743820278023538501439496490869951649542866302662163034540369528194003975413133161311528514289231644058420005456780712551691381115806887633142813708913962304644614538369223760488989972474973103825145911767197174998045774138152640628162876546283132457442355920692365451628058497510188887892526264003814971731602105055234818719107838886356959682060649964827951019132766391995843881370243949502688394360121574662387362829285756072646693984051210611564015138843502208887665563925574988404547634323051414714550860444803999362346842504852373348657580868280707134306894808698582449847968789094783822363454178076755828656450140031164739541422476900917580263194621455479525273900249469658344098031776686722596828921998300349077350029640688826299750423878798978699564278985686563794477690270252730681015512058434101506347167002791080508452943138765337884548339641218995418432300733512057744986193371001036792587086631343896442313427496361530956083717242339987774763434542732327941411809033639235512162268163975192903425063572703044483466885753452289537074669342943483597534944642923906486759987728914245046599668601990167960567967264486474200783090405672844871878943357041652889801956592460419371291957070980482003888240195591742170422714036242962873126519723755554309125264913966520882045061820440268836684965614063084372311823337969459831473373581241603551810711317668737190676840138466160533141282205242982052110279265585206135042053711149729232949444723481363085144504398104764554621621739793470969279842693820801004440178109566448519055631758790006579897275067504711398253058045705615224997091452792132741456599864714516725356212353146245236150920395600581197052850851140146897869419045725738539179267342100080380438274687584500073715470125113751158371646421559672903690837722114128516533870903323828206998964494268438297762425175694125238497778342928918184808119616601180421380762321784155942033530734811733183766471223838335941864040803977946917986111719449498134609282379018700898077393380394787240248157835760680422405170400175212982874151204848263714962414356213569439528766098980209839277444432989805915729320652333792795467772682741662187031097620823286577618713957008795099273401004918147686880915935752902935726571979899593497812819165033655326155937223654243918866996475437642968250228882881095390600833566137419536092403977749420844482288588608666320354466954709639746456396686354719437247139547612185447447844429258032649905580354783765002771378922269193319466736886245329527583373947527028207306481338810365981263064387121552573917049889544373108425827996879880319251902239799222485758222352266686829495660123251249819555280214954446159147488966364004672827794283179805076555739573747443491012549330630801413043430631559352096216330478081125280584868031018540880829189003032427432634502379340334998388768429542217546449932324947454943174021367876108453633517130620350922827476559540171044171720757264143314937479682337365620787118313120457531720598483812884240140968138449392671512765668123154786031582283855321363289227043172599722510680679491210311059930919051447262797704359165321083202571994672134239951525034253827512974402061000655607824661263738394805085661984307601887017471332189856418318717556294420087301367902918518986219289565274115136325019719718255371571071730106352484556250145755254593180069407636939013945170287884160099956865652794633202082759889880053878554128674693393574517549409291525900667717305195348608458162008345938765562654658435482241043229885013239260260934497153691572263850852945473511254106857488983998210810103689575625658957108073143908931533638183553142900822450143541318887707274769742324109614894795230253766024113617096667517601444832948779718056737432810005111634934571785051288650660775799158420126184936071583996379433507682223770755509729897304292563940930701081192610005359416409963258452334662953897805552304269619036646374462034866682834054071716186060657873239709629829260769184560702331154988025345824462370077914728946848125793854106183847021039283603683341182085994463028320671669931582867405674325283927074053293078667358300427151029814076856662722086044390555574402052799849351596380851480804974564092850288441836701117381262578711648380250347531066354105661040650669045325383933104797095994409994143419324419851651221915951821076229828827871532847247739797544425101756010239926904530509399329591693307458996088044859606499247220133680246521250139941891795301700412853870705655106830683963183080445289235950620108687174452251984758749095621431915311490758604683887872877862192572933211224093729144879090554330496784780291289412030896777225504991599630096270492360124201011465522615855714101408365265075789286397758277778326412013047738303748921460961822272763074013386950299347449978183468131962125441101344773216621533961127807630714207540092940650314088481608281794179161099654243837322194809624057123020096750707286193637438121176874973426631490000311706745919757027290302807127924246865803715768336845894042825086875757291018492832334096739056800895816340490818057121269669350286401156814978372960023092861407244781182704128899176355487832266117903864521594748102586124066260958308830911962711019622322227989641327068488581164607275870981739596179910187428644297276676725936524233646145187804906589570092519820167644099856777265978171722666035016698052384441532160026944709203621795654645711408036798593921214002357254741126764616226165231366379857237254388916034749901865888268313194140230572640844398450473344459162514165789308458497122798191680668364883436901899801790897109375190999826287342954826172107099867023687243044488741182994022449197043841254520697320154782265195195600059980532516617449689939149674410059652188296880439643923717875555609461876292722949906688754376617430473173604167070113032334775498690950941837377891959517612584470467630599072016484428433702054260481000072574957641100180832010382006080999884936897615254619217428531422948128764481099975168161859113291991236546485166360054028938088045666105959376093238089250875724345314946126338613160863139735204962511137413082802658453545773104565697967756835563330891130963795058671484926152600333911588494035825279706025045181472418551652110888328278068257412751370410459401706307156008695550183797095457937263463336179007644534708686979098699867342597294018091915081471279230521861941452204533399720666276267188728765411484206785652960571204994041653540903245195470345175135302303354362890393919792999586423994524796177950597328509816636469491756339816252165582740371856250008997343465405436871511433960572629273457690574568529069763064489644388067873182213790917310524896778947355097664202133140017746489105304814751184002326477592292882711864700513297342818189590050222380841821262414295703051852364741779338277816768964001342477353439497885689844286183833123526858476630387966054662726438674482459828986988743551814214259944129103951901837167822387684498153971259572244622583022603057334331793332217701824150371021487866490787799543644930408566024722240534433202334153508675416814754990235633028480720123461126884915660356301999846769997938897160098488183542446933682251921476137249978853166155527525762928346775039785203586370589677615353714555056807831359889509804635947203591439319253721793644938863053407413602485816395329675422350368997523484060140835685714677305132330413918613258468226678534787395866089555560721512026067089391522824887064553103293686252787722045329571012880733425789152714993515065475896868590011491613047870475801969172295742595546647503708062599674649365594948000381042582813859675657662262207110369363557450947375778685081334440678228499217056769817931865090141596139051120845564020942757184209216884668907821816927329681157223737482143990362162774714718059028826233739897859179184847870380477043476556772662681335514313701393835302534231766199506413145893538081057921464407953752152152466188785249172065678228231385797366006844671998544192185655114363113279699279687734818267288268450295294086936892418302522104097805869622109336407272103269470569254106856491672244877541093240686108552310695994235786360728492357117344838597229014947463773836991287972573077739289185178750172023200012729291483227918819139344879219234644698858300272325009787543453632979932010395151245999045988542284962776090932733276695501414931835677462267762496257651519794312344723017201493134372444147373483448838427053824667786432896684419965817859659278225629224104711729610171247470013884360372378250866809386773758931290944195533250294800388366404481334994972660544068487052862971038318125952952221266656230888503549510932822550104978623992848912737228900638213766482475313816628335230558078256577583872816207577206411958576703975670367239282145829652075223941028417996954228554662116361416545981396544923092833995348937568231847952456868264208542780958636296562296512681059045940793420743819684995664426433444364372468077056993782826477937962698075314476560639781199774999093480058071101367983523940180652102398466559676129222396620717971348046230669383045058321418315048804101873936917453925354751238683211614278879359577707932931547841397915011896802271195540369405947803302555097489994149436519585866360029069596827016540801858315473947735801163592524960057047153151462334744162860603553945295352641899348408023847463856952412996957520001436409162782582723567789747264046879345384789306923535910951705352734082309656703668138810555120128109846526329155976395502843888943999359511911944511000044816575267463865328446911221558949734398589574358232550402076583207889460067430313130318914680213184937175304035286442521552022442214280256728759731686116539509093115407025057793925860380810159744331521694801208444812893654295493906970546139756575543937826101990313468451636314664896112128953216197607340763188524269536241573661788795533122770754992761255886493738158709873207280895308714156655807492059457201549011952837895391597020502116873808733273837887820359548311327062393900685531144108899709054270698870670955718682663846272629497617794855616448293214274230393511194061102021530776483100665521107923871595711497155328703311140326373791204614842758165614806770957610350731365858305920499849843838940310930825818734480881780451713733471351518474226482493177752132463998210403663643700718188771186092584850735679067209634328163714795933158555576696028841137723137415404378516010769267208635592058909474420842642165819877171071936917378641195882269269089613403494360361766808846507028552437675211626145758495622465044704055848259065652867474779740359399938771814210472442935482766104690159766341274969631001679409674293979840576869065446443739858510967412494368205664767943232926058921605329598630206382788803827478822769780948691094780122518829267978522201366662147212451866113907862216290529612280724669882340792250720229363347941558318868392206538384434094866270586206182180257100201951885939289746163839296773495154341055793695526607996218692742713752667728109656241655028336108429469426081010185050395579035523458490647098271037522067060099139459541655067741111000392371430816201202837708223967561589025598776314795730561946268899364992104197790801325589764679898708965638028546442148876319116669067360434341298843947159416842070217544705297411281143178949172496978700005365278068339621848521422115715445850152130603441539828658565312739732965812926440450627857026261031781729341704320810741203498659702819707870208504622661946654488150088072722798066735017836549515246208711042761409512712803622431050029574862282207128429507638849308547357626704905897815342319042283214289034621570407592599497164306292075511593628968502052919343951525565810010648837504835305222743662624667832392322060810691962454588096312499799138396373520741507795953319083662971026640218051070282553434014895675499215127934162912745972184858203642844227050593361079113952768949606536996785596460878337067471222759643974653246991920865792626204849669242113660638209542916439857862004106774478657053881127637206543795622456625937135637517606842748901784831452297594281561816472907723916063958200216593159317564841818578810377221322420792956117314303997105587830693313155685347118123828821891308593582898985659622273267328020305358500873328310520996446562305759133927029055418095319855618043792758509176884127208763329966073221668272512192379412414888923758174420357559195566886953341782605751900782716320388043241633137809902127257055207136730594560969199701767926949379272381553859032077105935124258603291611491468563595580107849013467483251070405535908000413587348834917975008983379006423032736725435308133232670701004555152600869066632444317409473018659370169554362499526328233311586339612305247595282607922165990326055504210932883228237210279569037090398728709708792380649679178790103987588507883054533970631331405733116335218024376456146862040572042769149531635124850928761871056375614250590496539974686710414245871398178326489964416768377405265389606730066416269561718459262035698253866258358269727180862719562537631535970405634835288111400540685865940744348298797179700642272390305765628709085485331995998881439837596050767753577254922530702245329018858598274235970621140901606963317621959685348480871734137815963130983613545595955611303976819711484729588060825978486919635772257032472793890257911399504934606378742258656347306389277575547308978219623614066328716167064997437355782313470274208225811493883336456503709130385258356852333749235612220448562916421100463482601608517467074722842457893471774684894845488309399260340076946733677434307644637972585935651136371967565545056795319009107480009314821805482835539908116378037378851958333358861531712610566344197477232464573814601621508364941105092920962336819058174612453423688548936714840862928418574768255765006860796506931527704492197901211182575740867337966671059257412445262955677133599908076070190948675406361774213319534247121019445924422469451370447683440366540665191224244775391900072290905076828765240529751662460880817381001480221427681492457887573771785909121495587929305820355179008022337476868800151862899359830113957536925042115388653867519975432109926147098364083977726344218340724463129037720361020180656730648147127781827441094272939615922384958082811309317743400505342065637571899596909893912144189020632714210097588901782140031192415466799213969489949023456297428492352215760115958676402534840933879832300594090906728425720095503729219280120813485188045550955097574835196300668913948685394050243155373446050415496851619814694355524354393963415898955748196918575124639686606261084323952543952954376633408280328887771703528092086423626596989563534927567913002284271773241207729098151967136917516980307139873276519562004450277668286867556784081420157546364597131075481198125957232154973022807919258807742890502586456890157155005911381208889802670980904973138220778573321023776412934515236969903334340004543237615824740138452977767305148811654765158056740764645737506028497831009933261553909886742201440804409700727728098119363658170387329946625135359001059731116600566534160916575417028270236804575672330755226813486686330324778198807485998187922400794388849254841891129516749632495442104959209918292769043226185019275754338770030345725130209264975029693050099855591403833756073178893506170285700154123539751272707971496738152757732073571943375843905591512803741216384006006657341604796207067484269920834829310031543440955770431798603763911808459968851865750867549700107681983611900129007236993090453625742241461439539835476961058941984772639995456932954961310054888630049605723200735744078204722234221231544532257474116257191040537530672362313491705629432886056717863878656

7:4:0:1
7:4:1:7
7:4:2:823543
# 7:4:3:37598235267837885389...10357073612870132343 (a 695975 digit number)

8:4:0:1
8:4:1:8
8:4:2:16777216
# 8:4:3:60145207536513920379...25397555555421126656 (a 15151336 digit number)

9:4:0:1
9:4:1:9
9:4:2:387420489
# 9:4:3:42812477317574704803...99359681422627177289  (a 369693100 digit number)

10:4:0:1
10:4:1:10
10:4:2:10000000000
# 10:4:3:... (a 10000000000 digit number)

11:4:0:1
11:4:1:11
11:4:2:285311670611
# 11:4:3:... (a 297121486765 digit number)

12:4:0:1
12:4:1:12
12:4:2:8916100448256
# 12:4:3:... (a 9622088391635 digit number)

13:4:0:1
13:4:1:13
13:4:2:302875106592253
# 13:4:3:... (a 337385711567665 digit number)

14:4:0:1
14:4:1:14
14:4:2:11112006825558016
# 14:4:3:... (a 12735782555419959 digit number)

15:4:0:1
15:4:1:15
15:4:2:437893890380859375
# 15:4:3:... (a 515003176870814403 digit number)

16:4:0:1
16:4:1:16
16:4:2:18446744073709551616
# 16:4:3:... (a 22212093154093386895 digit number)

17:4:0:1
17:4:1:17
17:4:2:827240261886336764177
# 17:4:3:... (a 1017876887958722011919 digit number)

18:4:0:1
18:4:1:18
18:4:2:39346408075296537575424
# 18:4:3:... (a 49390464231494343541444 digit number)

19:4:0:1
19:4:1:19
19:4:2:1978419655660313589123979
# 19:4:3:... (a 2529911258871477183123293 digit number)

20:4:0:1
20:4:1:20
20:4:2:104857600000000000000000000
# 20:4:3:... (a 136422882873335218863214297 digit number)

21:4:0:1
21:4:1:21
21:4:2:5842587018385982521381124421
# 21:4:3:... (a 7725181286871851536584161699 digit number)

22:4:0:1
22:4:1:22
22:4:2:341427877364219557396646723584
# 22:4:3:... (a 458340526438710225132843406543 digit number)

23:4:0:1
23:4:1:23
23:4:2:20880467999847912034355032910567
# 23:4:3:... (a 28433514504467439831900993035955 digit number)

24:4:0:1
24:4:1:24
24:4:2:1333735776850284124449081472843776
# 24:4:3:... (a 1840837112681720683942395084381216 digit number)

25:4:0:1
25:4:1:25
25:4:2:88817841970012523233890533447265625
# 25:4:3:... (a 124162014773790739964431599731838212 digit number)

26:4:0:1
26:4:1:26
26:4:2:6156119580207157310796674288400203776
# 26:4:3:... (a 8710745132914411485514673420443810093 digit number)

27:4:0:1
27:4:1:27
27:4:2:443426488243037769948249630619149892803
# 27:4:3:... (a 634704607339354284876828904762819985796 digit number)

28:4:0:1
28:4:1:28
28:4:2:33145523113253374862572728253364605812736
# 28:4:3:... (a 47966809976383689164978385644364084203503 digit number)

29:4:0:1
29:4:1:29
29:4:2:2567686153161211134561828214731016126483469
# 29:4:3:... (a 3754979089615820440662116198963008711183407 digit number)

30:4:0:1
30:4:1:30
30:4:2:205891132094649000000000000000000000000000000
# 30:4:3:... (a 304126167375299121454244837378097999976300328 digit number)

31:4:0:1
31:4:1:31
31:4:2:17069174130723235958610643029059314756044734431
# 31:4:3:... (a 25456312443947506419913530706783844472823216353 digit number)

32:4:0:1
32:4:1:32
32:4:2:1461501637330902918203684832716283019655932542976
# 32:4:3:... (a 2199779157743111490588923283107086827417053752053 digit number)

33:4:0:1
33:4:1:33
33:4:2:129110040087761027839616029934664535539337183380513
# 33:4:3:... (a 196055395651457624070747411885057553879849564955563 digit number)

34:4:0:1
34:4:1:34
34:4:2:11756638905368616011414050501310355554617941909569536
# 34:4:3:... (a 18005044618850738018040884096438161566983607509140556 digit number)

35:4:0:1
35:4:1:35
35:4:2:1102507499354148695951786433413508348166942596435546875
# 35:4:3:... (a 1702346598409269964330866750261687948077899768864682678 digit number)

36:4:0:1
36:4:1:36
36:4:2:106387358923716524807713475752456393740167855629859291136
# 36:4:3:... (a 165570912743006692163267611092812105074488614008940964618 digit number)

37:4:0:1
37:4:1:37
37:4:2:10555134955777783414078330085995832946127396083370199442517
# 37:4:3:... (a 16552580835410493917255137076161060316511419991143722758700 digit number)

38:4:0:1
38:4:1:38
38:4:2:1075911801979993982060429252856123779115487368830416064610304
# 38:4:3:... (a 1699707816174424954321799266374566733167183523201848870889698 digit number)

39:4:0:1
39:4:1:39
39:4:2:112595147462071192539789448988889059930192105219196517009951959
# 39:4:3:... (a 179146154049830695635173738661912935811919990301877037742029069 digit number)

40:4:0:1
40:4:1:40
40:4:2:12089258196146291747061760000000000000000000000000000000000000000
# 40:4:3:... (a 19367716880879590104496859546004717199408818522283304763563550218 digit number)

41:4:0:1
41:4:1:41
41:4:2:1330877630632711998713399240963346255985889330161650994325137953641
# 41:4:3:... (a 2146417957953844823376201545581268787491595643213033406665376497189 digit number)

42:4:0:1
42:4:1:42
42:4:2:150130937545296572356771972164254457814047970568738777235893533016064
# 42:4:3:... (a 243699937837173716717726828717860034703775970860235550846981072974342 digit number)

43:4:0:1
43:4:1:43
43:4:2:17343773367030267519903781288812032158308062539012091953077767198995507
# 43:4:3:... (a 28330506695765243281623175267881552459593254038006373203060553323364140 digit number)

44:4:0:1
44:4:1:44
44:4:2:2050773823560610053645205609172376035486179836520607547294916966189367296
# 44:4:3:... (a 3370349729198490580651375043417911949914536054787130437661383001826156033 digit number)

45:4:0:1
45:4:1:45
45:4:2:248063644451341145494649182395412689744530581492654164321720600128173828125
# 45:4:3:... (a 410101921219674011473927237729199519279738684312248843030133150910423792460 digit number)

46:4:0:1
46:4:1:46
46:4:2:30680346300794274230660433647640397899788170645078853280082659754365153181696
# 46:4:3:... (a 51013986090348393998474945010096958955257976345093525672914575644029102371800 digit number)

47:4:0:1
47:4:1:47
47:4:2:3877924263464448622666648186154330754898344901344205917642325627886496385062863
# 47:4:3:... (a 6484268854175837242486802784110235149754653600611235824254236359662614978276766 digit number)

48:4:0:1
48:4:1:48
48:4:2:500702078263459319174537025249570888246709955377400223021257741084821677152403456
# 48:4:3:... (a 841800981616184880487268534079734532078120989957118390561691624001515952416172873 digit number)

49:4:0:1
49:4:1:49
49:4:2:66009724686219550843768321818371771650147004059278069406814190436565131829325062449
# 49:4:3:... (a 111569377908409484669478841232440867814957170723180995217259334164911961178605030730 digit number)

50:4:0:1
50:4:1:50
50:4:2:8881784197001252323389053344726562500000000000000000000000000000000000000000000000000
# 50:4:3:... (a 15089884935690772673530649371062523685706555590284695099921533545613226521640191055100 digit number)

# pentation

0:5:0:1
0:5:1:0
0:5:2:1
0:5:3:0
0:5:4:1
0:5:5:0
0:5:6:1
0:5:7:0
0:5:8:1
0:5:9:0

1:5:0:1
1:5:1:1
1:5:2:1
1:5:3:1
1:5:4:1
1:5:5:1
1:5:6:1
1:5:7:1
1:5:8:1
1:5:9:1

2:5:0:1
2:5:1:2
2:5:2:4
2:5:3:65536

3:5:0:1
3:5:1:3
3:5:2:7625597484987
# 3:5:3:... (a 3638334640025 digit number)

# hexation

0:6:0:1
0:6:1:0
0:6:2:1
0:6:3:0
0:6:4:1
0:6:5:0
0:6:6:1
0:6:7:0
0:6:8:1
0:6:9:0

1:6:0:1
1:6:1:1
1:6:2:1
1:6:3:1
1:6:4:1
1:6:5:1
1:6:6:1
1:6:7:1
1:6:8:1
1:6:9:1

2:6:0:1
2:6:1:2
2:6:2:4

3:6:0:1
3:6:1:3

4:6:0:1
4:6:1:4

5:6:0:1
5:6:1:5

6:6:0:1
6:6:1:6

7:6:0:1
7:6:1:7

8:6:0:1
8:6:1:8

9:6:0:1
9:6:1:9

# heptation / septation

0:7:0:1
0:7:1:0
0:7:2:1
0:7:3:0
0:7:4:1
0:7:5:0
0:7:6:1
0:7:7:0
0:7:8:1
0:7:9:0

1:7:0:1
1:7:1:1
1:7:2:1
1:7:3:1
1:7:4:1
1:7:5:1
1:7:6:1
1:7:7:1
1:7:8:1
1:7:9:1

2:7:0:1
2:7:1:2
2:7:2:4

3:7:0:1
3:7:1:3

4:7:0:1
4:7:1:4

5:7:0:1
5:7:1:5

6:7:0:1
6:7:1:6

7:7:0:1
7:7:1:7

8:7:0:1
8:7:1:8

9:7:0:1
9:7:1:9

# octation

0:8:0:1
0:8:1:0
0:8:2:1
0:8:3:0
0:8:4:1
0:8:5:0
0:8:6:1
0:8:7:0
0:8:8:1
0:8:9:0

1:8:0:1
1:8:1:1
1:8:2:1
1:8:3:1
1:8:4:1
1:8:5:1
1:8:6:1
1:8:7:1
1:8:8:1
1:8:9:1

2:8:0:1
2:8:1:2
2:8:2:4

3:8:0:1
3:8:1:3

4:8:0:1
4:8:1:4

5:8:0:1
5:8:1:5

6:8:0:1
6:8:1:6

7:8:0:1
7:8:1:7

8:8:0:1
8:8:1:8

9:8:0:1
9:8:1:9

# nontation / enneation

0:9:0:1
0:9:1:0
0:9:2:1
0:9:3:0
0:9:4:1
0:9:5:0
0:9:6:1
0:9:7:0
0:9:8:1
0:9:9:0

1:9:0:1
1:9:1:1
1:9:2:1
1:9:3:1
1:9:4:1
1:9:5:1
1:9:6:1
1:9:7:1
1:9:8:1
1:9:9:1

2:9:0:1
2:9:1:2
2:9:2:4

3:9:0:1
3:9:1:3

4:9:0:1
4:9:1:4

5:9:0:1
5:9:1:5

6:9:0:1
6:9:1:6

7:9:0:1
7:9:1:7

8:9:0:1
8:9:1:8

9:9:0:1
9:9:1:9

# decation

10:10:0:1
10:10:1:10

# undecation

11:11:0:1
11:11:1:11

# doedecation

12:12:0:1
12:12:1:12

# tredecation

13:13:0:1
13:13:1:13

# higher operators

1234:1234:0:1
1234:1234:1:1234

1234567:1234567:0:1
1234567:1234567:1:1234567

1234567898:1234567898:0:1
1234567898:1234567898:1:1234567898

1234567898765:1234567898765:0:1
1234567898765:1234567898765:1:1234567898765

1234567898765432:1234567898765432:0:1
1234567898765432:1234567898765432:1:1234567898765432

1234567898765432101:1234567898765432101:0:1
1234567898765432101:1234567898765432101:1:1234567898765432101

1234567898765432101234:1234567898765432101234:0:1
1234567898765432101234:1234567898765432101234:1:1234567898765432101234

1234567898765432101234567:1234567898765432101234567:0:1
1234567898765432101234567:1234567898765432101234567:1:1234567898765432101234567
