CREATE TABLE projects (
    id INT NOT NULL PRIMARY KEY,
    parent_id INTEGER,
    name VARCHAR(40) NOT NULL,
    title VARCHAR(1024) NOT NULL,
    path VARCHAR collate nocase,
    fullpath VARCHAR collate nocase,
    project_status_id INTEGER NOT NULL DEFAULT -1,
    hub_id INTEGER,
    local INTEGER NOT NULL DEFAULT 0,
    num_changes INTEGER,
    FOREIGN KEY(id) REFERENCES topics(id)
        ON DELETE CASCADE
    FOREIGN KEY(parent_id) REFERENCES projects(id)
        ON DELETE CASCADE,
    FOREIGN KEY(hub_id) REFERENCES hubs(id)
        ON DELETE CASCADE
    FOREIGN KEY(project_status_id,id) REFERENCES project_status(id,project_id)
        DEFERRABLE INITIALLY DEFERRED
);

CREATE UNIQUE INDEX projects_path_hub_id ON projects(path,hub_id);

CREATE TRIGGER
    projects_bd_1
BEFORE DELETE ON
    projects
FOR EACH ROW
BEGIN
    SELECT debug(
        OLD.id
    );


    /*
        Any issues that were created in this project need to go, even
        if they have been pushed to other projects because a) they
        won't make sense without the original history, and be not
        deleting them can cause issue.title to be set to undef.
    */

    DELETE FROM
        issues
    WHERE
        id IN (
            SELECT
                id.issue_id
            FROM
                project_issues pi
            INNER JOIN
                topics t
            ON
                t.id = pi.issue_id
            INNER JOIN
                issue_deltas id
            ON
                id.change_id = t.first_change_id AND
                id.project_id = OLD.id
            WHERE
                pi.project_id = OLD.id
        )
    ;

    /*
        The following is necessary, because although FK relationships
        do result in the remove of rows from [project_]issues_tomerge,
        the deletion of rows from project/issue_deltas just inserts
        more rows. So we remove those trigger-happy rows first.
    */


    DELETE FROM
        project_issues
    WHERE
        project_id = OLD.id
    ;


    DELETE FROM
        project_deltas
    WHERE
        project_id = OLD.id
    ;


END;

CREATE TRIGGER
    projects_ad_1
AFTER DELETE ON
    projects
FOR EACH ROW
BEGIN
    SELECT debug(
        OLD.id
    );

    DELETE FROM
        topics
    WHERE
        id = OLD.id
    ;

END;


CREATE TRIGGER
    projects_au_hub_id_1
AFTER UPDATE OF
    hub_id
ON
    projects
FOR EACH ROW
BEGIN

    INSERT INTO
        hub_related_projects(
            hub_id,
            project_id
        )
    SELECT
        NEW.hub_id,
        NEW.id
    ;

END;


CREATE TRIGGER
    projects_au_path_1
AFTER UPDATE OF
    path,hub_id
ON
    projects
FOR EACH ROW
BEGIN
    
    UPDATE
        projects
    SET
        fullpath = (
            SELECT
                COALESCE( h.name || '/','') || p.path
            FROM
                projects p
            LEFT JOIN
                hubs h
            ON
                h.id = p.hub_id
            WHERE
                p.id = NEW.id
        )
    WHERE
        id = NEW.id
    ;

END;


/*
    A hierachical data (tree) implementation in SQL using triggers as
    described here:

        http://www.depesz.com/index.php/2008/04/11/my-take-on-trees-in-sql/

    Generated by App::sqltree version 0.0.5_2 with the following
    paramters on Wed Jun 12 21:25:18 2013

        driver:     SQLite
        table:      projects
        pk:         id
        parent:     parent_id
        type:       INTEGER
        path:       path
        path_from:  name
        order:      
*/

/*
 Triggers in SQLite run in the reverse order to which they are defined.
 Actions happen from the bottom up.
 */

CREATE TRIGGER
    projects_ai_tree_3
AFTER INSERT ON
    projects
FOR EACH ROW WHEN
    NEW.parent_id IS NOT NULL
BEGIN

    UPDATE
        projects
    SET
        path = (
            SELECT
                path || '/' || NEW.name
            FROM
                projects
            WHERE
                id = NEW.parent_id
        )
    WHERE
        id = NEW.id
    ;

END;

CREATE TRIGGER
    projects_ai_tree_2
AFTER INSERT ON
    projects
FOR EACH ROW WHEN
    NEW.parent_id IS NULL
BEGIN

    UPDATE
        projects
    SET
        path = name
    WHERE
        id = NEW.id
    ;

END;

CREATE TRIGGER
    projects_ai_tree_1
AFTER INSERT ON
    projects
FOR EACH ROW 
BEGIN

    /*
     Insert a matching row in projects_tree where both parent and child
     are set to the id of the newly inserted object. Depth is set to 0
     as both child and parent are on the same level.
     */

    INSERT INTO
        projects_tree (
            parent,
            child,
            depth
        )
    VALUES (
        NEW.id,
        NEW.id,
        0
    );

    /*
     Copy all rows that our parent had as its parents, but we modify
     the child id in these rows to be the id of currently inserted row,
     and increase depth by one.
     */

    INSERT INTO
        projects_tree (
            parent,
            child,
            depth
        )
    SELECT
        x.parent,
        NEW.id,
        x.depth + 1
    FROM
        projects_tree x
    WHERE
        x.child = NEW.parent_id
    ;

END;

/*
    Handle parent_id changes

    4. Change all affected child rows with the new parent path
*/

CREATE TRIGGER
    projects_au_tree_5
AFTER UPDATE OF
    parent_id
ON
    projects
FOR EACH ROW WHEN
    NEW.parent_id IS NOT NULL AND
    ( OLD.parent_id IS NULL OR NEW.parent_id != OLD.parent_id )
BEGIN

    UPDATE
        projects
    SET
        path = (
            SELECT
                path
            FROM
                projects
            WHERE
                id = NEW.parent_id
        ) || '/' || path
    WHERE
        id IN (
            SELECT
                child
            FROM
                projects_tree
            WHERE parent = NEW.parent_id AND depth > 0
        )
    ;

END;

/*
    Handle parent_id changes

    3. Insert the new tree data relating to the new parent
*/

CREATE TRIGGER
    projects_au_tree_4
AFTER UPDATE OF
    parent_id
ON
    projects
FOR EACH ROW WHEN
    NEW.parent_id IS NOT NULL AND 
    ( OLD.parent_id IS NULL OR NEW.parent_id != OLD.parent_id )
BEGIN

    INSERT INTO
        projects_tree (parent, child, depth)
    SELECT
        r1.parent, r2.child, r1.depth + r2.depth + 1
    FROM
        projects_tree r1
    INNER JOIN
        projects_tree r2
    ON
        r2.parent = NEW.id
    WHERE
        r1.child = NEW.parent_id
    ;

END;

/*
    Handle parent_id changes

    2. Remove the tree data relating to the old parent
*/

CREATE TRIGGER
    projects_au_tree_3
AFTER UPDATE OF
    parent_id
ON
    projects
FOR EACH ROW WHEN
    OLD.parent_id IS NOT NULL AND 
    ( NEW.parent_id IS NULL OR NEW.parent_id != OLD.parent_id)
BEGIN
    DELETE FROM
        projects_tree
    WHERE
        treeid IN (
            SELECT
                r2.treeid
            FROM
                projects_tree r1
            INNER JOIN
                projects_tree r2
            ON
                r1.child = r2.child AND r2.depth > r1.depth
            WHERE
                r1.parent = NEW.id
        )
    ;
END;

/*
 FIXME: Also trigger when column 'path_from' changes. For the moment,
 the user work-around is to temporarily re-parent the row.
*/

/*
    Handle parent_id changes

    1. Remove parent part of the path
*/

CREATE TRIGGER
    projects_au_tree_2
AFTER UPDATE OF
    parent_id
ON
    projects
FOR EACH ROW WHEN
    OLD.parent_id IS NOT NULL
BEGIN
    UPDATE
        projects
    SET
        path = substr(path, (
            SELECT
                length(path || '/') + 1
            FROM
                projects
            WHERE
                id = OLD.parent_id
        ))
    WHERE
        id IN (
            SELECT
                child
            FROM
                projects_tree
            WHERE
                parent = OLD.parent_id AND depth > 0
        )
    ;

END;

/*
 Handle changes to the name column
*/

CREATE TRIGGER
    projects_au_tree_1
AFTER UPDATE OF
    name
ON
    projects
FOR EACH ROW WHEN
    OLD.name != NEW.name
BEGIN

    /*
        First of all the current row
    */

    UPDATE
        projects
    SET
        path = 
            CASE WHEN
                NEW.parent_id IS NOT NULL
            THEN
                (SELECT
                    path
                 FROM
                    projects
                 WHERE
                    id = NEW.parent_id
                ) || '/' || name
            ELSE
                name
            END
    WHERE
        id = OLD.id
    ;

    /*
        Then all of the child rows
    */

    UPDATE
        projects
    SET
        path = (
            SELECT
                path
            FROM
                projects
            WHERE
                id = OLD.id
        ) || SUBSTR(path, LENGTH(OLD.path) + 1)
    WHERE
        id IN (
            SELECT
                child
            FROM
                projects_tree
            WHERE
                parent = OLD.id AND depth > 0
        )
    ;

END;

/*
 Forbid moves that would create loops:
*/

CREATE TRIGGER
    projects_bu_tree_2
BEFORE UPDATE OF
    parent_id
ON
    projects
FOR EACH ROW WHEN
    NEW.parent_id IS NOT NULL AND
    (SELECT
        COUNT(child) > 0
     FROM
        projects_tree
     WHERE
        child = NEW.parent_id AND parent = NEW.id
    )
BEGIN
    SELECT RAISE (ABORT,
        'Change blocked, because it would create loop in tree.');
END;

/*
 This implementation doesn't support changes to the primary key
*/

CREATE TRIGGER
    projects_bu_tree_1
BEFORE UPDATE OF
    id
ON
    projects
FOR EACH ROW WHEN
    OLD.id != NEW.id
BEGIN
    SELECT RAISE (ABORT, 'Changing ids is forbidden.');
END;
