package Apache::TestMM;

use strict;
use warnings FATAL => 'all';

use Config;
use Apache::TestConfig ();
use Apache::TestTrace;

sub import {
    my $class = shift;

    for my $section (@_) {
        unless (defined &$section) {
            die "unknown Apache::TestMM section: $section";
        }
        no strict 'refs';
        my $sub = "MY::$section";
        # Force aliasing, since previous WriteMakefile might have
        # moved it
        undef &$sub if defined &$sub;
        *$sub = \&{$section};
    }
}

sub add_dep {
    my($string, $targ, $add) = @_;
    $$string =~ s/($targ\s+::)/$1 $add /;
}

sub clean {
    my $self = shift;
    my $string = $self->MM::clean(@_);
    add_dep(\$string, clean => 'test_clean');
    $string;
}

sub test {

    my $env = Apache::TestConfig->passenv_makestr();

    my $preamble = Apache::TestConfig::WIN32 ? "" : <<EOF;
PASSENV = $env
EOF

    return $preamble . <<'EOF';
TEST_VERBOSE = 0
TEST_FILES =

test_clean :
	$(FULLPERL) -I$(INST_ARCHLIB) -I$(INST_LIB) \
	t/TEST -clean

run_tests : test_clean
	$(PASSENV) \
	$(FULLPERL) -I$(INST_ARCHLIB) -I$(INST_LIB) \
	t/TEST -bugreport -verbose=$(TEST_VERBOSE) $(TEST_FILES)

test :: pure_all run_tests test_clean

cmodules:
	cd c-modules && $(MAKE) all

cmodules_clean:
	cd c-modules && $(MAKE) clean
EOF

}

sub generate_script {
    my $file = shift;

    unlink $file if -e $file;

    my $body = "BEGIN { eval { require blib; } }\n";

    $body .= Apache::TestConfig->modperl_2_inc_fixup;

    if (@Apache::TestMM::Argv) {
        $body .= "\n\%Apache::TestConfig::Argv = qw(@Apache::TestMM::Argv);\n";
    }

    my $in = Symbol::gensym();
    open $in, "$file.PL" or die "Couldn't open $file.PL: $!";
    {
        local $/;
        $body .= <$in>;
    }
    close $in;

    info "generating script $file";
    Apache::Test::config()->write_perlscript($file, $body);
}

sub filter_args {
    my($argv, $vars) =
        Apache::TestConfig::filter_args(\@ARGV, \%Apache::TestConfig::Usage);
    @ARGV = @$argv;
    @Apache::TestMM::Argv = %$vars;
}

1;

=head1 NAME

Apache::TestMM - Provide MakeMaker Wrapper Methods

=head1 SYNOPSIS

  require Apache::TestMM;
  
  # import MY::test and MY::clean overrides for MM
  Apache::TestMM->import(qw(test clean));
  
  # parse command line args
  Apache::TestMM::filter_args();
  
  # autogenerate the script
  Apache::TestMM::generate_script('t/TEST');

=head1 DESCRIPTION

C<Apache::TestMM> provides wrappers for the C<ExtUtils::MakeMaker>
craft, making it easier to extend the autogenerated F<Makefile> with
C<Apache::Test>.

=head1 FUNCTIONS

=head2 C<import>

  use Apache::TestMM qw(test clean);

or:

  Apache::TestMM->import(qw(test clean));

Imports C<MY::> overrides for the default C<ExtUtils::MakeMaker>
I<test> and I<clean> targets, as if you have defined:

  sub MY::test {...}
  sub MY::clean {...}

in F<Makefile.PL>. C<Apache::TestMM> does this for you so that these Makefile
targets will run the Apache server and the tests for it, and clean up after
its mess.

=head2 C<filter_args>

  push @ARGV, '-apxs', $apxs_path;
  Apache::TestMM::filter_args();
  WriteMakefile(...);

When C<WriteMakefile()> is called it parses C<@ARGV>, hoping to find special
options like C<PREFIX=/home/stas/perl>. C<Apache::Test> accepts a lot of
options of its own. When C<Apache::TestMM::filter_args()> is called, it
removes any C<Apache::Test>-specific options from C<@ARGV> and stores them
internally, so when C<WriteMakefile()> is called they aren't in C<@ARGV> and
thus won't be processed by C<WriteMakefile()>.

The options can be set when F<Makefile.PL> is called:

  % perl Makefile.PL -apxs /path/to/apxs

Or you can push them manually to C<@ARGV> from the code:

  push @ARGV, '-apxs', $apxs_path;

When:

  Apache::TestMM::generate_script('t/TEST');

is called, C<Apache::Test>-specific options extracted by
C<Apache::TestMM::filter_args()> are written to the autogenerated
file. In our example, the autogenerated F<t/TEST> will include:

  %Apache::TestConfig::Argv = qw(apxs /path/to/apxs);

which is going to be used by the C<Apache::Test> runtime.

=head2 C<generate_script>

  Apache::TestMM::generate_script('t/TEST');

C<generate_script()> accepts the name of the script to generate and
will look for a template with the same name and suffix I<.PL>. So in
our example it'll look for F<t/TEST.PL>. The autogenerated script
F<t/TEST> will include the contents of F<t/TEST.PL>, and special
directives, including any configuration options passed via
C<L<filter_args()|/C_filter_args_>> called from F<Makefile.PL>, special
fixup code, etc. If no argument is passed to C<generate_script()>,
it will create a file named F<t/TEST> by default.

=cut
