# status.tcl --
#
#	Application-direct URLs to give out status of the server.
# 	Tcl procedures of the form Status/hello implement URLS
#	of the form /status/hello
#	Note also the Debug/source procedure, which you can use
#	to get the server to reload files from its script library.
#
# Brent Welch (c) Copyright 1997 Sun Microsystems, Inc.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#
package provide status 1.0

proc Status_Url {dir} {
    Direct_Url $dir Status
}
proc Debug_Url {dir} {
    Direct_Url $dir Debug
}

proc Status/hello {args} {return hello}

proc Status/doc {{pattern *} {sort number}} {
    global counter
    if {[string compare $sort "number"] == 0} {
	set numcheck checked
	set namecheck ""
    } else {
	set numcheck ""
	set namecheck checked
    }
    set result ""
    append result "<h1>Document Hits</h1>"
    append result "<form action=/status/doc>"
    append result "Pattern <input type=text name=pattern value=$pattern><br>"
    append result "Sort by Hit Count <input type=radio name=sort value=number $numcheck> or Name <input type=radio name=sort value=name $namecheck><br>"
    append result "<input type=submit name=submit value=\"Again\"><p>"
    append result "<a href=/status/notfound>(See document misses ...)</a>"
    append result <pre>\n
    append result [format "%6s %s\n" Hits Url]
    set list {}
    foreach i [lsort [array names counter cachehit,$pattern]] {
	if ![catch {set counter($i)} value] {
	    regsub {cachehit,} $i {} j
	    lappend list [list $value $j]
	}
    }
    if {[string compare $sort "number"] == 0} {
	if [catch {lsort -index 0 -integer -decreasing $list} newlist] {
	    set newlist [lsort -command StatusSort $list]
	}
    } else {
	if [catch {lsort -index 1 -integer -decreasing $list} newlist] {
	    set newlist [lsort -command StatusSortName $list]
	}
    }
    foreach k $newlist {
	set url [lindex $k 1]
	append result [format "%6d %s" [lindex $k 0] $url]
	append result \n
    }
    append result </pre>\n
    return $result
}
proc StatusSort {a b} {
    set 1 [lindex $a 0]
    set 2 [lindex $b 0]
    if {$1 == $2} {
	return [string compare $a $b]
    } elseif {$1 < $2} {
	return 1
    } else {
	return -1
    }
}
proc StatusSortName {a b} {
    set 1 [lindex $a 1]
    set 2 [lindex $b 1]
    return [string compare $1 $2]
}

proc Status/notfound {args} {
    global Doc Referer
    set result ""
    append result "<h1>Documents Not Found</h1>"
    append result "<a href=/status/doc>(See document hits...)</a>"
    append result <pre>\n
    append result [format "%6s %s\n" Miss Url]
    set list {}
    foreach i [lsort [array names Doc notfound,*]] {
	if ![catch {set Doc($i)} value] {
	    regsub {notfound,} $i {} j
	    lappend list [list $value $j]
	}
    }
    if [catch {lsort -index 0 -integer -decreasing $list} newlist] {
	set newlist [lsort -command StatusSort $list]
    }
    foreach k $newlist {
	set url [lindex $k 1]
	append result [format "%6d <a href=/admin/redirect?old=%s>%s</a>" \
	    [lindex $k 0] [lindex $k 1] [lindex $k 1]]
	if {[info exists Referer($url)]} {
	    set i 0
	    foreach r $Referer($url) {
		append result " <a href=\"$r\">[incr i]</a>"
	    }
	}
	append result \n
    }
    append result </pre>\n
    append result "<a href=/status/notfound/reset>Reset counters</a>"
    return $result
}
proc Status/notfound/reset {args} {
    global Doc Referer
    foreach i [array names Doc notfound,*] {
	unset Doc($i)
    }
    catch {unset Referer}
    return <h1>ok</h1>
}
proc Status/size {args} {
    return [Status/datasize][Status/codesize]
}
proc Status/datasize {args} {
    set ng 0
    set nv 0
    set size 0
    foreach g [info globals *] {
	upvar #0 $g gg
	incr ng
	if [array exists gg] {
	    foreach {name value} [array get gg] {
		set size [expr {$size + [string length $name] + [string length $value]}]
		incr nv
	    }
	} else {
	    set size [expr {$size + [string length $g] + [string length $gg]}]
	    incr nv
	}
    }
    return "<h1>Data Size</h1>\n\
		Num Globals $ng<br>\n\
		Num Values $nv<br>\n\
		Data Bytes $size"
}
proc Status/codesize {args} {
    set np 0
    set size 0
    foreach g [info procs *] {
	incr np
	set size [expr {$size + [string length $g] +
			    [string length [info args $g]] +
			    [string length [info body $g]]}]
    }
    return "<h1>Code Size</h1>\n\
		Num Procs $np<br>\n\
		Code Bytes $size"
}
proc StatusMainTable {} {
    global Httpd Doc counter status tcl_patchLevel tcl_platform
    set html "<title>Tcl HTTPD Status</title>\n"
    append html "<img src=/images/PWRD150.GIF align=left width=95 height=150>\n"
    append html "<H1>$Httpd(name):$Httpd(port)</h1>\n"
    append html "<H2>Server Info</h2>"
    append html "<table border=0>"
    append html "<tr><td>Start Time</td><td>[clock format [Counter_StartTime]]</td></tr>\n"
    append html "<tr><td>Current Time</td><td>[clock format [clock seconds]]</td></tr>\n"
    append html "<tr><td>Server</td><td>$Httpd(server)</td></tr>\n"
    append html "<tr><td>Tcl Version</td><td>$tcl_patchLevel</td></tr>"
    switch $tcl_platform(platform) {
	unix {
	    append html "<tr><td colspan=2>[exec uname -a]</td></tr>"
	}
	macintosh -
	windows {
	    append html ""<tr><td colspan=2>$tcl_platform(os) $tcl_platform(osVersion)</td></tr>"
	}
    }

    append html </table>

    append html "<br><br><br><h3>Counters</h3>\n"
    append html "<table border>\n"

    foreach {c label} {
	    urlhits "URL Requests"
	    urlreply "URL Replies"
	    cachehit,/ "Home Page Hits"
	    connections "Active Requests"
	    sockets "Open Sockets"
	    cgihits "CGI Hits"
	    tclhits "Tcl Safe-CGIO Hits"
	    maphits "Image Map Hits"
	    cancel	"Timeouts"
	    errors	"Errors"
	    Status	"Status"
	    } {
	if [info exists counter($c)] {
	    append html "<tr><td>$label</td><td>$counter($c)</td></tr>\n"
	}
    }
    append html </table>\n
    return $html
}

proc Status/all {args} {
    global CntMinuteurlhits CntHoururlhits CntDayurlhits counter
    set html [StatusMainTable]
    append html "<p>The following bar charts are created with a table of horizontal rules that may not display correctly on your browser.<br><a href=/status/text>Text only view.</a>\n"
    append html [StatusMinuteHist CntMinuteurlhits "Per Minute Url Hits" $counter(basetime)]
    if [info exists CntHoururlhits] {
	append html [StatusMinuteHist CntHoururlhits "Per Hour Url Hits" $counter(hour,1) hour]
    }
    if [info exists CntDayurlhits] {
	append html [StatusMinuteHist CntDayurlhits "Per Day Url Hits" $counter(day,1) day]
    }
    return $html
}
proc Status/text {args} {
    global CntMinuteurlhits CntHoururlhits CntDayurlhits counter
    set html [StatusMainTable]
    append html "<p><a href=/status/all>Bar Chart View.</a>"
    append html [StatusTimeText CntMinuteurlhits "Per Minute Url Hits" Min Hits $counter(basetime)]
    if [info exists CntHoururlhits] {
	append html [StatusTimeText CntHoururlhits "Per Hour Url Hits" Hour Hits $counter(hour,1)]
    }
    if [info exists CntDayurlhits] {
	append html [StatusTimeText CntDayurlhits "Per Day Url Hits" Day Hits $counter(day,1)]
    }
    return $html
}

proc Status/ {args} [info body Status/all]
proc Status {args} [info body Status/all]

proc Debug/source {source} {
    global Httpd
    set source [file tail $source]
    set error [catch {uplevel #0 [list source [file join $Httpd(library) $source]]} result]
    set html "<title>Source $source</title>\n"
    if {$error} {
	append html "<H1>Error in $source</H1>\n"
    } else {
	append html "<H1>Reloaded $source</H1>\n"
    }
    append html "<pre>$result</pre>"
    return $html
}

proc Debug/parray {aname} {
    global $aname
    set html "<title>Array $aname</title>\n"
    append html "<H1>Array $aname</H1>\n"
    append html "<pre>[parray $aname]</pre>"
    return $html
}

proc Debug/raise {args} {
    error $args
}
proc Debug/goof {args} {
    set goof
}

proc Debug/after {} {
    global tcl_version
    set html "<title>After Queue</title>\n"
    append html "<H1>After Queue</H1>\n"
    append html "<pre>"
    if [catch {after info} afterlist] {
	append html "\"after info\" not supported in Tcl $tcl_version"
    } else {
	foreach a $afterlist {
	    append html "$a [after info $a]\n"
	}
    }
    append html </pre>
    return $html
}

proc Debug/echo {title args} {
    set html "<title>$title</title>\n"
    append html "<H1>$title</H1>\n"
    append html <dl>
    foreach {name value} $args {
	append html "<dt>$name<dd>$value"
    }
    append html </dl>
    return $html
}

proc Version {} {
    global tcl_patchLevel Httpd
    append html "$Httpd(server)"
    append html "<br>Tcl version $tcl_patchLevel"
    return $html
}
proc Debug/errorInfo {title errorInfo} {
    set html "<title>$title</title>\n"
    append html "<H1>$title</H1>\n"
    append html "<p>[Version]"
    append html "<br>Webmaster: [Doc_Webmaster]"
    append html <pre>$errorInfo</pre>
    return $html
}

proc StatusMinuteHist {array title time {unit minute}} {
    global counter
    upvar #0 $array data
    regsub ^Cnt $array Age agebitName
    upvar #0 $agebitName agebit

    set total 0
    set max 0
    set base 100
    foreach {name value} [array get data] {
	setmax max $value
    }
    switch $unit {
	minute	{set bar 3}
	hour	{set bar 5}
	day	{set bar 5}
    }
    append result "<h3>$title ($max max)</h3>"
    append result <ul>
    append result "<h4>Starting at [clock format $time]</h4>"
    append result "<table cellpadding=0 cellspacing=0><tr>\n"
    set skip 0
    append result "<td valign=top>$max</td>\n"
    foreach t [lsort -integer [array names data]] {
	if {!$skip && [info exists agebit($t)]} {

	    # Indicate the hourly wrap-around point with a zero value.

	    set skip 1
	    set marker 1
	} else {
	    set marker 0
	}

	if {$unit == "hour" && ($t == $counter(mergehour))} {
	    set marker 1
	}
	set value $data($t)
	if {[catch {expr {round($value * 100.0 / $max)}} percent]} {
	    puts "Skipping $percent"
	    continue
	}
	set width [expr {$percent * $base / 100}]
	if {$marker} {
	    append result "<td valign=bottom><hr size=$width width=$bar></td>\n"
	} else {
	    append result "<td valign=bottom><hr NOSHADE size=$width width=$bar></td>\n"
	}
    }
    append result "</tr>"

    switch $unit {
	minute	{#do nothing}
	hour	{
	    append result "<tr><td> </td>"
	    foreach t [lsort -integer [array names data]] {
		set tag td
		append result "<td><font size=1>[clock format $time -format %k]</font></td>"
		incr time 3600
	    }
	    append result </tr>
	}
	day {
	    append result "<tr><td> </td>"
	    set skip 4
	    set i 0
	    foreach t [lsort -integer [array names data]] {
		if {($i % $skip) == 0} {
		    append result "<td colspan=$skip><font size=1>[clock format $time -format "%h %e"]</font></td>"
		}
		incr time [expr 3600 * 24]
		incr i
	    }
	    append result </tr>
	}
    }
    append result "</table>"
    append result </ul>
    return $result
}

proc StatusTimeText {array title unit what time} {
    global counter
    upvar #0 $array data
    regsub ^Cnt $array Age agebitName
    upvar #0 $agebitName agebit
    set total 0
    set max 0
    set base 100
    foreach {name value} [array get data] {
	setmax max $value
    }
    switch $unit {
	Min	{set delta 60 ; set fmt %R}
	Hour	{set delta 3600 ; set fmt "%h %e %R"}
	Day	{set delta 86400 ; set fmt %D}
    }

    append result "<h3>$title ($max max)</h3>"
    append result <ul>
    append result "<h4>Starting at [clock format $time]</h4>"
    append result "<table cellpadding=2 cellspacing=2 border><tr>\n"
    append result "<tr><th>$unit</th><th>$what</th></tr>"
    foreach t [lsort -integer [array names data]] {
	set value $data($t)

	# Minutes time we infer from the starting time and the agebits,
	# which indicate minute buckets for the previous hour.

	if [info exists agebit($t)] {
	    set tt [expr $time - 3600]
	} else {
	    set tt $time
	}

	# Hours have their own base time in counter(hour,$hour)

	if {$unit == "Hour"} {
	    set tt $counter(hour,$t)
	}

	# Wrap separator

	if {[info exists lasttime] && ($lasttime > $tt)} {
	    append result "<tr><td><hr></td><td><hr></td></tr>"
	}
	set lasttime $tt
	append result "<tr><td>[clock format $tt -format $fmt]</td><td>$value</td></tr>"
	incr time $delta
    }
    append result "</table>"
    append result </ul>
    return $result
}

# Handle .stat templates. (NOTUSED)
# First process the incoming form data in an Status-specific way,
# then do a normal Subst into a safe interpreter
#   path:	The path name of the document
#   suffix:     The file suffix
#   sock:	The name of the socket, and a handle for the socket state

# It turns out this is not used, but you could use it as a template
# for your own application's template processor.

proc Doc_application/x-tcl-status {path suffix sock} {
    global status
    upvar #0 Httpd$sock data

    append data(query) ""
    set queryList [Url_DecodeQuery $data(query)]

    # Put the query data into an array.
    # If a key appears multiple times, the resultant array value will
    # be a list of all the values.

    foreach {name value} $queryList {
    	lappend query($name) $value
    }

    if ![info exists status(session)] {
	set status(session) [session_create Status]
    }

    # Process the query data from the previous page.

    if [catch {StatusProcess $session $queryList} result] {
	Httpd_ReturnData $sock text/html $result
	return
    } 

    # Expand the page in the correct session interpreter, or treat
    # the page as ordinary html if the session has ended.

    if {$result} {
	Doc_Subst $sock $path interp$session
    } else {
	Httpd_ReturnFile $sock text/html $path
    }
}

