# label_bind.tcl --
#
# This file defines the default bindings for Tk text widgets and provides
# procedures that help in implementing the bindings.
#
# SCCS: @(#) text.tcl 1.44 96/04/16 11:42:24
#
# Copyright (c) 1992-1994 The Regents of the University of California.
# Copyright (c) 1994-1996 Sun Microsystems, Inc.
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.
#

#-------------------------------------------------------------------------
# Elements of tkPriv that are used in this file:
#
# afterId -		If non-null, it means that auto-scanning is underway
#			and it gives the "after" id for the next auto-scan
#			command to be executed.
# char -		Character position on the line;  kept in order
#			to allow moving up or down past short lines while
#			still remembering the desired position.
# mouseMoved -		Non-zero means the mouse has moved a significant
#			amount since the button went down (so, for example,
#			start dragging out a selection).
# prevPos -		Used when moving up or down lines via the keyboard.
#			Keeps track of the previous insert position, so
#			we can distinguish a series of ups and downs, all
#			in a row, from a new up or down.
# selectMode -		The style of selection currently underway:
#			char, word, or line.
# x, y -		Last known mouse coordinates for scanning
#			and auto-scanning.
#-------------------------------------------------------------------------

# tkLabelClipboardKeysyms --
# This procedure is invoked to identify the keys that correspond to
# the "copy", "cut", and "paste" functions for the clipboard.
#
# Arguments:
# copy -	Name of the key (keysym name plus modifiers, if any,
#		such as "Meta-y") used for the copy operation.
# cut -		Name of the key used for the cut operation.
# paste -	Name of the key used for the paste operation.

proc tkLabelClipboardKeysyms {page copy cut paste} {
    $page bind text <$copy> {tk_textCopy %W}
    $page bind text <$cut> {tk_textCut %W}
    $page bind text <$paste> {tk_textPaste %W}
}

#-------------------------------------------------------------------------
# The code below creates the default class bindings for entries.
#-------------------------------------------------------------------------

# Standard Motif bindings:

proc label_bind {page} {
    global tk_strictMotif
    $page bind text <1> {
	tkLabelSetFocus %W current
	tkLabelButton1 %W %x %y
	[winfo command %W] select clear
    }

#    $page bind text <B1-Motion> {
#	set tkPriv(x) %x
#	set tkPriv(y) %y
#	tkLabelSelectTo %W %x %y
#    }
#    $page bind text <Double-1> {
#	set tkPriv(selectMode) word
#	tkLabelSelectTo %W %x %y
#	catch {%q icursor current sel.first}
#    }
#    $page bind text <Triple-1> {
#	set tkPriv(selectMode) line
#	tkLabelSelectTo %W %x %y
#	catch {%q icursor current sel.first}
#    }
#    $page bind text <Shift-1> {
#	tkLabelResetAnchor %W @%x,%y
#	set tkPriv(selectMode) char
#	tkLabelSelectTo %W %x %y
#    }
#    $page bind text <Double-Shift-1>	{
#	set tkPriv(selectMode) word
#	tkLabelSelectTo %W %x %y
#    }
#    $page bind text <Triple-Shift-1>	{
#	set tkPriv(selectMode) line
#	tkLabelSelectTo %W %x %y
#    }
    $page bind text <B1-Leave> {
	set tkPriv(x) %x
	set tkPriv(y) %y
#	tkLabelAutoScan %W
    }
    $page bind text <B1-Enter> {
	tkCancelRepeat
    }
    $page bind text <ButtonRelease-1> {
	tkCancelRepeat
    }
    $page bind text <Control-1> {
	%q icursor focus @%x,%y
    }
# We may need to change this.  Let's see how it acts first.
#    $page bind text <ButtonRelease-2> {
#	if {!$tkPriv(mouseMoved) || $tk_strictMotif} {
#	    tkLabelPaste %W %x %y
#	}
#    }
    $page bind text <Left> {
	tkLabelSetCursor %W [expr [%q index focus insert] - 1]
    }
    $page bind text <Right> {
	tkLabelSetCursor %W [expr [%q index focus insert] + 1]
    }
    $page bind text <Up> {
	tkLabelSetCursor %W [tkLabelUpDownLine %W -1]
    }
    $page bind text <Down> {
	tkLabelSetCursor %W [tkLabelUpDownLine %W 1]
    }
#    $page bind text <Shift-Left> {
#	tkLabelKeySelect %W [[%q index focus insert] -1]
#    }
#    $page bind text <Shift-Right> {
#	tkLabelKeySelect %W [expr [%q index focus insert] + 1]
#    }
#    $page bind text <Shift-Up> {
#	tkLabelKeySelect %W [tkLabelUpDownLine %W -1]
#    }
#    $page bind text <Shift-Down> {
#	tkLabelKeySelect %W [tkLabelUpDownLine %W 1]
#    }

    $page bind text <Home> {
	tkLabelSetCursor %W 0
    }
    $page bind text <Shift-Home> {
	tkLabelKeySelect %W 0
    }
    $page bind text <End> {
	catch {tkLabelSetCursor %W [string length [%q itemcget focus -text]]}
    }
    $page bind text <Shift-End> {
	catch {tkLabelKeySelect %W [string length [%q itemcget focus -text]]}
    }
    $page bind text <Control-Home> {
	tkLabelSetCursor %W 0
    }
    $page bind text <Control-Shift-Home> {
	tkLabelKeySelect %W 0
    }
    $page bind text <Control-End> {
	catch {tkLabelSetCursor %W [string length [%q itemcget focus -text]]}
    }
    $page bind text <Control-Shift-End> {
	catch {tkLabelKeySelect %W [string length [%q itemcget focus -text]]}
    }

#    $page bind text <Tab> {
#	tkLabelInsert %W \t
#	focus %W
#	break
#    }
#    $page bind text <Shift-Tab> {
#	# Needed only to keep <Tab> binding from triggering;  doesn't
#	# have to actually do anything.
#    }
#    $page bind text <Control-Tab> {
#	focus [tk_focusNext %W]
#    }
#    $page bind text <Control-Shift-Tab> {
#	focus [tk_focusPrev %W]
#    }
    $page bind text <Control-i> {
	tkLabelInsert %W \t
    }
    $page bind text <Return> {
	tkLabelInsert %W \n
    }
    $page bind text <Delete> {
	if {[%q select item] == [%q find withtag focus] &&
	    [%q select item] != {}} {
	    catch {%q dchars focus sel.first sel.last}
	} else {
	    %q dchars focus [%q index focus insert]
	}
    }
    $page bind text <BackSpace> {
	if {[%q select item] == [%q find withtag focus]} {
	    catch {%q dchars focus sel.first sel.last}
	} elseif {[%q index focus insert] != 0} {
	    %q dchars focus [expr [%q index focus insert] -1]
	}
#	if {[string length [%q itemcget focus -text]] == 0} {
#	    if {[%q index focus insert] == 0} {
#		%q delete focus
#	    }
#	}
    }

#    $page bind text <Control-space> {
#	%q select from focus insert
#    }
#    $page bind text <Select> {
#	%q select from focus insert
#    }
#    $page bind text <Control-Shift-space> {
#	set tkPriv(selectMode) char
#	tkLabelKeyExtend %W [%q index focus insert]
#    }
#    $page bind text <Shift-Select> {
#	set tkPriv(selectMode) char
#	tkLabelKeyExtend %W [%q index focus insert]
#    }
#    $page bind text <Control-slash> {
#	%q select from focus 0
#	%q select to focus end
#    }
#    $page bind text <Control-backslash> {
#	if {[%q select item] == [%q find withtag focus] &&
#	    [%q select item] != ""} {
#	    %q select clear
#	}
#    }
    tkLabelClipboardKeysyms $page F16 F20 F18
    $page bind text <Insert> {
	catch {tkLabelInsert %W [selection get -displayof %W]}
    }
    $page bind text <KeyPress> {
	tkLabelInsert %W %A
    }

# Ignore all Alt, Meta, and Control keypresses unless explicitly bound.
# Otherwise, if a widget binding for one of these is defined, the
# <KeyPress> class binding will also fire and insert the character,
# which is wrong.  Ditto for <Escape>.

#    $page bind text <Alt-KeyPress> {# nothing }
#    $page bind text <Meta-KeyPress> {# nothing}
    $page bind text <Control-KeyPress> {# nothing}
    $page bind text <Escape> {# nothing}
    $page bind text <KP_Enter> {# nothing}

# Additional emacs-like bindings:

    $page bind text <Control-a> {
	if !$tk_strictMotif {
	    catch {set text [%q itemcget focus -text]}
	    tkLabelSetCursor %W [tkLabelLinestart $text [%q index focus insert]]
	}
    }
    $page bind text <Control-b> {
	if !$tk_strictMotif {
	    tkLabelSetCursor %W [expr [%q index focus insert] -1]
	}
    }
    $page bind text <Control-d> {
	if !$tk_strictMotif {
	    %q dchars focus insert
	}
    }
    $page bind text <Control-e> {
	if !$tk_strictMotif {
	    catch {set text [%q itemcget focus -text]}
	    tkLabelSetCursor %W [tkLabelLineend $text [%q index focus insert]]
	}
    }
    $page bind text <Control-f> {
	if !$tk_strictMotif {
	    tkLabelSetCursor %W [expr [%q index focus insert] +1]
	}
    }
    $page bind text <Control-k> {
	if !$tk_strictMotif {
	    catch {set text [%q itemcget focus -text]}
	    set insert [%q index focus insert]
	    if {$insert == [tkLabelLineend $text $insert]} {
		%q dchars focus $insert
	    } else {
		%q dchars focus $insert [tkFocusLineend $text $insert]
	    }
	}
    }
    $page bind text <Control-n> {
	if !$tk_strictMotif {
	    tkLabelSetCursor %W [tkLabelUpDownLine %W 1]
	}
    }
    $page bind text <Control-o> {
	if !$tk_strictMotif {
	    %q insert focus insert \n
	    %q icursor focus [expr [%q index focus insert] -1]
	}
    }
    $page bind text <Control-p> {
	if !$tk_strictMotif {
	    tkLabelSetCursor %W [tkLabelUpDownLine %W -1]
	}
    }
    $page bind text <Control-t> {
	if !$tk_strictMotif {
	    tkLabelTranspose %W
	}
    }
#    $page bind text <Control-v> {
#	if !$tk_strictMotif {
#	    tkLabelScrollPages %W 1
#	}
#    }
# I don't want to bother with these.  If someone requests them, I'll do it.
#    $page bind text <Meta-b> {
#	if !$tk_strictMotif {
#	    tkLabelSetCursor %W {insert - 1c wordstart}
#	}
#    }
#    $page bind text <Meta-d> {
#	if !$tk_strictMotif {
#	    %q delete insert {insert wordend}
#	}
#    }
#    $page bind text <Meta-f> {
#	if !$tk_strictMotif {
#	    tkLabelSetCursor %W {insert wordend}
#	}
#    }
    $page bind text <Meta-less> {
	if !$tk_strictMotif {
	    tkLabelSetCursor %W 0
	}
    }
    $page bind text <Meta-greater> {
	if !$tk_strictMotif {
	    tkLabelSetCursor %W end
	}
    }
#    $page bind text <Meta-BackSpace> {
#	if !$tk_strictMotif {
#	    %q delete {insert -1c wordstart} insert
#	}
#    }
#    $page bind text <Meta-Delete> {
#	if !$tk_strictMotif {
#	    %q delete {insert -1c wordstart} insert
#	}
#    }
    if !$tk_strictMotif {
	tkLabelClipboardKeysyms $page Meta-w Control-w Control-y
    }

# A few additional bindings of my own.

    $page bind text <Control-h> {
	if !$tk_strictMotif {
	    if {[%q index focus insert] != 0} {
		%q dchars focus [expr [%q index focus insert] - 1]
	    }
	}
    }
    $page bind text <2> {
	if !$tk_strictMotif {
	    set tkPriv(x) %x
	    set tkPriv(y) %y
	    set tkPriv(mouseMoved) 0
	}
    }
    $page bind text <B2-Motion> {
	if !$tk_strictMotif {
	    if {(%x != $tkPriv(x)) || (%y != $tkPriv(y))} {
		set tkPriv(mouseMoved) 1
	    }
	}
    }
}

set tkPriv(prevPos) {}

# tkLabelClosestGap --
# Given x and y coordinates, this procedure finds the closest boundary
# between characters to the given coordinates and returns the index
# of the character just after the boundary.
#
# Arguments:
# w -		The text window.
# x -		X-coordinate within the window.
# y -		Y-coordinate within the window.

proc tkLabelClosestGap {w x y} {
    set q [winfo command $w]
    if {[$q find withtag focus] == ""} {
	set pos [$q index current @$x,$y]
    } else {
	set pos [$q index focus @$x,$y]
    }

    set bbox [$q bbox $w]
    sputs "bbox = $bbox"
    if ![string compare $bbox ""] {
	sputs "nearest to $x, $y is $pos"
	return $pos
    }
    if {($x - [lindex $bbox 0]) < ([lindex $bbox 2]/2)} {
	sputs "nearest to $x, $y is $pos"
	return $pos
    }
    sputs "nearest to $x, $y is $pos"
    return [expr $pos]
#    $q index "$pos + 1 char"
}

# tkLabelButton1 --
# This procedure is invoked to handle button-1 presses in text
# widgets.  It moves the insertion cursor, sets the selection anchor,
# and claims the input focus.
#
# Arguments:
# w -		The text window in which the button was pressed.
# x -		The x-coordinate of the button press.
# y -		The x-coordinate of the button press.

proc tkLabelButton1 {w x y} {
    global tkPriv

    set q [winfo command $w]

    set tkPriv(selectMode) char
    set tkPriv(mouseMoved) 0
    set tkPriv(pressX) $x
    $q icursor focus [tkLabelClosestGap $w $x $y]
    $q select from focus insert
}

# tkLabelSelectTo --
# This procedure is invoked to extend the selection, typically when
# dragging it with the mouse.  Depending on the selection mode (character,
# word, line) it selects in different-sized units.  This procedure
# ignores mouse motions initially until the mouse has moved from
# one character to another or until there have been multiple clicks.
#
# Arguments:
# w -		The text window in which the button was pressed.
# x -		Mouse x position.
# y - 		Mouse y position.

proc tkLabelSelectTo {w x y} {
    global tkPriv

    catch {set text [$w itemcget focus -text]}

    set q [winfo command $w]
    set cur [tkLabelClosestGap $w $x $y]
    if {[$q select item]!=[$q find withtag focus] || [$q select item] == ""} {
	$q select from focus @$x,$y
	sputs "set anchor \[$q index focus @$x,$y\]"
	sputs "set anchor [$q index focus @$x,$y]"
	set anchor [$q index focus @$x,$y]
    } else {
	sputs "set anchor \[$q index focus sel.first\]"
	sputs "set anchor [$q index focus sel.first]"
	set anchor [$q index focus sel.first]
    }

    sputs "cur, anchor = $cur, $anchor"
    if {$cur != $anchor || (abs($tkPriv(pressX) - $x) >= 3)} {
	set tkPriv(mouseMoved) 1
    }
    sputs "selectmode = $tkPriv(selectMode)"
    switch $tkPriv(selectMode) {
	char {
	    if {$cur < $anchor} {
		set first $cur
		set last $anchor
	    } else {
		set first $anchor
		set last $cur
	    }
	}
	word {
	    if {$cur < $anchor} {
		set first [string wordstart $text $cur]
		set last [string wordstart $text [expr $anchor - 1]]
	    } else {
		set first [string wordstart $text $anchor]
		set last [string wordstart $text [expr $cur - 1]]
	    }
	}
	line {
	    if {$cur < $anchor} {
		set first [string last [string range $text 0 $cur] \n]
		if {$first < 0} {set first 0}
		set last [string last [string range $text $anchor end] \n]
		if {$last < 0} {set last [string length $text]}
	    } else {
		set first [string last [string range $text 0 $anchor] \n]
		if {$first < 0} {set first 0}
		set last [string last [string range $text $cur end] \n]
		if {$last < 0} {set last [string length $text]}
	    }
	}
    }

    if {$tkPriv(mouseMoved) || ($tkPriv(selectMode) != "char")} {
	$q select to focus $cur
	update idletasks
    }
}

# tkLabelKeyExtend --
# This procedure handles extending the selection from the keyboard,
# where the point to extend to is really the boundary between two
# characters rather than a particular character.
#
# Arguments:
# w -		The text window.
# index -	The point to which the selection is to be extended.

proc tkLabelKeyExtend {w index} {
    global tkPriv

    set q [winfo command $w]
    set cur [$q index $index]
    if [catch {$q index focus sel.first}] {
	$q select from focus $cur
    }
    set anchor [$q index sel.first]
    if {$cur < $anchor} {
	set first $cur
	set last $anchor
    } else {
	set first $anchor
	set last $cur
    }
    $q select from $first
    $q select to $last
}

# tkLabelSetFocus --
# This procedure sets the page focus to the given item, and sets the focus
# to the page itself.  It also tags the item with the tag "focus" and removes
# the "focus" tag from every other item.
#
# Arguments:
# w -		The Page
# t -		a tag to describe the item

proc tkLabelSetFocus {w t} {
    focus $w
    $w focus $t

    $w dtag all focus
    $w addtag focus withtag $t
}

# tkLabelPaste --
# This procedure sets the insertion cursor to the mouse position,
# inserts the selection, and sets the focus to the window.
#
# Arguments:
# w -		The text window.
# x, y - 	Position of the mouse.

proc tkLabelPaste {w x y} {
    set q [winfo command $w]

    ;# Use current here and not focus!

    $q icursor current [tkLabelClosestGap $w $x $y]

    catch {$q insert current insert [selection get -displayof $w]}
    if {[$q cget -state] == "normal"} {
	tkLabelSetFocus $w current
    }
}

# tkLabelAutoScan --
# This procedure is invoked when the mouse leaves a text window
# with button 1 down.  It scrolls the window up, down, left, or right,
# depending on where the mouse is (this information was saved in
# tkPriv(x) and tkPriv(y)), and reschedules itself as an "after"
# command so that the window continues to scroll until the mouse
# moves back into the window or the mouse button is released.
#
# Arguments:
# w -		The text window.

proc tkLabelAutoScan {w} {
    global tkPriv
    if {![winfo exists $w]} return
    set q [winfo command $w]

    tkLabelSelectTo $w $tkPriv(x) $tkPriv(y)
    set tkPriv(afterId) [after 50 tkLabelAutoScan $w]
}

# tkLabelSetCursor
# Move the insertion cursor to a given position in a text.  Also
# clears the selection, if there is one in the text, and makes sure
# that the insertion cursor is visible.  Also, don't let the insertion
# cursor appear on the dummy last line of the text.
#
# Arguments:
# w -		The text window.
# pos -		The desired new position for the cursor in the window.

proc tkLabelSetCursor {w pos} {
    global tkPriv

    set q [winfo command $w]

    set length [string length [$q itemcget focus -text]]
    if {$pos > $length} {
	set pos $length
    }
    $q icursor focus $pos
    $q select clear
}

# tkLabelKeySelect
# This procedure is invoked when stroking out selections using the
# keyboard.  It moves the cursor to a new position, then extends
# the selection to that position.
#
# Arguments:
# w -		The text window.
# new -		A new position for the insertion cursor (the cursor hasn't
#		actually been moved to this position yet).

proc tkLabelKeySelect {w new} {
    global tkPriv

    set q [winfo command $w]
    if {[$q select item] == ""} {
	if {$new < [$q index focus insert]} {
	    $q select from focus 0
	    $q select to focus insert
	} else {
	    $q select from focus insert
	    $q select to focus 0
	}
    } else {
	if {$new < [$q index focus sel.from]} {
	    set first $new
	    set last [$q index focus sel.from]
	} else {
	    set first [$q index focus sel.from]
	    set last $new
	}
	$q select clear
	$q select from focus $first $last
    }
    $q icursor focus $new
    update idletasks
}

# tkLabelResetAnchor --
# Set the selection anchor to whichever end is farthest from the
# index argument.  One special trick: if the selection has two or
# fewer characters, just leave the anchor where it is.  In this
# case it doesn't matter which point gets chosen for the anchor,
# and for the things like Shift-Left and Shift-Right this produces
# better behavior when the cursor moves back and forth across the
# anchor.
#
# Arguments:
# w -		The text widget.
# index -	Position at which mouse button was pressed, which determines
#		which end of selection should be used as anchor point.

proc tkLabelResetAnchor {w index} {
    global tkPriv

    catch {set text [$q itemcget focus -text]}

    set q [winfo command $w]
    if {[$q select item] == ""} {
	$q select from current $index
	return
    }
    set a [$q index $index]
    set b [$q index sel.first]
    set c [$q index sel.last]
    if {$a < $b} {
	$q select from current $a
	return
    }
    if {$a > $c} {
	$q select to current $c
	return
    }
    for {set i 0; set count 0} {$i < [string length $text]} {incr i} {
	if [string match [string index $i] \n] { 
	    incr count 
	    set pos 0
	}
	if {$i == $a} {set lineA $count; set chA $pos}
	if {$i == $b} {set lineB $count; set chB $pos}
	if {$i == $c} {set lineC $count; set chC $pos}
    }
    if {$lineB < $lineC+2} {
	set total [string length [string range $b $c]]
	if {$total <= 2} {
	    return
	}
	if {[string length [string range $b $a]] < ($total/2)} {
	    $q select from current sel.last
	} else {
	    $q select from current sel.first
	}
	return
    }
    if {($lineA-$lineB) < ($lineC-$lineA)} {
	$q select from sel.last
    } else {
	$q select from sel.first
    }
}

# tkLabelInsert --
# Insert a string into a text at the point of the insertion cursor.
# If there is a selection in the text, and it covers the point of the
# insertion cursor, then delete the selection before inserting.
#
# Arguments:
# w -		The text window in which to insert the string
# s -		The string to insert (usually just a single character)

proc tkLabelInsert {w s} {
    set q [winfo command $w]
    if {($s == "") || ([$q cget -state] == "disabled")} {
	return
    }
    catch {
	if {[$q index focus sel.first] <= [$q index focus insert]]
		&& [$q index focus sel.last] >= [$q index focus insert]} {
	    $q delete sel.first sel.last
	    $q dchars focus sel.first sel.last
	}
    }
    $q insert focus insert $s
}

# tkLabelCoordToPos --
# Converts the given line and column index to a position from the
# start of the string.  If the given character position is off the end
# of the given line, the position of the last character in the line
# is returned.  If the given coordinates are past the end of the last line
# then the position of the last character is returned. 
#
# Arguments:
# text -	The string to use
# line -	The line number of the position to convert
# ch   -	The column number of the position to convert
# a    -	The name of the variable in which to store the result

proc tkLabelCoordToPos {text line ch a} {
    for {set i 0;set pos 0;set count 0} {$i < [string length $text]} {incr i;incr pos} {
	if [string match [string index $text $i] \n] { 
	    incr count 
	}
	if [string match [string index $text [expr $i - 1]] \n] {
	    set pos 0
	}
	if {$count == $line && $pos == $ch} {
	    break
	}
	if {$count > $line} {
	    # Make sure to count newlines
	    break
	}
    }

    uplevel set $a $i
}

# tkLabelPosToCoord -
# Convert a distance from the beginning of a string into a set of line/column
# coordinates.  If the position is past the end of the string, the
# coordinates of the last character in the string are returned.
#
# Arguments:
# text -	The string to use
# a -		The absolute position to convert
# line -	the name of the variable in which to store the line position
# ch   -	the name of the variable in which to store the column position

proc tkLabelPosToCoord {text a line ch} {
    for {set i 0;set pos 0;set count 0} {$i < [string length $text]} {incr i;incr pos} {
	if [string match [string index $text $i] \n] { 
	    incr count 
	    set pos 0
	}
	if {$i == $a} {
	    break
	}
    }

    uplevel set $line $count
    uplevel set $ch $pos
}

# tkLabelLineend -
# Returns the index of the start of the line containing the given position
#
# Arguments:
# text - 	String to analyze
# pos  - 	position to use

proc tkLabelLineend {text pos} {
    while {$pos <= [string length text]} {
	if [string match [string index $text $pos] \n] {
	    return expr $pos
	}
	incr pos
    }
}

# tkLabelLinestart -
# Returns the index of the start of the line containing the given position
#
# Arguments:
# text - 	String to analyze
# pos  - 	position to use

proc tkLabelLinestart {text pos} {
    while {$pos >= 0} {
	if [string match [string index $text $pos] \n] {
	    return [expr $pos + 1]
	}
	incr pos -1
    }
}

# tkLabelUpDownLine --
# Returns the index of the character one line above or below the
# insertion cursor.  There are two tricky things here.  First,
# we want to maintain the original column across repeated operations,
# even though some lines that will get passed through don't have
# enough characters to cover the original column.  Second, don't
# try to scroll past the beginning or end of the text.
#
# Arguments:
# w -		The text window in which the cursor is to move.
# n -		The number of lines to move: -1 for up one line,
#		+1 for down one line.

proc tkLabelUpDownLine {w n} {
    global tkPriv
    set q [winfo command $w]
    set a [$q index focus insert]
    catch {set text [$q itemcget focus -text]}

    tkLabelPosToCoord $text $a lineA chA

    if [string compare $tkPriv(prevPos) $a] {
	set tkPriv(char) $chA
    }

    tkLabelCoordToPos $text [expr $lineA + $n] $tkPriv(char) new

    for {set i [$q index focus insert]} {$i >= 0} { incr i -1 } {
	if [string match [string index $text $i] \n] break
    }

    if {$new == [string length $text] || $new == $i} {
	set new $a
    }
    set tkPriv(prevPos) $new
    return $new
}

# tkLabelPrevPara --
# Returns the index of the beginning of the paragraph just before a given
# position in the text (the beginning of a paragraph is the first non-blank
# character after a blank line).
#
# Arguments:
# w -		The text window in which the cursor is to move.
# pos -		Position at which to start search.

proc tkLabelPrevPara {w pos} {
#    set q [winfo command $w]
#    set pos [$q index "$pos linestart"]
#    while 1 {
#	if {(([$q get "$pos - 1 line"] == "\n") && ([$q get $pos] != "\n"))
#		|| ($pos == "1.0")} {
#	    if [regexp -indices {^[ 	]+(.)} [$q get $pos "$pos lineend"] \
#		    dummy index] {
#		set pos [$q index "$pos + [lindex $index 0] chars"]
#	    }
#	    if {[$q compare $pos != insert] || ($pos == "1.0")} {
#		return $pos
#	    }
#	}
#	set pos [$q index "$pos - 1 line"]
#    }
}

# tkLabelNextPara --
# Returns the index of the beginning of the paragraph just after a given
# position in the text (the beginning of a paragraph is the first non-blank
# character after a blank line).
#
# Arguments:
# w -		The text window in which the cursor is to move.
# start -	Position at which to start search.

proc tkLabelNextPara {w start} {
#    set q [winfo command $w]
#    set pos [$q index "$start linestart + 1 line"]
#    while {[$q get $pos] != "\n"} {
#	if [$q compare $pos == end] {
#	    return [$q index "end - 1c"]
#	}
#	set pos [$q index "$pos + 1 line"]
#    }
#    while {[$q get $pos] == "\n"} {
#	set pos [$q index "$pos + 1 line"]
#	if [$q compare $pos == end] {
#	    return [$q index "end - 1c"]
#	}
#    }
#    if [regexp -indices {^[ 	]+(.)} [$q get $pos "$pos lineend"] \
#	    dummy index] {
#	return [$q index "$pos + [lindex $index 0] chars"]
#    }
#    return $pos
}

# tkLabelScrollPages --
# This is a utility procedure used in bindings for moving up and down
# pages and possibly extending the selection along the way.  It scrolls
# the view in the widget by the number of pages, and it returns the
# index of the character that is at the same position in the new view
# as the insertion cursor used to be in the old view.
#
# Arguments:
# w -		The text window in which the cursor is to move.
# count -	Number of pages forward to scroll;  may be negative
#		to scroll backwards.

proc tkLabelScrollPages {w count} {
#    set q [winfo command $w]
#    set bbox [$q bbox insert]
#    $q yview scroll $count pages
#    if {$bbox == ""} {
#	return [$q index @[expr [winfo height $w]/2],0]
#    }
#    return [$q index @[lindex $bbox 0],[lindex $bbox 1]]
}

# tkLabelTranspose --
# This procedure implements the "transpose" function for text widgets.
# It tranposes the characters on either side of the insertion cursor,
# unless the cursor is at the end of the line.  In this case it
# transposes the two characters to the left of the cursor.  In either
# case, the cursor ends up to the right of the transposed characters.
#
# Arguments:
# w -		Text window in which to transpose.

proc tkLabelTranspose w {
    set q [winfo command $w]
    set pos [$q index focus insert]
    catch {set text [$q itemcget focus -text]}

    if [string match [string index $text [expr $pos]] \n] {
	incr pos
    }
    set new [string index $text [expr $pos-1]][string index $text [expr $pos-2]]
    if {$pos == 0} {
	return
    }
    $q dchars focus [expr $pos - 2] [$expr $pos - 1]
    $q insert focus insert $new
}

# tk_textCopy --
# This procedure copies the selection from a text widget into the
# clipboard.
#
# Arguments:
# w -		Name of a text widget.

proc tk_textCopy {w} {
    if {[selection own -displayof $w] == "$w"} {
	clipboard clear -displayof $w
	catch {
	    clipboard append -displayof $w [selection get -displayof $w]
	}
    }
}

# tk_textCut --
# This procedure copies the selection from a text widget into the
# clipboard, then deletes the selection (if it exists in the given
# widget).
#
# Arguments:
# w -		Name of a text widget.

proc tk_textCut {w} {
    set q [winfo command $w]
    if {[selection own -displayof $w] == "$w"} {
	clipboard clear -displayof $w
	catch {
	    clipboard append -displayof $w [selection get -displayof $w]
	    if {[$q select item] != ""} {
		$q dchars [$q select item] sel.first sel.last
	    }
	}
    }
}

# tk_textPaste --
# This procedure pastes the contents of the clipboard to the insertion
# point in a text widget.
#
# Arguments:
# w -		Name of a text widget.

proc tk_textPaste w {
    set q [winfo command $w]
    catch {
	$q insert focus insert [selection get -displayof $w \
		-selection CLIPBOARD]
    }
}
