;;; Mouse Settings to make tagnames and filenames "mouseable"

;; Left = This Window;  Middle = Other Window

;; Shift = Tag
(define-key mouse-map x-button-s-left          'x-find-tag-default)
(define-key mouse-map x-button-s-left-up       'x-mouse-ignore)
(define-key mouse-map x-button-s-middle        'x-find-tag-default-other-window)
(define-key mouse-map x-button-s-middle-up     'x-mouse-ignore)
;; Control = File
(define-key mouse-map x-button-c-left          'x-goto-file)
(define-key mouse-map x-button-c-left-up       'x-mouse-ignore)
(define-key mouse-map x-button-c-middle        'x-goto-file-other-window)
(define-key mouse-map x-button-c-middle-up     'x-mouse-ignore)


(autoload 'find-tag-default "tags" "Find potential tag at point.")

(defun x-find-tag (arg)
  (x-mouse-set-point arg)
  (let ((tag (find-tag-default))) 
    (find-tag tag)
    ;; Wait for and discard the button-up key so the message is not flushed.
    (sit-for 1)
    (discard-input)
    (message "Find tag: %s" tag)))


(defun x-find-tag-default (arg)
  (x-mouse-set-point arg)
  (let ((tag (find-tag-default))) 
    (message "Find tag: %s" tag)
    (find-tag tag) ))


(defun x-find-tag-default-other-window (arg)
  (x-mouse-set-point arg)
  (let ((tag (find-tag-default))) 
    (message "Find tag: %s" tag)
    (find-tag-other-window tag) ))


(defun x-goto-file (arg)
  (x-mouse-set-point arg)
  (let ((goto-file-other-window-p nil))
    (goto-file) ) )


(defun x-goto-file-other-window (arg)
  (x-mouse-set-point arg)
  (let ((goto-file-other-window-p t))
    (goto-file) ) )



;;;===== Mouse Command Defuns

(defvar x-auto-mouse-select nil
  "When non-nil, always select the window containing the mouse.")

;;; Redefined from x-mouse.el - dont leave the minibuffer via the mouse
(defun x-mouse-select (arg)
  "Select Emacs window the mouse is on."
  (let ((start-w (selected-window))
	(done nil)
	(w (selected-window))
	(rel-coordinate nil))
    (while (and (not done)
		(null (setq rel-coordinate
			    (coordinates-in-window-p arg w))))
      (setq w (next-window w))
      (if (eq w start-w)
	  (setq done t)))
    ;; Dont allow the user to exit the minibuffer using the mouse.
    (if (and (eq (selected-window) (minibuffer-window))
	     (not (eq w (minibuffer-window))))
	(error ""))
    (select-window w)
    rel-coordinate))


(defun x-scroll-up (arg)
  "Scroll up the window the mouse is over."
  (let ((owin (selected-window)))
    (if (x-mouse-select arg)
	(progn
	  (scroll-up nil)
	  (or (eq owin (selected-window))
	      x-auto-mouse-select
	      (select-window owin))))))


(defun x-scroll-down (arg)
  "Scroll down the window the mouse is over."
  (let ((owin (selected-window)))
    (if (x-mouse-select arg)
	(progn
	  (scroll-down nil)
	  (or (eq owin (selected-window))
	      x-auto-mouse-select
	      (select-window owin))))))


(defun x-line-to-top (arg)
  "Scroll line at the mouse to top of window."
  (let ((owin (selected-window)))
    (if (x-mouse-select arg)
	(progn
	  (save-excursion
	    (x-mouse-set-point arg)
	    (line-to-top-of-window))
	  (or (eq owin (selected-window))
	      x-auto-mouse-select
	      (select-window owin))))))


(defun x-line-to-bottom (arg)
  "Scroll line at the mouse to bottom of window."
  (let ((owin (selected-window)))
    (if (x-mouse-select arg)
	(progn
	  (save-excursion
	    (x-mouse-set-point arg)
	    (line-to-bottom-of-window))
	  (or (eq owin (selected-window))
	      x-auto-mouse-select
	      (select-window owin))))))


(defun x-scroll-up-one (arg)
  "Scroll the window at the mouse one line up."
  (let ((owin (selected-window)))
    (if (x-mouse-select arg)
	(progn
	  (scroll-one-line-up 1)
	  (or (eq owin (selected-window))
	      x-auto-mouse-select
	      (select-window owin))))))


(defun x-scroll-down-one (arg)
  "Scroll the window at the mouse one line up."
  (let ((owin (selected-window)))
    (if (x-mouse-select arg)
	(progn
	  (scroll-one-line-down 1)
	  (or (eq owin (selected-window))
	      x-auto-mouse-select
	      (select-window owin))))))


(defun x-enlarge-window (arg)
  "Select Emacs window mouse is on, then grow it by one line."
  (if (x-mouse-select arg)
      (enlarge-window 1)))


;;; Redefined to blink cursor around region
(defun x-cut-text (arg &optional kill)
  "Copy text between point and mouse position into window system cut buffer.
Save in Emacs kill ring also."
  (if (coordinates-in-window-p arg (selected-window))
      (save-excursion
	(let ((opoint (point))
	      beg end)
	  (x-mouse-set-point arg)
	  (sit-for 1)
	  (setq beg (min opoint (point))
		end (max opoint (point)))
	  (x-store-cut-buffer (buffer-substring beg end))
	  (copy-region-as-kill beg end)
	  (if kill (delete-region beg end))))
    (message "Mouse not in selected window")))


(defun x-cut-sexp (arg &optional kill)
  "Copy sexp starting at mouse into window system cut buffer.
Save in Emacs kill ring also."
  (save-window-excursion
    (x-mouse-select arg)
    (save-excursion
      (x-mouse-set-point arg)
      (let ((beg (point))
	    end)
	(discard-input)
	(sit-for 1)
	(forward-sexp 1)
	(sit-for 1)
	(setq end (point))
	(x-store-cut-buffer (buffer-substring beg end))
	(copy-region-as-kill beg end)
	(if kill (delete-region beg end))
	))))


(defun x-paste-sexp (arg)
  "Copy sexp at mouse into cut buffer and then paste at cursor."
  (x-cut-sexp arg)
  (insert (x-get-cut-buffer)))


(defun x-cut-and-wipe-word (arg)
  "Kill the word at the mouse."
  (x-mouse-set-point arg)
  (let ((beg (point))
	(end (save-excursion (forward-word 1) (point))))
    (x-store-cut-buffer (buffer-substring beg end))
    (copy-region-as-kill beg end)
    (delete-region beg end)))


(defun x-cut-and-wipe-sexp (arg)
  "Kill the sexp at the mouse."
  (x-mouse-set-point arg)
  (let ((beg (point))
	(end (save-excursion (forward-sexp 1) (sit-for 1) (point))))
    (x-store-cut-buffer (buffer-substring beg end))
    (copy-region-as-kill beg end)
    (delete-region beg end)))


(defun x-find-tag (arg)
  (x-mouse-set-point arg)
  (let ((tag (find-tag-default))) 
    (find-tag tag)
    ;; Wait for and discard the button-up key so the message is not flushed.
    (sit-for 1)
    (discard-input)
    (message "Find tag: %s" tag)))


(defun x-find-tag-default (arg)
  (x-mouse-set-point arg)
  (let ((tag (find-tag-default))) 
    (message "Find tag: %s" tag)
    (find-tag tag) ))


(defun x-find-tag-default-other-window (arg)
  (x-mouse-set-point arg)
  (let ((tag (find-tag-default))) 
    (message "Find tag: %s" tag)
    (find-tag-other-window tag) ))


(defun x-goto-file (arg)
  (x-mouse-set-point arg)
  (let ((goto-file-other-window-p nil))
    (goto-file) ) )


(defun x-goto-file-other-window (arg)
  (x-mouse-set-point arg)
  (let ((goto-file-other-window-p t))
    (goto-file) ) )


(defun x-search-forward (arg)
  (x-mouse-set-point arg)
  (skip-chars-forward " \t")
  (let* ((end (progn (forward-sexp 1) (point)))
	 (start (save-excursion (forward-sexp -1) (point)))
	 (string (buffer-substring start end)))
    (search-forward string)))


(defun x-search-backward (arg)
  (x-mouse-set-point arg)
  (skip-chars-forward " \t")
  (let* ((end (progn (forward-sexp 1) (point)))
	 (start (progn (forward-sexp -1) (point)))
	 (string (buffer-substring start end)))
    (search-backward string)))


;; Redefined to prevent clobbering "last-command" which is used by
;; x-search-forward/backward

(defun x-flush-mouse-queue () 
  "Process all queued mouse events."
  ;; A mouse event causes a special character sequence to be given
  ;; as keyboard input.  That runs this function, which process all
  ;; queued mouse events and returns.
  (interactive)
  (while (> (x-mouse-events) 0)
    (x-proc-mouse-event)
    (and (boundp 'x-process-mouse-hook)
	 (symbol-value 'x-process-mouse-hook)
	 (funcall x-process-mouse-hook x-mouse-pos x-mouse-item)))
  
  )


;; the following function may look very much like x-buffer-menu
(defun x-command-history-menu (arg)
  "Pop up a menu of command history for selection with the mouse."
  (let ((menu
         (list "Command History Menu"
               (cons "Select Command"
                     (let ((tail command-history)
                           (prev "^ "); non existent command
                           head)
                       (while tail
                         (let ((elt (car tail)))
                           (if (not (string-match prev
                                                  (prin1-to-string elt)))
                               (setq head (cons
                                           (cons
                                             (setq prev (prin1-to-string elt))
                                            elt)
                                           head))))
                         (setq tail (cdr tail)))
                       (if head (reverse head)
                         (setq head (cons (cons "command-history empty"
                                                (prin1-to-string nil)) head)))
                       )))))
    (eval (x-popup-menu arg menu))))
