;;; class-manual.el
;;; Class Manual Generation
;;;   Code to use hier-mode (for hierarchies output by hier++) and etags++
;;;   functionality to build a class manual using the header comments in 
;;;   the code itself.  The formatting is in standard texinfo.
;;;   This is a hack that has been helpful for us, given our coding 
;;;   guidelines.  It will likely need modification to produce good
;;;   output for you.  (Header comments are those placed between the 
;;;   signature and the body of functions or classes.  If you don't follow
;;;   this convention, then this code will need modification.)

;;; Copyright (C) 1993, Intellection Inc.
;;;
;;; Author: Brian M Kennedy (kennedy@intellection.com)
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 1, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; A copy of the GNU General Public License can be obtained from the
;;; Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;; 92/09     Brian M Kennedy  Original 

(provide 'class-manual)
(require 'hier-mode)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Manual Generation

(defvar class-manual-copyright-string "1992 Intellection, Inc."
  "The desired copyright string to be put on the title page entry 
   generated by hier-insert-header.")


(defun insert-class-manual-header (title)
  "Insert the leading texinfo header necessary to for the other
   insert-class-manual- commands."
  (interactive "sTitle for document: ")
  (insert "\\input texinfo	@c -*-texinfo-*-" ?\n
	  "@comment %**start of header" ?\n
	  "@settitle " title ?\n
	  "@setchapternewpage off" ?\n
	  "@comment %**end of header" ?\n
	  "" ?\n
	  "@titlepage" ?\n
	  "@sp 8" ?\n
	  "@center @titlefont{" title "}" ?\n
	  "" ?\n
	  "@vskip 0pt plus 1filll" ?\n
	  "@center Class Manual generated via hier++ on " (current-time-string) ?\n
	  "@center Copyright @copyright{} " class-manual-copyright-string ?\n
	  "@end titlepage" ?\n)
  )


(defun insert-all-class-manual-entries (only-if-commented-p members-p)
  "For every class in the named hier++ generated hierarchy file, in 
   alphabetic order, call hier-insert-class-entry to insert a manual
   entry about the class into the current buffer."
  (interactive (list (y-or-n-p "Generate entries only if commented? ")
		     (y-or-n-p "Generate entries for class members? ") ))
  (if (not hier-file-name)
      (prompt-for-hier-file-name))
  (let (class-list)
    (save-excursion
      (message "Collecting class names...")
      (find-file hier-file-name)
      (goto-char (point-min))
      (while (search-forward "* " nil t)
	(setq class-list
	      (add-unique-string (buffer-substring (point) 
						   (progn (while (looking-at "\\sw\\|\\s_")
							    (forward-char 1))
							  (point)))
				 class-list) ) ) )
    (insert "@chapter Classes Descriptions" ?\n)
    (while class-list
      (insert-class-manual-entry (car class-list) only-if-commented-p members-p)
      (setq class-list (cdr class-list)) ) ) )


(defun insert-class-manual-entry (class-name only-if-commented-p members-p)
  "Insert manual entry for given class into current buffer at point.
   It will contain the class name, a list of its direct base classes,
   a list of its direct descendants, a description, and a list of 
   entries for each of its members."
  (interactive (list (prompt-for-tag "Insert manual entry for class: ")
		     (y-or-n-p "Generate entries only if commented? ")
		     (y-or-n-p "Generate entries for class members? ") ))
  (if (not hier-file-name)
      (prompt-for-hier-file-name))
  (message "Generating manual entry for %s..." class-name)

  (let (base-list derived-list comments)
    ;; collect class info
    (save-excursion
      (find-file hier-file-name)
      (hier-find class-name)
      (setq base-list (sort (hier-base-list) 'string<))
      (setq derived-list (sort (hier-derived-list) 'string<))
      (if (find-tag-if-present class-name) 
	  (let ((end (save-excursion (re-search-forward "^[^/\n]*\\(;\\|{\\)" nil t) (point))))
	    (while (search-forward "//" end t)
	      (if (looking-at " ") (forward-char 1))
	      (setq comments (concat comments "\n"
				     (buffer-substring (point)
						       (progn (end-of-line) (point)) ) )) ) 
	    ) ) )
    ;; print class info 
    (if (or comments (not only-if-commented-p))
	(progn
	  (insert ?\n ?\n ?\n "@section " class-name ?\n)
  
	  (insert ?\n ?\n "@subsection Immediate Class Hierarchy" ?\n)
	  ;; Base Classes
	  (insert "The class " class-name " has ")
	  (if (not base-list)
	      (insert "no base classes." ?\n)
	    (let ((num (length base-list)))
	      (insert (int-to-string num) 
		      (if (> num 1) " base classes:  " " base class:  ") 
		      (car base-list) ) )
	    (while (setq base-list (cdr base-list))
	      (insert (if (cdr base-list) ", " ", and ") (car base-list)) ) 
	    (insert "." ?\n) )
	  ;; Derived Classes
	  (insert "The class " class-name " has ")
	  (if (not derived-list)
	      (insert "no derived classes." ?\n)
	    (let ((num (length derived-list)))
	      (insert (int-to-string num) 
		      (if (> num 1) " derived classes:  " " derived class:  ") 
		      (car derived-list) ) )
	    (while (setq derived-list (cdr derived-list))
	      (insert (if (cdr derived-list) ", " ", and ") (car derived-list)) ) 
	    (insert "." ?\n) )
	  ;; Comment-based Description
	  (insert ?\n ?\n "@subsection Description" ?\n
		  ?\n (if comments (texify comments) "[No class comments found.]") ?\n)
	  ;; Members
	  (if members-p
	      (let (member-list)
		;; collect members
		(save-excursion
		  (visit-tags-table-buffer)
		  (while (search-forward (concat "\C-a" class-name "::") nil t)
		    (skip-chars-backward "^\C-a")
		    (setq member-list 
			  (add-unique-string (buffer-substring (point)
							       (progn (end-of-line) (point))) 
					     member-list))
		    (forward-line 1) ))
		;; print members
		(insert ?\n ?\n "@subsection Members" ?\n)
		(while member-list
		  (insert-member-manual-entry (car member-list) only-if-commented-p)
		  (setq member-list (cdr member-list)) )
		) )
	  )
      )
    )
  )


(defun insert-member-manual-entry (tag only-if-commented-p)
  "Insert the member description for class member `tag`."
  (interactive (list (prompt-for-tag "Insert manual entry for member: ")))
  (let (entry comments)
    ;; collect info
    (save-excursion
       (if (find-tag-if-present tag) 
	   (let ((start (save-excursion 
			  (re-search-backward "^[ \t\f]*$\\|;\\|{\\|}" nil t) 
			  (forward-line 1)
			  (point)))
		 (h-end (save-excursion 
			  (re-search-forward "^[ \t\f]*$\\|)\\|;\\|^[ \t]*:\\|{\\|}" nil t) 
			  (skip-chars-backward ":{;")
			  (point))) 
		 (c-end (save-excursion 
			  (re-search-forward "^[^/\n]*\\(;\\|{\\|}\\)" nil t) (point))) )
	     (setq entry (buffer-substring start h-end)) 
	     (while (search-forward "//" c-end t)
	       (if (looking-at " ") (forward-char 1))
	       (setq comments (concat comments "\n"
				      (buffer-substring (point)
							(progn (end-of-line) (point)) ) )) )
	     ) ) )
    ;; print info
    (if (or comments (not only-if-commented-p))
	(insert ?\n "@example" ?\n
		(if entry entry "[This tag not found.]") ?\n
		"@end example" ?\n
		?\n (if comments (texify comments) "[No comments found.]") ?\n) )
    )
  )


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Internal Functions

(defun add-unique-string (string list)
  "Add `string` into `list` in alphabetic order, 
   only if `string` is not already in `list`.
   Assumes that `list` is a sorted list of strings."
  (if (or (not list) (string-lessp string (car list)))
      (cons string list)
    (let ((prev list))
      (while (and (cdr prev) (string< (car (cdr prev)) string))
	(setq prev (cdr prev)))
      (if (or (not (cdr prev)) (not (string= (car (cdr prev)) string)))
	  (setcdr prev (cons string (cdr prev))) )
      list)
    )
  )


(defun find-tag-if-present (tagname &optional next other-window)
  "Do find-tag with args.  If no error, return t; otherwise return nil."
  (condition-case signals
      (progn (find-tag tagname next other-window)
	     t)
    (error nil) ) )


(defun texify (string)
  "Return string with TeX special chars @, {, } changed to spaces."
  (let ((pos 0))
    (while pos
      (setq pos (string-match "{\\|}\\|@\\|\\\\" string pos))
      (if pos (aset string pos ? )) ) 
    string) )
