/***********************************************************************
 *								       *
 * Copyright (c) David L. Mills 1999				       *
 *								       *
 * Permission to use, copy, modify, and distribute this software and   *
 * its documentation for any purpose and without fee is hereby	       *
 * granted, provided that the above copyright notice appears in all    *
 * copies and that both the copyright notice and this permission       *
 * notice appear in supporting documentation, and that the name	       *
 * University of Delaware not be used in advertising or publicity      *
 * pertaining to distribution of the software without specific,	       *
 * written prior permission. The University of Delaware makes no       *
 * representations about the suitability this software for any	       *
 * purpose. It is provided "as is" without express or implied          *
 * warranty.							       *
 *								       *
 **********************************************************************/

/*
 * This header file complies with "Pulse-Per-Second API for UNIX-like
 * Operating Systems, Version 1.0", draft-mogul-pps-api-05.txt
 */
#ifndef _SYS_TIMEPPS_H_
#define _SYS_TIMEPPS_H_

#include <sys/ioccom.h>
#include <sys/stropts.h>

#define PPS_API_VERS_1	1	/* API version number */

/*
 * Device/implementation parameters (mode, edge)
 */
#define PPS_CAPTUREASSERT	0x01
#define PPS_CAPTURECLEAR	0x02
#define PPS_CAPTUREBOTH		0x03
#define PPS_OFFSETASSERT	0x10
#define PPS_OFFSETCLEAR		0x20
#define PPS_CANWAIT		0x100
#define PPS_CANPOLL		0x200

/*	
 * Kernel actions (mode)
 */
#define PPS_ECHOASSERT		0x40
#define PPS_ECHOCLEAR		0x80

/*
 * Timestamp formats (mode, tsformat)
 */
#define PPS_TSFMT_TSPEC		0x1000
#define PPS_TSFMT_NTPFP		0x2000

/*
 * Kernel discipline actions (kernel_consumer)
 */
#define PPS_KC_HARDPPS		0
#define PPS_KC_HARDPPS_PLL	1
#define PPS_KC_HARDPPS_FLL	2

/*
 * IOCTL definitions
 */
#ifdef __STDC__
#define PPS_IOC_CREATE		_IO('1', 1)
#define PPS_IOC_DESTROY		_IO('1', 2)
#define PPS_IOC_SETPARAMS	_IOW('1', 3, pps_params_t)
#define PPS_IOC_GETPARAMS	_IOR('1', 4, pps_params_t)
#define PPS_IOC_GETCAP		_IOR('1', 5, int)
#define PPS_IOC_FETCH		_IOR('1', 6, pps_info_t)
#define PPS_IOC_KPCBIND		_IOW('1', 7, int)
#else
#define PPS_IOC_CREATE		_IO(1, 1)
#define PPS_IOC_DESTROY		_IO(1, 2)
#define PPS_IOC_SETPARAMS	_IOW(1, 3, pps_params_t)
#define PPS_IOC_GETPARAMS	_IOR(1, 4, pps_params_t)
#define PPS_IOC_GETCAP		_IOR(1, 5, int)
#define PPS_IOC_FETCH		_IOR(1, 6, pps_info_t)
#define PPS_IOC_KPCBIND		_IOW(1, 7, int)
#endif

#define PPSAPISTR		"ppsapi" /* STREAMS module name */

/*
 * Type definitions
 */
typedef int pps_handle_t;	/* represents a PPS source */	
typedef unsigned pps_seq_t;	/* sequence number (at least 32 bits) */

struct timespec {		/* POSIX timestamp format */
	long tv_sec;		/* seconds */
	long tv_nsec;		/* nanoseconds */
};

typedef struct ntp_fp {		/* NTP timestamp format */
	unsigned int integral;	/* seconds units */
	unsigned int fractional; /* seconds fraction */
} ntp_fp_t;

typedef union pps_timeu {	/* timestamp format */
	struct timespec	tspec;
	ntp_fp_t ntpfp;
	unsigned long longpair[2];
} pps_timeu_t;

/*
 * Timestamp information structure
 */
typedef struct {
	pps_seq_t assert_sequence; /* assert sequence number */
	pps_seq_t clear_sequence; /* clear sequence number */
	pps_timeu_t assert_tu;	/* assert timestamp */
	pps_timeu_t clear_tu;	/* clear timestamp */
	int current_mode;	/* current mode bits */
} pps_info_t;

#define assert_timestamp        assert_tu.tspec
#define clear_timestamp         clear_tu.tspec

#define assert_timestamp_ntpfp  assert_tu.ntpfp
#define clear_timestamp_ntpfp   clear_tu.ntpfp

/*
 * Parameter structure
 */
typedef struct {		/* parameter structure */
	int api_version;	/* API version number */
	int mode;		/* mode bits */
	pps_timeu_t assert_off_tu; /* assert offset */
	pps_timeu_t clear_off_tu; /* clear offset */
} pps_params_t;

#define assert_offset		assert_off_tu.tspec
#define clear_offset		clear_off_tu.tspec

#define assert_offset_ntpfp	assert_off_tu.ntpfp
#define clear_offset_ntpfp	clear_off_tu.ntpfp

#if !defined(KERNEL)

/*
 * Create PPS API
 */
static __inline int
time_pps_create(
	int filedes,
	pps_handle_t *handle)
{
	int error;

	*handle = -1;
	error = ioctl(filedes, I_PUSH, PPSAPISTR);
	if (error < 0) 
		return (-1);
	*handle = filedes;
	return (0);
}

/*
 * Destroy PPS API
 */
static __inline int
time_pps_destroy(
	pps_handle_t handle)
{
	return (ioctl(handle, I_POP, "ppsclock"));
}

/*
 * Set parameters
 */
static __inline int
time_pps_setparams(
	pps_handle_t handle,
	const pps_params_t *ppsparams)
{
	return (ioctl(handle, PPS_IOC_SETPARAMS, ppsparams));
}

/*
 * Get parameters
 */
static __inline int
time_pps_getparams(
	pps_handle_t handle,
	pps_params_t *ppsparams)
{
	return (ioctl(handle, PPS_IOC_GETPARAMS, ppsparams));
}

/*
 * Get capabilities
 */
static __inline int 
time_pps_getcap(
	pps_handle_t handle,
	int *mode)
{
	return (ioctl(handle, PPS_IOC_GETCAP, mode));
}

/*
 * Fetch timestamps
 */
static __inline int
time_pps_fetch(
	pps_handle_t handle,
	const int tsformat,
	pps_info_t *ppsinfobuf,
 	const struct timespec *timeout)
{
	return (ioctl(handle, PPS_IOC_FETCH, ppsinfobuf));
}

/*
 * Define PPS discipline and mode
 */
static __inline int
time_pps_kcbind(
	pps_handle_t handle,
	const int kernel_consumer,
	const int edge,
	const int tsformat)
{
	return (ioctl(handle, PPS_IOC_KPCBIND, &edge));
}

#endif /* !KERNEL */
#endif /* _SYS_TIMEPPS_H_ */
