package WebService::Eulerian::Analytics::Website::Order;

# $Id: Order.pm,v 1.2 2008-09-03 15:22:48 cvscore Exp $

use strict;

use WebService::Eulerian::Analytics();

our @ISA        = qw/ WebService::Eulerian::Analytics /;

=pod

=head1	NAME 

WebService::Eulerian::Analytics::Website::Order - access to the Order service for a given website

=head1 DESCRIPTION

This module allow you to access the Order service holding information about all
the orders generated by the given website.

=head1 SYNOPSIS

	use WebService::Eulerian::Analytics::Website::Order;
	#
	my $api = new WebService::Eulerian::Analytics::Website::Order(
	 apikey	=> 'THE KEY PROVIDED BY YOUR ACCOUNT MANAGER FOR API ACCESS',
	 host	=> 'THE HOST ON WHICH THE API IS HOSTED'
	);

=cut

sub new {
 my $proto      = shift;
 my $class      = ref($proto) || $proto;
 return         $class->SUPER::new(@_, service => 'Website/Order');
}


=pod

=head1 METHODS

=head2 replay : replay an order at a given time and inject it in Eulerian Analytics (in direct access channel) with detailled information.

=head3 input

=over 4

=item * hash reference with the following parameters :

o reference : order reference, mandatory and must be unique for the website

o amount : amount of the order, mandatory and must be a number

o epoch : date at which the order was made, mandatory and the format muse be epoch time

o type : type of the order, it's a string describing the type of the order, it's optionnal (retail, holiday, flight, vacation, etc ...)

o payment : type of the order, it's a string describing the type of the payment, it's optionnal (creditcard, check, wire, etc ...)

=back

=head3 output

=over 4

=item * $order_id if everything is fine, otherwise 0 and check fault method

=back

=head3 sample

	my $order_id = $api->replay( $my_website_id, {
	 reference	=> 'ORDER_REFERENCE_TO_REPLAYED',
	 amount		=> 'ORDER_AMOUNT',
	 epoch		=> 'ORDER_DATE_IN_EPOCH_FORMAT',
	 type		=> 'ORDER_TYPE',
	 payment	=> 'ORDER_PAYMENT',
	});
	#
	if ( $api->fault ) {
	 die $api->faultstring();
	}
	#
	print "Order id following successfull replay : $order_id\n";

=cut

sub replay	{ return shift()->call('replay', @_);	}

=pod

=head1 METHODS

=head2 search : generate a search method on the order database.

=head3 input

=over 4

=item * id of the website

=item * hash reference with the following parameters :

o order-from-epoch : from epoch date

o order-to-epoch : to epoch date

o order-from : from date (dd/mm/yyyy)

o order-to : to date (dd/mm/yyyy)

o order-ref : order reference

=back

=head3 output

=over 4

=item * total : total number of orders for given search

=item * result : array ref of hash ref for each order

=back

=head3 sample

	my $rh_order = $api->search( $my_website_id, {
	 'order-ref'	=> 'XXXXXXX'
	});
	#
	if ( $api->fault ) {
	 die $api->faultstring();
	}
	#
	for ( @{ $rh_order->{result} } ) {
	 print "Order ref : ".$_->{order_ref}." at ".localtime($_->{order_date})."\n";
	}

=cut

sub search	{ return shift()->call('search', @_);	}

=pod

=head1 METHODS

=head2 valid : validate a list of orders

=head3 input

=over 4

=item * id of the website

=item * array of order references to validate

=back

=head3 output

=over 4

=item * totalerror : total number of errors

=item * totaltotreat : total number of reference to treat

=item * totaltreated : total number of treated references

=item * result : array ref of hash ref for each order

=back

=head3 sample

	my $rh_order = $api->valid( $my_website_id, 'XXXXX', 'YYYYY');
	#
	if ( $api->fault ) {
	 die $api->faultstring();
	}
	#
	for ( @{ $rh_order->{result} } ) {
	 print "Order ref : ".$_->{order_ref}." at ".localtime($_->{order_date})."\n";
	}

=cut

sub valid	{ return shift()->call('valid', @_);	}

=pod

=head1 METHODS

=head2 cancel : cancel a list of orders

=head3 input

=over 4

=item * id of the website

=item * array of order references to cancel 

=back

=head3 output

=over 4

=item * totalerror : total number of errors

=item * totaltotreat : total number of reference to treat

=item * totaltreated : total number of treated references

=item * result : array ref of hash ref for each order

=back

=head3 sample

	my $rh_order = $api->cancel( $my_website_id, 'XXXXX', 'YYYYY');
	#
	if ( $api->fault ) {
	 die $api->faultstring();
	}
	#
	for ( @{ $rh_order->{result} } ) {
	 print "Order ref : ".$_->{order_ref}." at ".localtime($_->{order_date})."\n";
	}

=cut

sub cancel	{ return shift()->call('cancel', @_);	}

=pod

=head1 SEE ALSO

L<WebService::Eulerian::Analytics|WebService::Eulerian::Analytics>

=head1 AUTHOR

Mathieu Jondet <mathieu@eulerian.com>

=head1 COPYRIGHT

Copyright (c) 2008 Eulerian Technologies Ltd L<http://www.eulerian.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

=cut

1;
__END__
