#!/usr/bin/perl
#-------------------------------------------------------------------------------
# Encode a Unicode string in Perl and decode it in Java
# Philip R Brenan at gmail dot com, Appa Apps Ltd, 2017
#-------------------------------------------------------------------------------

package Encode::Unicode::PerlDecodeJava;
require v5.16.0;
use warnings FATAL => qw(all);
use strict;
use Carp;

our $VERSION = '2017.224';

sub encode93($)                                                                   # Encode a string
 {my ($i) = @_;
  my $s;
  my $n = length($i);
  for(split //, $i)
   {$s .=  /[a-z '\(\)\[\]\{\}<>~`!@#\$%^&*_\-+=:;|.?\/]/i ? $_ : ord($_).',';
   }
  $s
 }

sub decode93($)                                                                   # Decode a string
 {my ($i) = @_;
  my $s;
  my $n = '';
  for(split //, $i)
   {if (   /[a-z '\(\)\[\]\{\}<>~`!@#\$%^&*_\-+=:;|.?\/]/i) {$s .= $_}
    elsif (/,/i)  {$s .= pack('U', $n); $n = ''}
    else          {$n .= $_}
   }
  $s
 }

#-------------------------------------------------------------------------------
# Test
#-------------------------------------------------------------------------------

sub test
 {eval join('', <Encode::Unicode::PerlDecodeJava::DATA>) || die $@
 }

test unless caller();

# Documentation
#extractDocumentation unless caller;

#-------------------------------------------------------------------------------
# Export
#-------------------------------------------------------------------------------

require Exporter;

use vars qw(@ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

@ISA          = qw(Exporter);
@EXPORT       = qw(decode93 encode93);
@EXPORT_OK    = qw();
%EXPORT_TAGS  = (all=>[@EXPORT, @EXPORT_OK]);

1;

=pod

=encoding utf-8

=head1 Name

Encode::Unicode::PerlDecodeJava - Encode a Unicode string in Perl and decode it in Java

=head1 Synopsis

 use Encode::Unicode::PerlDecodeJava;

 ok $_ eq decode93(encode93($_)) for(qw(aaa (𝝰𝝱𝝲) aaa𝝰𝝱𝝲aaa yüz))

=head1 Description

 encode93($input)

encodes any Perl string given as $input, even one containing Unicode
characters, using only the 93 well known ASCII characters below:

 abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ
 0123456789 '()[]{}<>~`!@#$%^&*_-+=:;|.,?\

and returns the resulting encoded string.

Such a string can be easily compressed and transported using software
restricted to ASCII data and then reconstituted as a Unicode string in Perl by
using decode93() or in Java by using the code reproduced further below.

 decode93($input)

takes an $input string encoded by encode93() and returns the decoded string.

The following Java code takes a string encoded by encode93() and returns the
decoded string to Java:

  String decode(String input)                                                   // Decode string
   {final StringBuilder s = new StringBuilder();
    final StringBuilder n = new StringBuilder();
    final int N           = input.length();

    for(int i = 0; i < N; ++i)
     {final char c = input.charAt(i);
      if (Character.isDigit(c)) n.append(c);
      else if (c == ',')
       {final int p = Integer.parseInt(n.toString());
        s.appendCodePoint(p);
        n.setLength(0);
       }
      else s.append(c);
     }

    return s.toString();
   }

=head1 Installation

Standard Module::Build process for building and installing modules:

  perl Build.PL
  ./Build
  ./Build test
  ./Build install

=head1 Author

philiprbrenan@gmail.com

http://www.appaapps.com

=head1 Copyright

Copyright (c) 2017 Philip R Brenan.

This module is free software. It may be used, redistributed and/or modified
under the same terms as Perl itself.

=cut

__DATA__
use Test::More tests=>4;
ok $_ eq decode93(encode93($_)) for(qw(aaa (𝝰𝝱𝝲) aaa𝝰𝝱𝝲aaa yüz));

1
