use strict;
use warnings;
use Test::More;
use Test::Moose;
use lib './lib';
chdir '../../../../../' if !-d '_build';
die if !-d '_build';
use_ok 'Net::BitTorrent::Torrent';
my $torrent = new_ok('Net::BitTorrent::Torrent',
                     [path =>
                          't/90000_data/95000_torrents/OOo_3.2.0_Win32Intel_install_en-US.exe.torrent'
                     ]
);
require MIME::Base64;
isa_ok($torrent->info_hash, 'Bit::Vector',
       'Infohash is a Bit::Vector object');
is($torrent->info_hash->to_Hex,
    '0630481FA0023579BF37EDDAD3FF4AB54BAB9567',
    'Infohash is 0630481F...');
is_deeply(
    $torrent->metadata,
    {announce        => 'http://torrent.projects.ooodev.org:6969/announce',
     'announce-list' => [
                  ['http://borft.student.utwente.nl:6969/announce',
                   'http://core-tracker.enlist-a-distro.net:9800/announce',
                   'http://core-tracker.depthstrike.com:9800/announce',
                   'http://clients-tracker.enlist-a-distro.net:9800/announce',
                   'http://clients-tracker.depthstrike.com:9800/announce',
                   'http://torrent.projects.ooodev.org:6969/announce'
                  ]
     ],
     azureus_properties => {dht_backup_enable => 1},
     "creation date"    => 1265821992,
     info               => {
         length         => 141933136,
         name           => 'OOo_3.2.0_Win32Intel_install_en-US.exe',
         "piece length" => 131072,
         pieces =>
             MIME::Base64::decode(
             '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'
             )
     },
     nodes => [['router.bitcomet.com',   554],
               ['router.bittorrent.com', 6881],
               ['router.utorrent.com',   6881],
     ],
     sources => [
         'http://ftp.osuosl.org/pub/openoffice/stable/3.2.0/OOo_3.2.0_Win32Intel_install_en-US.exe',
         'http://openoffice.mirrors.tds.net/pub/openoffice/stable/3.2.0/OOo_3.2.0_Win32Intel_install_en-US.exe',
         'http://vesta.informatik.rwth-aachen.de/ftp/pub/mirror/OpenOffice/stable/3.2.0/OOo_3.2.0_Win32Intel_install_en-US.exe',
         'http://openoffice.virginmedia.com/stable/3.2.0/OOo_3.2.0_Win32Intel_install_en-US.exe',
         'ftp://ftp.kr.freebsd.org/pub/openoffice/stable/3.2.0/OOo_3.2.0_Win32Intel_install_en-US.exe'
     ]
    }
);

#use Data::Dumper;
#warn Dumper $torrent->metadata;
#meta_ok($class_or_obj, "... Foo has a ->meta");
#does_ok($class_or_obj, $role, "... Foo does the Baz role");
#has_attribute_ok($class_or_obj, $attr_name, "... Foo has the 'bar' attribute");
done_testing;

=pod

=head1 Author

Sanko Robinson <sanko@cpan.org> - http://sankorobinson.com/

CPAN ID: SANKO

=head1 License and Legal

Copyright (C) 2008-2010 by Sanko Robinson <sanko@cpan.org>

This program is free software; you can redistribute it and/or modify it under
the terms of
L<The Artistic License 2.0|http://www.perlfoundation.org/artistic_license_2_0>.
See the F<LICENSE> file included with this distribution or
L<notes on the Artistic License 2.0|http://www.perlfoundation.org/artistic_2_0_notes>
for clarification.

When separated from the distribution, all original POD documentation is
covered by the
L<Creative Commons Attribution-Share Alike 3.0 License|http://creativecommons.org/licenses/by-sa/3.0/us/legalcode>.
See the
L<clarification of the CCA-SA3.0|http://creativecommons.org/licenses/by-sa/3.0/us/>.

Neither this module nor the L<Author|/Author> is affiliated with BitTorrent,
Inc.

=for rcs $Id: File.t 0c58774 2010-06-15 16:14:08Z sanko@cpan.org $

=cut
