/*
     Filter - Packet filtering software for the Drawbridge package
     Copyright (C) 1993 David K. Hess, Douglas Lee Schales, David R. Safford

     Please see the file `COPYING' for the complete copyright notice.

     FITLER.H - Version 1.0 - 4/21/93
*/

#define YES           1
#define NO            0

#define NUMINTERFACES          2
#define VERSION                "1.1"

#define INSIDE_MASK            0x1
#define OUTSIDE_MASK           0x2

// The maximum number of networks must always be a power of two due
//   to the hash function.
#define MAX_NUM_NETWORKS       32
#define NETWORK_HASH_MASK      0x00F80000UL

// The maximum number of ethernet address entries must also always
//   be a power of two.
#define MAX_NUM_ETHERS         8192
#define ETHERNET_HASH_MASK     0x1FFF

#define MAX_NUM_ACCESS_LISTS   256
#define MAX_NUM_ACCESS_RANGES  32
#define MAX_NUM_REJECT_ENTRIES 32
#define MAX_NUM_ALLOW_ENTRIES  8
#define MAX_NUM_NEW_NETWORKS   8

#define NETWORK_EXTENSION      "net"
#define ACCESS_LIST_FILE       "class.tbl"
#define REJECT_LIST_FILE       "reject.tbl"
#define ALLOW_LIST_FILE        "allow.tbl"
#define DES_KEY_FILE           "des.key"

// Ethernet protocols. Byte reversed of course.
#define FILTERPROT    0x01A0
#define IPPROT        0x0008

// Ip protocols.
#define ICMPPROT      1
#define TCPPROT       6
#define UDPPROT       17

// Constants for TCP
#define TH_FIN  0x01
#define TH_SYN  0x02
#define TH_RST  0x04
#define TH_PUSH 0x08
#define TH_ACK  0x10
#define TH_URG  0x20

// Constants defining the cards installed in the PC.
#define CAMPUSPORT    0x280
#define CAMPUSMEM     0x0D000
#define INTERNETPORT  0x300
#define INTERNETMEM   0x0CC00
#define DATAPATH      16

// Class macros.
#define IN_CLASSA(i)  (((unsigned long)(i) & 0x80000000UL) == 0)
#define IN_CLASSB(i)  (((unsigned long)(i) & 0xC0000000UL) == 0x80000000UL)
#define IN_CLASSC(i)  (((unsigned long)(i) & 0xE0000000UL) == 0xC0000000UL)
#define IN_CLASSD(i)  (((unsigned long)(i) & 0xF0000000UL) == 0xE0000000UL)

// Class Constants
#define CLASSB_HOST      0x0000FFFFUL
#define CLASSC_HOST      0x000000FFUL

#define CLASSB_NETWORK   0xFFFF0000UL
#define CLASSC_NETWORK   0xFFFFFF00UL

// This is a kludge. I need to look at the offset field in the ip header
//   but it is bit fielded with some flags. However since this stupid
//   architecture is byte reversed, I can't define a bit field structure
//   in C so that I can easily get to it. I always end up with two
//   pieces of it that I have to put back together. Luckily since
//   I don't need to know the value of the offset, just if it is
//   0 I can mask off the flag bits and do an int compare to 0.
#define IP_OFF_MASK      0xFF1F

// Message types.
#define FM_M_SYNC        0
#define FM_M_SYNCACK     1
#define FM_M_WRITE       2
#define FM_M_WRITEACK    3
#define FM_M_REBOOT      4
#define FM_M_REBOOTACK   5
#define FM_M_NEWKEY      6
#define FM_M_NEWKEYACK   7
#define FM_M_QUERY       8
#define FM_M_QUERYACK    9
#define FM_M_LOAD        10
#define FM_M_LOADACK     11
#define FM_M_ERROR       12
#define FM_M_RELEASE     13
#define FM_M_RELEASEACK  14

// Message flags.
#define FM_F_CRYPTED      0x01

// Miscellaneous constants used by the different messages.
#define FM_QUERY_NETWORK  0
#define FM_QUERY_HOST     1
#define FM_QUERY_REJECT   2
#define FM_QUERY_ALLOW    3
#define FM_QUERY_CLASS    4

#define FM_LOAD_NETWORK       0
#define FM_LOAD_REJECT        1
#define FM_LOAD_ALLOW         2
#define FM_LOAD_CLASS         3

#define FM_RELEASE_CLASSES    0
#define FM_RELEASE_REJECT     1
#define FM_RELEASE_ALLOW      2
#define FM_RELEASE_NETWORK    3

#define FM_LOAD_FLAGS_BEGIN  0x01
#define FM_LOAD_FLAGS_END    0x02

#define FM_ERROR_INSECURE        0
#define FM_ERROR_SECURE          1
#define FM_ERROR_DESFILE         2
#define FM_ERROR_DESWRITE        3
#define FM_ERROR_NONETWORK       4
#define FM_ERROR_NOMEMORY        5
#define FM_ERROR_COMMAND         6
#define FM_ERROR_DATAFILE        7
#define FM_ERROR_DATAWRITE       8
#define FM_ERROR_INVALIDNETWORK  9
#define FM_ERROR_LOADBUFFER     10
#define FM_ERROR_NODES          11

typedef struct _WDPacketHeader {
    unsigned char status;
    unsigned char next;
    unsigned short length;
} WDPacketHeader;

typedef union _EtherAddress {
	unsigned char bytes[6];
	unsigned short words[3];
	struct {
	    unsigned char bytes[6];
	} addr;
} EtherAddress;

typedef struct _WDInterface {
    unsigned short ioAddress;
    unsigned char *bufferAddress;
    EtherAddress etherAddress;
    int cardID;
    int numPages;
    int currBuffer;
    int transmitting;
    int lengthBuf1;
    int lengthBuf2;
    int packetWaiting;
    int packetGood;
    int packetLength;
    unsigned char packet[1536];
} WDInterface;

typedef struct _EtherHeader {
    EtherAddress etherDestHost;
    EtherAddress etherSrcHost;
    unsigned short etherType;
} EtherHeader;

// Internet address structure.
typedef union _in_addr {
    struct {
	unsigned char s_b1;
	unsigned char s_b2;
	unsigned char s_b3;
	unsigned char s_b4;
    } S_un_b;
    struct {
	unsigned short s_w1;
	unsigned short s_w2;
    } S_un_w;
    unsigned long S_addr;
} in_addr;

typedef struct _IpHeader {
    unsigned int   ip_hl:4,                /* header length */
		   ip_v:4;                 /* version */
    unsigned char  ip_tos;                 /* type of service */
    short ip_len;                          /* total length */
    unsigned short ip_id;                  /* identification */
    short          ip_off;                 /* fragment offset field */
    unsigned char  ip_ttl;                 /* time to live */
    unsigned char  ip_p;                   /* protocol */
    unsigned short ip_sum;                 /* checksum */
    in_addr ip_src,ip_dst;                 /* source and dest address */
} IpHeader;

typedef unsigned long tcp_seq;

typedef struct _TcpHeader {
    unsigned short th_sport;               /* source port */
    unsigned short th_dport;               /* destination port */
    tcp_seq th_seq;                        /* sequence number */
    tcp_seq th_ack;                        /* acknowledgement number */
    unsigned int   th_x2:4,                /* (unused) */
		   th_off:4;               /* data offset */
    unsigned char  th_flags;
    unsigned short th_win;                 /* window */
    unsigned short th_sum;                 /* checksum */
    unsigned short th_urp;                 /* urgent pointer */
} TcpHeader;

typedef struct _UdpHeader {
    unsigned short uh_sport;               /* source port */
    unsigned short uh_dport;               /* destination port */
    short   uh_ulen;                       /* udp length */
    unsigned short uh_sum;                 /* udp checksum */
} UdpHeader;

//
// Structure for an address table entry.
//
typedef struct _AddrTableEntry {
    in_addr network;
    unsigned char dirty;
    unsigned char *hostTable;
    unsigned char *realHostTable;
} AddrTableEntry;

typedef struct _AccessListTableEntry {
    unsigned short begin;
    unsigned short end;
} AccessListTableEntry;

typedef struct _RejectTableEntry {
    in_addr network;
    unsigned long mask;
} RejectTableEntry;

typedef struct _EtherHashEntry {
    EtherAddress address;
    unsigned char interface;
    unsigned char reserved;
} EtherHashEntry;

typedef struct _AllowTableEntry {
    in_addr network;
    unsigned long mask;
    AccessListTableEntry access[MAX_NUM_ACCESS_RANGES];
} AllowTableEntry;

typedef struct _DESauth {
    unsigned long fmSeq;
    unsigned long filterSeq;
} DESauth;

//
// Structure representing a filter protocol header.
//
typedef struct _FiltHeader {
    unsigned char   type;
    unsigned char   flags;
    DESauth auth;
} FiltHeader;

//
// Structure representing a filter protocol message.
//
typedef struct _QueryPacket {
    unsigned char type;
    unsigned char dummy[3];
    union {
	unsigned char index;
	in_addr addr;
    } queryValue;
    union {
	in_addr networks[MAX_NUM_NETWORKS];
	unsigned char index;
	RejectTableEntry reject[MAX_NUM_REJECT_ENTRIES];
	AllowTableEntry allow[MAX_NUM_ALLOW_ENTRIES];
	struct {
	    AccessListTableEntry in[MAX_NUM_ACCESS_RANGES];
	    AccessListTableEntry out[MAX_NUM_ACCESS_RANGES];
	    AccessListTableEntry src[MAX_NUM_ACCESS_RANGES];
	    AccessListTableEntry udp[MAX_NUM_ACCESS_RANGES];
	} accessList;
    } queryResult;
} QueryPacket;

typedef struct _LoadPacket {
    unsigned char type;
    unsigned char flags;
    unsigned char dummy[2];
    union {
	unsigned char index;
	struct {
	    in_addr network;
	    unsigned long offset;
	} networkBlock;
    } loadValue;
    union {
	RejectTableEntry reject[MAX_NUM_REJECT_ENTRIES];
	AllowTableEntry allow[MAX_NUM_ALLOW_ENTRIES];
	struct {
	    AccessListTableEntry in[MAX_NUM_ACCESS_RANGES];
	    AccessListTableEntry out[MAX_NUM_ACCESS_RANGES];
	    AccessListTableEntry src[MAX_NUM_ACCESS_RANGES];
	    AccessListTableEntry udp[MAX_NUM_ACCESS_RANGES];
	} accessList;
	unsigned char networkBlock[1024];
    } loadData;
} LoadPacket;

typedef struct _ReleasePacket {
    unsigned char type;
    unsigned char dummy[3];
    in_addr network;
} ReleasePacket;

typedef struct _ErrorPacket {
    unsigned char errorCode;
} ErrorPacket;

// From MESSAGE.C
void buildPacket(
	int side,
	EtherAddress *toAddr,
	unsigned char *data,
	int length,
	int type,
	unsigned long *theFmSeq,
	unsigned long *theFilterSeq);
int installNewKey(unsigned char *newKey);
void handleReboot(EtherHeader *etherHead,FiltHeader *filtHead,int from);
void handleNewkey(EtherHeader *etherHead,FiltHeader *filtHead,int from);
void handleQuery(EtherHeader *etherHead,FiltHeader *filtHead,int from);
void handleLoad(EtherHeader *etherHead,FiltHeader *filtHead,int from);
void handleWrite(EtherHeader *etherHead,FiltHeader *filtHead,int from);
void handleRelease(EtherHeader *etherHead,FiltHeader *filtHead,int from);
void filtMessage(EtherHeader *etherHead,int from);

// From WD.C
WDInterface *WDDefine(unsigned short ioAddress,unsigned char *bufferAddress,int dataPath);
unsigned char *WDGetReadAccess(WDInterface *interface);
void WDFree(WDInterface *interface,unsigned char next);
unsigned char WDIsBusy(WDInterface *interface);
void WDWrite(WDInterface *interface,unsigned short length);
void WDMove(
	int length,
	unsigned char *fromBuffer,
	WDInterface *fromInterface,
	unsigned char *toBuffer);
void WDCopy(
	int length,
	unsigned char *fromBuffer,
	WDInterface *fromInterface,
	WDInterface *toInterface,
	int bufnum);

// From FILTER.C
int checkIncomingTcp(
	in_addr srcAddr,
	in_addr dstAddr,
	unsigned short srcPort,
	unsigned short dstPort);
int checkOutgoingTcp(
	in_addr srcAddr,
	in_addr dstAddr,
	unsigned short srcPort,
	unsigned short dstPort);
int checkIncomingUdp(
	in_addr srcAddr,
	in_addr dstAddr,
	unsigned short srcPort,
	unsigned short dstPort);
int checkOutgoingUdp(
	in_addr srcAddr,
	in_addr dstAddr,
	unsigned short srcPort,
	unsigned short dstPort);
int checkIncomingPacket(unsigned char *packet,int length,WDInterface *interface);
int checkOutgoingPacket(unsigned char *packet,int length,WDInterface *interface);
int bridge(
	WDInterface *fromInterface,
	EtherHeader *etherHead);
void interfaceUpdate(WDInterface *interface);
void interfaceLoad(
	WDInterface *toInterface,
	WDInterface *fromInterface,
	unsigned char *packet,
	unsigned short length,
	unsigned char next);
void bufferLoad(WDInterface *toInterface);
void forward(
	WDInterface *fromInterface,
	WDInterface *toInterface,
	int (*checkFunction)(unsigned char *packet,int length,WDInterface *interface));
void initMemory(void);
void initTables(void);
void initNetworks(void);
void init(void);
void usage(void);
int main(int argc,char *argv[]);

// From CRYPT.C
void encrypt(unsigned char *cipher,unsigned long *method,unsigned char *plain);
void decrypt(unsigned char *plain,unsigned long *method,unsigned char *cipher);

// Global variables defined in FILTER.C and need to be visible.
extern AddrTableEntry addrTable[MAX_NUM_NETWORKS];
extern AccessListTableEntry *in;
extern AccessListTableEntry *out;
extern AccessListTableEntry *source;
extern AccessListTableEntry *udp;
extern RejectTableEntry rejectTable[MAX_NUM_REJECT_ENTRIES];
extern AllowTableEntry allowTable[MAX_NUM_ALLOW_ENTRIES];

extern AddrTableEntry newAddrTable[MAX_NUM_NEW_NETWORKS];
extern AccessListTableEntry *newIn;
extern AccessListTableEntry *newOut;
extern AccessListTableEntry *newSource;
extern AccessListTableEntry *newUdp;

extern int accessTableDirty;
extern int rejectTableDirty;
extern int allowTableDirty;

extern unsigned char desKey[128];
extern int desKeyLoaded;

extern unsigned char rebootRequested;

extern WDInterface *campus;
extern WDInterface *internet;


#define moveWords(length,to,from) \
{ \
    unsigned short toSeg = FP_SEG(to); \
    unsigned short toOff = FP_OFF(to); \
    unsigned short fromSeg = FP_SEG(from); \
    unsigned short fromOff = FP_OFF(from); \
    unsigned short savedDS; \
    unsigned short savedES; \
 \
    /* Save off the segment registers. */\
    asm mov savedDS,DS; \
    asm mov savedES,ES; \
 \
    /* Perform the move. */\
    asm mov DS,fromSeg; \
    asm mov SI,fromOff; \
    asm mov ES,toSeg; \
    asm mov DI,toOff; \
    asm mov CX,length; \
    asm rep movsw; \
 \
    /* Restore the segment registers. */\
    asm mov DS,savedDS; \
    asm mov ES,savedES; \
 \
    /*fprintf(stdout,"Done\n"); */\
}

#define SWAP_LONG(w) { \
    asm mov ax,WORD PTR w; \
    asm mov bx,WORD PTR (w + 2); \
    asm xchg al,ah; \
    asm xchg bl,bh; \
    asm mov WORD PTR w,bx; \
    asm mov WORD PTR (w + 2),ax; \
}

#define SWAP_WORD(w) { \
    asm mov ax,w; \
    asm xchg al,ah; \
    asm mov w,ax; \
}



