// 
//   Copyright (C) 2005, 2006, 2007 Free Software Foundation, Inc.
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 3 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

// 
//

#ifndef GNASH_AS_PROP_FLAGS_H
#define GNASH_AS_PROP_FLAGS_H

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

namespace gnash {

/// Flags defining the level of protection of a member
class as_prop_flags
{

	/// Numeric flags
	int _flags;

	/// if true, this value is protected (internal to gnash)
	bool _protected;

public:

	/// Actual flags
	enum Flags {

		/// Protect from enumeration
		dontEnum	= 1 << 0,

		/// Protect from deletion
		dontDelete	= 1 << 1,

		/// Protect from assigning a value
		readOnly	= 1 << 2
	};

	/// mask for flags
	//
	/// TODO: make private.
	///       Currently used by as_global_assetpropflags in Global.cpp
	///
	static const int as_prop_flags_mask = dontEnum|dontDelete|readOnly;


	/// Default constructor
	as_prop_flags() : _flags(0), _protected(false)
	{
	}

	/// Constructor
	as_prop_flags(const bool read_only, const bool dont_delete, const bool dont_enum)
		:
		_flags(((read_only) ? readOnly : 0) |
				((dont_delete) ? dontDelete : 0) |
				((dont_enum) ? dontEnum : 0)),
		_protected(false)
	{
	}

	/// Constructor, from numerical value
	as_prop_flags(const int flags)
		: _flags(flags), _protected(false)
	{
	}

	bool operator== (const as_prop_flags& o) const
	{
		return ( _flags == o._flags ) && ( _protected == o._protected );
	}

	bool operator!= (const as_prop_flags& o) const
	{
		return ( _flags != o._flags ) || ( _protected != o._protected );
	}

	/// Get "read-only" flag 
	bool get_read_only() const { return (((_flags & readOnly)!=0)?true:false); }

	/// Set "read-only" flag 
	void set_read_only() { _flags |= readOnly; }

	/// Clear "read-only" flag 
	void clear_read_only() { _flags &= ~readOnly; }

	/// Get "don't delete" flag
	bool get_dont_delete() const { return (((_flags & dontDelete)!=0)?true:false); }

	/// Set "don't delete" flag
	void set_dont_delete() { _flags |= dontDelete; }

	/// Clear "don't delete" flag 
	void clear_dont_delete() { _flags &= ~dontDelete; }

	/// Get "don't enum" flag
	bool get_dont_enum() const { return (((_flags & dontEnum)!=0)?true:false);	}

	/// Set "don't enum" flag
	void set_dont_enum() { _flags |= dontEnum; }

	/// Clear "don't enum" flag 
	void clear_dont_enum() { _flags &= ~dontEnum; }

	/// accesor to the numerical flags value
	int get_flags() const { return _flags; }

	/// Get "protected" flag
	bool get_is_protected() const { return _protected; }

	/// Set "protected" flag
	//
	/// @@ why isn't this a bitflag like the others ?
	///
	void set_is_protected(const bool is_protected) { _protected = is_protected; }

	/// set the numerical flags value (return the new value )
	/// If unlocked is false, you cannot un-protect from over-write,
	/// you cannot un-protect from deletion and you cannot
	/// un-hide from the for..in loop construct
	///
	/// @param setTrue
	///	the set of flags to set
	///
	/// @param setFalse
	///	the set of flags to clear
	///
	/// @return true on success, false on failure (is protected)
	///
	bool set_flags(const int setTrue, const int setFalse = 0)
	{
		if (get_is_protected()) return false;

		_flags &= ~setFalse;
		_flags |= setTrue;

		return true;
	}
};



} // namespace gnash

#endif // GNASH_AS_PROP_FLAGS_H
