/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*   (c) 1993.  The Regents of the University of California.  All rights   */
/*   reserved.                                                             */
/*                                                                         */
/*   This work was produced at the University of California, Lawrence      */
/*   Livermore National Laboratory (UC LLNL) under contract no.            */
/*   W-7405-ENG-48 (Contract 48) between the U.S. Department of Energy     */
/*   (DOE) and The Regents of the University of California (University)    */
/*   for the operation of UC LLNL.  Copyright is reserved to the           */
/*   University for purposes of controlled dissemination,                  */
/*   commercialization through formal licensing, or other disposition      */
/*   under terms of Contract 48; DOE policies, regulations and orders;     */
/*   and U.S. statutes.  The rights of the Federal Government are          */
/*   reserved under Contract 48 subject to the restrictions agreed upon    */
/*   by the DOE and University.                                            */
/*                                                                         */
/*                                                                         */
/*                              DISCLAIMER                                 */
/*                                                                         */
/*   This software was prepared as an account of work sponsored by an      */
/*   agency of the United States Government.  Neither the United States    */
/*   Government nor the University of California nor any of their          */
/*   employees, makes any warranty, express or implied, or assumes any     */
/*   liability or responsibility for the accuracy, completeness, or        */
/*   usefulness of any information, apparatus, product, or process         */
/*   disclosed, or represents that its specific commercial products,       */
/*   process, or service by trade name, trademark, manufacturer, or        */
/*   otherwise, does not necessarily constitute or imply its               */
/*   endorsement, recommendation, or favoring by the United States         */
/*   Government or the University of California. The views and opinions    */
/*   of the authors expressed herein do not necessarily state or reflect   */
/*   those of the United States Government or the University of            */
/*   California, and shall not be used for advertising or product          */
/*   endorsement purposes.                                                 */
/*                                                                         */
/*   Permission to use, copy, modify and distribute this software and its  */
/*   documentation for any non-commercial purpose, without fee, is         */
/*   hereby granted, provided that the above copyright notice and this     */
/*   permission notice appear in all copies of the software and            */
/*   supporting documentation, and that all UC LLNL identification in      */
/*   the user interface remain unchanged.  The title to copyright LLNL     */
/*   XFTP shall at all times remain with The Regents of the University     */
/*   of California and users agree to preserve same. Users seeking the     */
/*   right to make derivative works with LLNL XFTP for commercial          */
/*   purposes may obtain a license from the Lawrence Livermore National    */
/*   Laboratory's Technology Transfer Office, P.O. Box 808, L-795,         */
/*   Livermore, CA 94550.                                                  */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/

#include <stdio.h>
#include <Xm/Xm.h>
#include <Xm/SelectioB.h>
#include <Xm/Label.h>
#include <Xm/Form.h>
#include <Xm/Scale.h>
#include "xftp.h"
#include "xfer.h"

static struct {
	Widget w_dialog;
	Widget w_controlArea;
	Widget w_nameLabel;
	Widget w_name;
	Widget w_lengthLabel;
	Widget w_length;
	Widget w_bytesMovedLabel;
	Widget w_bytesMoved;
	Widget w_scale;
} monitor;

extern Widget w_toplev;
extern struct xfer_ctrl_block xfer_ctrl;
extern Display *display;

void cb_monitor_close();
void cb_monitor_abort();


/*
 * show_monitor - Pop up file transfer monitor.
 */
show_monitor()
{
    create_monitor_dialog();
	XtManageChild(monitor.w_dialog);
	add_dialog_to_list(monitor.w_dialog);
	force_update(monitor.w_dialog);
}


/*
 * create_monitor_dialog - Create file transfer monitor.
 */
create_monitor_dialog()
{
	static int initialized = False;
	XmString label;
	Widget widget;

    /* Create dialog only once */
    if (initialized)
        return;
    initialized = True;

    /* Create prompt dialog for connection to remote host */
    monitor.w_dialog = XmCreatePromptDialog(w_toplev, "monitor", NULL, 0);
    XtAddCallback(monitor.w_dialog, XmNokCallback, cb_monitor_abort, NULL);

	/* Don't need all the prompt dialog's widgets */
	widget = XmSelectionBoxGetChild(monitor.w_dialog, XmDIALOG_SELECTION_LABEL);
	XtUnmanageChild(widget);
	widget = XmSelectionBoxGetChild(monitor.w_dialog, XmDIALOG_TEXT);
	XtUnmanageChild(widget);
	widget = XmSelectionBoxGetChild(monitor.w_dialog, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(widget);
	widget = XmSelectionBoxGetChild(monitor.w_dialog, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(widget);

	/* Add callback for the WM_DELETE_WINDOW protocol */
	add_wm_delete_window_cb(monitor.w_dialog, cb_monitor_close, NULL);

	/* Kludge to add title bar under OLWM */
	AddOLWMDialogFrame(monitor.w_dialog);

	/* Create form for control area */
    monitor.w_controlArea = XtVaCreateWidget(
        "controlArea",
        xmFormWidgetClass,
        monitor.w_dialog,
        NULL
    );

	/* Create file name label */
	monitor.w_nameLabel = XtVaCreateManagedWidget(
        "nameLabel",
        xmLabelWidgetClass,
        monitor.w_controlArea,
		XmNtopAttachment, 	XmATTACH_FORM,
		XmNtopOffset,       10,
		XmNleftAttachment, 	XmATTACH_FORM,
        NULL
    );

	/* Create file name */
    label = XmStringCreateSimple("");
    monitor.w_name= XtVaCreateManagedWidget(
        "name",
        xmLabelWidgetClass,
        monitor.w_controlArea,
        XmNlabelString,     label,
		XmNtopAttachment,	XmATTACH_OPPOSITE_WIDGET,
		XmNtopWidget,		monitor.w_nameLabel,
		XmNleftAttachment,	XmATTACH_FORM,
        NULL
    );
    XmStringFree(label);

	/* Create file length label */
	monitor.w_lengthLabel = XtVaCreateManagedWidget(
        "lengthLabel",
        xmLabelWidgetClass,
        monitor.w_controlArea,
		XmNtopAttachment, 	XmATTACH_WIDGET,
		XmNtopWidget,		monitor.w_nameLabel,
		XmNleftAttachment, 	XmATTACH_FORM,
        NULL
    );

	/* Create file length */
    label = XmStringCreateSimple("");
    monitor.w_length = XtVaCreateManagedWidget(
        "length",
        xmLabelWidgetClass,
        monitor.w_controlArea,
        XmNlabelString,     label,
		XmNtopAttachment,	XmATTACH_OPPOSITE_WIDGET,
		XmNtopWidget,		monitor.w_lengthLabel,
		XmNleftAttachment,	XmATTACH_FORM,
        NULL
    );
    XmStringFree(label);

	/* Create bytes moved label */
	monitor.w_bytesMovedLabel = XtVaCreateManagedWidget(
        "bytesMovedLabel",
        xmLabelWidgetClass,
        monitor.w_controlArea,
		XmNtopAttachment, 	XmATTACH_WIDGET,
		XmNtopWidget,		monitor.w_lengthLabel,
		XmNleftAttachment, 	XmATTACH_FORM,
        NULL
    );

	/* Create bytes moved */
    label = XmStringCreateSimple("");
    monitor.w_bytesMoved = XtVaCreateManagedWidget(
        "bytesMoved",
        xmLabelWidgetClass,
        monitor.w_controlArea,
        XmNlabelString,     label,
		XmNtopAttachment,	XmATTACH_OPPOSITE_WIDGET,
		XmNtopWidget,		monitor.w_bytesMovedLabel,
		XmNleftAttachment,	XmATTACH_FORM,
        NULL
    );
    XmStringFree(label);

	/* Create scale for displaying percentage of file moved */
	label = XmStringCreateSimple("Percent Transferred Unavailable");
	monitor.w_scale = XtVaCreateManagedWidget(
		"scale",
		xmScaleWidgetClass,
		monitor.w_controlArea,
		XmNtitleString,			label,
		XmNshowValue,			True,
		XmNorientation,			XmHORIZONTAL,
		XmNtopAttachment,		XmATTACH_WIDGET,
		XmNtopWidget,			monitor.w_bytesMovedLabel,
		XmNbottomAttachment,	XmATTACH_FORM,
		XmNleftAttachment,		XmATTACH_FORM,
		XmNrightAttachment,		XmATTACH_FORM,
		XmNtopOffset,			15,
		NULL
	);
	XmStringFree(label);

	XtManageChild(monitor.w_controlArea);
}


/*
 * initialize_monitor1 - Initialize the file transfer monitor using
 *                       the file name in xfer_ctrl.  initialize_monitor2()
 *                       should be invoked after the file transfer is
 *                       underway to fill in the rest of the fields in
 *                       the monitor.
 */
initialize_monitor1()
{
	XmString string;

	/* Display file name */
	string = XmStringCreateSimple(xfer_ctrl.rel_path);
	XtVaSetValues(monitor.w_name, XmNlabelString, string, NULL);
	XmStringFree(string);

	/* Don't know file length yet */
	string = XmStringCreateSimple("Unavailable");
	XtVaSetValues(monitor.w_length, XmNlabelString, string, NULL);
	XmStringFree(string);

	/* Don't know yet if bytes transferred will be known */
	string = XmStringCreateSimple("Unavailable");
	XtVaSetValues(monitor.w_bytesMoved, XmNlabelString, string, NULL);
	XmStringFree(string);

	/* Don't know percent transferred yet */
	string = XmStringCreateSimple("Percent Transferred Unavailable");
	XtVaSetValues(monitor.w_scale,
		XmNvalue,		0,
		XmNshowValue,	True,
		XmNtitleString, string,
		NULL
	);
	XmStringFree(string);

    XSync(display, 0);
    XmUpdateDisplay(w_toplev);
}


/*
 * initialize_monitor2 - Finish initializing the file transfer monitor
 *                       using information in xfer_ctrl.  A value of -1
 *                       for xfer_ctrl.file_index indicates that the 
 *                       number of bytes transferred will be unavailable.
 *                       A value of -1 for xfer_ctrl.file_len indicates
 *                       that the file length is unavailable.
 */
initialize_monitor2()
{
	XmString string;
	char buf[20];

	/* Display file length */
	if (xfer_ctrl.file_len != -1) {
		sprintf(buf, "%ld", xfer_ctrl.file_len);
		string = XmStringCreateSimple(buf);
		XtVaSetValues(monitor.w_length, XmNlabelString, string, NULL);
		XmStringFree(string);
	}

	/* Initialize bytes transferred */
	if (xfer_ctrl.file_index != -1) {
		string = XmStringCreateSimple("0");
		XtVaSetValues(monitor.w_bytesMoved, XmNlabelString, string, NULL);
		XmStringFree(string);
	}

	/* Initialize percent transferred */
	if (xfer_ctrl.file_len != -1 && xfer_ctrl.file_index != -1) {
		string = XmStringCreateSimple("Percent Transferred");
		XtVaSetValues(monitor.w_scale,
			XmNtitleString,	string,
			NULL
		);
		XmStringFree(string);
	}

    XSync(display, 0);
    XmUpdateDisplay(w_toplev);
}


/*
 * clear_monitor - Erases information from the file transfer monitor.
 */
clear_monitor()
{
	XmString string;

	/* Clear text fields */
	string = XmStringCreateSimple("");
	XtVaSetValues(monitor.w_name, XmNlabelString, string, NULL);
	XtVaSetValues(monitor.w_length, XmNlabelString, string, NULL);
	XtVaSetValues(monitor.w_bytesMoved, XmNlabelString, string, NULL);
	XmStringFree(string);

	/* Initialize percent transferred */
	string = XmStringCreateSimple("Percent Transferred");
	XtVaSetValues(monitor.w_scale,
		XmNvalue,		0,
		XmNshowValue,	True,
		XmNtitleString,	string,
		NULL
	);
	XmStringFree(string);
    XSync(display, 0);
    XmUpdateDisplay(w_toplev);
}


/*
 * Update the file transfer monitor.
 */
update_monitor()
{
	char buf[20];
	XmString string;
	int prev_percent = xfer_ctrl.percent;
	int update_bytes_xferred = True;

	/* Update percent transferred */
	if (xfer_ctrl.file_len != -1 && xfer_ctrl.file_index != -1) {
		if (xfer_ctrl.file_len > 0) {
			xfer_ctrl.percent =
				((100.*(float)xfer_ctrl.file_index)/(float)xfer_ctrl.file_len);
			xfer_ctrl.percent = MIN(100, xfer_ctrl.percent);
		} else
			xfer_ctrl.percent = 100;
		if (xfer_ctrl.percent > prev_percent)
			XtVaSetValues(monitor.w_scale, XmNvalue, xfer_ctrl.percent, NULL);
		else
			update_bytes_xferred = False;
	}

    /* Update bytes transferred */
    if (xfer_ctrl.file_index != -1 && update_bytes_xferred) {
		sprintf(buf, "%ld", xfer_ctrl.file_index);
        string = XmStringCreateSimple(buf);
    	XtVaSetValues(monitor.w_bytesMoved, XmNlabelString, string, NULL);
    	XmStringFree(string);
	}

/*
    XSync(display, 0);
    XmUpdateDisplay(w_toplev);
*/
}


/*
 * hide_monitor - Hide file transfer monitor dialog.
 */
hide_monitor()
{
	XtUnmanageChild(monitor.w_dialog);
}


/*
 * cb_monitor_abort - Callback for file transfer abort button.
 */
void
cb_monitor_abort(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	xfer_ctrl.abort_requested = True;
	show_abort_dialog();
}


/*
 * cb_monitor_close - Callback to handle closes from window manager.
 *                    A close is not allowed.
 */
void
cb_monitor_close(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	warning_error("Can't close monitor while transfer is in progress.");
}

