/*

sshstrmapping.h

Author: Vesa Suontama <vsuontam@ssh.fi>

Copyright (c) 1999 Vesa Suontama <vsuontam@ssh.fi>, Espoo, Finland
                   All rights reserved

Created: Thu Sep  2 10:50:06 1999 vsuontam
Last modified: Thu Sep  2 13:50:32 1999 vsuontam

Defines a c style string mapped has table. Uses sshmapping.
See sshmapping.h.

String mappings are useful, if you need to map c style strings to a
fixed context. Mapping stores the strings too, so the
application does not need to keep string allocated it is using as keys
in the mapping table.


*/
#include "sshmapping.h"

#ifndef SSH_STR_MAPPING_H
#define SSH_STR_MAPPING_H

/* Allocates the string mapping.  This function returns pointer to
   allocated SshMapping set or NULL in case of error. */
SshMapping ssh_str_mapping_allocate(void);

/* Frees the memory allocated for the set. If a destructor for elements is
   needed, use ssh_str_mapping_destroy. */
void ssh_str_mapping_free(SshMapping set);

/* Removes all mappings from the set, making it empty. */
void ssh_str_mapping_clear(SshMapping set);


/* A type of this callback is passed to the ssh_str_mapping_destroy.
   A function of this type is called once for each item in the array. */
typedef void (*SshStrMappingDestructorCB)(const char *key, 
                                          void *value,
                                          void *context);

/* Removes all the elements from the mapping and calls the destructor function
   for each of them. If destructor_cb is NULL, this correspons the
   ssh_str_mapping_free. */
void ssh_str_mapping_destroy(SshMapping set, 
                             SshStrMappingDestructorCB destructor_cb,
                             void *context);

/* Adds a mapping for a key. Duplicate keys are stored only once.  If
   mapping for this key already exists, the old value is replaced with
   the new one. In that case this returns TRUE, otherwise FALSE.  */
Boolean ssh_str_mapping_put(SshMapping set, const char *key, void *value);

/* Removes a mapping for a key. If there was no mapping for specified
   key, FALSE is returned. If mapping existed, this returns TRUE and
   if `value_return' is not NULL, also copies the old mapped value to
   it. */
Boolean ssh_str_mapping_remove(SshMapping set, const char *key,
                               void *value_return);

/* Checks the existance of key in set and returns TRUE if it is found,
   FALSE otherwise.  If `value_return' is not NULL, copies also the
   mapped value to buffer pointer in it. */
Boolean ssh_str_mapping_get(const SshMapping set, const char *key,
                            void *value_return);

/* Returns the number of mappings in the set. */
unsigned int ssh_str_mapping_count(const SshMapping set);

/* Returns the size of the hash table underlying the set; this can be
   seen as an overestimate of the number of keys in the set. */
unsigned int ssh_str_mapping_size(const SshMapping set);

/* These two functions provide a way to loop through the entire
   mapping, one key (and associated value) at time. At first the
   ssh_str_mapping_reset_index should be called to reset the internal
   index in mapping. After that the repeated calls to
   ssh_str_mapping_get_next return all the existing mappings from the
   set. When all mappings are processed mapping_get_next will return
   FALSE, otherwise TRUE.
   
   It must be noted that no operations that can possibly change
   the size of the mapping (put and remove) can be made during
   this procedure.  */
void ssh_str_mapping_reset_index(SshMapping set);

Boolean ssh_str_mapping_get_next(SshMapping set, char **key_return,
                                 void *value_return);

/* Refresh lock allows/disallows refreshes when removing entries. */

/* Set refresh lock. Changes the refresh lock to the given flag value.
   Function returns the previous flag value. */
Boolean ssh_str_mapping_set_refresh_lock(SshMapping set, Boolean flag);

/* The following macros are defined for functions that can be directly
   mapped to ssh_mapping_functions. */
#define ssh_str_mapping_set_refresh_lock ssh_mappig_set_refresh_lock
#define ssh_str_mapping_reset_index ssh_mapping_reset_index
#define ssh_str_mapping_free ssh_mapping_free
#define ssh_str_mapping_clear ssh_mapping_clear
#define ssh_str_mapping_count ssh_mapping_count
#define ssh_str_mapping_size ssh_mapping_size




#endif /* SSH_STR_MAPPING_H */


