/*
 ***********************************************************************
 *                                                                     *
 * Program to control LORAN-C radio                                    *
 *                                                                     *
 * This program controls a special-purpose radio designed to receive   *
 * transmissions from the US Coast Guard LORAN-C navigation system.    *
 * These stations operate on an assigned radio frequency of 100 kHz    *
 * and can be received over the continental US, adjacent coastal areas *
 * and significant areas elsewhere in the world.                       *
 *                                                                     *
 * The radio, which is contained in a separate, shielded box, receives *
 * the signals, which consist of an eight-pulse biphase-modulated      *
 * pulse group transmitted at a 1-kHz rate. Each of these pulse groups *
 * is repeated at an interval characteristic of the particular LORAN-C *
 * chain, which consists of a master station and up to three slave     *
 * stations. The radio includes a synchronous detector driven by a     *
 * quadrature-phase clock, two integrators with adjustable gain and    *
 * an signal-level detector used to derive the agc voltage.            *
 *                                                                     *
 * The radio is controlled by this program using a special-purpose     *
 * interface, which converts the receiver signals using an             *
 * analog/digital converter and multiplexor, and generates the digital *
 * timing and analog control signals using an AMD 9513A System Timing  *
 * Controller (STC) chip, two digital/analog converters and            *
 * miscellaneous logic components. The radio provides three analog     *
 * signals, one for the in-phase integrator, another for the           *
 * quadrature-phase integrator and a third for the agc. This program   *
 * computes the master oscillator frequency adjustment and receiver    *
 * agc voltage.                                                        *
 *                                                                     *
 * The reciever includes a precision, oven-controlled crystal          *
 * oscillator used to derive all timing signals used by the receiver   *
 * and this program. The 5-MHz output of this oscillator is adjusted   *
 * over a small range by this program to coincide with the LORAN-C     *
 * signal as broadcast to within a few parts in 10e10 and is suitable  *
 * for use as a laboratory frequency standard. The oscillator itself   *
 * should have good intrinsic stability and setability to within less  *
 * than 2.5 Hz at 5 MHz (0.5 ppm), since it must maintain the master   *
 * clock to within 100 us over the pulse-code scan interval of several *
 * minutes.                                                            *
 *                                                                     *
 * The PC running this program generates the control signals necessary *
 * to run the radio and produces a 1-pps signal synchronized to        *
 * UTC(LORAN) to within a fraction of a microsecond. When manually     *
 * adjusted using time-of-coincidence (TOC) data published by US Naval *
 * Observatory, this signal is suitable for use as a precision source  *
 * of standard time. The system can generate all sorts of external     *
 * signals as well, as programmed in the 9513A.                        *
 *                                                                     *
 *      David L. Mills (mills@udel.edu) 2 April 1992                   *
 *                                                                     *
 ***********************************************************************
 *
 * Current LORAN-C chains by gri (master listed first)
 *
 * 9990 North Pacific		St. Paul Island, Attu, Port Clarence,
 *							Narrow Cape
 * 9980 Icelandic Sea
 * 9970 North West Pacific	Iwo Jima, Marcus Island, Hokkaido, Gesashi,
 *							Yap Island
 * 9960 North East US		Seneca, Caribou, Nantucket, Carolina Beach,
 *							Dana
 * 9940 West Central US		Fallon, George, Middletown, Searchlight
 * 9610 South Central US
 * 8970 Great Lakes US		Dana, Malone, Seneca, Baudette
 * 8290 North Central US
 * 7990 Mediterranian Sea	Sellia Marina, Lampedusa, Kargabarun,
 *							Estartit
 * 7980 South East US		Malone, Grangeville, Raymondville, Jupiter,
 *							Carolina Beach
 * 7970 Norwegian Sea		Ejde, Sylt, B0, Sandur, Jan Mayen
 * 7960 Gulf of Alaska		Tok, Narrow Cape, Shoal Cove
 * 7930 Labrador Sea		Angissoq, Sandur, Ejde, Cape Race
 * 5990 West Central Canada	Williams Lake, Shoal Cove, Port Hardy
 * 5930 East Central Canada	Caribou, Nantucket, Cape Race
 * 4990 Central Pacific		Johnson Island, Upolu Point, Kure Island
 */

#include "loran.h"

/*
 * Function declarations
 */
void status(double, double, char *);
void receive(struct station *);

/*
 * Program variables (units)
 */
int ptrenv = 0;					/* index of display cycle */
char mode = '1';				/* operating mode */
int par = 0x0a;					/* mode register */
char kbd = ' ';					/* latest keystroke */
char test[2000];				/* test vector */
int epoch = 0;

/*
 * System timing and related data (units). These values provide the
 * precise offset of the reference phase relative to the epoch when
 * the STC chip was last reset by this program. All timing calculations
 * are performed relative to this epoch. The freq and phase variables
 * are computed in various places in the program, but take effect only
 * at the end of the processing cycle, so that all timing calculations
 * can be performed with respect to the epoch the system is actually at.
 */
int frame = 0;					/* offset to reference frame (2*gri) */
int offset = 0;					/* offset to reference gri (10 us) */
int strobe = 0;					/* offset to reference cycle (10 us) */
int gri = 9960;					/* group repetition interval (10 us) */
int pcx = 1500;					/* pulse-code interval (200 ns) */
int freq = 0;					/* frequency offset (10 us/frame) */
int phase = 0;					/* phase offset (10 us) */
int step = 1;					/* phase step (10 us) */
int stb[MMAX];					/* strobe median filter */
int sgate = SGATE;				/* strobe noise (max-min) */
char codesw;					/* gri a/b switch */
char pllsw = '1';				/* pll switch */
int cindex = 0;					/* chain station index */

/*
 * Various controls normally preset, but can be adjusted by keyboard
 * commands. After all, this is a prototype device.
 */
double vco = VCO;				/* vco dac signal */
double vcodac = VCO;			/* vco dac bias (dac a) */
double agc = AGC;				/* agc dac signal */
double agcdac = AGC;			/* agc dac bias (dac b) */

/*
 * Raw and processed data input from receiver. The raw data are received
 * directly from the adc and summed for both the a and b gri intervals.
 * The offset data are computed during the receiver calibration modes
 * and used to remove bias from the raw data to produce the net signal
 * valid at the end of the frame. As the envelope pointer cycles through
 * all 30 100-us cycles of the pulse gate, the i and q signal envelopes
 * are averaged separately for use in the cycle-identification and
 * phase-tracking processes. The median filters are used to suppress
 * impulse-noise and pulse-dropout. The rms error signal produced from
 * the i and q signals is used during the pulse-code identification
 * process.
 */
struct station {
	char code[2];				/* gri a and b pulse codes */
	double isig;				/* i-signal (a+b) (adc chan 0) */
	double qsig;				/* q-signal (a+b) (adc chan 1) */
	double fmax;				/* max pulse-group signal */
	int mindex;					/* index of max pulse-group signal */
	double rms[NRMS];			/* pulse-group signal shift register */
	};

struct station chain[] = {{MPCA, MPCB}, {SPCA, SPCB}}; /* loran-c chain */

double pmed[NENV][MMAX];		/* phase median filter */
double emed[NENV][MMAX];		/* envelope median filter */
double mgate = 0;				/* envelope median filter span */

double pcyc[NENV];				/* cycle phase */
double pfac = PMIN;				/* cycle phase weight */
double ecyc[NENV];				/* cycle amplitude */
double efac = EMIN;				/* cycle amplitude weight */
double erms[NENV];				/* cycle rms error */

double iofs = 307;				/* i-integrator offset */
double qofs = 308;				/* q-integrator offset */
double agcraw = 0;				/* receiver agc (adc chan 2) */
double agcavg = 0;				/* receiver agc smoothed signal */
double agcofs = 317;			/* receiver agc offset (zero signal) */
double agcmax = 355;			/* receiver agc max signal (overload) */
double gain = RGAIN;			/* program gain */
char report[257] = "\0";		/* report string for display */

double freqx = 0;				/* type II frequency */
double phasex = 0;				/* type II phase */
double tau = TAU;				/* type II time constant */

/*
 * Event counters. These tally the synchronization events of interest
 * for debugging and monitoring.
 */
int pgcnt = 0;					/* pulse-group search events */
int encnt = 0;					/* envelope search events */
int cscnt = 0;					/* cycle-slip events */
int sscnt = 0;					/* strobe-slip events */
int pncnt = 0;					/* pulse-group noise events */

/*
 * Signal/noise ratios. These reveal signal quality and health of the
 * tracking processes.
 */
double psnr = 0;				/* pulse-group max-envelope/rms */
double esnr = 0;				/* envelope rms-max/rms-min */

/*
 * Main program
 *
 * Programming note: There is usually enough time between gri intervals
 * for one print statement, but not two, at least on a 286.
 */
void main(argc, argv) int argc; char *argv[]; {
	int mindex = 0;				/* index of min cycle in envelope */
	int maxdex = 0;				/* index of max cycle in envelope */
	int cycle = 0;				/* cycle counter */
	int count = 0;				/* utility counter */
	int icnt = 0;				/* integration cycle counter */
	int mcnt = 0;				/* median counter */
	int ecnt = 0;				/* envelope counter */
	int env = 0;				/* envelope scan pointer */
	int envbot = 0;				/* first cycle in envelope scan */
	int envtop = NENV-1;		/* last cycle in envelope scan */
	int i, j, temp;				/* utility temps */
	double dtemp, etemp, ftemp, gtemp; /* utility doubles */
	double fmax, fmin;			/* utility max/min values */
	char msg[80] = "\0";		/* status message */
	int tmp[MMAX];				/* int temporary list */
	double ftmp[MMAX];			/* double temporary list */
	int osc = 1;				/* oscillator (1 = int, 5 = ext) */
	char ctemp;					/* char temp */
	struct station *sptr;		/* which station */

/*
 * Decode command-line arguments
 *
 * usage: <program name> <gri> <codes> <agc> <vco>
 * <gri>	assigned LORAN-C group repitition interval (default 9960)
 * <osc>	1 for internal oscillator, 5 for external oscillator
 * <agc>	initial agc dac (0-255) (default AGC parameter)
 * <vco>	initial vco dac (0-255) (default VCO parameter)
 */
	if (argc > 1)
		sscanf(argv[1], "%i", &gri);
	if (argc > 2)
		sscanf(argv[2], "%i", &osc);
	if (argc > 3)
		sscanf(argv[3], "%lf", &agcdac);
	if (argc > 4)
		sscanf(argv[4], "%lf", &vcodac);

	for (i = 0; i < 2000; i++)
		test[i]=0;
	for (i = 100; i < 108; i++)
		test[i]=1;

/*
 * Initialization
 *
 * This section runs only once. It resets the timing generator,
 * loads its registers with default values and clears arrays. The
 * program then simulates a "1" keystroke and sets the receiver gain
 * at minimum to begin receiver calibration.
 */
	init[0*3] = init[0*3] & 0x0f00 || osc << 8; /* set STC source */
	init[2*3] = init[0*3] & 0x0f00 || osc << 8;
	init[3*3] = init[0*3] & 0x0f00 || osc << 8;
	init[4*3] = init[0*3] & 0x0f00 || osc << 8;
	outp(TGC, RESET); outp(TGC, LOAD+0x1f); /* reset STC chip */
	outp(TGC, LOADDP+MASTER); outp(TGD, 0xf0); outp(TGD, 0x8a);
	outp(TGC, LOADDP+1);
	for (i = 0; i < 5*3; i++) {
		outp(TGD, init[i]); outp(TGD, init[i] >> 8);
		}
	outp(TGC, LOADARM+0x1f);	/* let the good times (mode 1) roll */
	sprintf(report, "Calibrating receiver");

	/*
	 * Main loop
	 *
	 * This is the main receiver loop and runs until escaped by a ^C
	 * signal. The main loop runs twice per frame or once each gri
	 * (pulse groups a and b) and performs the main receiver update
	 * between the end of pulse group b and the beginning of pulse group
	 * a.
	 */
	cindex = 0;
	while (1) {

		/*
		 * Process i-phase, q-phase and agc
		 *
		 * Note that a LORAN frame consists of two gri intervals a and
		 * b, each with individual pulse codes. The receiver integrates
		 * each gri using the assigned pulse codes. There are two sets
		 * of pulse codes, one for the master station and the other for
		 * slave stations, of which there may be as many as four. Each
		 * LORAN chain is assigned a unique gri interval in the range
		 * 40-100 ms.
		 */
		sptr = &chain[cindex];
		receive(sptr);
		if (codesw == 0) {
			count++;

			/*
			 * gri a processing
			 *
			 * This section processes the i, q and agc samples from the
			 * previous frame and computes the receiver vco and agc dac
			 * values. It begins by computing the average in-phase,
			 * quadrature-phase and agc signals. Note the offset and
			 * sign corrections, since the adc operates in unsigned,
			 * inverted mode.
			 */
			sptr->isig = -(sptr->isig - iofs) * gain;
			sptr->qsig = -(sptr->qsig - qofs) * gain;
			agcavg += (agcraw - agcofs) / AGCFAC;
			if (agcavg < 0) agcavg = 0;

			/*
			 * In calibrate mode "1" the receiver gain is set to
			 * minimum. The program waits for the averages to settle
			 * and then calculates the integrator offsets and agc
			 * minimum value, which takes a few seconds. When done, the
			 * program simulates a '2' keystroke to complete receiver
			 * calibration.
			 *
			 * In this mode the vco dac is clamped and the agc dac is
			 * set to minimum (0).
			 */
			if (mode == '1') {
				vco = vcodac; agc = 0; env = 0;
				iofs -= sptr->isig / AGCFAC;
				qofs -= sptr->qsig / AGCFAC;
				dtemp = agcraw - agcofs; agcofs += dtemp/AGCFAC;
				if (sptr->isig * sptr->isig +
					sptr->qsig * sptr->qsig +
					dtemp * dtemp < 3 && count > AGCFAC * 2) {
					mode = '2';		/* continue in mode '2' */
					count = 0;
					sprintf(report, "iofs %4.1lf  qofs %4.1lf  agcofs %4.1lf  gain %4.2lf",
						iofs, qofs, agcofs, gain);
					}
				}

			/*
			 * In calibrate mode 2 the receiver gain is set to  maximum.
			 * The program waits for the averages to settle and then
			 * calculates the agc maximum value, which takes a few
			 * seconds. When done, the program simulates a '3' keystroke
			 * to establish the normal receiver gain configuration.
			 *
			 * In this mode the vco dac is clamped and the agc dac is
			 * set to maximum (255).
			 */
			else if (mode == '2') {
				vco = vcodac; agc = 255;
				dtemp = agcraw - agcmax; agcmax += dtemp/AGCFAC;
				if (dtemp*dtemp < 1 && count > AGCFAC * 2) {
					mode = '3';		/* continue in mode 3 */
					count = 0; agcavg = 0;
					}
				}

			/*
			 * In calibrate mode 3 the receiver gain is determined by
			 * the receiver agc, which is a peak-indicating type
			 * sensitive to the peak pulse power in the 90-110 kHz
			 * spectrum. This step, which takes a few seconds, is
			 * necessary only to insure the receiver operates at the
			 * best signal/noise ratio and without overload during the
			 * scanning process. When done, the program simulates a '4'
			 * keystroke to begin the pulse-group scan.
			 *
			 * In this mode the vco dac is clamped and the agc dac is
			 * computed from the receiver agc smoothed output.
			 *
			 * *** This part not finished yet ***
			 */
			else if (mode == '3') {
				vco = vcodac; agc = agcdac;
				if (count > AGCFAC * 2) {
					mode = '4';		/* continue in mode 4 */
					count = 0; fmax = 0; fmin = 0; pgcnt++;
					sprintf(report, "agcmax %4.1lf  agcavg %4.1lf\nSearching for signal",
						agcmax, agcavg);
					}
				}

			/*
			 * In acquisition mode 4 the radio scans at 100 us per frame
			 * over the entire frame (2*gri), which takes up to about
			 * seven minutes. The program accumulates the 12 most recent
			 * received rms signal samples in a pulse-gate filter. The
			 * first two samples are used as a noise gate. In the three
			 * most recent samples, including two in the filter and the
			 * most recent, either the first or last must be at least
			 * 1/3 the second, or the second is most likely a noise
			 * pulse, which could be due to a pulse code from another
			 * chain just happening to appear in the pulse-code window.
			 * In addition, the program computes a weighted sum which
			 * favors the two samples near the middle of the last ten
			 * stages in the filter. The program saves the value and
			 * index of the highest sample received. Note that sample
			 * selection from the pulse-group filter is valid only after
			 * 12 samples have been received. Therefore, the program
			 * waits for 12 samples before updating a bin and wraps
			 * around for 12 bins beyond the end of the frame. In order
			 * to improve the estimated position, the program
			 * interpolates between the intervals just before and just
			 * after the selected interval. When the scan is complete,
			 * the program simulates a '5' keystroke to begin the
			 * envelope scan.
			 *
			 * In this mode the vco dac is clamped and the agc dac
			 * remains at the value computed in the previous mode.
			 */
			else if (mode == '4') {
				vco = vcodac; agc = agcdac;
				if (count > 2 * gri / 10 + NRMS) {
					fmin = sqrt(fmin / count);
					if (sptr->fmax <= 4 * fmin) {
						status(sptr->fmax, fmin, " low signal");
						count = 0; sptr->fmax = 0; fmin = 0; pgcnt++;
						}
					else {
						status(sptr->fmax, fmin, " cycle search");
						if (fmin > 0)
							psnr = sptr->fmax / fmin;
						count = 0;
						phase = sptr->mindex - offset-OFFSET; freq = 0;
						mode = '5';		/* continue in mode 5 */
						envbot = 0; envtop = NENV-1;
						env = 0; ptrenv = 0; par = 0x0c;
						for (i = 0; i < MMAX; i++)
							stb[i] = 0;
						efac = EMIN; pfac = PMIN;
						esnr = 0; mcnt = 0; ecnt = 0;
						strobe = 0; fmin = 999; sptr->fmax = 0;
						icnt = 0; encnt++;
						}
					}
				else {
					freq = 10;
					dtemp = sptr->isig * sptr->isig +
						sptr->qsig * sptr->qsig;
					fmin += dtemp;
					dtemp = sqrt(dtemp);
					if (sptr->rms[0] > dtemp * PGATE ||
						sptr->rms[0] > sptr->rms[1] * PGATE) {
						sptr->rms[0] = (dtemp + sptr->rms[1]) / 2;
						pncnt++;
						}
					for (i = 0; i < NRMS - 1; i++)
						sptr->rms[i+1] = sptr->rms[i];
					sptr->rms[0] = dtemp;
					dtemp = sptr->rms[4] + sptr->rms[5] + sptr->rms[6];
					dtemp -= (sptr->rms[1] + sptr->rms[2] +
						sptr->rms[3] + sptr->rms[7] +
						sptr->rms[8] + sptr->rms[9])/6;
					if (count > NRMS) {
						if (dtemp > sptr->fmax) {
							sptr->fmax = dtemp;
							dtemp = (sptr->rms[6] - sptr->rms[4]) /
								(sptr->rms[6] + sptr->rms[4]);
							sptr->mindex = offset + (int)dtemp * 10;
							}
						}
					if (((count+1)%100) == 0)
						status(sptr->fmax, sqrt(fmin/count), "\0");
					}
				}

			/*
			 * In tracking mode 5, which is the default, the program
			 * scans at 10 us per frame over the 300-us receiver
			 * pulse-gate interval. The program integrates the q-
			 * channel and i-channel envelopes separately in one-cycle
			 * (10 us) bins and determines the reference cycle as the
			 * minimum rms error relative to a prestored model envelope.
			 * If sufficient integrated signal amplitude is not found
			 * within a minute or so, the search is abandoned and the
			 * program reverts to the pulse-group search.
			 *
			 * In this mode the vco dac follows the i channel and the
			 * agc follows the q channel, both averaged in complicated
			 * ways.
			 */
			else {

				return;

				/*
				 * This is the envelope scan. The q signal represents
				 * the amplitude and the i signal the phase-correction.
				 * The program actually uses only the envelope
				 * amplitude. Median filters for both the amplitude and
				 * phase signals for each cycle of the pulse are used to
				 * help cope with cross-rate interference and dual-rate
				 * transmitter blanking. The phase signal is saved for
				 * each cycle separately, while the envelope signal is
				 * averaged for each cycle separately.
				 */
				if (mcnt < MMIN)
					mcnt++;
				for (i = 0; i < mcnt-1; i++) {
					pmed[env][i+1] = pmed[env][i];
					emed[env][i+1] = emed[env][i];
					}
				pmed[env][0] = sptr->isig;
				emed[env][0] = sqrt(sptr->isig * sptr->isig+
					sptr->qsig * sptr->qsig);
				for (i = 0; i < mcnt; i++){
					ftmp[i] = pmed[env][i];
					for (j = 0; j < i; j++) {
						if (ftmp[i] < ftmp[j]) {
							gtemp = ftmp[i]; ftmp[i] = ftmp[j];
							ftmp[j] = gtemp;
							}
						}
					}
				pcyc[env] = ftmp[mcnt/2];
				for (i = 0; i < mcnt; i++){
					ftmp[i] = emed[env][i];
					for (j = 0; j < i; j++) {
						if (ftmp[i] < ftmp[j]) {
							gtemp = ftmp[i]; ftmp[i] = ftmp[j];
							ftmp[j] = gtemp;
							}
						}
					}
				ecyc[env] += (ftmp[mcnt/2] - ecyc[env])/efac;
				mgate = ftmp[mcnt-1] - ftmp[0];

				/*
				 * Compute rms envelope error and find cycles of max
				 * amplitude and min error.
				 */
				if (ecyc[env] > fmax && env >= 8) {
					fmax = ecyc[env]; maxdex = env;
					}
				if (fmax != 0)
					ftemp = 100 / fmax;
				else
					ftemp = 0;
				etemp = 0;
				for (i = 0; i < 7; i++) {
					j = env+i;
					if (j > envtop)
						dtemp = ecyc[envtop] * ftemp;
					else
						dtemp = ecyc[j] * ftemp;
					dtemp -= envcyc[i]; etemp += dtemp * dtemp;
					}
				erms[env] = sqrt(etemp / 7);
				if (erms[env] < fmin &&
					env >= maxdex - 8 && env <= maxdex-4) {
					fmin = erms[env]; mindex = env;
					}

				/*
				 * This section is entered at the end of the envelope
				 * scan. It establishes the cycle position and
				 * calculates the signal/noise ratio and phase-
				 * correction signal and receiver gain control. The
				 * cycle position is determined from the envelope cycle
				 * of minimum rms error using a median filter. The
				 * phase-correction signal is extracted from the third
				 * carrier cycle, while the signal/noise ratio is
				 * computed as the ratio of the amplitude of the seventh
				 * carrier cycle divided by the rms error of the first
				 * cycle. The span of the values in the strobe filter is
				 * calculated for later use as a noise gate. In order to
				 * maintain critical damping (damping factor of 0.707),
				 * the pll gain must be controlled so that the product
				 * of the loop time constant and loop gain is equal to
				 * 1/2.
				 */
				if (env == envbot) {
					fmin = 999; fmax = 0;
					for (i = 0; i < mcnt-1; i++)
						stb[i+1] = stb[i];
					if (maxdex < 8)
						stb[0] = 0;
					else if (mindex < maxdex-8 || mindex > maxdex-4)
						stb[0] = maxdex-6;
					else
						stb[0] = mindex;
					for (i = 0; i < mcnt; i++){
						tmp[i] = stb[i];
						for (j = 0; j < i; j++) {
							if (tmp[i] < tmp[j]) {
								temp = tmp[i]; tmp[i] = tmp[j];
								tmp[j] = temp;
								}
							}
						}
					cycle = tmp[mcnt/2];

					/*
					 * Determine loop type and compute phase, frequency
					 * and gain. Controls select type-I pll, type-II pll
					 * or open-loop.
					 */
					ftemp = pcyc[cycle+2];
					if (pllsw == '1') {
						gtemp = 1 / (2*VGAIN*pfac);
						if (gtemp > 1)
							gtemp = 1;
						phasex += (gtemp * ftemp - phasex) / pfac;
						freqx = phasex;
						}
					else if (pllsw == '2') {
						gtemp = 16 / VGAIN;
						freqx += gtemp * 1.8 * ftemp / (tau * tau);
						phasex = gtemp * ftemp / tau + freqx;
						}
					else {
						gtemp = freqx = phasex = 0;
						}
					vco = vcodac - phasex;
					if (cycle < 1 || cycle > 17)
						sgate = SGATE;
					else
						sgate = tmp[mcnt-1] - tmp[0];
					dtemp = ecyc[cycle + 6]; etemp = erms[cycle];
					if (etemp > 0)
						esnr = dtemp / etemp;
					else
						esnr = 0;
					strcpy(msg, "\0");

					/*
					 * This section is entered at the end of each
					 * integration cycle. It checks the signal quality,
					 * cycle position within the 300-us envelope gate
					 * and strobe position within the 90-us cycle-
					 * identification window. The receiver gain is
					 * determined from the amplitude of the seventh
					 * cycle in the window. If reliable cycle
					 * identification has not been achieved in a
					 * reasonable time, the program punts back to the
					 * pulse-group scan. The integration time constant
					 * is increased only if all samples in the strobe
					 * median filter are identical.
					 */
					icnt++;
					if (icnt >= mcnt) {
						icnt = 0;
						if (mcnt < (int)(pfac/8) && mcnt < MMAX)
							mcnt++;
						if (sgate >= SGATE) {
							if (count < WATCHDOG) {
								strcpy(msg, " strobe noise");
								if (strobe != 0) {
									mode = '5';	/* continue in mode 5 */
									envbot = 0; envtop = NENV-1;
									env = 0; ptrenv = 0; par = 0x0c;
									for (i = 0; i < MMAX; i++)
										stb[i] = 0;
									efac = EMIN; pfac = PMIN;
									esnr = 0; mcnt = 0; ecnt = 0;
									strobe = 0; fmin = 999; fmax = 0;
									encnt++;
									}
								}
							else {
								strcpy (msg,
									" resume pulse-group search");
								mode = '4';	/* continue in mode 4 */
								count = 0; fmax = 0; fmin = 0; pgcnt++;
								par = 0x0a;
								}
							}
						else {
							if (cycle < 3 || cycle >= 15) {
								strcpy(msg, " cycle slip");
								phase = -(9 - cycle); cscnt++;
								mode = '5';	/* continue in mode 5 */
								envbot = 0; envtop = NENV-1;
								env = 0; ptrenv = 0; par = 0x0c;
								for (i = 0; i < MMAX; i++)
									stb[i] = 0;
								efac = EMIN; pfac = PMIN;
								esnr = 0; mcnt = 0; ecnt = 0;
								strobe = 0; fmin = 999; fmax = 0;
								cscnt++;
							}
							else if (cycle + 2 != strobe &&
							  pllsw == '1') {
								strcpy(msg, " strobe slip");
								mode = '6'; sscnt++;
								strobe = cycle+2;
								}
							}
						}
					ecnt++;
					if (ecnt >= (int)efac) {
						ecnt = 0;
						if (strobe != 0) {
							if (dtemp > 100)
								agc--;
							else
								agc++;
							}
						if (sgate == 0) {
							if (efac < EMAX)
								efac *= EFAC;
							if (pfac < PMAX)
								pfac *= PFAC;
							}
						}

					/*
					 * This section determines what to display about the
					 * status of the receiver and tracking information.
					 * The envelope amplitude phase-correction and rms
					 * error signals can be desplayed for every cycle
					 * scanned. In coarse-search mode before the strobe
					 * position has been determined only the
					 * signal/noise ratio and strobe span are displayed;
					 * while, in fine-tracking mode only the three
					 * cycles before and five cycles after the strobe
					 * are displayed. This includes one cycle before the
					 * cycle-posigion window and one after. This reduces
					 * the integrator bumps in case of a strobe slip,
					 * which normally is not more than +-1 cycle.
					 */
					if (strobe == 0) {
						strcat(msg, " ?");
						dtemp = esnr; etemp = sgate;
						ptrenv = cycle + 2;
						}
					else {
						count = 0;
						dtemp = ecyc[ptrenv]; etemp = erms[ptrenv];
						if (ptrenv == strobe)
								strcat(msg, " x");
						if (kbd == 'v')
							etemp = phasex;
						else if (kbd == 'f')
							etemp = freqx;
						if (ptrenv == envtop-1) {
							strcat(msg, " agc");
							etemp = agc;
							}
						else if (ptrenv == envtop) {
							strcat(msg, " vco");
							etemp = vco;
							}
						}
					status(dtemp, etemp, msg);
					if (strobe != 0) {
						envbot = strobe - 3; envtop = strobe + 5;
						}
					else {
						envbot = 0; envtop = NENV-1;
						}
					ptrenv++;
					if (ptrenv > envtop) ptrenv = envbot;
					if (ptrenv < envbot) ptrenv = envtop;
					}
				}

			/*
			 * Compute the interval to the next gri. Normally, this is
			 * fixed at the gri interval specified less 800 (8 ms), but
			 * can be adjusted in 10-us steps to align the timing
			 * generator to the transmitted LORAN-C signal. The
			 * adjustments can be in the form of a frequency offset, in
			 * 10 us/frame increments, or a one-time phase adjustment,
			 * in 10-us increments. Note that the interval to the next
			 * gri has already been loaded in the counters at this
			 * point. The value loaded here actually applies to the gri
			 * after that. Since this is the end of a b interval and the
			 * next a interval has already been established, frequency
			 * and phase adjustments will take effect at the beginning
			 * of the next b interval after that, so correct timing
			 * between the a and b pulse groups for a single frame are
			 * preserved.
			 */
			offset += freq + phase;		/* adjust epoch */
			while (offset >= 2 * gri) {
				offset -= 2 * gri; frame++;
				}
			while (offset < 0) {
				offset += 2 * gri; frame--;
				}
			temp = gri - 800 + freq + phase;	/* gri counter */
			while (temp >= 2 * gri)
				temp -= 2 * gri;
			while (temp < FGUARD || (temp < DGUARD &&
				report[0] != '\0'))
				temp += 2 * gri;
			outp(TGC, LOADDP+0x0a);
			outp(TGD, temp); outp(TGD, temp >> 8); phase = 0;

			/*
			 * Load the vco and agc dacs and initialize the pulse code
			 * and internal integrators for the next frame. Also step
			 * the envelope strobe one cycle (10 us) for the next frame.
			 */
			dtemp = vco * 16;			/* vco dac */
			if (dtemp > 4095)
				dtemp = 4095;
			if (dtemp < 0)
				dtemp = 0;
			temp = (int)dtemp;
			outp(PAR, par & !MSB);
			outp(DACA, temp);
			outp(PAR, par | MSB);
			outp(DACA, temp >> 8);
			dtemp = agc * 16;			/* agc dac */
			if (dtemp > 4095)
				dtemp = 4095;
			if (dtemp < 0)
				dtemp = 0;
			temp = (int)dtemp;
			outp(PAR, par & !MSB);
			outp(DACB, temp);
			outp(PAR, par | MSB);
			outp(DACB, temp >> 8);
			temp = 5000 - pcx;			/* envelope scan window */
			outp(TGC, LOADDP + 0x0b);
			outp(TGD, pcx); outp(TGD, pcx >> 8);
			outp(TGD, temp); outp(TGD, temp >> 8);
			outp(TGC, LOADDP + 0x0c);
			env--;						/* envelope scan strobe */
			if (env < envbot)
				env = envtop;
			if (env > envtop)
				env = envbot;
			outp(CODE, sptr->code[0]);	/* pulse code */
			temp = env * 50 + RCVDELAY;	/* gri counter */
			outp(TGD, temp); outp(TGD, temp >> 8);
			sptr->isig = 0; sptr->qsig = 0; agcraw = 0;
			}
		else {

			/*
			 * gri b processing
			 *
			 * This section sets up for the b pulse group. It resets the
			 * gri (counter 2) load register to delay exactly one gri
			 * less 800 (8 ms), which will be used for the subsequent a
			 * interval following the next b interval. It also sets the
			 * b code for the next b interval.
			 */
			outp(CODE, sptr->code[1]);	/* pulse code */
			temp = gri - 800;			/* gri counter */
			outp(TGC, LOADDP + 0x0a);
			outp(TGD, temp); outp(TGD, temp >> 8);
			}
		codesw = (char)!codesw;
		}
	}

/*
 * Subroutine to wait for next gri and accumulate i, q, agc
 *
 * This soutine first enables automatic adc start at the end of
 * the next gri. It then reads the i integrator (adc channel 0),
 * q integrator (adc channel 1) and agc (adc channel 2). These
 * values are summed for the a and b pulse groups and processed
 * at the end of the b pulse group. Keyboard commands and status
 * displays are processed during the b pulse group.
 */
void receive(sptr)
	struct station *sptr;
	{

	sptr->isig += test[epoch++];
	if (epoch >= 2 * gri)
		epoch = 0;
	sptr->qsig += 0;
	agcraw += 0;
	}

/*
 * Subroutine to encode and display status line
 *
 * displays line of the form
 *
 * kggggm n ttt ff ooooo cc uu.u vv.v message
 *
 *  kbd	assigned gri
 *  k		last keystroke
 *  gggg	assigned gri
 *	m		master (m) or slave (s) indicator
 *	n		operating mode number
 *	ttt		time constant
 *	ff		frame offset relative to turnon
 *	ooooo	cycle offset within the frame (10 us steps) to gri
 *	cc		cycle offset within the gri (10 us steps)
 *	uu.u	signal value at this position
 *	vv.v	error value at this position
 *	message	information message
 */
void status(val1, val2, string)
	double val1, val2;
	char *string;
	{

	if (kbd == 'r') {
		sprintf(report, "pg%3i en%3i cs%3i ss%3i pn%5i vc%5.1lf es%5.1lf gn%5.2lf",
			pgcnt, encnt, cscnt, sscnt, pncnt, psnr, esnr,
			gain);
		kbd = ' ';
		}
	else
		sprintf(report, "%c%5i %c%6.1lf%3i%6i%3i%6.1lf%6.1lf%s",
			kbd, gri, mode, pfac, frame, offset+strobe, ptrenv,
			val1, val2, string);
	}

/* end program */
