#include <stdio.h>
#include <math.h>
#include <malloc.h>
#include <string.h>
#include <X11/libsx.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>


/* format of the plot taken from SAWatch */

/*
For satellites
	Green is good
	Red is bad,
	Heavy Border implies used in solution.
For Path/points
	black is DOP < 2
	blue  is DOP < 3
	Red   is DOP > 3
*/

#define uchar unsigned char

extern Dimension pixmap_xmax, pixmap_ymax;
extern double	X0, Y0, Xmax, Ymax;

/*
 * MY coordinate system is (0,0) at lower left
 *			   (5,5) at upper right.
 * LIBSX coordinate system is in pixels,
 *	measured from upper left.
 */

/*
 * NB. Navigators measure Azimuth from the North, to the East,
 *     viz ccw, so
 *		x = R*sin(Az)
 *		y = R*cos(Az)
 *     Astronomers measure Azimuth from the South to the West.
 */

struct point {
	struct point *next;
	double	dLat;
	double	dLon;
	double	dz;
	float	dop;
} *Point;

struct Sat {
	int	active;
	int	az;
	int	el;
	int	health;
	int	used;
} sat[33];

void circle(double x, double y, double R);
void draw_lines(void);
void draw_point(struct point *);
void draw_mean(void);
void draw_path(void);
void draw_sats(void);
void frame(void);
void Text(double, double, char *);
void quit(Widget w, void *data);
void print_ps(Widget w, void *data);
void nada(Widget w, int width, int height, void *data);
void set_color(void);

void radio0(Widget w, void *data);
void radio1(Widget w, void *data);
void radio2(Widget w, void *data);
void radio3(Widget w, void *data);
void radio4(Widget w, void *data);
void radio5(Widget w, void *data);
void radio6(Widget w, void *data);
void radio7(Widget w, void *data);
void radio8(Widget w, void *data);
void radio9(Widget w, void *data);
void radio10(Widget w, void *data);

double	pi = 3.1415926535;
double	Rearth = 6.371e6;	/* m */
double	RRmax = 1.6875;

int	plot_choice = 0;
int	size;
char	*file;
uchar	*Buf;

XFont	f;
int	fd;

int	lines = 1;
int	deg_ft = 1;
int	show_sats = 1;
int	show_path = 1;

int	ht_type0;
double	Lat0, Lon0, Ht0, Rmax = 100.;
double	MdLat, MdLon, MdZ;
int	Mn;

int	year, mo, day, hour, minute, second;
int	timeout = 11000;       /* miliseconds */
int	Ea_seen, As_seen;


Widget wb, wd, wt[11], wp;

void	plots(), axis(), plot(), line(), satellite();
void	draw(), read_data();
void	oncore_msg_do_Ea(), oncore_msg_Ea(), oncore_msg_As(), oncore_msg_Bb(), do_time();

struct Msg {
	char	     c[5];
	unsigned int seq;
	void	     (*go_to)(uchar *);
};

struct Msg Hdr[] = { {"@@Bb", 0, &oncore_msg_Bb},
		     {"@@Ea", 0, &oncore_msg_Ea},
		     {"@@As", 0, &oncore_msg_As}};

/*
 * Colors
 *   1 red
 *   2 blue
 *   3 green
 *   4 black
 *   5 white
 *   6 magenta
 *   7 orange
 *   8 gray
 */

int	Red = 1, Blue = 2, Green = 3, Black = 4, White = 5, Magenta = 6, Orange = 7, Gray = 8, Color; /* flags */
int	red, blue, green, black, white, magenta, orange, gray;	/* libsx colors */

/* size of drawing area in pixels */
#define XSIZE	500
#define YSIZE	500

#define w32(buf)      (((buf)[0]&0xff) << 24 | \
		       ((buf)[1]&0xff) << 16 | \
		       ((buf)[2]&0xff) <<  8 | \
		       ((buf)[3]&0xff) )

	/* from buffer, char *buf, result to an int */
#define buf_w32(buf) (((buf)[0]&0200) ? (-(~w32(buf)+1)) : w32(buf))

struct stat statbuf;


int
main(int argc, char *argv[])
{
	argc = OpenDisplay(argc, argv);
	if (argc == 0)
		exit(5);

	wt[0] = MakeToggle("300m", FALSE,  NULL, radio0, NULL);
	wt[1] = MakeToggle("150m", FALSE, wt[0], radio1, NULL);
	wt[2] = MakeToggle("100m",  TRUE, wt[0], radio2, NULL);
	wt[3] = MakeToggle( "50m", FALSE, wt[0], radio3, NULL);

	wt[4] = MakeToggle("Lines",  TRUE,  NULL,  radio4, NULL);
	wt[5] = MakeToggle("Points", FALSE, wt[4], radio5, NULL);

	wt[6] = MakeToggle("Offset-Deg",  TRUE,  NULL,  radio6, NULL);
	wt[7] = MakeToggle("Offset-M",    FALSE, wt[6], radio7, NULL);

	wt[10]= MakeToggle("Both",        TRUE,  NULL,   radio10, NULL);
	wt[8] = MakeToggle("Sat-only",    FALSE, wt[10], radio8,  NULL);
	wt[9] = MakeToggle("Path-only",   FALSE, wt[10], radio9,  NULL);

	wb = MakeButton("Quit", quit, NULL);

	wd = MakeDrawArea(XSIZE, YSIZE, nada, NULL);

	wp = MakeButton("Print", print_ps, NULL);

	SetWidgetPos(wb, PLACE_UNDER, wt[0], NO_CARE, NULL);
	SetWidgetPos(wd, PLACE_UNDER, wb, NO_CARE, NULL);
	SetWidgetPos(wp, PLACE_RIGHT, wt[7], NO_CARE, NULL);

	SetWidgetPos(wt[1],  PLACE_RIGHT, wt[0], NO_CARE, NULL);
	SetWidgetPos(wt[2],  PLACE_RIGHT, wt[1], NO_CARE, NULL);
	SetWidgetPos(wt[3],  PLACE_RIGHT, wt[2], NO_CARE, NULL);
	SetWidgetPos(wt[4],  PLACE_RIGHT, wt[3], NO_CARE, NULL);
	SetWidgetPos(wt[5],  PLACE_RIGHT, wt[4], NO_CARE, NULL);
	SetWidgetPos(wt[6],  PLACE_RIGHT, wt[5], NO_CARE, NULL);
	SetWidgetPos(wt[7],  PLACE_RIGHT, wt[6], NO_CARE, NULL);
	SetWidgetPos(wt[8],  PLACE_RIGHT, wb,	 PLACE_UNDER, wt[0]);
	SetWidgetPos(wt[9],  PLACE_RIGHT, wt[8], PLACE_UNDER, wt[0]);
	SetWidgetPos(wt[10], PLACE_RIGHT, wt[9], PLACE_UNDER, wt[0]);

	red = GetNamedColor("red");
	black = GetNamedColor("black");
	white = GetNamedColor("white");
	blue = GetNamedColor("blue");
	green = GetNamedColor("green");
	magenta = GetNamedColor("magenta");
	orange = GetNamedColor("orange");
	gray = GetNamedColor("gray");

	ShowDisplay();
	GetStandardColors();

	if (!(f=GetFont("-adobe-helvetica-medium-r-*-*-15-*-*-*-*-*-*-*"))) {
		fprintf(stderr, "Cant get Helvetica 8pt\n");
		exit(1);
	}

	file = "/var/adm/ntpstats/ONCORE";
	if ((fd=open(file, O_RDONLY)) < 0) {
		fprintf(stderr, "Cant open %s\n", file);
		exit(1);
	}

	if (stat(file, &statbuf) < 0) {
		fprintf(stderr, "Cant stat %s\n", file);
		exit(1);
	}

	size = statbuf.st_size;
	if ((Buf=mmap(0, size, PROT_READ, MAP_SHARED, fd, (off_t) 0)) < 0) {
		fprintf(stderr, "MMAP failed\n");
		exit(1);
	}

	AddTimeOut(100, read_data, NULL);

	MainLoop();
	exit(0);
}


void
quit (Widget w, void *data)
{
	exit(0);
}

void
nada (Widget w, int width, int height, void *data)
{
	draw();
}

void
radio0 (Widget w, void *data)
{
	Rmax = 300.;
	draw();
}

void
radio1 (Widget w, void *data)
{
	Rmax = 150.;
	draw();
}

void
radio2 (Widget w, void *data)
{
	Rmax = 100;
	draw();
}

void
radio3 (Widget w, void *data)
{
	Rmax = 50.;
	draw();
}

void
radio4 (Widget w, void *data)
{
	lines = 1;
	draw();
}

void
radio5 (Widget w, void *data)
{
	lines = 0;
	draw();
}

void
radio6 (Widget w, void *data)
{
	deg_ft = 1;
	draw();
}

void
radio7 (Widget w, void *data)
{
	deg_ft = 0;
	draw();
}

void
radio8 (Widget w, void *data)
{
	show_sats = 1;
	show_path = 0;
	draw();
}

void
radio9 (Widget w, void *data)
{
	show_sats = 0;
	show_path = 1;
	draw();
}

void
radio10(Widget w, void *data)
{
	show_sats = 1;
	show_path = 1;
	draw();
}

void
read_data()
{
	int	i, j, k, n, iseq, jseq;
	uchar	*cp, *cp1;


	for(cp=Buf+1; (n = 256*(*cp) + *(cp+1)) != 0;  cp+=(n+3)) {
		for (k=0; k<sizeof(Hdr)/sizeof(Hdr[0]);  k++) {
			if (!strncmp(cp+3, Hdr[k].c, 4)) {	/* am I interested? */
				iseq = *(cp+2);
				jseq = Hdr[k].seq;
				Hdr[k].seq = iseq;
				if (iseq > jseq) {		/* has it changed? */
					/* verify checksum */
					j = 0;
					cp1 = cp+3;		/* points to start of oncore response */
					for (i=2; i<n-3; i++)
						j ^= cp1[i];
					if (j == cp1[n-3]) {	/* good checksum */
						Hdr[k].go_to(cp1);
					} else {
						fprintf(stderr, "Bad Checksum for %s\n", Hdr[k].c);
						break;
					}
				}
			}
		}
		if (!strncmp(cp+3, "@@Ea", 4))
			cp += 3*(n+3);
		if (!strncmp(cp+3, "@@Cb", 4))
			cp += 34*(n+3);
	}
	draw();
	AddTimeOut(timeout, read_data, NULL);
}


void
oncore_msg_Bb(uchar *buf)
{
	int i, j, id, nsat;

	for (i=0; i<33; i++) {
		sat[i].active = 0;
	}
	nsat = buf[4];
	nsat = (nsat > 12) ? 12 : nsat; /* manual lies, this can be 13 */
	j = 4;
	for (i=0; i<nsat; i++) {
		id = buf[j+1];
		if (id < 1 || id > 32) {
			fprintf(stderr, "Error in sat id = %d\n", id);
		} else {
			sat[id].el = buf[j+4];
			sat[id].az = 256*buf[j+5] + buf[j+6];
			sat[id].health = buf[j+7];
			sat[id].active = 1;
		}
		j += 7;
	}
}


/*
 * get Position hold position
 */
void
oncore_msg_As(u_char *buf)
{
	long lat, lon, ht;

	As_seen = 1;
	lat	 = buf_w32(&buf[4]);
	lon	 = buf_w32(&buf[8]);
	ht	 = buf_w32(&buf[12]);
	ht_type0 = buf[16];

	Lat0 = lat/3600000.;	/* to degrees */
	Lon0 = lon/3600000.;	/* to degrees */
	Ht0  = ht/100.; 	/* to m */
}


void
oncore_msg_Ea(uchar *buf)
{
	int	i, j, k, n, iseq, jseq;
	static int Jseq[3];
	uchar	 *cp, *cp1;

	cp = buf-3;
	n = 256*(*cp) + *(cp+1);

	for (k=0; k<3; k++) {	/* look at next three */
		cp += (n+3);
		iseq = *(cp+2);
		jseq = Jseq[k];
		Jseq[k] = iseq;
		if (iseq > jseq) {		/* has it changed? */
			/* verify checksum */
			j = 0;
			cp1 = cp+3;		/* points to start of oncore response */
			for (i=2; i<n-3; i++)
				j ^= cp1[i];
			if (j == cp1[n-3]) {	/* good checksum */
				oncore_msg_do_Ea(cp1);
			} else {
				fprintf(stderr, "Bad Checksum for Ea[%d]\n", k);
				break;
			}
		}
	}
}


void
oncore_msg_do_Ea(uchar *buf)
{
	double	LLat, LLon, HHt, ht_gps, ht_sl, dLat, dLon;
	double	dz, xdop, Lat, Lon, Ht;
	struct point *pt;
	int	i, dop, dop_type, id;


	if (!(buf[72]&040))
		return;
	Ea_seen = 1;
	LLat  = buf_w32(&buf[15]);
	LLon  = buf_w32(&buf[19]);
	ht_gps= buf_w32(&buf[23]);  /* GPS ellipse */
	ht_sl = buf_w32(&buf[27]);  /* MSL */
	HHt   = (ht_type0) ? ht_sl : ht_gps;

	Lat = LLat/3600000.;		/* from mili arc sec  to deg */
	Lon = LLon/3600000.;
	Ht  = HHt/100.; 		/* from CM to m */

	/********************************/

	mo   = buf[4];
	day  = buf[5];
	year = buf[6]*256+buf[7];
	hour = buf[8];
	minute = buf[9];
	second = buf[10];

	/********************************/

	if (!As_seen)
		return;

	dop = 256*buf[35]+buf[36];
	dop_type = buf[37];
	xdop = dop;
	xdop /= 10.;

	dLat = Lat - Lat0;
	dLon = Lon - Lon0;
	dz = Ht - Ht0;

	/********************************/

	/* compute running mean */

	MdLat = (Mn*MdLat + dLat)/(Mn+1);
	MdLon = (Mn*MdLon + dLon)/(Mn+1);
	MdZ   = (Mn*MdZ   + dz	)/(Mn+1);
	Mn++;

	/********************************/

	for (i=1; i<33; i++)
		sat[i].used = 0;

	for (i=40; i<72; i+=4) {
		id = buf[i];
		if (id < 1 || id > 32)
			continue;

		sat[id].used = buf[i+3]&0x80;
	}

	/********************************/

	pt = Point;
	Point = (struct point *) malloc(sizeof(struct point));
	Point->next = pt;
	Point->dLat = dLat;
	Point->dLon = dLon;
	Point->dz   = dz;
	Point->dop  = xdop;
}


void
draw_point(struct point *pt)
{
	double dx, dy, dz, xdop, rscal, dLat, dLon;
	double dLat_rad, dLon_rad, Lat_rad;

	if (!pt)
		return;

	rscal = RRmax/Rmax;
	dLat = pt->dLat;
	dLon = pt->dLon;
	dz   = pt->dz;
	xdop = pt->dop;

	dLat_rad = 2.0*pi*(dLat/360.);
	dy = Rearth * dLat_rad;

	dLon_rad = 2.0*pi*(dLon/360.);
	Lat_rad  = 2.0*pi*(Lat0/360.);
	dx = Rearth*cos(Lat_rad)*dLon_rad;

	if (dx*dx+dy*dy < Rmax*Rmax) {
		if (pt->dop < 2.0) {
			Color = Black;
		} else if (pt->dop < 3.0) {
			Color = Blue;
		} else {
			Color = Red;
		}
		set_color();
		plot (2.5+rscal*dx-0.02, 2.5+rscal*dy, 3);
		plot (2.5+rscal*dx+0.02, 2.5+rscal*dy, 2);
		plot (2.5+rscal*dx, 2.5+rscal*dy-0.02, 3);
		plot (2.5+rscal*dx, 2.5+rscal*dy+0.02, 2);
	}
	Color = Black;
	set_color();
}


void
draw_path()
{
	struct point	*pt;

	if (!show_path)
		return;

	if (!Point)
		return;


	if (lines)
		draw_lines();
	else
		for (pt=Point; pt; pt=pt->next)
			draw_point(pt);
}

void
draw_lines()
{
	struct point	*pt;
	double	rscal, dLat_rad, dLat, dLon, Lat_rad;
	double	dLon_rad, x0, y0, x1, y1, x3, y3, tmp, a, b;
	int	edge;

	rscal = RRmax/Rmax;

	dLat = Point->dLat;
	dLon = Point->dLon;

	dLat_rad = 2.0*pi*(dLat/360.);
	y0 = Rearth * dLat_rad;

	dLon_rad = 2.0*pi*(dLon/360.);
	Lat_rad  = 2.0*pi*(Lat0/360.);
	x0 = Rearth*cos(Lat_rad)*dLon_rad;

	edge = 1;
	if (x0*x0+y0*y0 < Rmax*Rmax) {
		edge = 0;
		plot (2.5+rscal*x0, 2.5+rscal*y0, 3);
	}

	for (pt=Point->next; pt; pt=pt->next) {

		if (pt->dop < 2.0) {
			Color = Black;
		} else if (pt->dop < 3.0) {
			Color = Blue;
		} else {
			Color = Red;
		}
		set_color();

		dLat = pt->dLat;
		dLon = pt->dLon;

		dLat_rad = 2.0*pi*(dLat/360.);
		y1 = Rearth * dLat_rad;

		dLon_rad = 2.0*pi*(dLon/360.);
		Lat_rad  = 2.0*pi*(Lat0/360.);
		x1 = Rearth*cos(Lat_rad)*dLon_rad;

		/* do the business at the edge of the graph correctly */

		if (x1*x1+y1*y1 > Rmax*Rmax)
			edge = edge + 2;

	/*	edge ((x1,y1),	(x0,y0))
	 *	00 = 0 : inside->inside
	 *	01 = 1 : outside->inside
	 *	10 = 2 : inside->outside
	 *	11 = 3 : outside->outside
	 */

		switch(edge) {
		case 0:
			plot (2.5+rscal*x1, 2.5+rscal*y1, 2);
			break;
		case 3:
			break;
		case 1: 		/* x1 -> x0 */
			tmp= x0;	/* exchange points */
			x0 = x1;	/* x0 on inside */
			x1 = tmp;
			tmp= y0;
			y0 = y1;
			y1 = tmp;
		case 2: 		/* x0 -> x1 */
			if (x0 == x1) { /* unlikely special case */
				x3 = x0;
				y3 = sqrt(Rmax*Rmax - x3*x3);
				if (y1 < 0.)	/* outside point */
					y3 = -y3;
			} else {	/* normal case */
				double x3a, x3b;

				b = (y1-y0)/(x1-x0);
				a = y0-b*x0;		/* nb, y = ax+b */
				tmp = sqrt((1.0+b*b)*Rmax*Rmax -a*a);
				x3 = (-a*b+tmp)/(1.0+b*b);
				x3a = x3;
				x3b = (-a*b-tmp)/(1.0+b*b);

				if (((x3 >= x0) && (x3 >= x1)) || ((x3 <= x0) && (x3 <= x1)))
					x3 = (-a*b-tmp)/(1.0+b*b);
				y3 = a+b*x3;
			}
			switch (edge) {
			case 1:
				plot(2.5+rscal*x3, 2.5+rscal*y3, 3);
				plot(2.5+rscal*x0, 2.5+rscal*y0, 2);
				x1 = x0;	/* exchange twice */
				y1 = y0;
				break;
			case 2:
				plot(2.5+rscal*x3, 2.5+rscal*y3, 2);
				break;
			}
			break;
		}
		x0 = x1;
		y0 = y1;
		edge = edge >> 1;
	}
	Color = Black;
	set_color();
}


void
draw()
{
	plots(0., 0., 5., 5.);
	ClearDrawArea();

	axis();
	draw_sats();
	draw_path();
	draw_mean();

	SyncDisplay();
}


void
print_ps(Widget w, void *data)
{
	plot_choice = 2;
	plots(0., 0., 5., 5.);

	axis();
	draw_sats();
	draw_path();
	draw_mean();

	frame();
	plot_choice = 0;
}

void
draw_sats()
{
	int	i;

	if (!show_sats)
		return;

	for (i=1; i<33; i++) {
		if (sat[i].active) {
			Color = sat[i].health ? Red : Green;	/* Sat OK? */
			satellite(i, sat[i].az, sat[i].el, sat[i].used);
		}
	}
}


void
draw_mean()
{
	int	i;
	struct point *pt, *pt1, *pt2;
	double	dx, dy, dLat_rad, Lat_rad, dLon_rad, rscal;

	pt = Point;
	i = 0;
	for (pt=Point; pt; pt=pt->next)
		i++;

	if (!i)
		return;

#if 1
	if (i > 3000) { /* trim to 1000 */
		i = 0;
		for (pt=Point; pt && i<1000; pt=pt->next)
			i++;
		pt2 = pt;

		for(pt=pt->next; pt; pt=pt1) {
			pt1 = pt->next;
			free(pt);
		}
		pt2->next = 0;
	}
#endif
	/* convert to inches */

	dLat_rad = 2.0*pi*(MdLat/360.);
	dy = Rearth * dLat_rad;

	dLon_rad = 2.0*pi*(MdLon/360.);
	Lat_rad  = 2.0*pi*(Lat0/360.);
	dx = Rearth*cos(Lat_rad)*dLon_rad;

	rscal = RRmax/Rmax;

	Color = Magenta;
	set_color();
	circle(2.5+rscal*dx, 2.5+rscal*dy, 0.1);
}


void
axis()
{
	int	j;
	double x, y;
	double RR[3] = { 1.6875, 1.125, 0.5625};
	char	line1[10], line2[10], line3[10];

	Color = Black;
	set_color();
	SetBgColor(wd, white);

	plot(2.5, 2.5-1.8, 3);
	plot(2.5, 2.5+1.8, 2);

	plot(2.5-1.8, 2.5, 3);
	plot(2.5+1.8, 2.5, 2);

	plot(2.5+0.9, 2.5+1.56, 3);
	plot(2.5-0.9, 2.5-1.56, 2);

	plot(2.5-0.9, 2.5+1.56, 3);
	plot(2.5+0.9, 2.5-1.56, 2);

	plot(2.5+1.56, 2.5+0.9, 3);
	plot(2.5-1.56, 2.5-0.9, 2);

	plot(2.5-1.56, 2.5+0.9, 3);
	plot(2.5+1.56, 2.5-0.9, 2);

	Text( 2.49,  4.25, "N");
	Text( 2.49,  0.6 , "S");
	Text( 0.6 ,  2.45, "W");
	Text( 4.35,  2.45, "E");

	x = 4.5;
	y = 2.5-RRmax;
	Text(x, y, "Horiz");

	y = 2.5-1.125;
	Text(x, y, "30 deg");

	y = 2.5-0.5625;
	Text(x, y, "60 deg");

	y = 2.5-0.15;
	Text(x, y, "90 deg");

	y = 2.5+0.05;
	Text(x, y, "0 m");

	sprintf(line1, "%d m", (int) (Rmax/3.0));
	sprintf(line2, "%d m", (int) (2.0*Rmax/3.0));
	sprintf(line3, "%d m", (int) Rmax);

	y = 2.5+0.5625;
	Text(x, y, line1);

	y = 2.5+1.125;
	Text(x, y, line2);

	y = 2.5+RRmax;
	Text(x, y, line3);

	for (j=0; j<3; j++)
		circle(2.5, 2.5, RR[j]);

	if (Ea_seen && As_seen) {
		int	lat_sign, lon_sign;
		long	lat, lon, latms, lonms, latm, lonm, latd, lond;
		double	Lats, Lons;
		char	line[100];
		char	*Mo[12] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jly", "Aug", "Sep", "Oct", "Nov", "Dec"};
		time_t	t;
		struct tm tm, *tmp;

		lat = 3600000*Lat0;	/* mas */
		lon = 3600000*Lon0;

		lat_sign = (lat) ? +1 : -1;
		lon_sign = (lon) ? +1 : -1;

		lat = (lat>0) ? lat : -lat;
		lon = (lon>0) ? lon : -lon;

		latms = lat % 60000;
		lonms = lon % 60000;

		Lats = latms/1000.;
		Lons = lonms/1000.;

		latm = lat/60000;
		lonm = lon/60000;

		latd = latm/60;
		lond = lonm/60;

		latm %= 60;
		lonm %= 60;

		Text(0.25, 5.0-0.15, "Time:");
		sprintf(line, "%2d%3s%4d %02d:%02d:%02d UTC", day, Mo[mo-1], year, hour, minute, second);
		Text(0.25, 5.0-0.30, line);
		tm.tm_sec = second;
		tm.tm_min = minute;
		tm.tm_hour = hour;
		tm.tm_mday = day;
		tm.tm_mon  = mo-1;
		tm.tm_year = year-1900;
		t = mktime(&tm);
		t -= timezone;
		tmp = localtime(&t);
		sprintf(line, "%2d%3s%4d %02d:%02d:%02d %s", tmp->tm_mday, Mo[tmp->tm_mon], tmp->tm_year+1900, tmp->tm_hour, tmp->tm_min, tmp->tm_sec, ((tmp->tm_isdst > 0) ? tzname[1] : tzname[0]));
		Text(0.25, 5.0-0.45, line);
		Text(0.25, 0.55, "Assumed Posn:");
		sprintf(line, "Lat = %c %3lddeg %2ldm %6.3fs", (lat_sign>0) ? 'N' : 'S', latd, latm, Lats);
		Text(0.25, 0.40, line);
		sprintf(line, "Lon = %c %3lddeg %2ldm %6.3fs", (lon_sign>0) ? 'W' : 'E', lond, lonm, Lons);
		Text(0.25, 0.25, line);
		sprintf(line, "Ht  = %.3fm", Ht0);
		Text(0.25, 0.10, line);

		sprintf(line, "Mean Offset: (N = %d)", Mn);
		Text(3.0, 0.55, line);

		if (deg_ft) {
			lat = 3600000*MdLat;	/* mas */
			lon = 3600000*MdLon;

			lat_sign = (lat>0) ? +1 : -1;
			lon_sign = (lon>0) ? +1 : -1;

			lat = (lat>0) ? lat : -lat;
			lon = (lon>0) ? lon : -lon;

			latms = lat % 60000;
			lonms = lon % 60000;

			Lats = latms/1000.;
			Lons = lonms/1000.;

			latm = lat/60000;
			lonm = lon/60000;

			latd = latm/60;
			lond = lonm/60;

			latm %= 60;
			lonm %= 60;

			sprintf(line, "dLat = %c %3lddeg %2ldm %6.3fs", (lat_sign>0) ? '+' : '-', latd, latm, Lats);
			Text(3.0, 0.40, line);
			sprintf(line, "dLon = %c %3lddeg %2ldm %6.3fs", (lon_sign>0) ? '+' : '-', lond, lonm, Lons);
			Text(3.0, 0.25, line);
			sprintf(line, "dHt  = %.3fm", MdZ);
			Text(3.0, 0.10, line);
		} else {
			double dLat, dLon, dLat_rad, dLon_rad, Lat_rad, dx, dy;

			dLat = MdLat;
			dLon = MdLon;

			dLat_rad = 2.0*pi*(dLat/360.);
			dy = Rearth * dLat_rad;

			dLon_rad = 2.0*pi*(dLon/360.);
			Lat_rad  = 2.0*pi*(Lat0/360.);
			dx = Rearth*cos(Lat_rad)*dLon_rad;
			sprintf(line, "dLat = %8.3fm", dy);
			Text(3.0, 0.40, line);
			sprintf(line, "dLon = %8.3fm", dx);
			Text(3.0, 0.25, line);
			sprintf(line, "dHt  = %8.3fm", MdZ);
			Text(3.0, 0.10, line);
		}
	}
}
