/* 
This software is being provided to you, the LICENSEE, by the
Massachusetts Institute of Technology (M.I.T.) under the following
license.  By obtaining, using and/or copying this software, you agree
that you have read, understood, and will comply with these terms and
conditions:

Permission to use, copy, modify and distribute, including the right to
grant others the right to distribute at any tier, this software and
its documentation for any purpose and without fee or royalty is hereby
granted, provided that you agree to comply with the following
copyright notice and statements, including the disclaimer, and that
the same appear on ALL copies of the software and documentation,
including modifications that you make for internal use or for
distribution:

Copyright 1992,1993,1994 by the Massachusetts Institute of Technology.
                    All rights reserved.

THIS SOFTWARE IS PROVIDED "AS IS", AND M.I.T. MAKES NO REPRESENTATIONS
OR WARRANTIES, EXPRESS OR IMPLIED.  By way of example, but not
limitation, M.I.T. MAKES NO REPRESENTATIONS OR WARRANTIES OF
MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE USE
OF THE LICENSED SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY THIRD
PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.

The name of the Massachusetts Institute of Technology or M.I.T. may
NOT be used in advertising or publicity pertaining to distribution of
the software.  Title to copyright in this software and any associated
documentation shall at all times remain with M.I.T., and USER agrees
to preserve same.
*/
/*
 * Gregory D. Troxel
 */

#include "pd.h"

dataset_t ntploop;		/* ntp local clock logs */
dataset_t raw[NCLOCKS];		/* raw aoffset observations */
dataset_t *ntpp = &ntploop;

int clock_n = 0;
clock_state_t clock_state[NCLOCKS];

/* convert strings to numbers, assigning numbers as needed */
int
clock_parse(char *s)
{
  int i;

  if ( strlen(s) > CLOCKLEN - 1)
    {
      fprintf(stderr, "CLOCK <%s> too long\n", s);
      s[CLOCKLEN - 1] = '\0';
    }

  for ( i = 0; i <= CLOCK_MAX(); i++ )
    if ( ! strcmp(s, clock_state[i].name) )
      return i;			/* matched, so return index */

  if ( clock_n > NCLOCKS - 1 )
    panic("Too many clocks");

  strncpy(clock_state[clock_n].name, s, CLOCKLEN-1);
  clock_state[clock_n].flags = 0;
  return clock_n++;	/* newly added index */
}

/*
 * read clock config file
 * format is "name flags offset"
 * flags 0x1 integrated 0x2 dont write
 * offset is systematic amount for integrated fit
 */
void
clock_config(FILE *fin, FILE *flog)
{
  char inputbuf[1024];
  char clockname[1024];
  double offset;
  int flags;
  int i;

  fprintf(flog, "CLOCK_CONFIG:\n");
  while ( ! feof(fin) )
    {
      /* read line and trim newline */
      {
	char *p;
	
	inputbuf[0] = 0;
	errno = 0;
	
	if ( fgets(inputbuf, 1024, fin) == NULL )
	  {
	    if ( errno != 0 )
	      panic("reading clock config file");
	    continue;
	  }
	
	/* skip blank lines and comments*/
	if ( inputbuf[0] == 0 || inputbuf[0] == '\n' || inputbuf[0] == '#' )
	  continue;
	
	/* trim newline */
	p = &inputbuf[strlen(inputbuf) - 1];
	if ( *p != '\n' )
	  panic("line too long");
	*p = '\0';
      }

      /* read clock name */
      i = sscanf(inputbuf, "%s %d %lf", clockname, &flags, &offset);
      if ( i != 2 && i != 3)
	{
	  fprintf(flog, "couldn't read flags on <%s>\n", inputbuf);
	  continue;
	}
      if ( i != 3 )
	offset = 0.0;

      i = clock_parse(clockname);
      clock_state[i].flags = flags;

      /* offset is in ms */
      clock_state[i].offset = offset / 1E3;

      fprintf(flog, "Clock %2d [%15s] flags %d offset %6.3f ms\n",
	      i,  clockname, flags, clock_state[i].offset * 1E3);
    }
}

void
clock_state_print(FILE *finfo, int cl)
{
  fprintf(finfo,
	  "CLOCK_STATE %2d [%15s] flags %2d",
	  cl, clock_unparse(cl), clock_state[cl].flags);
  if ( clock_state[cl].offset_valid )
    fprintf(finfo, " offset %6.3f ms\n", clock_state[cl].offset * 1E3);
  else
    if ( clock_state[cl].offset != 0 )
      fprintf(finfo, " offset %6.3f ms INVALID\n", clock_state[cl].offset*1E3);
    else
      fprintf(finfo, "\n");

  if ( clock_state[cl].valid )
    {
      fprintf(finfo, "  BEGIN %s END %s\n",
	      mytime(Tntp2unix(clock_state[cl].begin)),
	      mytime(Tntp2unix(clock_state[cl].end)));
      if ( clock_state[cl].count > 0 )
	fprintf(finfo, "  NEWBEGIN %s COUNT %d\n",
		mytime(Tntp2unix(clock_state[cl].newbegin)),
		clock_state[cl].count);
    }
}

void
clock_state_read(FILE *fin, FILE *flog)
{
  int i;
  char inputbuf[1024];
  int cl;
  char clockname[1024];
  int offset_valid;
  ntp_t begin, end, newbegin;
  int count;
  double offset;

  fprintf(flog, "CLOCK_STATE_READ\n");
  while ( ! feof(fin) )
    {
      /* read line and trim newline */
      {
	char *p;
	
	inputbuf[0] = 0;
	errno = 0;
	
	if ( fgets(inputbuf, 1024, fin) == NULL )
	  {
	    if ( errno != 0 )
	      panic("reading clock state file");
	    continue;
	  }
	
	/* skip blank lines */
	if ( inputbuf[0] == 0 || inputbuf[0] == '\n' )
	  continue;
	
	/* trim newline */
	p = &inputbuf[strlen(inputbuf) - 1];
	if ( *p != '\n' )
	  panic("line too long");
	*p = '\0';
      }

      /* read clock name */
      i = sscanf(inputbuf, "%s %u %u %u %d %d %lf",
		 clockname,
		 &begin, &end, &newbegin, &count,
		 &offset_valid, &offset);
      if ( i != 7 )
	{
	  fprintf(flog, "clockstate: couldn't read <%s>\n", inputbuf);
	  continue;
	}

      cl = clock_parse(clockname);

      clock_state[cl].valid = 1;
      clock_state[cl].begin = begin;
      clock_state[cl].end = end;
      clock_state[cl].newbegin = newbegin;
      clock_state[cl].count = count;

      if ( flags.debug > 0 )
	clock_state_print(flog, cl);

      if ( offset_valid )
	{
	  clock_state[cl].offset_valid = 1;
	  clock_state[cl].offset = offset;
	  if ( flags.debug > 0 )
	    fprintf(flog, " offset %6.3f ms\n", clock_state[cl].offset * 1E3);
	}
    }
}

void
clock_state_write(FILE *fout, FILE *flog)
{
  int cl;

  for ( cl = 0; cl <= CLOCK_MAX(); cl++ )
    if ( clock_state[cl].valid )
      fprintf(fout, "%s %u %u %u %d %d %.6f\n",
	      clock_unparse(cl),
	      clock_state[cl].begin,
	      clock_state[cl].end,
	      clock_state[cl].newbegin,
	      clock_state[cl].count,
	      clock_state[cl].offset_valid,
	      clock_state[cl].offset);
}

/*
 * if no clocks valid for integrated, use them all
 */
void
clock_state_makevalid(FILE *finfo)
{
  int i, nvalid;

  for ( i = 0, nvalid = 0; i <= CLOCK_MAX(); i++ )
    {
      clock_state_print(finfo, i);
      if ( clock_state[i].flags & CLOCK_USE_INTEGRATED )
	nvalid++;
    }

  if ( nvalid > 0 )
    fprintf(finfo, "VALID CLOCKS = %d\n", nvalid);
  else
    {
      for ( i = 0; i <= CLOCK_MAX(); i++ )
	clock_state[i].flags |= CLOCK_USE_INTEGRATED;
      fprintf(finfo, "VALID CLOCKS = 0  ==> setting all clocks valid\n");
    }
  fflush(finfo);
}

void
Invalid_adjweight(FILE *finfo, int clock, double frac)
{
  if ( ! flags.predict )
    fprintf(finfo, "Invalid fraction = %f\tWeight before=%f\t",
	    frac, raw[clock].weight);
  raw[clock].weight /= 1 + pow(frac/0.25, 3.0);
  if ( ! flags.predict )
    fprintf(finfo, "Weight after=%f\n",
	    raw[clock].weight);
}

void
sdev_adjweight(FILE *finfo, int clock, double sdev)
{
  if ( 1 || ! flags.predict )
    fprintf(finfo, "SDEV = %f\tWeight before=%f\t",
	    sdev, raw[clock].weight);
  raw[clock].weight /= sdev;
  if ( 1 || ! flags.predict )
    fprintf(finfo, "Weight after=%f\n",
	    raw[clock].weight);
}

#ifdef THIS_NOT_USED
void
mul_adjweight(FILE *finfo, int clock, double mul)
{
  if ( 1 || ! flags.predict )
    fprintf(finfo, "ADJUST weight by %f\tWeight before=%f\t",
	    mul, raw[clock].weight);
  raw[clock].weight *= mul;
  if ( 1 || ! flags.predict )
    fprintf(finfo, "Weight after=%f\n",
	    raw[clock].weight);
}
#endif
