/* 
This software is being provided to you, the LICENSEE, by the
Massachusetts Institute of Technology (M.I.T.) under the following
license.  By obtaining, using and/or copying this software, you agree
that you have read, understood, and will comply with these terms and
conditions:

Permission to use, copy, modify and distribute, including the right to
grant others the right to distribute at any tier, this software and
its documentation for any purpose and without fee or royalty is hereby
granted, provided that you agree to comply with the following
copyright notice and statements, including the disclaimer, and that
the same appear on ALL copies of the software and documentation,
including modifications that you make for internal use or for
distribution:

Copyright 1992,1993,1994 by the Massachusetts Institute of Technology.
                    All rights reserved.

THIS SOFTWARE IS PROVIDED "AS IS", AND M.I.T. MAKES NO REPRESENTATIONS
OR WARRANTIES, EXPRESS OR IMPLIED.  By way of example, but not
limitation, M.I.T. MAKES NO REPRESENTATIONS OR WARRANTIES OF
MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE USE
OF THE LICENSED SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY THIRD
PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.

The name of the Massachusetts Institute of Technology or M.I.T. may
NOT be used in advertising or publicity pertaining to distribution of
the software.  Title to copyright in this software and any associated
documentation shall at all times remain with M.I.T., and USER agrees
to preserve same.
*/
/*
 * Gregory D. Troxel
 */

/*
 * initialize "constants" controlling the program
 * to change a value, create a file 'CONFIG' in the machine's directory
 * and place lines
 * "foo value"
 * where foo is EXACTLY the string in the CFG_* macro calls below,
 * and containing exactly one space before the value
 * for example, a line "CONFIG_DELAY 0" will turn off marking invalid
 * high-delay points
 */

#include "pd.h"

config_t config;

/*
 * make a table describing config names,
 */

typedef struct config_read_s
{
  char *name;
  void *ptr;
  int typecode;
  int intval;
  ntp_t ntpval;
  double dblval;
} config_read_t;


#define CFG_INT_T	0
#define CFG_NTP_T	1
#define CFG_DBL_T	2

#define CFG_INT(n, v)	{ #n, &n, CFG_INT_T, v, 0, 0}
#define CFG_NTP(n, v)	{ #n, &n, CFG_NTP_T, 0, v, 0}
#define CFG_DBL(n, v)	{ #n, &n, CFG_DBL_T, 0, 0, v}

config_read_t config_read[] =
{
  CFG_NTP(SIM_BEGIN, 3600),	/* start at 1 hour */
  CFG_NTP(SIM_INCR, 4*3600),	/* advance by 4 hours */

  CFG_INT(EXTENT_MIN, 4*24*3600), /* examine back at least a day */
  CFG_INT(EXTENT_MAX, 7*24*3600), /* and at most a week */
  CFG_DBL(CONFIG_SYSTEMATIC, 0.8), /* update systematic offset estimates */

  CFG_INT(HOURLY_EARLIER_COUNT, 5), /*  */

  CFG_INT(INTEGRATED_MIN_TIME, 8*3600),	/*  */

  CFG_INT(OUTPUT_MIN_POINTS, 5), /* min points to write data */
  CFG_INT(OUTPUT_SHOWNEXTPHASE, 1), /* show phase of this fitseg at next */

  CFG_DBL(FREQ_BOGUS, 1E-3),	/* 1000 ppm */
  CFG_DBL(AGING_BOGUS, 0.1 / ( 86400.0 * 2.0 * KK )), /* 0.1 ppm/day */
  CFG_INT(AGING_TIME_MIN, 3600*48), /* no aging on segments < 48 hours */

  CFG_INT(CONFIG_TRIM, 1),	/* use trim */
  CFG_INT(CONFIG_DELAY, 1),	/* use low delay selection */
  CFG_INT(CONFIG_SDEV_ADJ, 0),	/* adjust weight by sdev */
  CFG_INT(CONFIG_AUTOCORRELATION, 0), /* don't compute autocorrelation */

  CFG_INT(TRIM_POINTS, 20),	/* punt runs with < 20 points */
  CFG_NTP(TRIM_TIME, 2*3600),	/* punt runs with < 2 hours */

  CFG_DBL(DELAY_BASE, 0.05),
  CFG_DBL(DELAY_ADD, 0.50),
  CFG_DBL(DELAY_MUL, 0.75),
  CFG_NTP(DELAY_DECAY, 12*3600), /* age low-delay samples over 12 hours */

  CFG_DBL(HIGHDEV_RATIO, 2.0),
  CFG_DBL(OUTLIER_SELFS_RATIO, 2.0),
  CFG_DBL(OUTLIER_SELFL_RATIO, 2.0),

  CFG_DBL(CONFIG_COMBINE_RATIO_PREDICTION, 2.5), /*  */
  CFG_DBL(CONFIG_COMBINE_RATIO_ANALYSIS, 4.0), /*  */

  CFG_INT(CONFIG_OPTIMIZE_PIECEWISE, 0), /* don't optimize joining */

  CFG_INT(BREAK_POINTS_MIN, 16), /* segment at least 16 points */
  CFG_INT(BREAK_POINTS_MAX, 128), /* segment at most 128 points */
  CFG_NTP(BREAK_TIME_MAX, 4 * 3600), /* segment at most 4 hours */

  CFG_INT(HOURLY_INTEGRATED_LIMIT_N, 16), /* have > 16 valid points */
  CFG_NTP(HOURLY_INTEGRATED_LIMIT_TE, 4*3600), /* cover 4 hours */
  CFG_NTP(HOURLY_INTEGRATED_LIMIT_TR, 4*3600), /* valid point within 4 h */

};

#define CONFIG_READ_N (sizeof(config_read) / sizeof(config_read_t))

/*
 * fill in config with default values from table
 */
void
config_init()
{
  int i;

  for ( i = 0; i < CONFIG_READ_N; i++ )
    switch(config_read[i].typecode)
      {
      case CFG_INT_T:
	* (int *) config_read[i].ptr = config_read[i].intval;
	break;
      case CFG_NTP_T:
	* (ntp_t *) config_read[i].ptr = config_read[i].ntpval;
	break;
      case CFG_DBL_T:
	* (double *) config_read[i].ptr = config_read[i].dblval;
	break;
	
      default:
	panic("unknown type code in config_read");
      }
}

void
config_readfile(FILE *fp, FILE *fout, FILE *ferror)
{
  char buf[1024];
  char *ptr;
  int i;

  while ( fgets(buf, 1024, fp) )
    {
      if ( (ptr = strchr(buf, '\n')) != NULL )
	*ptr = '\0';

      if ( buf[0] == '\0' || buf[0] == '#' || buf[0] == ';' )
	continue;

      if ( (ptr = strchr(buf, ' ')) == NULL )
	{
	  fprintf(ferror, "line <%s> no space\n", buf);
	  continue;
	}
      *ptr++ = '\0';
      for ( i = 0; i < CONFIG_READ_N; i++ )
	if ( strcmp(buf, config_read[i].name) == 0 )
	  {
	    switch(config_read[i].typecode)
	      {
	      case CFG_INT_T:
		* (int *) config_read[i].ptr = atoi(ptr);
		i = CONFIG_READ_N+1;
		break;
	      case CFG_NTP_T:
		sscanf(ptr, "%ui", (unsigned int *) config_read[i].ptr);
		i = CONFIG_READ_N+1;
		break;
	      case CFG_DBL_T:
		* (double *) config_read[i].ptr = atof(ptr);
		i = CONFIG_READ_N+1;
		break;

	      default:
		panic("unknown type code in config_read");
	      }

	    fprintf(fout, "%s %s\n", buf, ptr);

	  }
      if ( i == CONFIG_READ_N )
	  fprintf(ferror, "unknown keyword <%s>, rest <%s>\n", buf, ptr);

    }

}




