/* 
This software is being provided to you, the LICENSEE, by the
Massachusetts Institute of Technology (M.I.T.) under the following
license.  By obtaining, using and/or copying this software, you agree
that you have read, understood, and will comply with these terms and
conditions:

Permission to use, copy, modify and distribute, including the right to
grant others the right to distribute at any tier, this software and
its documentation for any purpose and without fee or royalty is hereby
granted, provided that you agree to comply with the following
copyright notice and statements, including the disclaimer, and that
the same appear on ALL copies of the software and documentation,
including modifications that you make for internal use or for
distribution:

Copyright 1992,1993,1994 by the Massachusetts Institute of Technology.
                    All rights reserved.

THIS SOFTWARE IS PROVIDED "AS IS", AND M.I.T. MAKES NO REPRESENTATIONS
OR WARRANTIES, EXPRESS OR IMPLIED.  By way of example, but not
limitation, M.I.T. MAKES NO REPRESENTATIONS OR WARRANTIES OF
MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE USE
OF THE LICENSED SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY THIRD
PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.

The name of the Massachusetts Institute of Technology or M.I.T. may
NOT be used in advertising or publicity pertaining to distribution of
the software.  Title to copyright in this software and any associated
documentation shall at all times remain with M.I.T., and USER agrees
to preserve same.
*/
/*
 * Gregory D. Troxel
 */

#include "pd.h"

/*
 * print statistics about delays
 * invalidate points with high delay
 */
void
select_low_delay(FILE *finfo, dataset_t *ds, int cl)
{
  int curpoint, direction;
  datapoint_t *dp;
  real_t low_delay;
  ntp_t low_time;
  real_t low, high, p_base, p_add, p_mul;
  real_t low_ok, add_ok, mul_ok;
  int origvalid, invalidated;
  median_t *md;

  if ( ds == NULL )
    panic("NULL ds passed to select_low_delay");

  if ( cl < 0 || cl > CLOCK_MAX() )
    panic("non-raw clock value passed to select_low_delay");
  
  if ( ds->n == 0 )
    {
      fprintf(finfo, "Dataset with 0 points passed to select_low_delay.\n");
      return;
    }

  fprintf(finfo, "\n");

  origvalid = invalidated = 0;
  md = median_init();

  /* iterate over the points, adding them to median object */
  for ( curpoint = ds->extent_begin; curpoint < ds->extent_end; curpoint++ )
    {
      dp = & ds->dptr[curpoint];

#ifdef DIAGNOSTIC
      if ( dp->observation == 0 )
	  panic("select_low_delay called with non-observation");
#endif

      if ( dp->valid != VALID_OK )
	continue;

      median_add(md, dp->delay);
      origvalid += 1;
    }

  if ( origvalid <= 0 )
    {
      fprintf(finfo, "SELECT_LOW_DELAY: No valid points.\n");
      return;
    }

  clock_state[cl].low_delay = low = median_extract(md, 0.0);
  clock_state[cl].high_delay = high = median_extract(md, 1.0);
  
  fprintf(finfo, "LOW DELAY %7.3f ms  HIGH DELAY %7.3f ms\n",
	  low*1E3, high*1E3);
    
  fprintf(finfo,
	  "1%%\t5%%\t10%%\t25%%\t50%%\t75%%\t85%%\t90%%\t95%%\t99%%\n");
  fprintf(finfo,
	  "%3.2f\t%3.2f\t%3.2f\t%3.2f\t%3.2f\t%3.2f\t%3.2f\t%3.2f\t%3.2f\t%3.2f\n",
	  median_extract(md, 0.01) * 1E3,
	  median_extract(md, 0.05) * 1E3,
	  median_extract(md, 0.10) * 1E3,
	  median_extract(md, 0.25) * 1E3,
	  median_extract(md, 0.50) * 1E3,
	  median_extract(md, 0.75) * 1E3,
	  median_extract(md, 0.85) * 1E3,
	  median_extract(md, 0.90) * 1E3,
	  median_extract(md, 0.95) * 1E3,
	  median_extract(md, 0.99) * 1E3);
  
  if ( high == low )
    {
      fprintf(finfo, "ALL DELAYS EQUAL ==> NOT INVALIDATING\n");
      return;
    }
  
  clock_state[cl].base_delay = p_base = median_extract(md, DELAY_BASE);
  clock_state[cl].add_delay = p_add = median_extract(md, DELAY_ADD);
  clock_state[cl].mul_delay = p_mul = median_extract(md, DELAY_MUL);
  
  /* don't let notion of 'low' go too low - avoid being
     hosed by outlying very low delay points */
  low_ok = p_base;
  
  /* allow 50% of points through on additive criteria */
  add_ok = p_add - p_base;
  
  /* decay from 5th to 75th percentile over 12 hours */
  mul_ok = (p_mul - p_base) / p_base;
  fprintf(finfo, "p_base %3.3f p_add %3.3f p_mul %3.3f add %3.3f mul %1.6f\n",
	  p_base * 1E3, p_add * 1E3, p_mul * 1E3,
	  add_ok * 1E3, mul_ok);
  
  /* if not invalidating high-delay points, we are done */
  if ( CONFIG_DELAY == 0 )
    return;

#ifdef DEBUG_LOW_DELAY
  fprintf(finfo, "FORWARD DELAY SCAN\n");
#endif
  direction = 0;		/* forward */
  curpoint = ds->extent_begin;
  low_delay = HUGEDELAY;
  low_time = 0;
  while ( 1 )
    {
      real_t timedistance;
      real_t addexcess, mulexcess;
      real_t mulallowed;

      dp = & ds->dptr[curpoint];

      if ( dp->valid != VALID_OK || dp->observation == 0 ||
	  curpoint >= ds->extent_end )
	goto ITER;

      timedistance = (real_t) dp->time - (real_t) low_time;
      timedistance = fabs(timedistance);
      addexcess = (real_t) dp->delay - (real_t) low_delay;
      mulexcess = addexcess / low_delay;

      /* add delay penalty to low-delay for being far away in time
	 penalty is mul_ok at DELAY_DECAY, and proportional */
      mulallowed = 1 + (mul_ok *
			 (timedistance / DELAY_DECAY));

      if (dp->delay <= low_delay ||
	  dp->delay < low_delay * mulallowed )
	{
	  low_time = dp->time;
	  /* don't let stored low delay drop below low_ok */
	  low_delay = dp->delay > low_ok ? dp->delay : low_ok;
	
/* #define DEBUG_LOW_DELAY */
#ifdef DEBUG_LOW_DELAY
	  fprintf(finfo,
		  "low %d %u %.6f offset %.3f mulexcess %f mulallowed %f\n",
		  curpoint, low_time, low_delay, dp->aoffset,
		  mulexcess, mulallowed);
#endif
	}
      else if ( dp->delay <=
	       low_delay * mulallowed + add_ok )
	{
/* #define DEBUG_MED_DELAY */
#ifdef DEBUG_MED_DELAY
	  fprintf(finfo,
		  "med %d %u %.6f offset %.3f\n",
		  curpoint, dp->time, dp->delay, dp->aoffset);
#endif
	}
      else
	{
	  dp->valid = VALID_DELAY;
	  invalidated += 1;

/* #define DEBUG_HIGH_DELAY */
#ifdef DEBUG_HIGH_DELAY
	  fprintf(finfo,
		  "high %d %u %.6f offset %.3f\n",
		  curpoint, dp->time, dp->delay,
		  dp->aoffset);
#endif
	}

    ITER:
      if ( direction == 0 )	/* going forward */
	{
	  curpoint++;
	  if ( curpoint >= ds->extent_end )
	    {
#ifdef DEBUG_LOW_DELAY
	      fprintf(finfo, "BACKWARD DELAY SCAN\n");
#endif
	      curpoint = ds->extent_end - 1;
	      direction = 1;
	    }
	}
      else			/* going in reverse */
	curpoint--;

      if ( curpoint < ds->extent_begin )
	break;
    }

  /* free median resources */
  if ( median_n(md) > 0 )
    {
      fprintf(finfo,
	      "DELAY: ORIG %d OK %d INVALID %d (%0.3f invalidated)\n",
	      origvalid, origvalid - invalidated,
	      invalidated,
	      invalidated * 1.0 / origvalid);
    }
  median_free(md);
  md = NULL;
}
