/* 
This software is being provided to you, the LICENSEE, by the
Massachusetts Institute of Technology (M.I.T.) under the following
license.  By obtaining, using and/or copying this software, you agree
that you have read, understood, and will comply with these terms and
conditions:

Permission to use, copy, modify and distribute, including the right to
grant others the right to distribute at any tier, this software and
its documentation for any purpose and without fee or royalty is hereby
granted, provided that you agree to comply with the following
copyright notice and statements, including the disclaimer, and that
the same appear on ALL copies of the software and documentation,
including modifications that you make for internal use or for
distribution:

Copyright 1992,1993,1994 by the Massachusetts Institute of Technology.
                    All rights reserved.

THIS SOFTWARE IS PROVIDED "AS IS", AND M.I.T. MAKES NO REPRESENTATIONS
OR WARRANTIES, EXPRESS OR IMPLIED.  By way of example, but not
limitation, M.I.T. MAKES NO REPRESENTATIONS OR WARRANTIES OF
MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE USE
OF THE LICENSED SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY THIRD
PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.

The name of the Massachusetts Institute of Technology or M.I.T. may
NOT be used in advertising or publicity pertaining to distribution of
the software.  Title to copyright in this software and any associated
documentation shall at all times remain with M.I.T., and USER agrees
to preserve same.
*/
/*
 * Gregory D. Troxel
 */

/* FIT_PARAM - describes rsadj as a function of time
   for all clocks and ntp data */
typedef struct fit_param_s
{
  double aging;
  double freq;
  double phase[NRUNS];
  double systematic[NCLOCKS];
  double phasedev[NRUNS];
  double sysdev[NCLOCKS];	/* sdev in s per clock */
  char phasevalid[NRUNS];
  char systematicvalid[NCLOCKS];
} fit_param_t;

/* FIT_QUALITY - describes how well FIT_PARAM fits a SEGMENT */
typedef struct fit_quality_s
{
  int n;			/* number of points added for stats */
  double sumweights;		/* sum of weights used */
  double sumresid;
  double sumsqresid;
  double sdev;			/* sqrt(sumsqresid/sumweights) */
  double offset;		/* sumresid/sumweights */
} fit_quality_t;

/* FITSEG - wrapper for SEGMENT, FIT_PARAM, FIT_QUALITY
 *	  - contains next pointer for linking
 *        - contains memoized info regarding joining to next
 */
typedef struct fitseg_s
{
  /* link to next fitseg - contents NOT logically connected */
  struct fitseg_s *next;

  /* real data */
  segment_t *sp;		/* must exist */
  fit_param_t *fp;		/* may be NULL */
  fit_quality_t *fq;		/* may be NULL */

  /* memoization optimizations */
  int generation;		/* incremented when changed */
  struct fitseg_s *cand;	/* memoize combine attempts */
  int candgen;			/* generation of next segment attempted */
  double canddev;		/* saved result */
  double candaging, candfreq;	/* params of joint fit */
} fitseg_t;

/* FIT_PARAM routines */
double fit_param_aging(fit_param_t *fp);
double fit_param_freq(fit_param_t *fp, ntp_t t);
double fit_param_phase(fit_param_t *fp, ntp_t t, int run, int clock);
void fit_param_print(FILE *finfo, fit_param_t *fp);
void fit_param_merge(fit_param_t *fp, fit_param_t *fp1, fit_param_t *fp2);
double fit_param_residpoint(datapoint_t *dp, fit_param_t *fp);

/* FIT_QUALITY routines */
void fit_quality_print(FILE *finfo, fit_quality_t *fq);
void fit_quality_calc(segment_t *sp, fit_param_t *fp, fit_quality_t *fq,
		 int crossclockruns, int doingint, int includeinvalid);

/* FITSEG routines */
fitseg_t *fitseg_new();
void fitseg_free(fitseg_t *fs);
void fitseg_print(FILE *finfo, fitseg_t *fs);
fitseg_t *fitseg_last(fitseg_t *fs);
int fitseg_count_valid(fitseg_t *fs, int crossclockruns);

/* memoization consists of forward pointer, and generation counter
 * when checking, stored generation must match forward generation */
#ifdef __GNUC__
inline static
#endif
void fitseg_memoize_invalidate(fitseg_t *fs)
#if defined(PD_UTIL) || defined(__GNUC__)
{
  fs->generation++;
  fs->cand = NULL;
}
#else
;
#endif

/* unclassified */
void dofit_segment_single(FILE *finfo, segment_t *sp,
	      fit_param_t *fp, fit_quality_t *fq);
void dofit_segment_multiple(FILE *finfo, segment_t *sp, sumseg_t *ss,
       fit_param_t *fp, fit_quality_t *fq, int doingint);
void dofit_fitseg(FILE *finfo, fitseg_t *fs);
void doaging_check_fitseg(FILE *finfo, fitseg_t *fs);
