/* 
This software is being provided to you, the LICENSEE, by the
Massachusetts Institute of Technology (M.I.T.) under the following
license.  By obtaining, using and/or copying this software, you agree
that you have read, understood, and will comply with these terms and
conditions:

Permission to use, copy, modify and distribute, including the right to
grant others the right to distribute at any tier, this software and
its documentation for any purpose and without fee or royalty is hereby
granted, provided that you agree to comply with the following
copyright notice and statements, including the disclaimer, and that
the same appear on ALL copies of the software and documentation,
including modifications that you make for internal use or for
distribution:

Copyright 1992,1993,1994 by the Massachusetts Institute of Technology.
                    All rights reserved.

THIS SOFTWARE IS PROVIDED "AS IS", AND M.I.T. MAKES NO REPRESENTATIONS
OR WARRANTIES, EXPRESS OR IMPLIED.  By way of example, but not
limitation, M.I.T. MAKES NO REPRESENTATIONS OR WARRANTIES OF
MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE USE
OF THE LICENSED SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY THIRD
PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.

The name of the Massachusetts Institute of Technology or M.I.T. may
NOT be used in advertising or publicity pertaining to distribution of
the software.  Title to copyright in this software and any associated
documentation shall at all times remain with M.I.T., and USER agrees
to preserve same.
*/
/*
 * Gregory D. Troxel
 */

#include "pd.h"

/*
 * make list of segments for integrated run
 * clock argument as folows:
 *  CLOCK_ALL		ntp and each raw clock
 *  CLOCK_NTP		ntp clock
 *  CLOCK_ALLRAW	each raw clock
 *  CLOCK_ALLVALID	each raw clock with valid flag set
 * only include points if in time period for clock
 */

segment_t *
Integrated_segment_list(FILE *finfo, int clock, int alltimes)
{
  segment_t *head, *tail, *working;
  int cl, i;

  head = tail = working = NULL;

  for ( cl = CLOCK_NTP; cl <= CLOCK_MAX(); cl++ )
    if (cl == clock
	|| clock == CLOCK_ALL
	|| (clock == CLOCK_ALLRAW && cl != CLOCK_NTP)
	|| (clock == CLOCK_ALLVALID &&
	    cl != CLOCK_NTP &&
	    ((clock_state[cl].flags & CLOCK_INTEGRATED_MASK) ==
	     CLOCK_INTEGRATED_MASK )))
      {
	dataset_t *ds =  clock2dataset(cl);
	clock_state_t *cs = (cl == CLOCK_NTP) ? NULL : &clock_state[cl];
	if ( alltimes )
	  cs = NULL;
	for ( i = 0 ; i < ds->n; i++ )
	  {
	    datapoint_t *dp = &ds->dptr[i];
	
	    if ( cs )
	      {
		if ( dp->time < Tntp2run(cs->begin) )
		  continue;
		if ( dp->time > Tntp2run(cs->end) )
		  break;
	      }

	    /* if no current run or point doesn't fit in this run */
	    if ( working == NULL ||
		working->run != dp->run ||
		working->clock != cl)
	      {
		/* if we have a current segment, prepare to move on */
		if ( working != NULL )
		  {
		    tail->next = working;
		    tail = working;
		    working = NULL;	/* not needed */
		  }
	
		/* get new segment */
		if ( (working = NEW(segment_t)) == NULL )
		  panic("Integrated_segment_list malloc segment_t");
	
		working->clock = cl;
		working->run = dp->run;
		working->ds = ds;
		working->first = i;
		working->n = 0;
		working->samerun = working->next = NULL;
		
		if ( head == NULL )
		  head = tail = working;
	      }
	    /* n is extent of region - skipping valid so can't count*/
	    working->n = i - working->first + 1;
	  }
      }
  /* attach last segment if necessary */
  if ( working != NULL && tail != working )
    tail->next = working;
  return head;
}

/*
 * do integrated estimation on raw datasets, as determined by
 * clock_state: area of validity of each clock, allowed clocks,
 *  systematic offsets
 */
fitseg_t *
Integrated(FILE *finfo)
{
  fitseg_t *fs = fitseg_new();
  sumseg_t ss;
  int i, currentrun = 0;

#ifdef MALLOC_DEBUG
  free(malloc(0x3021));
#endif

  /* get linked list of segments to be included */
  fs->sp = Integrated_segment_list(finfo, CLOCK_ALLVALID, 0);

  if ( fs->sp == NULL )
    {
      fprintf(finfo, "EMPTY SEGMENT ==> INTEGRATED_INVALID\n");
      flags.intvalid = 0;
      fitseg_free(fs);
      return NULL;
    }

  segment_print(finfo, fs->sp, 1);

  /* estimate frequency, aging?, phases(runs) and systematics(clocks) */
  dofit_segment_multiple(finfo, fs->sp, &ss, fs->fp, fs->fq, 1);

  /* CHECK FOR REASONS RESULTS ARE NO GOOD */

  if ( fs->fq->sumweights <= 0.0 )
    {
      fprintf(finfo, "NO DATA ==> INTEGRATED_INVALID\n");
      flags.intvalid = 0;
      goto donevalid;
    }

  /* XXX print results for human inspection */
  fprintf(finfo, "\naging %f ppm/day freq %f %f ppm\n",
	  fs->fp->aging * 86400.0 * 2.0 * KK,
	  (fs->fp->freq + 2.0 * fs->fp->aging * ss.ltime) * KK,
	  (fs->fp->freq + 2.0 * fs->fp->aging * ss.htime) * KK);
  for ( i = 0; i <= RUN_MAX(); i++ )
    if ( fs->fp->phasevalid[i] )
      fprintf(finfo, "Run %d\t%f s\n", i, fs->fp->phase[i]);
  for ( i = 0; i <= CLOCK_MAX(); i++ )
    if ( fs->fp->systematicvalid[i] )
      fprintf(finfo,
	      "[%15s] #%2d offset %7.3f ms rms error %5.3f ms weight %.3e\n",
	      clock_unparse(i),
	      i,
	      /* XXX offsetvalid */
	      (fs->fp->systematic[i] + clock_state[i].offset) * 1000.0,
	      fs->fp->sysdev[i] * 1000.0,
	      raw[i].weight);

  /* check time since beginning of most recent run */
  {
    /* calculate beginning time of last run */
    ntp_t start_time = globals.runend[RUN_MAX()];
    for ( i = currentrun = 0; i <= RUN_MAX(); i++ )
	if ( globals.runbeg[i] < globals.stop )
	  {
	    start_time = globals.runbeg[i];
	    currentrun = i;
	  }

    if (start_time >= globals.stop /* CANT HAPPEN */ ||
	globals.stop - start_time < INTEGRATED_MIN_TIME )
      {
	fprintf(finfo,
		"\nINTERVAL %d TOO SHORT REQUIRES %d ==> INTEGRATED_INVALID\n",
		globals.stop - start_time, INTEGRATED_MIN_TIME);
	flags.intvalid = 1;
	goto donevalid;
      }
  }

  if ( ! fs->fp->phasevalid[currentrun] )
    {
      fprintf(finfo, "\nLAST RUN %d NOT COVERED ==> INTEGRATED_INVALID\n",
	      currentrun);
      flags.intvalid = 1;
      goto donevalid;
    }
  fprintf(finfo, "\nINTEGRATED_VALID\n");
  flags.intvalid = 2;
    
 donevalid:

  /* openloop command to poke daemon */
  {
    ntp_t basis = segment_multiple_end_time(fs->sp);
    
    fprintf(finfo, "%s 1 %u %f %f %f\n",
	    flags.intvalid > 1 ? "openloop" : "INVALIDLOOP",
	    Trun2ntp(basis),
	    fit_param_phase(fs->fp, basis, currentrun, CLOCK_NTP),
	    fit_param_freq(fs->fp, basis),
	    fit_param_aging(fs->fp));
  }

  /* if called for, update systematic offsets in clock state */
  if ( flags.predict && CONFIG_SYSTEMATIC > 0.0 )
    {
      double meansys, newfactor;
      int nsys;

      fprintf(finfo, "\nUpdating stored systematic offsets:\n");

      /* compute exponential averaging constant */
      newfactor = 1 - CONFIG_SYSTEMATIC;
      newfactor = 1 - pow( newfactor, SIM_INCR * 1.0 / SECSPERDAY);

      meansys = nsys = 0;
      for ( i = 0; i <= CLOCK_MAX(); i++ )
	if ( clock_state[i].offset_valid )
	  {
	    meansys += clock_state[i].offset;
	    nsys++;
	  }

      if ( nsys > 0 )
	{
	  double update;
	  meansys /= nsys;
	  update = meansys * newfactor;
	  fprintf(finfo,
		  "%d offsets, average stored offset %6.3f ms, update %6.3f ms\n",
		  nsys,
		  meansys * 1000,
		  update * 1000);
	  for ( i = 0; i <= CLOCK_MAX(); i++ )
	    if ( clock_state[i].offset_valid )
	      clock_state[i].offset -= update;
	}

      for ( i = 0; i <= CLOCK_MAX(); i++ )
	if ( fs->fp->systematicvalid[i] )
	  {
	    /* compute exponential averaging constant
	       originally in per/day units, convert to per/incr */
	    double update = fs->fp->systematic[i] * newfactor;
	    
	    fprintf(finfo,
		    "Clock %2d offset %6.3f ms sys %6.3f ms update %6.3f ms", 
		    i, clock_state[i].offset * 1000,
		    fs->fp->systematic[i] * 1000, update * 1000);
	    clock_state[i].offset_valid = 1;
	    clock_state[i].offset += update;
	    fs->fp->systematic[i] -= update; /* keep total the same */
	    fprintf(finfo, " new_offset %6.3f ms\n",
		    clock_state[i].offset * 1000);
	  }
    }

  /* add clock_state sys_offsets into fp
   * clock_state notion is local to integrated package */
  for ( i = 0 ; i <= CLOCK_MAX(); i++ )
    if ( fs->fp->systematicvalid[i] )
      fs->fp->systematic[i] += clock_state[i].offset;

  /* if analysis, make integrated fit valid even for runs that
     we didn't fit over (by finding a phase) so we can examine
     residuals from it with respect to earlier data */
  if ( ! (flags.simulate && flags.predict) )
    {
      segment_t *sp = Integrated_segment_list(finfo, CLOCK_ALLVALID, 1);
      fit_param_do_missing_runs(finfo, sp, fs->fp);
      segment_free(sp);

      /* do the same for missing clocks */
      sp = Integrated_segment_list(finfo, CLOCK_ALLRAW, 1);
      fit_param_do_missing_clocks(finfo, sp, fs->fp);
      segment_free(sp);
    }

  fputc('\n', finfo);
  if ( flags.noaging == 2 )
    doaging_check_fitseg(finfo, fs);
  fitseg_print(finfo, fs);

#ifdef MALLOC_DEBUG
  free(malloc(0x3022));
#endif

  return fs;
}
