/* 
This software is being provided to you, the LICENSEE, by the
Massachusetts Institute of Technology (M.I.T.) under the following
license.  By obtaining, using and/or copying this software, you agree
that you have read, understood, and will comply with these terms and
conditions:

Permission to use, copy, modify and distribute, including the right to
grant others the right to distribute at any tier, this software and
its documentation for any purpose and without fee or royalty is hereby
granted, provided that you agree to comply with the following
copyright notice and statements, including the disclaimer, and that
the same appear on ALL copies of the software and documentation,
including modifications that you make for internal use or for
distribution:

Copyright 1992,1993,1994 by the Massachusetts Institute of Technology.
                    All rights reserved.

THIS SOFTWARE IS PROVIDED "AS IS", AND M.I.T. MAKES NO REPRESENTATIONS
OR WARRANTIES, EXPRESS OR IMPLIED.  By way of example, but not
limitation, M.I.T. MAKES NO REPRESENTATIONS OR WARRANTIES OF
MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE USE
OF THE LICENSED SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY THIRD
PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.

The name of the Massachusetts Institute of Technology or M.I.T. may
NOT be used in advertising or publicity pertaining to distribution of
the software.  Title to copyright in this software and any associated
documentation shall at all times remain with M.I.T., and USER agrees
to preserve same.
*/
/*
 * Gregory D. Troxel
 */

#include "pd.h"

/* Join sp2 onto end of sp1.  After call, sp2 pointer must not be used. */
void
segment_join(FILE *finfo, segment_t *sp1, segment_t *sp2)
{
  segment_t *spt, *sptsr;

#ifdef DEBUG_JOIN
  fprintf(finfo, "joinsegs:\n");
  segment_print(finfo, sp1, 1); segment_print(finfo, sp2, 1);
#endif

  if ( sp1->clock != sp2->clock )
    panic("different clocks to joinsegs");

  /* find tail of sp1 */
  for (spt = sp1; spt->next != NULL; spt = spt->next)
    ;
  /* find tail of same run */
  for (sptsr = spt; sptsr->samerun != NULL; sptsr = sptsr->samerun)
    ;

  if ( sptsr->first + sptsr->n != sp2->first )
    {
      fprintf(finfo, "NONcontinguous to join\n");
      segment_print(finfo, sp1, 1); segment_print(finfo, sp2, 1);
    }

  if ( spt->run == sp2->run )	/* same run - graft sublists */
    {
      if ( sp2->next )
	{
	  spt->next = sp2->next;
	  sp2->next = NULL;
	}

      if ( sptsr->first + sptsr->n == sp2->first )
	{
	  /* just add points */
	  sptsr->n += sp2->n;
	  if ( sp2->samerun )
	    {
	      sptsr->samerun = sp2->samerun;
	      sp2->samerun = NULL;
	    }
	  /* not segment_free - moved sublists! */
	  free(sp2);
	}
      else
	sptsr->samerun = sp2;
    }
  else
    /* different run - use next rather than samerun */
    spt->next = sp2;

#ifdef DEBUG_JOIN
  segment_print(finfo, sp1, 1);
#endif
}

/* join fitsegs together, eliminating second one
 * after call, fs2 pointer may not be used
 * fs2 is freed, after moving contents as appropriate to fs1
 * if recompute true, compute fit for new fitseg
 */
void
fitseg_join(FILE *finfo, fitseg_t *fs1, fitseg_t *fs2, int recompute)
{
  if ( fs1->next != fs2 )
    panic("attempt to combine nonadjacent fitsegs");

  /* recover saved aging, freq */
  if ( recompute )
    if ( fs1->cand == fs2 && fs1->candgen == fs2->generation )
      {
	fs1->fp->aging = fs1->candaging;
	fs1->fp->freq = fs1->candfreq;
	/* fprintf(finfo, "Using candaging, candfreq\n"); */
      }

  segment_join(finfo, fs1->sp, fs2->sp);
  fs2->sp = NULL;

  /* make fs1's next be fs2's old next, free fs2 */
  fs1->next = fs2->next;
  fs2->next = NULL;
  fitseg_free(fs2);

  /* recompute fit_param and fit_quality */
  if ( recompute )
    dofit_fitseg(finfo, fs1);

  /* fs1's cand no longer valid, nor is anyone pointing to fs1 */
  fitseg_memoize_invalidate(fs1);

#ifdef DEBUG_JOIN
  fprintf(finfo, "-> ");
  segment_print(finfo, fs1->sp, 1);
#endif
}

/*
 * Given two fitsegs, return measure in seconds of how well
 * they fit together.
 * If runcombok is false and join would combine runs, return HUGEDEV
 * print info if 'print' is true or on exceptional conditions
 * dmin and dmax are 'local' functions
 */
#ifdef __GNUC__
inline static
#endif
double
dmax(double a, double b)
{
  return a > b ? a : b;
}

#ifdef __GNUC__
inline static
#endif
double
dmin(double a, double b)
{
  return a < b ? a : b;
}

double
fitseg_joinp(FILE *finfo, fitseg_t *fs1, fitseg_t *fs2,
	   int runcombok, int print)
{
  fitseg_t fs_joint;
  segment_t *seg_joint;
  fit_param_t fp_joint;
  fit_quality_t fq_joint;
  fit_quality_t fq_j1, fq_j2;
  double result_sdev;
  int runcomb = 0;

  /* check for memoized answer */
  if ( fs1->cand == fs2 && fs1->candgen == fs2->generation )
    return fs1->canddev;

  if ( fs1->sp->run != fs2->sp->run )
    if ( runcombok )
      runcomb = 1;
    else
      return HUGEDEV;

  bzero(&fs_joint, sizeof(fitseg_t));
  fs_joint.fp = &fp_joint;
  fs_joint.fq = &fq_joint;

  bzero(&fp_joint, sizeof(fit_param_t));
  bzero(&fq_joint, sizeof(fit_quality_t));
  bzero(&fq_j1, sizeof(fit_quality_t));
  bzero(&fq_j2, sizeof(fit_quality_t));

  /* if either has no points, return 0 */
  if ( fs1->fq->n == 0 || fs2->fq->n == 0 )
    {
      result_sdev = 0;
      goto DONE;
    }

  /* construct segment covering joined runs */
  seg_joint = segment_copy(fs1->sp);
  segment_join(finfo, seg_joint, segment_copy(fs2->sp));
	
  fs_joint.sp = seg_joint;

  /* guess at fit_param to make fit go faster */
  fit_param_merge(&fp_joint, fs1->fp, fs2->fp);

  /* compute joint fit, and quality of joint seg wrt it */
  dofit_fitseg(finfo, &fs_joint);
  
  /* compute quality of each segment wrt joint fit */
  fit_quality_calc(fs1->sp, &fp_joint, &fq_j1, 1, 0, 0);
  fit_quality_calc(fs2->sp, &fp_joint, &fq_j2, 1, 0, 0);

  /*
   * result >= joint
   * result <= greater of 1, 2 wrt joint fit
   * within those constraints, result is joint sdev *
   *  max over i of (sdev of i wrt joint / sdev of i wrt self)
   */
  {
    /* ratio of worsening of each segment (wrt joint vs orig) */
    double ratio1 = fs1->fq->sdev > 0 ? fq_j1.sdev / fs1->fq->sdev : 1;
    double ratio2 = fs2->fq->sdev > 0 ? fq_j2.sdev / fs2->fq->sdev : 1;
    
    result_sdev =
      dmax (fq_joint.sdev,
	    dmin(dmax(ratio1, ratio2) * fq_joint.sdev,
		 dmax(fq_j1.sdev, fq_j2.sdev)));
  }

  segment_free(seg_joint);

 DONE:
  if ( print )
    {
      fprintf(finfo, "ret %.3e j %.3e fs1 %.3e %.3e fs2 %.3e %.3e\n",
	      result_sdev, fs_joint.fq->sdev,
	      fs1->fq->sdev, fq_j1.sdev,
	      fs2->fq->sdev, fq_j2.sdev);
      fflush(finfo);
    }

  /* memoize results */
  fs1->cand = fs2;
  fs1->candgen = fs2->generation;
  fs1->canddev = result_sdev;

  /* save aging, freq, of joint fit - ABSTRACTION VIOLATION */
  fs1->candaging = fp_joint.aging;
  fs1->candfreq = fp_joint.freq;

  return result_sdev;
}

/*
 * migrate points across fitsegs if points fit better on other side
 * return 1 if migration occurred
 */
int
aux_fitseg_migrate(FILE *finfo, fitseg_t *fs1, fitseg_t *fs2, int invalidok)
{
  segment_t *sp1l, *sp2f;
  double resorig, resnew;
  int mig_from, mig_to, mig_limit, mig_limit_end;
  int i, n1, n2;

  /* only if migration is to be adjacent to same run */
  sp1l = segment_last(fs1->sp, 1);
  sp2f = fs2->sp;
  if ( sp1l->run != sp2f->run )
    return 0;

  if ( sp1l->first + sp1l->n != sp2f->first )
    {
      fprintf(finfo, "NONcontiguous segments to fitseg_migrate");
      segment_print(finfo, fs1->sp, 1); segment_print(finfo, fs2->sp, 1);
      return 0;
    }

  n1 = segment_count_valid(fs1->sp, 1);
  n2 = segment_count_valid(fs2->sp, 1);

  /* only migrate TO decent-size segments */
  if ( n1 > 64 || (n1 > n2 && n1 >= 8) )
    {
      /* consider moving points from front of fs2 to back of fs1 */
      mig_from = sp2f->first;	/* starting point */
      mig_limit_end = mig_limit = mig_from + sp2f->n - 1; /* max possible */
      if ( mig_limit > mig_from + 3 )
	mig_limit = mig_from + 3; /* at most 4 points */
      mig_to = mig_from - 1;	/* don't do any at first */
      for ( i = mig_from; i <= mig_limit ; i++ )
	{
	  /* invalid points just go with */
	  if ( !invalidok && sp2f->ds->dptr[i].valid != VALID_OK )
	    {
	      if ( mig_to >= mig_from )
		mig_to = i;	/* move these along if moved any */
	      continue;
	    }

	  resorig = fit_param_residpoint(&sp2f->ds->dptr[i], fs2->fp);
	  resnew = fit_param_residpoint(&sp2f->ds->dptr[i], fs1->fp);
	  if ( fabs(resorig) <= fabs(resnew) * 1.05 )
	    break;
	  else
	    mig_to = i;
	}

      if ( mig_to >= mig_from )	/* are we migrating any? */
	{
/* #define MIGRATE_DEBUG */
#ifdef MIGRATE_DEBUG
	  fprintf(finfo, "migrating backward %d %d\n", mig_from, mig_to);
	  segment_print(finfo, fs1->sp, 1);
	  segment_print(finfo, fs2->sp, 1);
#endif
	  if ( mig_to == mig_limit_end )
	    {
	      fprintf(finfo, "NOT migrating - whole segment (back)\n");
	      dofit_fitseg(finfo, fs1);
	      dofit_fitseg(finfo, fs2);
	      return 0;
	    }

	  segment_append(finfo, fs1->sp, sp2f, mig_from, mig_to);
	  segment_trim(sp2f, mig_from, mig_to);
	  fs2->sp = segment_fixup(fs2->sp);
	  goto FIXUP;
	}
    }
  /* only migrate TO if big enough */
  if ( n2 > 64 || (n2 > n1 && n2 >= 8) )
    {
      /* consider moving points from back of fs1 to front of fs2 */
      mig_to = sp1l->first + sp1l->n - 1;
      mig_limit_end = mig_limit = sp1l->first;
      mig_from = mig_to + 1;
      if ( mig_limit < mig_to - 3 )
	mig_limit = mig_to - 3;	/* at most 4 */
      for ( i = mig_to; i >= mig_limit ; i-- )
	{
	  if ( !invalidok && sp1l->ds->dptr[i].valid != VALID_OK )
	    {
	      if ( mig_from <= mig_to )
		mig_from = i;	/* move these along if moved any */
	      continue;
	    }

	  resorig = fit_param_residpoint(&sp1l->ds->dptr[i], fs1->fp);
	  resnew = fit_param_residpoint(&sp1l->ds->dptr[i], fs2->fp);
	  if ( fabs(resorig) <= fabs(resnew) * 1.05 )
	    break;
	  else
	    mig_from = i;
	}

      if ( mig_from <= mig_to )
	{
#ifdef MIGRATE_DEBUG
	  fprintf(finfo, "migrating forward %d %d\n", mig_from, mig_to);
	  segment_print(finfo, fs1->sp, 1);
	  segment_print(finfo, fs2->sp, 1);
#endif
	  if ( mig_from == mig_limit_end )
	    {
	      fprintf(finfo, "NOT migrating - whole segment (forward)\n");
	      dofit_fitseg(finfo, fs1);
	      dofit_fitseg(finfo, fs2);
	      return 0;
	    }

	  fs2->sp = segment_append(finfo, fs2->sp, sp1l, mig_from, mig_to);
	  segment_trim(sp1l, mig_from, mig_to);
	  fs1->sp = segment_fixup(fs1->sp);	
	  goto FIXUP;
	}
    }
  return 0;			/* didn't migrate any */

 FIXUP:
#ifdef MIGRATE_DEBUG
  segment_print(finfo, fs1->sp, 1);
  segment_print(finfo, fs2->sp, 1);
#endif

  fitseg_memoize_invalidate(fs1);
  fitseg_memoize_invalidate(fs2);

  /* recompute fits */
  dofit_fitseg(finfo, fs1);
  dofit_fitseg(finfo, fs2);

  return 1;
}

int
fitseg_migrate(FILE *finfo, fitseg_t *fs1, fitseg_t *fs2)
{
  int r;

  /* try migrating skipping valid points, moving them along with */
  if ( (r = aux_fitseg_migrate(finfo, fs1, fs2, 0)) != 0 )
    return r;

  /* if none, count invalid points and move them */
  r = aux_fitseg_migrate(finfo, fs1, fs2, 1);
  return r;
}
