/* 
This software is being provided to you, the LICENSEE, by the
Massachusetts Institute of Technology (M.I.T.) under the following
license.  By obtaining, using and/or copying this software, you agree
that you have read, understood, and will comply with these terms and
conditions:

Permission to use, copy, modify and distribute, including the right to
grant others the right to distribute at any tier, this software and
its documentation for any purpose and without fee or royalty is hereby
granted, provided that you agree to comply with the following
copyright notice and statements, including the disclaimer, and that
the same appear on ALL copies of the software and documentation,
including modifications that you make for internal use or for
distribution:

Copyright 1992,1993,1994 by the Massachusetts Institute of Technology.
                    All rights reserved.

THIS SOFTWARE IS PROVIDED "AS IS", AND M.I.T. MAKES NO REPRESENTATIONS
OR WARRANTIES, EXPRESS OR IMPLIED.  By way of example, but not
limitation, M.I.T. MAKES NO REPRESENTATIONS OR WARRANTIES OF
MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE USE
OF THE LICENSED SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY THIRD
PARTY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS.

The name of the Massachusetts Institute of Technology or M.I.T. may
NOT be used in advertising or publicity pertaining to distribution of
the software.  Title to copyright in this software and any associated
documentation shall at all times remain with M.I.T., and USER agrees
to preserve same.
*/
/*
 * Gregory D. Troxel
 */

#include "pd.h"

void
fit_param_do_missing_runs_aux(FILE *finfo,
			      segment_t *sp,
			      fit_param_t *fp,
			      int includeinvalid)
{
  int run;
  segment_t *spi;
  double sump, sumw;

  fit_quality_t fq;

  for ( run = 0; run <= RUN_MAX(); run++ )
    if ( ! fp->phasevalid[run] )
      {
	if ( flags.debug > 0 )
	  fprintf(finfo, "Adjusting run %d: ", run);
	sumw = sump = 0;

	for ( spi = sp; spi; spi = spi->next )
	  if ( spi->run == run )
	    {
	      /* sum run */
	      fit_quality_calc(spi, fp, &fq, 0, 0, includeinvalid);
	      sumw += fq.sumweights;
	      sump += fq.sumresid;
	      if ( flags.debug >= 3 )
		fprintf(finfo, "clock %d sumw %f sump %f\n",
			spi->clock, fq.sumweights, fq.sumresid);
	    }
	if ( sumw > 0 )
	  {
	    fp->phase[run] = sump / sumw;
	    fp->phasevalid[run] = 1;
	    if ( flags.debug > 0 )
	      fprintf(finfo, "%8.6f\n", fp->phase[run]);
	  }
	else
	  if ( flags.debug > 0 )
	    fprintf(finfo, "NO points\n");
      }
}


/*
 * for each run not not valid in fp,
 * sum all segments of that run in sp, and estimate phase
 * do this first for all valid points (normal method)
 * then, do with invalid
 * we estimate phase only based on invalid points when there are no valid
 * points to base the estimation on.   This way the output is reasonable
 * for the invalid points.
 */
void
fit_param_do_missing_runs(FILE *finfo,
			      segment_t *sp,
			      fit_param_t *fp)
{
  fit_param_do_missing_runs_aux(finfo, sp, fp, 0);
  fit_param_do_missing_runs_aux(finfo, sp, fp, 1);
}

/*
 * for each clock not not valid in fp,
 * sum all segments of that clock in sp, and estimate systematic
 */
void
fit_param_do_missing_clocks(FILE *finfo, segment_t *sp, fit_param_t *fp)
{
  int clock;
  segment_t *spi;
  double sump, sumw;

  fit_quality_t fq;

  for ( clock = 0; clock <= CLOCK_MAX(); clock++ )
    if ( ! fp->systematicvalid[clock] )
      {
	if ( flags.debug > 0 )
	  fprintf(finfo, "Adjusting clock %d [%s]: ",
		  clock, clock_unparse(clock));
	sumw = sump = 0;

	for ( spi = sp; spi; spi = spi->next )
	  if ( spi->clock == clock )
	    {
	      fit_quality_calc(spi, fp, &fq, 0, 0, 0); /* sum clock */
	      sumw += fq.sumweights;
	      sump += fq.sumresid;
	      if ( flags.debug >= 3 )
		fprintf(finfo, "clock %d sumw %f sump %f\n",
			spi->clock, fq.sumweights, fq.sumresid);
	    }
	if ( sumw > 0 )
	  {
	    fp->systematic[clock] = sump / sumw;
	    fp->systematicvalid[clock] = 1;
	    if ( flags.debug > 0 )
	      fprintf(finfo, "%8.6f (ms)\n", fp->systematic[clock]*1E3);
	  }
	else
	  if ( flags.debug > 0 )
	    fprintf(finfo, "NO points\n");
      }
}

void
calc_rsadjr_segment(segment_t *sp, fit_param_t *fp, int nrsadj, int sim)
{
  segment_t *spr, *spi;
  int i;

  if ( nrsadj == RSADJ_NULL )
    return;

  /* iterate over the points */
   for ( spr = sp; spr; spr = spr->next )
    for ( spi = spr; spi; spi = spi->samerun )
      for ( i = spi->first; i < spi->first + spi->n; i++ )
	{
	  datapoint_t *dp = &spi->ds->dptr[i];
	  int clock = spi->clock;
	  double t = dp->time;
	  double predrsadj;

	  predrsadj = (fp->aging * t + fp->freq) * t;

	  if ( fp->phasevalid[spi->run] )
	    predrsadj += fp->phase[spi->run];
	  else
	    fprintf(stderr,
		    "calc_rsadjr_segment: invalid phase for run %d\n",
		    spi->run);

	  if ( clock != CLOCK_NTP )
	    {
#ifdef OFFSET_FITS_BY_SYSTEMATICS
	      if ( fp->systematicvalid[clock] )
		predrsadj += fp->systematic[clock];
#endif
#ifdef OFFSET_FITS_BY_STORED_SYSTEMATICS
	      /* we don't do this, since this routine is used for
	       fits not related to integrated */
	      predrsadj += clock_state[clock].offset;
#endif
	    }

	  if ( sim )
	    dp->simrsadjr[nrsadj] = dp->simrsadj - predrsadj;
	  else
	    {
	      dp->rsadjr[nrsadj] = dp->rsadj - predrsadj;
	      dp->poffset[nrsadj] = dp->uoffset - predrsadj;
	    }
	}

  /* sim not true, and SIM ANAL */
  if ( sim == 0 && flags.simulate && ! flags.predict )
    calc_rsadjr_segment(sp, fp, nrsadj, 1);
}

void
calc_rsadjr_fitseg(fitseg_t *head, int rsadjn, int sim)
{
  fitseg_t *fs1;

  for( fs1 = head; fs1; fs1 = fs1->next )
    calc_rsadjr_segment(fs1->sp, fs1->fp, rsadjn, sim);
}

void
calc_rsadjr_dataset(dataset_t *ds, fit_param_t *fp,
		    int nrsadj, int clock, int sim)
{
  int i;

  if ( nrsadj == RSADJ_NULL )
    return;

  /* adjust every point in the dataset */
  for ( i = 0; i < ds->n; i++ )
  {
    datapoint_t *dp = &ds->dptr[i];
    double t = dp->time;
    double predrsadj;

    predrsadj = (fp->aging * t + fp->freq) * t;
    if ( fp->phasevalid[dp->run] )
      predrsadj += fp->phase[dp->run];
    else
      fprintf(stderr,
	      "calc_rsadjr_dataset: invalid phase for run %d\n",
	      dp->run);
    if ( clock != CLOCK_NTP )
      {
#ifdef OFFSET_FITS_BY_SYSTEMATICS
	if ( fp->systematicvalid[clock] )
	  predrsadj += fp->systematic[clock];
#endif
#ifdef OFFSET_FITS_BY_STORED_SYSTEMATICS
	/* we don't do this, since this routine is used for
	   fits not related to integrated */
	predrsadj += clock_state[clock].offset;
#endif
      }
    if ( sim )
      dp->simrsadjr[nrsadj] = dp->simrsadj - predrsadj;
    else
      {
	dp->rsadjr[nrsadj] = dp->rsadj - predrsadj;
	dp->poffset[nrsadj] = dp->uoffset - predrsadj;
      }
  }

  /* sim not true, and SIM ANAL */
  if ( sim == 0 && flags.simulate && ! flags.predict )
    calc_rsadjr_dataset(ds, fp, nrsadj, clock, 1);
}

/*
 * set simrsadj for times > globals.stop and <= (globals.stop + SIM_INCR)
 * according to trajectory specified by fp
 * ONLY set points for which runvalid is set and are the same run
 */
void
calc_simrsadj_dataset(FILE *flog, dataset_t *ds, fit_param_t *fp)
{
  datapoint_t *dp;
  int i, run;
  double t;
  double phase;

  /* no points? */
  if ( ds->n == 0 || ds->extent_end == 0 )
    return;

  i = ds->extent_end - 1;	/* last point of the extent */

  dp = &ds->dptr[i];
  run = dp->run;

  if ( fp->phasevalid[run] )
    {
      phase = fp->phase[run];
#ifdef DEBUG_SIMRSADJ
      fprintf(flog, "simrsadj stop %u i %d run %d phase %f\n",
	      globals.stop, i, run, phase);
#endif
    }
  else
    {
      fprintf(flog, "simrsadj: phase not valid for %d\n", run);
      return;
    }

  /* iterate over the rest of points */
  for ( ++i; i < ds->n; i++ )
  {
    dp = &ds->dptr[i];

    /* done if different run */
    if ( dp->run != run )
      {
#ifdef DIAGNOSTIC
	if ( dp->sim )		/* shouldn't be set */
	  {
	    fprintf(flog, "point %d run %d sim was true!",
		    i, dp->run);
	    dp->sim = 0;
	  }
	continue;
#else
	break;			/* done loop */
#endif
      }

    t = dp->time;

#ifdef ONLY_UNTIL_NEXT
    /* check for within time bounds */
    if ( t >= globals.stop + SIM_INCR )
      return;
#endif

    /* calculate simrsadj, and set 'simulated' bit */
    dp->simrsadj = (fp->aging * t + fp->freq) * t + phase;
    dp->simaoffset = dp->uoffset - dp->simrsadj;
    dp->sim = 1;
  }
}
