/*
 * ntp_crypto.c - NTP version 4 public key routines
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef OPENSSL
#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include <unistd.h>
#include <fcntl.h>

#include "ntpd.h"
#include "ntp_stdlib.h"
#include "ntp_unixtime.h"
#include "ntp_string.h"
#include "ntp_random.h"
#include "ntp_assert.h"

#include "openssl/asn1_mac.h"
#include "openssl/bn.h"
#include "openssl/err.h"
#include "openssl/evp.h"
#include "openssl/pem.h"
#include "openssl/rand.h"
#include "openssl/x509v3.h"

#ifdef KERNEL_PLL
#include "ntp_syscall.h"
#endif /* KERNEL_PLL */

/*
 * Extension field message format
 *
 * These are always signed and saved before sending in network byte
 * order. They must be converted to and from host byte order for
 * processing.
 *
 * +-------+-------+
 * |   op  |  len  | <- extension pointer
 * +-------+-------+
 * |    associd    |
 * +---------------+
 * |   timestamp   | <- value pointer
 * +---------------+
 * |   filestamp   |
 * +---------------+
 * |   value len   |
 * +---------------+
 * |               |
 * =     value     =
 * |               |
 * +---------------+
 * | signature len |
 * +---------------+
 * |               |
 * =   signature   =
 * |               |
 * +---------------+
 *
 * The CRYPTO_RESP bit is set to 0 for requests, 1 for responses.
 * Requests carry the association ID of the receiver; responses carry
 * the association ID of the sender. Some messages include only the
 * operation/length and association ID words and so have length 8
 * octets. Ohers include the value structure and associated value and
 * signature fields. These messages include the timestamp, filestamp,
 * value and signature words and so have length at least 24 octets. The
 * signature and/or value fields can be empty, in which case the
 * respective length words are zero. An empty value with nonempty
 * signature is syntactically valid, but semantically questionable.
 *
 * The filestamp represents the time when a cryptographic data file such
 * as a public/private key pair is created. It follows every reference
 * depending on that file and serves as a means to obsolete earlier data
 * of the same type. The timestamp represents the time when the
 * cryptographic data of the message were last signed. Creation of a
 * cryptographic data file or signing a message can occur only when the
 * creator or signor is synchronized to an authoritative source and
 * proventicated to a trusted authority.
 *
 * Note there are several conditions required for server trust. First,
 * the public key on the server certificate must be verified, which can
 * involve a hike along the certificate trail to a trusted host. Next,
 * the server trust must be confirmed by one of several identity
 * schemes. Valid cryptographic values are signed with attached
 * timestamp and filestamp. Individual packet trust is confirmed
 * relative to these values by a message digest with keys generated by a
 * reverse-order pseudorandom hash.
 *
 * State decomposition. These flags are lit in the order given. They are
 * dim only when the association is demobilized.
 *
 * CRYPTO_FLAG_ENAB	Lit upon acceptance of a CRYPTO_ASSOC message
 * CRYPTO_FLAG_CERT	Lit when a self-digned trusted certificate is
 *			accepted.
 * CRYPTO_FLAG_VRFY	Lit when identity is confirmed.
 * CRYPTO_FLAG_PROV	Lit when the first signature is verified.
 * CRYPTO_FLAG_COOK	Lit when a valid cookie is accepted.
 * CRYPTO_FLAG_AUTO	Lit when valid autokey values are accepted.
 * CRYPTO_FLAG_SIGN	Lit when the server signed certificate is
 *			accepted.
 * CRYPTO_FLAG_LEAP	Lit when the leapsecond values are accepted.
 */
/*
 * Cryptodefines
 */
#define TAI_1972	10	/* initial TAI offset (s) */
#define MAX_LEAP	100	/* max UTC leapseconds (s) */
#define VALUE_LEN	(6 * 4) /* min response field length */
#define YEAR		(60 * 60 * 24 * 365) /* seconds in year */

/*
 * Global cryptodata in host byte order
 */
u_int32	crypto_flags = 0x0;	/* status word */
char	*sys_hostname = NULL;	/* host name */
char	*sys_groupname = NULL;	/* group name */

/*
 * Global cryptodata in network byte order
 */
struct cert_info *cinfo = NULL;	/* certificate info/value cache */
struct cert_info *cert_host = NULL; /* host certificate */
struct pkey_info *pkinfo = NULL; /* key info/value cache */
struct value hostval;		/* host value */
struct value pubkey;		/* public key */
struct value tai_leap;		/* leapseconds values */
struct pkey_info *iffkey_info = NULL; /* IFF keys */
struct pkey_info *gqkey_info = NULL; /* GQ keys */
struct pkey_info *mvkey_info = NULL; /* MV keys */

/*
 * Private cryptodata in host byte order
 */
static char *passwd = NULL;	/* private key password */
static EVP_PKEY *host_pkey = NULL; /* host key */
static EVP_PKEY *sign_pkey = NULL; /* sign key */
static const EVP_MD *sign_digest = NULL; /* sign digest */
static u_int sign_siglen;	/* sign key length */
static char *rand_file = NULL;	/* random seed file */
static char *host_file = NULL;	/* host key file */
static char *sign_file = NULL;	/* sign key file */

/*
 * Cryptotypes
 */
static	int	crypto_verify	(struct exten *, struct value *,
				    struct peer *);
static	int	crypto_encrypt	(struct exten *, struct value *,
				    keyid_t *);
static	int	crypto_alice	(struct peer *, struct value *);
static	int	crypto_alice2	(struct peer *, struct value *);
static	int	crypto_alice3	(struct peer *, struct value *);
static	int	crypto_bob	(struct exten *, struct value *);
static	int	crypto_bob2	(struct exten *, struct value *);
static	int	crypto_bob3	(struct exten *, struct value *);
static	int	crypto_iff	(struct exten *, struct peer *);
static	int	crypto_gq	(struct exten *, struct peer *);
static	int	crypto_mv	(struct exten *, struct peer *);
static	u_int	crypto_send	(struct exten *, struct value *,
				    u_int *);
static	tstamp_t crypto_time	(void);
static	u_long	asn2ntp		(ASN1_TIME *);
static	struct cert_info *cert_parse (u_char *, long, tstamp_t);
static	int	cert_sign	(struct exten *, struct value *);
static	struct cert_info *cert_install (struct exten *, struct peer *);
static	int	cert_hike	(struct peer *, struct cert_info *);
static	void	cert_free	(struct cert_info *);
static	struct pkey_info *crypto_key (char *, char *, struct
				    sockaddr_storage *);
static	void	bighash		(BIGNUM *, BIGNUM *);
static	struct cert_info *crypto_cert (char *);

#ifdef SYS_WINNT
int
readlink(char * link, char * file, int len) {
	return (-1);
}
#endif

/*
 * session_key - generate session key
 *
 * This routine generates a session key from the source address,
 * destination address, key ID and private value. The value of the
 * session key is the MD5 hash of these values, while the next key ID is
 * the first four octets of the hash.
 *
 * Returns the next key ID or 0 if there is no destination address.
 */
keyid_t
session_key(
	struct sockaddr_storage *srcadr, /* source address */
	struct sockaddr_storage *dstadr, /* destination address */
	keyid_t	keyno,		/* key ID */
	keyid_t	private,	/* private value */
	u_long	lifetime 	/* key lifetime */
	)
{
	EVP_MD_CTX ctx;		/* message digest context */
	u_char dgst[EVP_MAX_MD_SIZE]; /* message digest */
	keyid_t	keyid;		/* key identifer */
	u_int32	header[10];	/* data in network byte order */
	u_int	hdlen, len;

	if (!dstadr)
		return 0;
	
	/*
	 * Generate the session key and key ID. If the lifetime is
	 * greater than zero, install the key and call it trusted.
	 */
	hdlen = 0;
	switch(srcadr->ss_family) {
	case AF_INET:
		header[0] =
		    ((struct sockaddr_in *)srcadr)->sin_addr.s_addr;
		header[1] =
		    ((struct sockaddr_in *)dstadr)->sin_addr.s_addr;
		header[2] = htonl(keyno);
		header[3] = htonl(private);
		hdlen = 4 * sizeof(u_int32);
		break;

	case AF_INET6:
		memcpy(&header[0], &GET_INADDR6(*srcadr),
		    sizeof(struct in6_addr));
		memcpy(&header[4], &GET_INADDR6(*dstadr),
		    sizeof(struct in6_addr));
		header[8] = htonl(keyno);
		header[9] = htonl(private);
		hdlen = 10 * sizeof(u_int32);
		break;
	}
	EVP_DigestInit(&ctx, EVP_md5());
	EVP_DigestUpdate(&ctx, (u_char *)header, hdlen);
	EVP_DigestFinal(&ctx, dgst, &len);
	memcpy(&keyid, dgst, 4);
	keyid = ntohl(keyid);
	if (lifetime != 0) {
		MD5auth_setkey(keyno, dgst, len);
		authtrust(keyno, lifetime);
	}
#ifdef DEBUG
	if (debug > 1)
		printf(
		    "session_key: %s > %s %08x %08x hash %08x life %lu\n",
		    stoa(srcadr), stoa(dstadr), keyno,
		    private, keyid, lifetime);
#endif
	return (keyid);
}


/*
 * make_keylist - generate key list
 *
 * Returns
 * XEVNT_OK	success
 * XEVNT_ERR	protocol error
 *
 * This routine constructs a pseudo-random sequence by repeatedly
 * hashing the session key starting from a given source address,
 * destination address, private value and the next key ID of the
 * preceeding session key. The last entry on the list is saved along
 * with its sequence number and public signature.
 */
int
make_keylist(
	struct peer *peer,	/* peer structure pointer */
	struct interface *dstadr /* interface */
	)
{
	EVP_MD_CTX ctx;		/* signature context */
	tstamp_t tstamp;	/* NTP timestamp */
	struct autokey *ap;	/* autokey pointer */
	struct value *vp;	/* value pointer */
	keyid_t	keyid = 0;	/* next key ID */
	keyid_t	cookie;		/* private value */
	long	lifetime;
	u_int	len, mpoll;
	int	i;

	if (!dstadr)
		return XEVNT_ERR;
	
	/*
	 * Allocate the key list if necessary.
	 */
	tstamp = crypto_time();
	if (peer->keylist == NULL)
		peer->keylist = emalloc(sizeof(keyid_t) *
		    NTP_MAXSESSION);

	/*
	 * Generate an initial key ID which is unique and greater than
	 * NTP_MAXKEY.
	 */
	while (1) {
		keyid = ntp_random() & 0xffffffff;
		if (keyid <= NTP_MAXKEY)
			continue;

		if (authhavekey(keyid))
			continue;
		break;
	}

	/*
	 * Generate up to NTP_MAXSESSION session keys. Stop if the
	 * next one would not be unique or not a session key ID or if
	 * it would expire before the next poll. The private value
	 * included in the hash is zero if broadcast mode, the peer
	 * cookie if client mode or the host cookie if symmetric modes.
	 */
	mpoll = 1 << min(peer->ppoll, peer->hpoll);
	lifetime = min(1 << sys_automax, NTP_MAXSESSION * mpoll);
	if (peer->hmode == MODE_BROADCAST)
		cookie = 0;
	else
		cookie = peer->pcookie;
	for (i = 0; i < NTP_MAXSESSION; i++) {
		peer->keylist[i] = keyid;
		peer->keynumber = i;
		keyid = session_key(&dstadr->sin, &peer->srcadr, keyid,
		    cookie, lifetime + mpoll);
		lifetime -= mpoll;
		if (auth_havekey(keyid) || keyid <= NTP_MAXKEY ||
		    lifetime < 0)
			break;
	}

	/*
	 * Save the last session key ID, sequence number and timestamp,
	 * then sign these values for later retrieval by the clients. Be
	 * careful not to use invalid key media. Use the public values
	 * timestamp as filestamp. 
	 */
	vp = &peer->sndval;
	if (vp->ptr == NULL)
		vp->ptr = emalloc(sizeof(struct autokey));
	ap = (struct autokey *)vp->ptr;
	ap->seq = htonl(peer->keynumber);
	ap->key = htonl(keyid);
	vp->tstamp = htonl(tstamp);
	vp->fstamp = hostval.tstamp;
	vp->vallen = htonl(sizeof(struct autokey));
	vp->siglen = 0;
	if (tstamp != 0) {
		if (vp->sig == NULL)
			vp->sig = emalloc(sign_siglen);
		EVP_SignInit(&ctx, sign_digest);
		EVP_SignUpdate(&ctx, (u_char *)vp, 12);
		EVP_SignUpdate(&ctx, vp->ptr, sizeof(struct autokey));
		if (EVP_SignFinal(&ctx, vp->sig, &len, sign_pkey))
			vp->siglen = htonl(len);
		else
			msyslog(LOG_ERR, "make_keys: %s",
			    ERR_error_string(ERR_get_error(), NULL));
		peer->flags |= FLAG_ASSOC;
	}
#ifdef DEBUG
	if (debug)
		printf("make_keys: %d %08x %08x ts %u fs %u poll %d\n",
		    peer->keynumber, keyid, cookie, ntohl(vp->tstamp),
		    ntohl(vp->fstamp), peer->hpoll);
#endif
	return (XEVNT_OK);
}


/*
 * crypto_recv - parse extension fields
 *
 * This routine is called when the packet has been matched to an
 * association and passed sanity, format and MAC checks. We believe the
 * extension field values only if the field has proper format and
 * length, the timestamp and filestamp are valid and the signature has
 * valid length and is verified. There are a few cases where some values
 * are believed even if the signature fails, but only if the proventic
 * bit is not set.
 *
 * Returns
 * XEVNT_OK	success
 * XEVNT_ERR	protocol error
 * XEVNT_LEN	bad field format or length
 */
int
crypto_recv(
	struct peer *peer,	/* peer structure pointer */
	struct recvbuf *rbufp	/* packet buffer pointer */
	)
{
	const EVP_MD *dp;	/* message digest algorithm */
	u_int32	*pkt;		/* receive packet pointer */
	struct autokey *ap, *bp; /* autokey pointer */
	struct exten *ep, *fp;	/* extension pointers */
	struct cert_info *xinfo; /* certificate info pointer */
	int	has_mac;	/* length of MAC field */
	int	authlen;	/* offset of MAC field */
	associd_t associd;	/* association ID */
	tstamp_t tstamp = 0;	/* timestamp */
	tstamp_t fstamp = 0;	/* filestamp */
	u_int	len;		/* extension field length */
	u_int	code;		/* extension field opcode */
	u_int	vallen = 0;	/* value length */
	X509	*cert;		/* X509 certificate */
	char	statstr[NTP_MAXSTRLEN]; /* statistics for filegen */
	keyid_t	cookie;		/* crumbles */
	int	hismode;	/* packet mode */
	int	rval = XEVNT_OK;
	u_char	*ptr;
	u_int32 temp32;

	/*
	 * Initialize. Note that the packet has already been checked for
	 * valid format and extension field lengths. First extract the
	 * field length, command code and association ID in host byte
	 * order. These are used with all commands and modes. Then check
	 * the version number, which must be 2, and length, which must
	 * be at least 8 for requests and VALUE_LEN (24) for responses.
	 * Packets that fail either test sink without a trace. The
	 * association ID is saved only if nonzero.
	 */
	authlen = LEN_PKT_NOMAC;
	hismode = (int)PKT_MODE((&rbufp->recv_pkt)->li_vn_mode);
	while ((has_mac = rbufp->recv_length - authlen) > MAX_MAC_LEN) {
		pkt = (u_int32 *)&rbufp->recv_pkt + authlen / 4;
		ep = (struct exten *)pkt;
		code = ntohl(ep->opcode) & 0xffff0000;
		len = ntohl(ep->opcode) & 0x0000ffff;
		associd = (associd_t)ntohl(pkt[1]);
		rval = XEVNT_OK;
#ifdef DEBUG
		if (debug)
			printf(
			    "crypto_recv: flags 0x%x ext offset %d len %u code 0x%x associd %d\n",
			    peer->crypto, authlen, len, code >> 16,
			    associd);
#endif

		/*
		 * Check version number and field length. If bad,
		 * quietly ignore the packet.
		 */
		if (((code >> 24) & 0x3f) != CRYPTO_VN || len < 8) {
			sys_badlength++;
			code |= CRYPTO_ERROR;
		}

		if (len >= VALUE_LEN) {
			tstamp = ntohl(ep->tstamp);
			fstamp = ntohl(ep->fstamp);
			vallen = ntohl(ep->vallen);
		}
		switch (code) {

		/*
		 * Install status word, host name, signature scheme and
		 * association ID. In OpenSSL the signature algorithm is
		 * bound to the digest algorithm, so the NID completely
		 * defines the signature scheme. Note the request and
		 * response are identical, but neither is validated by
		 * signature. The request is processed here only in
		 * symmetric modes. The server name field might be
		 * useful to implement access controls in future.
		 */
		case CRYPTO_ASSOC:

			/*
			 * If our state machine is running when this
			 * message arrives, the other fellow might have
			 * restarted. However, this could be an
			 * intruder, so just clamp the poll interval and
			 * find out for ourselves. Otherwise, pass the
			 * extension field to the transmit side.
			 */
			if (peer->crypto & CRYPTO_FLAG_CERT) {
				rval = XEVNT_ERR;
				break;
			}
			if (peer->cmmd) {
				if (peer->assoc != associd) {
					rval = XEVNT_ERR;
					break;
				}
			}
			fp = emalloc(len);
			memcpy(fp, ep, len);
			fp->associd = htonl(peer->associd);
			peer->cmmd = fp;
			/* fall through */

		case CRYPTO_ASSOC | CRYPTO_RESP:

			/*
			 * Discard the message if it has already been
			 * stored or the message has been amputated.
			 */
			if (peer->crypto) {
				if (peer->assoc != associd)
					rval = XEVNT_ERR;
				break;
			}
			if (vallen == 0 || vallen > MAXHOSTNAME ||
			    len < VALUE_LEN + vallen) {
				rval = XEVNT_LEN;
				break;
			}
#ifdef DEBUG
			if (debug)
				printf(
				    "crypto_recv: ident host 0x%x %d server 0x%x %d\n",
				    crypto_flags, peer->associd, fstamp,
				    peer->assoc);
#endif
			temp32 = crypto_flags & CRYPTO_FLAG_MASK;

			/*
			 * If the client scheme is PC, the server scheme
			 * must be PC. The public key and identity are
			 * presumed valid, so we skip the certificate
			 * and identity exchanges and move immediately
			 * to the cookie exchange which confirms the
			 * server signature.
			 */
			if (crypto_flags & CRYPTO_FLAG_PRIV) {
				if (!(fstamp & CRYPTO_FLAG_PRIV)) {
					rval = XEVNT_KEY;
					break;
				}
				fstamp |= CRYPTO_FLAG_CERT |
				    CRYPTO_FLAG_VRFY | CRYPTO_FLAG_SIGN;

			/*
			 * It is an error if either peer supports
			 * identity, but the other does not.
			 */
			} else if (hismode == MODE_ACTIVE || hismode ==
			    MODE_PASSIVE) {
				if ((temp32 && !(fstamp &
				    CRYPTO_FLAG_MASK)) ||
				    (!temp32 && (fstamp &
				    CRYPTO_FLAG_MASK))) {
					rval = XEVNT_KEY;
					break;
				}
			}

			/*
			 * Discard the message if the signature digest
			 * NID is not supported.
			 */
			temp32 = (fstamp >> 16) & 0xffff;
			dp =
			    (const EVP_MD *)EVP_get_digestbynid(temp32);
			if (dp == NULL) {
				rval = XEVNT_MD;
				break;
			}

			/*
			 * Save status word, host name and message
			 * digest/signature type. If this is from a
			 * broadcast and the association ID has changed,
			 * request the autokey values.
			 */
			peer->assoc = associd;
			if (peer->cast_flags & MDF_BCLNT)
				fstamp |= CRYPTO_FLAG_COOK;
			else if (hismode == MODE_SERVER)
				fstamp |= CRYPTO_FLAG_AUTO;
			if (!(fstamp & CRYPTO_FLAG_TAI))
				fstamp |= CRYPTO_FLAG_LEAP;
			RAND_bytes((u_char *)&peer->hcookie, 4);
			peer->crypto = fstamp;
			peer->digest = dp;
			if (peer->subject != NULL)
				free(peer->subject);
			peer->subject = emalloc(vallen + 1);
			memcpy(peer->subject, ep->pkt, vallen);
			peer->subject[vallen] = '\0';
			if (peer->issuer != NULL)
				free(peer->issuer);
			peer->issuer = emalloc(vallen + 1);
			strcpy(peer->issuer, peer->subject);
			snprintf(statstr, NTP_MAXSTRLEN,
			    "assoc %d %d host %s %s", peer->associd,
			    peer->assoc, peer->subject,
			    OBJ_nid2ln(temp32));
			record_crypto_stats(&peer->srcadr, statstr);
#ifdef DEBUG
			if (debug)
				printf("crypto_recv: %s\n", statstr);
#endif
			break;

		/*
		 * Decode X509 certificate in ASN.1 format and extract
		 * the data containing, among other things, subject
		 * name and public key. In the default identification
		 * scheme, the certificate trail is followed to a self
		 * signed trusted certificate.
		 */
		case CRYPTO_CERT | CRYPTO_RESP:

			/*
			 * Discard the message if empty or invalid.
			 */
			if (len < VALUE_LEN)
				break;

			if ((rval = crypto_verify(ep, NULL, peer)) !=
			    XEVNT_OK)
				break;

			/*
			 * Scan the certificate list to delete old
			 * versions and link the newest version first on
			 * the list. Then, verify the signature. If the
			 * certificate is bad or missing, just ignore
			 * it.
			 */
			if ((xinfo = cert_install(ep, peer)) == NULL) {
				rval = XEVNT_CRT;
				break;
			}
			if ((rval = cert_hike(peer, xinfo)) != XEVNT_OK)
				break;

			/*
			 * We plug in the public key and lifetime from
			 * the first certificate received. However, note
			 * that this certificate might not be signed by
			 * the server, so we can't check the
			 * signature/digest NID.
			 */
			if (peer->pkey == NULL) {
				ptr = (u_char *)xinfo->cert.ptr;
				cert = d2i_X509(NULL, &ptr,
				    ntohl(xinfo->cert.vallen));
				peer->pkey = X509_get_pubkey(cert);
				X509_free(cert);
			}
			peer->flash &= ~TEST8;
			temp32 = xinfo->nid;
			snprintf(statstr, NTP_MAXSTRLEN,
			    "cert %s %s 0x%x %s (%u) fs %u",
			    xinfo->subject, xinfo->issuer, xinfo->flags,
			    OBJ_nid2ln(temp32), temp32,
			    ntohl(ep->fstamp));
			record_crypto_stats(&peer->srcadr, statstr);
#ifdef DEBUG
			if (debug)
				printf("crypto_recv: %s\n", statstr);
#endif
			break;

		/*
		 * Schnorr (IFF) identity scheme. This scheme is
		 * designed for use with shared secret server group keys
		 * and where the certificate may be generated by a third
		 * party. The client sends a challenge to the server,
		 * which performs a calculation and returns the result.
		 * A positive result is possible only if both client and
		 * server contain the same secret group key.
		 */
		case CRYPTO_IFF | CRYPTO_RESP:

			/*
			 * Discard the message if invalid.
			 */
			if ((rval = crypto_verify(ep, NULL, peer)) !=
			    XEVNT_OK)
				break;

			/*
			 * If the challenge matches the response, the
			 * server public key, signature and identity are
			 * all verified at the same time. The server is
			 * declared trusted, so we skip further
			 * certificate exchanges and move immediately to
			 * the cookie exchange.
			 */
			if ((rval = crypto_iff(ep, peer)) != XEVNT_OK)
				break;

			peer->crypto |= CRYPTO_FLAG_VRFY |
			    CRYPTO_FLAG_PROV;
			peer->flash &= ~TEST8;
			snprintf(statstr, NTP_MAXSTRLEN, "iff %s fs %u",
			    peer->issuer, ntohl(ep->fstamp));
			record_crypto_stats(&peer->srcadr, statstr);
#ifdef DEBUG
			if (debug)
				printf("crypto_recv: %s\n", statstr);
#endif
			break;

		/*
		 * Guillou-Quisquater (GQ) identity scheme. This scheme
		 * is designed for use with public certificates carrying
		 * the GQ public key in an extension field. The client
		 * sends a challenge to the server, which performs a
		 * calculation and returns the result. A positive result
		 * is possible only if both client and server contain
		 * the same group key and the server has the matching GQ
		 * private key.
		 */
		case CRYPTO_GQ | CRYPTO_RESP:

			/*
			 * Discard the message if invalid
			 */
			if ((rval = crypto_verify(ep, NULL, peer)) !=
			    XEVNT_OK)
				break;

			/*
			 * If the challenge matches the response, the
			 * server public key, signature and identity are
			 * all verified at the same time. The server is
			 * declared trusted, so we skip further
			 * certificate exchanges and move immediately to
			 * the cookie exchange.
			 */
			if ((rval = crypto_gq(ep, peer)) != XEVNT_OK)
				break;

			peer->crypto |= CRYPTO_FLAG_VRFY |
			    CRYPTO_FLAG_PROV;
			peer->flash &= ~TEST8;
			snprintf(statstr, NTP_MAXSTRLEN, "gq %s fs %u",
			    peer->issuer, ntohl(ep->fstamp));
			record_crypto_stats(&peer->srcadr, statstr);
#ifdef DEBUG
			if (debug)
				printf("crypto_recv: %s\n", statstr);
#endif
			break;

		/*
		 * Mu-Varadharajan (MV) identity scheme. This scheme is
		 * designed for use with three levels of trust, trusted
		 * host, server and client. The trusted host key is
		 * opaque to servers and clients; the server keys are
		 * opaque to clients and each client key is different.
		 * Client keys can be revoked without requiring new key
		 * generations.
		 */
		case CRYPTO_MV | CRYPTO_RESP:

			/*
			 * Discard the message if invalid.
			 */
			if ((rval = crypto_verify(ep, NULL, peer)) !=
			    XEVNT_OK)
				break;

			/*
			 * If the challenge matches the response, the
			 * server public key, signature and identity are
			 * all verified at the same time. The server is
			 * declared trusted, so we skip further
			 * certificate exchanges and move immediately to
			 * the cookie exchange.
			 */
			if ((rval = crypto_mv(ep, peer)) != XEVNT_OK)
				break;

			peer->crypto |= CRYPTO_FLAG_VRFY |
			    CRYPTO_FLAG_PROV;
			peer->flash &= ~TEST8;
			snprintf(statstr, NTP_MAXSTRLEN, "mv %s fs %u",
			    peer->issuer, ntohl(ep->fstamp));
			record_crypto_stats(&peer->srcadr, statstr);
#ifdef DEBUG
			if (debug)
				printf("crypto_recv: %s\n", statstr);
#endif
			break;


		/*
		 * Cookie response in client and symmetric modes. If the
		 * cookie bit is set, the working cookie is the EXOR of
		 * the current and new values.
		 */
		case CRYPTO_COOK | CRYPTO_RESP:

			/*
			 * Discard the message if invalid or signature
			 * not verified with respect to the cookie
			 * values.
			 */
			if ((rval = crypto_verify(ep, &peer->cookval,
			    peer)) != XEVNT_OK)
				break;

			/*
			 * Decrypt the cookie, hunting all the time for
			 * errors.
			 */
			if (vallen == (u_int)EVP_PKEY_size(host_pkey)) {
				RSA_private_decrypt(vallen,
				    (u_char *)ep->pkt,
				    (u_char *)&temp32,
				    host_pkey->pkey.rsa,
				    RSA_PKCS1_OAEP_PADDING);
				cookie = ntohl(temp32);
			} else {
				rval = XEVNT_CKY;
				break;
			}

			/*
			 * Install cookie values and light the cookie
			 * bit. If this is not broadcast client mode, we
			 * are done here.
			 */
			key_expire(peer);
			if (hismode == MODE_ACTIVE || hismode ==
			    MODE_PASSIVE)
				peer->pcookie = peer->hcookie ^ cookie;
			else
				peer->pcookie = cookie;
			peer->crypto |= CRYPTO_FLAG_COOK;
			peer->flash &= ~TEST8;
			snprintf(statstr, NTP_MAXSTRLEN,
			    "cook %x ts %u fs %u", peer->pcookie,
			    ntohl(ep->tstamp), ntohl(ep->fstamp));
			record_crypto_stats(&peer->srcadr, statstr);
#ifdef DEBUG
			if (debug)
				printf("crypto_recv: %s\n", statstr);
#endif
			break;

		/*
		 * Install autokey values in broadcast client and
		 * symmetric modes. We have to do this every time the
		 * sever/peer cookie changes or a new keylist is
		 * rolled. Ordinarily, this is automatic as this message
		 * is piggybacked on the first NTP packet sent upon
		 * either of these events. Note that a broadcast client
		 * or symmetric peer can receive this response without a
		 * matching request.
		 */
		case CRYPTO_AUTO | CRYPTO_RESP:

			/*
			 * Discard the message if invalid or signature
			 * not verified with respect to the receive
			 * autokey values.
			 */
			if ((rval = crypto_verify(ep, &peer->recval,
			    peer)) != XEVNT_OK)
				break;

			/*
			 * Discard the message if a broadcast client and
			 * the association ID does not match. This might
			 * happen if a broacast server restarts the
			 * protocol. A protocol restart will occur at
			 * the next ASSOC message.
			 */
			if (peer->cast_flags & MDF_BCLNT &&
			    peer->assoc != associd)
				break;

			/*
			 * Install autokey values and light the
			 * autokey bit. This is not hard.
			 */
			if (peer->recval.ptr == NULL)
				peer->recval.ptr =
				    emalloc(sizeof(struct autokey));
			bp = (struct autokey *)peer->recval.ptr;
			peer->recval.tstamp = ep->tstamp;
			peer->recval.fstamp = ep->fstamp;
			ap = (struct autokey *)ep->pkt;
			bp->seq = ntohl(ap->seq);
			bp->key = ntohl(ap->key);
			peer->pkeyid = bp->key;
			peer->crypto |= CRYPTO_FLAG_AUTO;
			peer->flash &= ~TEST8;
			snprintf(statstr, NTP_MAXSTRLEN, 
			    "auto seq %d key %x ts %u fs %u", bp->seq,
			    bp->key, ntohl(ep->tstamp),
			    ntohl(ep->fstamp));
			record_crypto_stats(&peer->srcadr, statstr);
#ifdef DEBUG
			if (debug)
				printf("crypto_recv: %s\n", statstr);
#endif
			break;
	
		/*
		 * X509 certificate sign response. Validate the
		 * certificate signed by the server and install. Later
		 * this can be provided to clients of this server in
		 * lieu of the self signed certificate in order to
		 * validate the public key.
		 */
		case CRYPTO_SIGN | CRYPTO_RESP:

			/*
			 * Discard the message if invalid.
			 */
			if ((rval = crypto_verify(ep, NULL, peer)) !=
			    XEVNT_OK)
				break;

			/*
			 * Scan the certificate list to delete old
			 * versions and link the newest version first on
			 * the list.
			 */
			if ((xinfo = cert_install(ep, peer)) == NULL) {
				rval = XEVNT_CRT;
				break;
			}
			peer->crypto |= CRYPTO_FLAG_SIGN;
			peer->flash &= ~TEST8;
			temp32 = xinfo->nid;
			snprintf(statstr, NTP_MAXSTRLEN,
			    "sign %s %s 0x%x %s (%u) fs %u",
			    xinfo->subject, xinfo->issuer, xinfo->flags,
			    OBJ_nid2ln(temp32), temp32,
			    ntohl(ep->fstamp));
			record_crypto_stats(&peer->srcadr, statstr);
#ifdef DEBUG
			if (debug)
				printf("crypto_recv: %s\n", statstr);
#endif
			break;

		/*
		 * Install leapseconds values. While the leapsecond
		 * values epoch, TAI offset and values expiration epoch
		 * are retained, only the current TAI offset is provided
		 * via the kernel to other applications.
		 */
		case CRYPTO_LEAP | CRYPTO_RESP:

			/*
			 * Discard the message if invalid We can't
			 * compare the value timestamps here, as they
			 * can be updated by different servers.
			 */
			if ((rval = crypto_verify(ep, NULL, peer)) !=
			    XEVNT_OK)
				break;

			/*
			 * If the packet leap values are more recent
			 * than the stored ones, install the new leap
			 * values and recompute the signatures.
			 */
			if (ntohl(ep->pkt[2]) > leap_expire) {
				tai_leap.tstamp = ep->tstamp;
				tai_leap.fstamp = ep->fstamp;
				tai_leap.vallen = ep->vallen;
				leap_tai = ntohl(ep->pkt[0]);
				leap_sec = ntohl(ep->pkt[1]);
				leap_expire = ntohl(ep->pkt[2]);
				crypto_update();
			}
			peer->crypto |= CRYPTO_FLAG_LEAP;
			peer->flash &= ~TEST8;
			snprintf(statstr, NTP_MAXSTRLEN,
			    "leap TAI offset %d at %u expire %u fs %u",
			    ntohl(ep->pkt[0]), ntohl(ep->pkt[1]),
			    ntohl(ep->pkt[2]), ntohl(ep->fstamp));
			record_crypto_stats(&peer->srcadr, statstr);
#ifdef DEBUG
			if (debug)
				printf("crypto_recv: %s\n", statstr);
#endif
			break;

		/*
		 * We come here in symmetric modes for miscellaneous
		 * commands that have value fields but are processed on
		 * the transmit side. All we need do here is check for
		 * valid field length. Note that ASSOC is handled
		 * separately.
		 */
		case CRYPTO_CERT:
		case CRYPTO_IFF:
		case CRYPTO_GQ:
		case CRYPTO_MV:
		case CRYPTO_COOK:
		case CRYPTO_SIGN:
			if (len < VALUE_LEN) {
				rval = XEVNT_LEN;
				break;
			}
			/* fall through */

		/*
		 * We come here in symmetric modes for requests
		 * requiring a response (above plus AUTO and LEAP) and
		 * for responses. If a request, save the extension field
		 * for later; invalid requests will be caught on the
		 * transmit side. If an error or invalid response,
		 * declare a protocol error.
		 */
		default:
			if (code & (CRYPTO_RESP | CRYPTO_ERROR)) {
				rval = XEVNT_ERR;
			} else if (peer->cmmd == NULL) {
				fp = emalloc(len);
				memcpy(fp, ep, len);
				peer->cmmd = fp;
			}
		}

		/*
		 * The first error found terminates the extension field
		 * scan and we return the laundry to the caller.
		 */
		if (rval != XEVNT_OK) {
			snprintf(statstr, NTP_MAXSTRLEN,
			    "%04x %d %02x %s", htonl(ep->opcode),
			    associd, rval, eventstr(rval));
			record_crypto_stats(&peer->srcadr, statstr);
#ifdef DEBUG
			if (debug)
				printf("crypto_recv: %s\n", statstr);
#endif
			return (rval);
		}
		authlen += (len + 3) / 4 * 4;
	}
	return (rval);
}


/*
 * crypto_xmit - construct extension fields
 *
 * This routine is called both when an association is configured and
 * when one is not. The only case where this matters is to retrieve the
 * autokey information, in which case the caller has to provide the
 * association ID to match the association.
 *
 * Side effect: update the packet offset.
 *
 * Returns
 * XEVNT_OK	success
 * XEVNT_CRT	bad or missing certificate
 * XEVNT_ERR	protocol error
 * XEVNT_LEN	bad field format or length
 * XEVNT_PER	host certificate expired
 */
int
crypto_xmit(
	struct peer *peer,	/* peer structure pointer */
	struct pkt *xpkt,	/* transmit packet pointer */
	struct recvbuf *rbufp,	/* receive buffer pointer */
	int	*start,		/* offset to extension field */
	struct exten *ep,	/* extension pointer */
	keyid_t cookie		/* session cookie */
	)
{
	struct exten *fp;	/* extension pointers */
	struct cert_info *cp, *xp, *yp; /* cert info/value pointer */
	struct sockaddr_storage *srcadr_sin; /* source address */
	u_int32	*pkt;		/* packet pointer */
	u_int	opcode;		/* extension field opcode */
	char	certname[MAXHOSTNAME + 1]; /* subject name buffer */
	char	statstr[NTP_MAXSTRLEN]; /* statistics for filegen */
	tstamp_t tstamp;
	u_int	vallen;
	u_int	len;
	struct value vtemp;
	associd_t associd;
	int	rval;
	keyid_t tcookie;

	/*
	 * Generate the requested extension field request code, length
	 * and association ID. If this is a response and the host is not
	 * synchronized, light the error bit and go home.
	 */
	pkt = (u_int32 *)xpkt + *start / 4;
	if (peer != NULL)
		srcadr_sin = &peer->srcadr;
	else
		srcadr_sin = &rbufp->recv_srcadr;
	fp = (struct exten *)pkt;
	opcode = ntohl(ep->opcode);
	if (!(opcode & CRYPTO_RESP))
		peer->opcode = ep->opcode;
	associd = (associd_t) ntohl(ep->associd);
	fp->associd = ep->associd;
	len = 8;
	rval = XEVNT_OK;
	tstamp = crypto_time();
	switch (opcode & 0xffff0000) {

	/*
	 * Send association request and response with status word and
	 * host name. Note, this message is not signed and the filestamp
	 * contains only the status word.
	 */
	case CRYPTO_ASSOC:
	case CRYPTO_ASSOC | CRYPTO_RESP:
		rval = crypto_send(fp, &hostval, &len);
		fp->fstamp = htonl(crypto_flags);
		break;

	/*
	 * Send certificate request. Use the values from the extension
	 * field.
	 */
	case CRYPTO_CERT:
		memset(&vtemp, 0, sizeof(vtemp));
		vtemp.tstamp = ep->tstamp;
		vtemp.fstamp = ep->fstamp;
		vtemp.vallen = ep->vallen;
		vtemp.ptr = (u_char *)ep->pkt;
		rval = crypto_send(fp, &vtemp, &len);
		break;

	/*
	 * Send sign request. Use the host certificate, which is self-
	 * signed and may or may not be trusted.
	 */
	case CRYPTO_SIGN:
		if (tstamp < cert_host->first || tstamp >
		    cert_host->last)
			rval = XEVNT_PER;
		else
			rval = crypto_send(fp, &cert_host->cert, &len);
		break;

	/*
	 * Send certificate response. Use the name in the extension
	 * field to find the certificate in the cache. If the request
	 * contains no subject name, assume the name of this host. This
	 * is for backwards compatibility. Private certificates are
	 * never sent.
	 *
	 * There may be several certificates matching the request. First
	 * choice is a self-signed trusted certificate; second choice is
	 * any certificate signed by another host. There is no third
	 * choice. 
	 */
	case CRYPTO_CERT | CRYPTO_RESP:
		vallen = ntohl(ep->vallen);
		if (vallen == 0 || vallen > MAXHOSTNAME) {
			rval = XEVNT_LEN;
			break;

		} else {
			memcpy(certname, ep->pkt, vallen);
			certname[vallen] = '\0';
		}

		/*
		 * Find all public valid certificates with matching
		 * subject. If a self-signed, trusted certificate is
		 * found, use that certificate. If not, use the last non
		 * self-signed certificate.
		 */
		xp = yp = NULL;
		for (cp = cinfo; cp != NULL; cp = cp->link) {
			if (cp->flags & (CERT_PRIV | CERT_ERROR))
				continue;

			if (strcmp(certname, cp->subject) != 0)
				continue;

			if (strcmp(certname, cp->issuer) != 0)
				yp = cp;
			else if (cp ->flags & CERT_TRUST)
				xp = cp;
			continue;
		}

		/*
		 * Be careful who you trust. If the certificate is not
		 * found, return an empty response. Note that we dont
		 * enforce lifetimes here.
		 *
		 * The timestamp and filestamp are taken from the
		 * certificate value structure. For all certificates the
		 * timestamp is the latest signature update time. For
		 * host and imported certificates the filestamp is the
		 * creation epoch. For signed certificates the filestamp
		 * is the creation epoch of the trusted certificate at
		 * the root of the certificate trail. In principle, this
		 * allows strong checking for signature masquerade.
		 */
		if (xp == NULL)
			xp = yp;
		if (xp == NULL)
			break;

		if (tstamp == 0)
			break;

		rval = crypto_send(fp, &xp->cert, &len);
		break;

	/*
	 * Send challenge in Schnorr (IFF) identity scheme.
	 */
	case CRYPTO_IFF:
		if ((rval = crypto_alice(peer, &vtemp)) == XEVNT_OK) {
			rval = crypto_send(fp, &vtemp, &len);
			value_free(&vtemp);
		}
		break;

	/*
	 * Send response in Schnorr (IFF) identity scheme.
	 */
	case CRYPTO_IFF | CRYPTO_RESP:
		if ((rval = crypto_bob(ep, &vtemp)) == XEVNT_OK) {
			rval = crypto_send(fp, &vtemp, &len);
			value_free(&vtemp);
		}
		break;

	/*
	 * Send challenge in Guillou-Quisquater (GQ) identity scheme.
	 */
	case CRYPTO_GQ:
		if ((rval = crypto_alice2(peer, &vtemp)) == XEVNT_OK) {
			rval = crypto_send(fp, &vtemp, &len);
			value_free(&vtemp);
		}
		break;

	/*
	 * Send response in Guillou-Quisquater (GQ) identity scheme.
	 */
	case CRYPTO_GQ | CRYPTO_RESP:
		if ((rval = crypto_bob2(ep, &vtemp)) == XEVNT_OK) {
			rval = crypto_send(fp, &vtemp, &len);
			value_free(&vtemp);
		}
		break;

	/*
	 * Send challenge in MV identity scheme.
	 */
	case CRYPTO_MV:
		if ((rval = crypto_alice3(peer, &vtemp)) == XEVNT_OK) {
			rval = crypto_send(fp, &vtemp, &len);
			value_free(&vtemp);
		}
		break;

	/*
	 * Send response in MV identity scheme.
	 */
	case CRYPTO_MV | CRYPTO_RESP:
		if ((rval = crypto_bob3(ep, &vtemp)) == XEVNT_OK) {
			rval = crypto_send(fp, &vtemp, &len);
			value_free(&vtemp);
		}
		break;

	/*
	 * Send certificate sign response. The integrity of the request
	 * certificate has already been verified on the receive side.
	 * Sign the response using the local server key. Use the
	 * filestamp from the request and use the timestamp as the
	 * current time. Light the error bit if the certificate is
	 * invalid or contains an unverified signature.
	 */
	case CRYPTO_SIGN | CRYPTO_RESP:
		if ((rval = cert_sign(ep, &vtemp)) == XEVNT_OK) {
			rval = crypto_send(fp, &vtemp, &len);
			value_free(&vtemp);
		}
		break;

	/*
	 * Send public key and signature. Use the values from the public
	 * key.
	 */
	case CRYPTO_COOK:
		rval = crypto_send(fp, &pubkey, &len);
		break;

	/*
	 * Encrypt and send cookie and signature. Light the error bit if
	 * anything goes wrong.
	 */
	case CRYPTO_COOK | CRYPTO_RESP:
		if ((opcode & 0xffff) < VALUE_LEN) {
			rval = XEVNT_LEN;
			break;
		}
		if (PKT_MODE(xpkt->li_vn_mode) == MODE_SERVER) {
			tcookie = cookie;
		} else {
			tcookie = peer->hcookie;
		}
		if ((rval = crypto_encrypt(ep, &vtemp, &tcookie)) ==
		    XEVNT_OK) {
			rval = crypto_send(fp, &vtemp, &len);
			value_free(&vtemp);
		}
		break;

	/*
	 * Find peer and send autokey data and signature in broadcast
	 * server and symmetric modes. Use the values in the autokey
	 * structure. If no association is found, either the server has
	 * restarted with new associations or some perp has replayed an
	 * old message, in which case light the error bit.
	 */
	case CRYPTO_AUTO | CRYPTO_RESP:
		if (peer == NULL) {
			if ((peer = findpeerbyassoc(associd)) == NULL) {
				rval = XEVNT_ERR;
				break;
			}
		}
		peer->flags &= ~FLAG_ASSOC;
		rval = crypto_send(fp, &peer->sndval, &len);
		break;

	/*
	 * Send leapseconds values and signature. Use the values from
	 * the tai structure. If no table has been loaded, just send an
	 * empty request.
	 */
	case CRYPTO_LEAP | CRYPTO_RESP:
		rval = crypto_send(fp, &tai_leap, &len);
		break;

	/*
	 * Default - Fall through for requests; for unknown responses,
	 * flag as error.
	 */
	default:
		if (opcode & CRYPTO_RESP)
			rval = XEVNT_ERR;
	}
	fp->opcode = htonl((opcode & 0xffff0000) | len);

	/*
	 * In case of error, flame the log. If a request, toss the
	 * puppy; if a response, return so the sender can flame, too.
	 */
	if (rval != XEVNT_OK) {
		u_int32	uint32;

		uint32 = CRYPTO_ERROR;
		opcode |= uint32;
		fp->opcode |= htonl(uint32);
		snprintf(statstr, NTP_MAXSTRLEN,
		    "%04x %d %02x %s", opcode, associd, rval,
		    eventstr(rval));
		record_crypto_stats(srcadr_sin, statstr);
#ifdef DEBUG
		if (debug)
			printf("crypto_xmit: %s\n", statstr);
#endif
		if (!(opcode & CRYPTO_RESP))
			return (0);
	}
#ifdef DEBUG
	if (debug)
		printf(
		    "crypto_xmit: flags 0x%x ext offset %d len %u code 0x%x associd %d\n",
		    crypto_flags, *start, len, opcode >> 16, associd);
#endif
	*start += len;
	return (rval);
}


/*
 * crypto_verify - verify the extension field value and signature
 *
 * Returns
 * XEVNT_OK	success
 * XEVNT_ERR	protocol error
 * XEVNT_FSP	bad filestamp
 * XEVNT_LEN	bad field format or length
 * XEVNT_PUB	bad or missing public key
 * XEVNT_SGL	bad signature length
 * XEVNT_SIG	signature not verified
 * XEVNT_TSP	bad timestamp
 */
static int
crypto_verify(
	struct exten *ep,	/* extension pointer */
	struct value *vp,	/* value pointer */
	struct peer *peer	/* peer structure pointer */
	)
{
	EVP_PKEY *pkey;		/* server public key */
	EVP_MD_CTX ctx;		/* signature context */
	tstamp_t tstamp, tstamp1 = 0; /* timestamp */
	tstamp_t fstamp, fstamp1 = 0; /* filestamp */
	u_int	vallen;		/* value length */
	u_int	siglen;		/* signature length */
	u_int	opcode, len;
	int	i;

	/*
	 * We are extremely parannoyed. We require valid opcode, length,
	 * association ID, timestamp, filestamp, public key, digest,
	 * signature length and signature, where relevant. Note that
	 * preliminary length checks are done in the main loop.
	 */
	len = ntohl(ep->opcode) & 0x0000ffff;
	opcode = ntohl(ep->opcode) & 0xffff0000;

	/*
	 * Check for valid value header opcode, association ID and
	 & extension field length. The request and response opcodes must
	 & match and the response ID must match the association ID. The
	 * autokey values response is the exception, as it can be sent
	 * unsolicited.
	 */
	if (opcode & CRYPTO_ERROR)
		return (XEVNT_ERR);

 	if (len < VALUE_LEN)
		return (XEVNT_LEN);

	if (opcode == (CRYPTO_AUTO | CRYPTO_RESP)) {
		if (ntohl(ep->associd) != peer->assoc)
			return (XEVNT_ERR);
	} else {
		if (((htonl(peer->opcode) & 0x3fff0000) !=
		    (opcode & 0x3fff0000)) || ntohl(ep->associd) !=
		    peer->associd)
			return (XEVNT_ERR);
	}

	/*
	 * We have a valid value header. Check for valid value and
	 * signature field lengths. The extension field length must be
	 * long enough to contain the value header, value and signature.
	 * Note both the value and signature field lengths are rounded
	 * up to the next word.
	 */
	vallen = ntohl(ep->vallen);
	if (vallen == 0)
		return (XEVNT_LEN);

	i = (vallen + 3) / 4;
	siglen = ntohl(ep->pkt[i++]);
	if (len < VALUE_LEN + ((vallen + 3) / 4) * 4 + ((siglen + 3) /
	    4) * 4)
		return (XEVNT_LEN);

	/*
	 * Check for valid timestamp and filestamp. If the timestamp is
	 * zero, the sender is not synchronized and signatures are
	 * disregarded. If not, the timestamp must not precede the
	 * filestamp. The timestamp and filestamp must not precede the
	 * corresponding values in the value structure, if present. 	 	 */
	tstamp = ntohl(ep->tstamp);
	fstamp = ntohl(ep->fstamp);
	if (tstamp == 0)
		return (XEVNT_OK);

	if (tstamp < fstamp)
		return (XEVNT_TSP);

	if (vp != NULL) {
		tstamp1 = ntohl(vp->tstamp);
		fstamp1 = ntohl(vp->fstamp);
		if (tstamp1 != 0 && fstamp1 != 0) {
			if (tstamp < tstamp1)
				return (XEVNT_TSP);

			if ((tstamp < fstamp1 || fstamp < fstamp1))
				return (XEVNT_FSP);
		}
	}

	/*
	 * Check for valid signature length, public key and digest
	 * algorithm.
	 */
	if (crypto_flags & peer->crypto & CRYPTO_FLAG_PRIV)
		pkey = sign_pkey;
	else
		pkey = peer->pkey;
	if (siglen == 0 || pkey == NULL || peer->digest == NULL)
		return (XEVNT_OK);

	if (siglen != (u_int)EVP_PKEY_size(pkey))
		return (XEVNT_SGL);

	/*
	 * Darn, I thought we would never get here. Verify the
	 * signature. If the identity exchange is verified, light the
	 * proventic bit.
	 */
	EVP_VerifyInit(&ctx, peer->digest);
	EVP_VerifyUpdate(&ctx, (u_char *)&ep->tstamp, vallen + 12);
	if (!EVP_VerifyFinal(&ctx, (u_char *)&ep->pkt[i], siglen, pkey))
		return (XEVNT_SIG);

	if (peer->crypto & CRYPTO_FLAG_VRFY)
		peer->crypto |= CRYPTO_FLAG_PROV;
	return (XEVNT_OK);
}


/*
 * crypto_encrypt - construct encrypted cookie and signature from
 * extension field and cookie
 *
 * Returns
 * XEVNT_OK	success
 * XEVNT_CKY	bad or missing cookie
 * XEVNT_PUB	bad or missing public key
 */
static int
crypto_encrypt(
	struct exten *ep,	/* extension pointer */
	struct value *vp,	/* value pointer */
	keyid_t	*cookie		/* server cookie */
	)
{
	EVP_PKEY *pkey;		/* public key */
	EVP_MD_CTX ctx;		/* signature context */
	tstamp_t tstamp;	/* NTP timestamp */
	u_int32	temp32;
	u_int	len;
	u_char	*ptr;

	/*
	 * Extract the public key from the request.
	 */
	len = ntohl(ep->vallen);
	ptr = (u_char *)ep->pkt;
	pkey = d2i_PublicKey(EVP_PKEY_RSA, NULL, &ptr, len);
	if (pkey == NULL) {
		msyslog(LOG_ERR, "crypto_encrypt: %s",
		    ERR_error_string(ERR_get_error(), NULL));
		return (XEVNT_PUB);
	}

	/*
	 * Encrypt the cookie, encode in ASN.1 and sign.
	 */
	memset(vp, 0, sizeof(struct value));
	tstamp = crypto_time();
	vp->tstamp = htonl(tstamp);
	vp->fstamp = hostval.tstamp;
	len = EVP_PKEY_size(pkey);
	vp->vallen = htonl(len);
	vp->ptr = emalloc(len);
	ptr = vp->ptr;
	temp32 = htonl(*cookie);
	if (!RSA_public_encrypt(4, (u_char *)&temp32, ptr,
	    pkey->pkey.rsa, RSA_PKCS1_OAEP_PADDING)) {
		msyslog(LOG_ERR, "crypto_encrypt: %s",
		    ERR_error_string(ERR_get_error(), NULL));
		free(vp->ptr);
		EVP_PKEY_free(pkey);
		return (XEVNT_CKY);
	}
	EVP_PKEY_free(pkey);
	if (tstamp == 0)
		return (XEVNT_OK);

	vp->sig = emalloc(sign_siglen);
	EVP_SignInit(&ctx, sign_digest);
	EVP_SignUpdate(&ctx, (u_char *)&vp->tstamp, 12);
	EVP_SignUpdate(&ctx, vp->ptr, len);
	if (EVP_SignFinal(&ctx, vp->sig, &len, sign_pkey))
		vp->siglen = htonl(len);
	return (XEVNT_OK);
}


/*
 * crypto_ident - construct extension field for identity scheme
 *
 * This routine determines which identity scheme is in use and
 * constructs an extension field for that scheme.
 *
 * Returns
 * CRYTPO_IFF	IFF scheme
 * CRYPTO_GQ	GQ scheme
 * CRYPTO_MV	MV scheme
 * CRYPTO_NULL	no available scheme
 */
u_int
crypto_ident(
	struct peer *peer	/* peer structure pointer */
	)
{
	char	filename[MAXFILENAME];

	/*
	 * We come here after the group trusted host has been found; its
	 * name defines the group name. Search the key cache for all
	 * keys matching the same group name in order IFF, GQ and MV.
	 * Use the first one available.
	 */
	if (peer->crypto & CRYPTO_FLAG_IFF) {
		snprintf(filename, MAXFILENAME, "ntpkey_iffpar_%s",
		    peer->issuer);
		peer->ident_pkey = crypto_key(filename, NULL,
		    &peer->srcadr);
		if (peer->ident_pkey != NULL)
			return (CRYPTO_IFF);
	}
	if (peer->crypto & CRYPTO_FLAG_GQ) {
		snprintf(filename, MAXFILENAME, "ntpkey_gqpar_%s",
		    peer->issuer);
		peer->ident_pkey = crypto_key(filename, NULL,
		    &peer->srcadr);
		if (peer->ident_pkey != NULL)
			return (CRYPTO_GQ);
	}
	if (peer->crypto & CRYPTO_FLAG_MV) {
		snprintf(filename, MAXFILENAME, "ntpkey_mvpar_%s",
		    peer->issuer);
		peer->ident_pkey = crypto_key(filename, NULL,
		    &peer->srcadr);
		if (peer->ident_pkey != NULL)
			return (CRYPTO_MV);
	}
	msyslog(LOG_NOTICE,
	    "crypto_ident: no identity parameters found for group %s",
	    peer->issuer);
	return (CRYPTO_NULL);
}


/*
 * crypto_args - construct extension field from arguments
 *
 * This routine creates an extension field with current timestamps and
 * specified opcode, association ID and optional string. Note that the
 * extension field is created here, but freed after the crypto_xmit()
 * call in the protocol module.
 *
 * Returns extension field pointer (no errors)
 */
struct exten *
crypto_args(
	struct peer *peer,	/* peer structure pointer */
	u_int	opcode,		/* operation code */
	associd_t associd,	/* association ID */
	char	*str		/* argument string */
	)
{
	tstamp_t tstamp;	/* NTP timestamp */
	struct exten *ep;	/* extension field pointer */
	u_int	len;		/* extension field length */

	tstamp = crypto_time();
	len = sizeof(struct exten);
	if (str != NULL)
		len += strlen(str);
	ep = emalloc(len);
	memset(ep, 0, len);
	if (opcode == 0)
		return (ep);

	ep->opcode = htonl(opcode + len);
	ep->associd = htonl(associd);
	ep->tstamp = htonl(tstamp);
	ep->fstamp = hostval.tstamp;
	ep->vallen = 0;
	if (str != NULL) {
		ep->vallen = htonl(strlen(str));
		memcpy((char *)ep->pkt, str, strlen(str));
	}
	return (ep);
}


/*
 * crypto_send - construct extension field from value components
 *
 * The value and signature fields are zero-padded to a word boundary.
 * Note: it is not polite to send a nonempty signature with zero
 * timestamp or a nonzero timestamp with an empty signature, but those
 * rules are not enforced here.
 *
 * Returns
 * XEVNT_OK	success
 * XEVNT_LEN	bad field format or length
 */
u_int
crypto_send(
	struct exten *ep,	/* extension field pointer */
	struct value *vp,	/* value pointer */
	u_int	*start		/* buffer offset */
	)
{
	u_int	len, vallen, siglen;
	int	i, j;

	/*
	 * Calculate extension field length and check for buffer
	 * overflow. Leave room for the MAC.
	 */
	len = 16;
	vallen = ntohl(vp->vallen);
	len += ((vallen + 3) / 4 + 1) * 4; 
	siglen = ntohl(vp->siglen);
	len += ((siglen + 3) / 4 + 1) * 4; 
	if (*start + len > NTP_MAXEXTEN - LEN_PKT_NOMAC - MAX_MAC_LEN)
		return (XEVNT_LEN);

	/*
	 * Copy timestamps.
	 */
	ep->tstamp = vp->tstamp;
	ep->fstamp = vp->fstamp;
	ep->vallen = vp->vallen;

	/*
	 * Copy value. If the data field is empty or zero length,
	 * encode an empty value with length zero.
	 */
	i = 0;
	if (vallen > 0 && vp->ptr != NULL) {
		j = vallen / 4;
		if (j * 4 < vallen)
			ep->pkt[i + j++] = 0;
		memcpy(&ep->pkt[i], vp->ptr, vallen);
		i += j;
	}

	/*
	 * Copy signature. If the signature field is empty or zero
	 * length, encode an empty signature with length zero.
	 */
	ep->pkt[i++] = vp->siglen;
	if (siglen > 0 && vp->sig != NULL) {
		j = vallen / 4;
		if (j * 4 < siglen)
			ep->pkt[i + j++] = 0;
		memcpy(&ep->pkt[i], vp->sig, siglen);
		i += j;
	}
	*start += len;
	return (XEVNT_OK);
}


/*
 * crypto_update - compute new public value and sign extension fields
 *
 * This routine runs periodically, like once a day, and when something
 * changes. It updates the timestamps on three value structures and one
 * value structure list, then signs all the structures:
 *
 * hostval	host name (not signed)
 * pubkey	public key
 * cinfo	certificate info/value list
 * tai_leap	leap values
 *
 * Filestamps are proventic data, so this routine runs only when the
 * host is synchronized to a proventicated source. Thus, the timestamp
 * is proventic and can be used to deflect clogging attacks.
 *
 * Returns void (no errors)
 */
void
crypto_update(void)
{
	EVP_MD_CTX ctx;		/* message digest context */
	struct cert_info *cp;	/* certificate info/value */
	char	statstr[NTP_MAXSTRLEN]; /* statistics for filegen */
	u_int32	*ptr;
	u_int	len;

	hostval.tstamp = htonl(crypto_time());
	if (hostval.tstamp == 0)
		return;


	/*
	 * Sign public key and timestamps. The filestamp is derived from
	 * the host key file extension from wherever the file was
	 * generated. 
	 */
	if (pubkey.vallen != 0) {
		pubkey.tstamp = hostval.tstamp;
		pubkey.siglen = 0;
		if (pubkey.sig == NULL)
			pubkey.sig = emalloc(sign_siglen);
		EVP_SignInit(&ctx, sign_digest);
		EVP_SignUpdate(&ctx, (u_char *)&pubkey, 12);
		EVP_SignUpdate(&ctx, pubkey.ptr, ntohl(pubkey.vallen));
		if (EVP_SignFinal(&ctx, pubkey.sig, &len, sign_pkey))
			pubkey.siglen = htonl(len);
	}

	/*
	 * Sign certificates and timestamps. The filestamp is derived
	 * from the certificate file extension from wherever the file
	 * was generated. Note we do not throw expired certificates
	 * away; they may have signed younger ones.
	 */
	for (cp = cinfo; cp != NULL; cp = cp->link) {
		cp->cert.tstamp = hostval.tstamp;
		cp->cert.siglen = 0;
		if (cp->cert.sig == NULL)
			cp->cert.sig = emalloc(sign_siglen);
		EVP_SignInit(&ctx, sign_digest);
		EVP_SignUpdate(&ctx, (u_char *)&cp->cert, 12);
		EVP_SignUpdate(&ctx, cp->cert.ptr,
		    ntohl(cp->cert.vallen));
		if (EVP_SignFinal(&ctx, cp->cert.sig, &len, sign_pkey))
			cp->cert.siglen = htonl(len);
	}

	/*
	 * Sign leapseconds values and timestamps. Note it is not an
	 * error to return null values.
	 */
	tai_leap.tstamp = hostval.tstamp;
	tai_leap.fstamp = hostval.fstamp;
	len = 3 * sizeof(u_int32);
	if (tai_leap.ptr == NULL)
		tai_leap.ptr = emalloc(len);
	tai_leap.vallen = htonl(len);
	ptr = (u_int32 *)tai_leap.ptr;
	ptr[0] = htonl(leap_tai);
	ptr[1] = htonl(leap_sec);
	ptr[2] = htonl(leap_expire);
	if (tai_leap.sig == NULL)
		tai_leap.sig = emalloc(sign_siglen);
	EVP_SignInit(&ctx, sign_digest);
	EVP_SignUpdate(&ctx, (u_char *)&tai_leap, 12);
	EVP_SignUpdate(&ctx, tai_leap.ptr, len);
	if (EVP_SignFinal(&ctx, tai_leap.sig, &len, sign_pkey))
		tai_leap.siglen = htonl(len);
	if (leap_sec > 0)
		crypto_flags |= CRYPTO_FLAG_TAI;
	snprintf(statstr, NTP_MAXSTRLEN, "signature update ts %u",
	    ntohl(hostval.tstamp)); 
	record_crypto_stats(NULL, statstr);
#ifdef DEBUG
	if (debug)
		printf("crypto_update: %s\n", statstr);
#endif
}


/*
 * value_free - free value structure components.
 *
 * Returns void (no errors)
 */
void
value_free(
	struct value *vp	/* value structure */
	)
{
	if (vp->ptr != NULL)
		free(vp->ptr);
	if (vp->sig != NULL)
		free(vp->sig);
	memset(vp, 0, sizeof(struct value));
}


/*
 * crypto_time - returns current NTP time.
 *
 * Returns NTP seconds if in synch, 0 otherwise
 */
tstamp_t
crypto_time()
{
	l_fp	tstamp;		/* NTP time */

	L_CLR(&tstamp);
	if (sys_leap != LEAP_NOTINSYNC)
		get_systime(&tstamp);
	return (tstamp.l_ui);
}


/*
 * asn2ntp - convert ASN1_TIME time structure to NTP time.
 *
 * Returns NTP seconds (no errors)
 */
u_long
asn2ntp	(
	ASN1_TIME *asn1time	/* pointer to ASN1_TIME structure */
	)
{
	char	*v;		/* pointer to ASN1_TIME string */
	struct	tm tm;		/* used to convert to NTP time */

	/*
	 * Extract time string YYMMDDHHMMSSZ from ASN1 time structure.
	 * Note that the YY, MM, DD fields start with one, the HH, MM,
	 * SS fiels start with zero and the Z character is ignored.
	 * Also note that years less than 50 map to years greater than
	 * 100. Dontcha love ASN.1? Better than MIL-188.
	 */
	v = (char *)asn1time->data;
	tm.tm_year = (v[0] - '0') * 10 + v[1] - '0';
	if (tm.tm_year < 50)
		tm.tm_year += 100;
	tm.tm_mon = (v[2] - '0') * 10 + v[3] - '0' - 1;
	tm.tm_mday = (v[4] - '0') * 10 + v[5] - '0';
	tm.tm_hour = (v[6] - '0') * 10 + v[7] - '0';
	tm.tm_min = (v[8] - '0') * 10 + v[9] - '0';
	tm.tm_sec = (v[10] - '0') * 10 + v[11] - '0';
	tm.tm_wday = 0;
	tm.tm_yday = 0;
	tm.tm_isdst = 0;
	return (timegm(&tm) + JAN_1970);
}


/*
 * bigdig() - compute a BIGNUM MD5 hash of a BIGNUM number.
 *
 * Returns void (no errors)
 */
static void
bighash(
	BIGNUM	*bn,		/* BIGNUM * from */
	BIGNUM	*bk		/* BIGNUM * to */
	)
{
	EVP_MD_CTX ctx;		/* message digest context */
	u_char dgst[EVP_MAX_MD_SIZE]; /* message digest */
	u_char	*ptr;		/* a BIGNUM as binary string */
	u_int	len;

	len = BN_num_bytes(bn);
	ptr = emalloc(len);
	BN_bn2bin(bn, ptr);
	EVP_DigestInit(&ctx, EVP_md5());
	EVP_DigestUpdate(&ctx, ptr, len);
	EVP_DigestFinal(&ctx, dgst, &len);
	BN_bin2bn(dgst, len, bk);
	free(ptr);
}


/*
 ***********************************************************************
 *								       *
 * The following routines implement the Schnorr (IFF) identity scheme  *
 *								       *
 ***********************************************************************
 *
 * The Schnorr (IFF) identity scheme is intended for use when
 * certificates are generated by some other trusted certificate
 * authority and the certificate cannot be used to convey public
 * parameters. There are two kinds of files: encrypted server files that
 * contain private and public values and nonencrypted client files that
 * contain only public values. New generations of server files must be
 * securely transmitted to all servers of the group; client files can be
 * distributed by any means. The scheme is self contained and
 * independent of new generations of host keys, sign keys and
 * certificates.
 *
 * The IFF values hide in a DSA cuckoo structure which uses the same
 * parameters. The values are used by an identity scheme based on DSA
 * cryptography and described in Stimson p. 285. The p is a 512-bit
 * prime, g a generator of Zp* and q a 160-bit prime that divides p - 1
 * and is a qth root of 1 mod p; that is, g^q = 1 mod p. The TA rolls a
 * private random group key b (0 < b < q) and public key v = g^b, then
 * sends (p, q, g, b) to the servers and (p, q, g, v) to the clients.
 * Alice challenges Bob to confirm identity using the protocol described
 * below.
 *
 * How it works
 *
 * The scheme goes like this. Both Alice and Bob have the public primes
 * p, q and generator g. The TA gives private key b to Bob and public
 * key v to Alice.
 *
 * Alice rolls new random challenge r (o < r < q) and sends to Bob in
 * the IFF request message. Bob rolls new random k (0 < k < q), then
 * computes y = k + b r mod q and x = g^k mod p and sends (y, hash(x))
 * to Alice in the response message. Besides making the response
 * shorter, the hash makes it effectivey impossible for an intruder to
 * solve for b by observing a number of these messages.
 * 
 * Alice receives the response and computes g^y v^r mod p. After a bit
 * of algebra, this simplifies to g^k. If the hash of this result
 * matches hash(x), Alice knows that Bob has the group key b. The signed
 * response binds this knowledge to Bob's private key and the public key
 * previously received in his certificate.
 *
 * crypto_alice - construct Alice's challenge in IFF scheme
 *
 * Returns
 * XEVNT_OK	success
 * XEVNT_ID	bad or missing group key
 * XEVNT_PUB	bad or missing public key
 */
static int
crypto_alice(
	struct peer *peer,	/* peer pointer */
	struct value *vp	/* value pointer */
	)
{
	DSA	*dsa;		/* IFF parameters */
	BN_CTX	*bctx;		/* BIGNUM context */
	EVP_MD_CTX ctx;		/* signature context */
	tstamp_t tstamp;
	u_int	len;

	/*
	 * The identity parameters must have correct format and content.
	 */
	if (peer->ident_pkey == NULL)
		return (XEVNT_ID);

	if ((dsa = peer->ident_pkey->pkey->pkey.dsa) == NULL) {
		msyslog(LOG_NOTICE, "crypto_alice: defective key");
		return (XEVNT_PUB);
	}

	/*
	 * Roll new random r (0 < r < q).
	 */
	if (peer->iffval != NULL)
		BN_free(peer->iffval);
	peer->iffval = BN_new();
	len = BN_num_bytes(dsa->q);
	BN_rand(peer->iffval, len * 8, -1, 1);	/* r mod q*/
	bctx = BN_CTX_new();
	BN_mod(peer->iffval, peer->iffval, dsa->q, bctx);
	BN_CTX_free(bctx);

	/*
	 * Sign and send to Bob. The filestamp is from the local file.
	 */
	memset(vp, 0, sizeof(struct value));
	tstamp = crypto_time();
	vp->tstamp = htonl(tstamp);
	vp->fstamp = htonl(peer->ident_pkey->fstamp);
	vp->vallen = htonl(len);
	vp->ptr = emalloc(len);
	BN_bn2bin(peer->iffval, vp->ptr);
	if (tstamp == 0)
		return (XEVNT_OK);

	vp->sig = emalloc(sign_siglen);
	EVP_SignInit(&ctx, sign_digest);
	EVP_SignUpdate(&ctx, (u_char *)&vp->tstamp, 12);
	EVP_SignUpdate(&ctx, vp->ptr, len);
	if (EVP_SignFinal(&ctx, vp->sig, &len, sign_pkey))
		vp->siglen = htonl(len);
	return (XEVNT_OK);
}


/*
 * crypto_bob - construct Bob's response to Alice's challenge
 *
 * Returns
 * XEVNT_OK	success
 * XEVNT_ERR	protocol error
 * XEVNT_ID	bad or missing group key
 */
static int
crypto_bob(
	struct exten *ep,	/* extension pointer */
	struct value *vp	/* value pointer */
	)
{
	DSA	*dsa;		/* IFF parameters */
	DSA_SIG	*sdsa;		/* DSA signature context fake */
	BN_CTX	*bctx;		/* BIGNUM context */
	EVP_MD_CTX ctx;		/* signature context */
	tstamp_t tstamp;	/* NTP timestamp */
	BIGNUM	*bn, *bk, *r;
	u_char	*ptr;
	u_int	len;

	/*
	 * If the IFF parameters are not valid, something awful
	 * happened or we are being tormented.
	 */
	if (iffkey_info == NULL) {
		msyslog(LOG_NOTICE, "crypto_bob: scheme unavailable");
		return (XEVNT_ID);
	}
	dsa = iffkey_info->pkey->pkey.dsa;

	/*
	 * Extract r from the challenge.
	 */
	len = ntohl(ep->vallen);
	if ((r = BN_bin2bn((u_char *)ep->pkt, len, NULL)) == NULL) {
		msyslog(LOG_ERR, "crypto_bob: %s",
		    ERR_error_string(ERR_get_error(), NULL));
		return (XEVNT_ERR);
	}

	/*
	 * Bob rolls random k (0 < k < q), computes y = k + b r mod q
	 * and x = g^k mod p, then sends (y, hash(x)) to Alice.
	 */
	bctx = BN_CTX_new(); bk = BN_new(); bn = BN_new();
	sdsa = DSA_SIG_new();
	BN_rand(bk, len * 8, -1, 1);		/* k */
	BN_mod_mul(bn, dsa->priv_key, r, dsa->q, bctx); /* b r mod q */
	BN_add(bn, bn, bk);
	BN_mod(bn, bn, dsa->q, bctx);		/* k + b r mod q */
	sdsa->r = BN_dup(bn);
	BN_mod_exp(bk, dsa->g, bk, dsa->p, bctx); /* g^k mod p */
	bighash(bk, bk);
	sdsa->s = BN_dup(bk);
	BN_CTX_free(bctx);
	BN_free(r); BN_free(bn); BN_free(bk);
#ifdef DEBUG
	if (debug > 1)
		DSA_print_fp(stdout, dsa, 0);
#endif

	/*
	 * Encode the values in ASN.1 and sign. The filestamp is from
	 * the local file.
	 */
	len = i2d_DSA_SIG(sdsa, NULL);
	if (len == 0) {
		msyslog(LOG_ERR, "crypto_bob: %s",
		    ERR_error_string(ERR_get_error(), NULL));
		DSA_SIG_free(sdsa);
		return (XEVNT_ERR);
	}
	memset(vp, 0, sizeof(struct value));
	tstamp = crypto_time();
	vp->tstamp = htonl(tstamp);
	vp->fstamp = htonl(iffkey_info->fstamp);
	vp->vallen = htonl(len);
	ptr = emalloc(len);
	vp->ptr = ptr;
	i2d_DSA_SIG(sdsa, &ptr);
	DSA_SIG_free(sdsa);
	if (tstamp == 0)
		return (XEVNT_OK);

	vp->sig = emalloc(sign_siglen);
	EVP_SignInit(&ctx, sign_digest);
	EVP_SignUpdate(&ctx, (u_char *)&vp->tstamp, 12);
	EVP_SignUpdate(&ctx, vp->ptr, len);
	if (EVP_SignFinal(&ctx, vp->sig, &len, sign_pkey))
		vp->siglen = htonl(len);
	return (XEVNT_OK);
}


/*
 * crypto_iff - verify Bob's response to Alice's challenge
 *
 * Returns
 * XEVNT_OK	success
 * XEVNT_FSP	bad filestamp
 * XEVNT_ID	bad or missing group key
 * XEVNT_PUB	bad or missing public key
 */
int
crypto_iff(
	struct exten *ep,	/* extension pointer */
	struct peer *peer	/* peer structure pointer */
	)
{
	DSA	*dsa;		/* IFF parameters */
	BN_CTX	*bctx;		/* BIGNUM context */
	DSA_SIG	*sdsa;		/* DSA parameters */
	BIGNUM	*bn, *bk;
	u_int	len;
	const u_char *ptr;
	int	temp;

	/*
	 * If the IFF parameters are not valid or no challenge was sent,
	 * something awful happened or we are being tormented.
	 */
	if (peer->ident_pkey == NULL) {
		msyslog(LOG_NOTICE, "crypto_iff: scheme unavailable");
		return (XEVNT_ID);
	}
	if (ntohl(ep->fstamp) != peer->ident_pkey->fstamp) {
		msyslog(LOG_NOTICE, "crypto_iff: invalid filestamp %u",
		    ntohl(ep->fstamp));
		return (XEVNT_FSP);
	}
	if ((dsa = peer->ident_pkey->pkey->pkey.dsa) == NULL) {
		msyslog(LOG_NOTICE, "crypto_iff: defective key");
		return (XEVNT_PUB);
	}
	if (peer->iffval == NULL) {
		msyslog(LOG_NOTICE, "crypto_iff: missing challenge");
		return (XEVNT_ID);
	}

	/*
	 * Extract the k + b r and g^k values from the response.
	 */
	bctx = BN_CTX_new(); bk = BN_new(); bn = BN_new();
	len = ntohl(ep->vallen);
	ptr = (u_char *)ep->pkt;
	if ((sdsa = d2i_DSA_SIG(NULL, &ptr, len)) == NULL) {
		BN_free(bn); BN_free(bk); BN_CTX_free(bctx);
		msyslog(LOG_ERR, "crypto_iff: %s",
		    ERR_error_string(ERR_get_error(), NULL));
		return (XEVNT_ERR);
	}

	/*
	 * Compute g^(k + b r) g^(q - b)r mod p.
	 */
	BN_mod_exp(bn, dsa->pub_key, peer->iffval, dsa->p, bctx);
	BN_mod_exp(bk, dsa->g, sdsa->r, dsa->p, bctx);
	BN_mod_mul(bn, bn, bk, dsa->p, bctx);

	/*
	 * Verify the hash of the result matches hash(x).
	 */
	bighash(bn, bn);
	temp = BN_cmp(bn, sdsa->s);
	BN_free(bn); BN_free(bk); BN_CTX_free(bctx);
	BN_free(peer->iffval);
	peer->iffval = NULL;
	DSA_SIG_free(sdsa);
	if (temp == 0)
		return (XEVNT_OK);

	msyslog(LOG_NOTICE, "crypto_iff: identity not verified");
	return (XEVNT_ID);
}


/*
 ***********************************************************************
 *								       *
 * The following routines implement the Guillou-Quisquater (GQ)        *
 * identity scheme                                                     *
 *								       *
 ***********************************************************************
 *
 * The Guillou-Quisquater (GQ) identity scheme is intended for use when
 * the certificate can be used to convey public parameters. The scheme
 * uses a X509v3 certificate extension field do convey the public key of
 * a private key known only to servers. There are two kinds of files:
 * encrypted server files that contain private and public values and
 * nonencrypted client files that contain only public values. New
 * generations of server files must be securely transmitted to all
 * servers of the group; client files can be distributed by any means.
 * The scheme is self contained and independent of new generations of
 * host keys and sign keys. The scheme is self contained and independent
 * of new generations of host keys and sign keys.
 *
 * The GQ parameters hide in a RSA cuckoo structure which uses the same
 * parameters. The values are used by an identity scheme based on RSA
 * cryptography and described in Stimson p. 300 (with errors). The 512-
 * bit public modulus is n = p q, where p and q are secret large primes.
 * The TA rolls private random group key b as RSA exponent. These values
 * are known to all group members.
 *
 * When rolling new certificates, a server recomputes the private and
 * public keys. The private key u is a random roll, while the public key
 * is the inverse obscured by the group key v = (u^-1)^b. These values
 * replace the private and public keys normally generated by the RSA
 * scheme. Alice challenges Bob to confirm identity using the protocol
 * described below.
 *
 * How it works
 *
 * The scheme goes like this. Both Alice and Bob have the same modulus n
 * and some random b as the group key. These values are computed and
 * distributed in advance via secret means, although only the group key
 * b is truly secret. Each has a private random private key u and public
 * key (u^-1)^b, although not necessarily the same ones. Bob and Alice
 * can regenerate the key pair from time to time without affecting
 * operations. The public key is conveyed on the certificate in an
 * extension field; the private key is never revealed.
 *
 * Alice rolls new random challenge r and sends to Bob in the GQ
 * request message. Bob rolls new random k, then computes y = k u^r mod
 * n and x = k^b mod n and sends (y, hash(x)) to Alice in the response
 * message. Besides making the response shorter, the hash makes it
 * effectivey impossible for an intruder to solve for b by observing
 * a number of these messages.
 * 
 * Alice receives the response and computes y^b v^r mod n. After a bit
 * of algebra, this simplifies to k^b. If the hash of this result
 * matches hash(x), Alice knows that Bob has the group key b. The signed
 * response binds this knowledge to Bob's private key and the public key
 * previously received in his certificate.
 *
 * crypto_alice2 - construct Alice's challenge in GQ scheme
 *
 * Returns
 * XEVNT_OK	success
 * XEVNT_ID	bad or missing group key
 * XEVNT_PUB	bad or missing public key
 */
static int
crypto_alice2(
	struct peer *peer,	/* peer pointer */
	struct value *vp	/* value pointer */
	)
{
	RSA	*rsa;		/* GQ parameters */
	BN_CTX	*bctx;		/* BIGNUM context */
	EVP_MD_CTX ctx;		/* signature context */
	tstamp_t tstamp;
	u_int	len;

	/*
	 * The identity parameters must have correct format and content.
	 */
	if (peer->ident_pkey == NULL)
		return (XEVNT_ID);

	if ((rsa = peer->ident_pkey->pkey->pkey.rsa) == NULL) {
		msyslog(LOG_NOTICE, "crypto_alice2: defective key");
		return (XEVNT_PUB);
	}

	/*
	 * Roll new random r (0 < r < n).
	 */
	if (peer->iffval != NULL)
		BN_free(peer->iffval);
	peer->iffval = BN_new();
	len = BN_num_bytes(rsa->n);
	BN_rand(peer->iffval, len * 8, -1, 1);	/* r mod n */
	bctx = BN_CTX_new();
	BN_mod(peer->iffval, peer->iffval, rsa->n, bctx);
	BN_CTX_free(bctx);

	/*
	 * Sign and send to Bob. The filestamp is from the local file.
	 */
	memset(vp, 0, sizeof(struct value));
	tstamp = crypto_time();
	vp->tstamp = htonl(tstamp);
	vp->fstamp = htonl(peer->ident_pkey->fstamp);
	vp->vallen = htonl(len);
	vp->ptr = emalloc(len);
	BN_bn2bin(peer->iffval, vp->ptr);
	if (tstamp == 0)
		return (XEVNT_OK);

	vp->sig = emalloc(sign_siglen);
	EVP_SignInit(&ctx, sign_digest);
	EVP_SignUpdate(&ctx, (u_char *)&vp->tstamp, 12);
	EVP_SignUpdate(&ctx, vp->ptr, len);
	if (EVP_SignFinal(&ctx, vp->sig, &len, sign_pkey))
		vp->siglen = htonl(len);
	return (XEVNT_OK);
}


/*
 * crypto_bob2 - construct Bob's response to Alice's challenge
 *
 * Returns
 * XEVNT_OK	success
 * XEVNT_ERR	protocol error
 * XEVNT_ID	bad or missing group key
 */
static int
crypto_bob2(
	struct exten *ep,	/* extension pointer */
	struct value *vp	/* value pointer */
	)
{
	RSA	*rsa;		/* GQ parameters */
	DSA_SIG	*sdsa;		/* DSA parameters */
	BN_CTX	*bctx;		/* BIGNUM context */
	EVP_MD_CTX ctx;		/* signature context */
	tstamp_t tstamp;	/* NTP timestamp */
	BIGNUM	*r, *k, *g, *y;
	u_char	*ptr;
	u_int	len;

	/*
	 * If the GQ parameters are not valid, something awful
	 * happened or we are being tormented.
	 */
	if (gqkey_info == NULL) {
		msyslog(LOG_NOTICE, "crypto_bob2: scheme unavailable");
		return (XEVNT_ID);
	}
	rsa = gqkey_info->pkey->pkey.rsa;

	/*
	 * Extract r from the challenge.
	 */
	len = ntohl(ep->vallen);
	if ((r = BN_bin2bn((u_char *)ep->pkt, len, NULL)) == NULL) {
		msyslog(LOG_ERR, "crypto_bob2: %s",
		    ERR_error_string(ERR_get_error(), NULL));
		return (XEVNT_ERR);
	}

	/*
	 * Bob rolls random k (0 < k < n), computes y = k u^r mod n and
	 * x = k^b mod n, then sends (y, hash(x)) to Alice. 
	 */
	bctx = BN_CTX_new(); k = BN_new(); g = BN_new(); y = BN_new();
	sdsa = DSA_SIG_new();
	BN_rand(k, len * 8, -1, 1);		/* k */
	BN_mod(k, k, rsa->n, bctx);
	BN_mod_exp(y, rsa->p, r, rsa->n, bctx); /* u^r mod n */
	BN_mod_mul(y, k, y, rsa->n, bctx);	/* k u^r mod n */
	sdsa->r = BN_dup(y);
	BN_mod_exp(g, k, rsa->e, rsa->n, bctx); /* k^b mod n */
	bighash(g, g);
	sdsa->s = BN_dup(g);
	BN_CTX_free(bctx);
	BN_free(r); BN_free(k); BN_free(g); BN_free(y);
#ifdef DEBUG
	if (debug > 1)
		RSA_print_fp(stdout, rsa, 0);
#endif
 
	/*
	 * Encode the values in ASN.1 and sign. The filestamp is from
	 * the local file.
	 */
	len = i2d_DSA_SIG(sdsa, NULL);
	if (len <= 0) {
		msyslog(LOG_ERR, "crypto_bob2: %s",
		    ERR_error_string(ERR_get_error(), NULL));
		DSA_SIG_free(sdsa);
		return (XEVNT_ERR);
	}
	memset(vp, 0, sizeof(struct value));
	tstamp = crypto_time();
	vp->tstamp = htonl(tstamp);
	vp->fstamp = htonl(gqkey_info->fstamp);
	vp->vallen = htonl(len);
	ptr = emalloc(len);
	vp->ptr = ptr;
	i2d_DSA_SIG(sdsa, &ptr);
	DSA_SIG_free(sdsa);
	if (tstamp == 0)
		return (XEVNT_OK);

	vp->sig = emalloc(sign_siglen);
	EVP_SignInit(&ctx, sign_digest);
	EVP_SignUpdate(&ctx, (u_char *)&vp->tstamp, 12);
	EVP_SignUpdate(&ctx, vp->ptr, len);
	if (EVP_SignFinal(&ctx, vp->sig, &len, sign_pkey))
		vp->siglen = htonl(len);
	return (XEVNT_OK);
}


/*
 * crypto_gq - verify Bob's response to Alice's challenge
 *
 * Returns
 * XEVNT_OK	success
 * XEVNT_ERR	protocol error
 * XEVNT_FSP	bad filestamp
 * XEVNT_ID	bad or missing group keys
 * XEVNT_PUB	bad or missing public key
 */
int
crypto_gq(
	struct exten *ep,	/* extension pointer */
	struct peer *peer	/* peer structure pointer */
	)
{
	RSA	*rsa;		/* GQ parameters */
	BN_CTX	*bctx;		/* BIGNUM context */
	DSA_SIG	*sdsa;		/* RSA signature context fake */
	BIGNUM	*y, *v;
	const u_char *ptr;
	long	len;
	u_int	temp;

	/*
	 * If the GQ parameters are not valid or no challenge was sent,
	 * something awful happened or we are being tormented. Note that
	 * the filestamp on the local key file can be greater than on
	 * the remote parameter file if the keys have been refreshed.
	 */
	if (peer->ident_pkey == NULL) {
		msyslog(LOG_NOTICE, "crypto_gq: scheme unavailable");
		return (XEVNT_ID);
	}
	if (ntohl(ep->fstamp) < peer->ident_pkey->fstamp) {
		msyslog(LOG_NOTICE, "crypto_gq: invalid filestamp %u",
		    ntohl(ep->fstamp));
		return (XEVNT_FSP);
	}
	if ((rsa = peer->ident_pkey->pkey->pkey.rsa) == NULL) {
		msyslog(LOG_NOTICE, "crypto_gq: defective key");
		return (XEVNT_PUB);
	}
	if (peer->iffval == NULL) {
		msyslog(LOG_NOTICE, "crypto_gq: missing challenge");
		return (XEVNT_ID);
	}

	/*
	 * Extract the y = k u^r and hash(x = k^b) values from the
	 * response.
	 */
	bctx = BN_CTX_new(); y = BN_new(); v = BN_new();
	len = ntohl(ep->vallen);
	ptr = (u_char *)ep->pkt;
	if ((sdsa = d2i_DSA_SIG(NULL, &ptr, len)) == NULL) {
		BN_CTX_free(bctx); BN_free(y); BN_free(v);
		msyslog(LOG_ERR, "crypto_gq: %s",
		    ERR_error_string(ERR_get_error(), NULL));
		return (XEVNT_ERR);
	}

	/*
	 * Compute v^r y^b mod n.
	 */
	if (peer->grpkey == NULL) {
		msyslog(LOG_NOTICE, "crypto_gq: missing group key");
		return (XEVNT_ID);
	}
	BN_mod_exp(v, peer->grpkey, peer->iffval, rsa->n, bctx);
						/* v^r mod n */
	BN_mod_exp(y, sdsa->r, rsa->e, rsa->n, bctx); /* y^b mod n */
	BN_mod_mul(y, v, y, rsa->n, bctx);	/* v^r y^b mod n */

	/*
	 * Verify the hash of the result matches hash(x).
	 */
	bighash(y, y);
	temp = BN_cmp(y, sdsa->s);
	BN_CTX_free(bctx); BN_free(y); BN_free(v);
	BN_free(peer->iffval);
	peer->iffval = NULL;
	DSA_SIG_free(sdsa);
	if (temp == 0)
		return (XEVNT_OK);

	msyslog(LOG_NOTICE, "crypto_gq: identity not verified");
	return (XEVNT_ID);
}


/*
 ***********************************************************************
 *								       *
 * The following routines implement the Mu-Varadharajan (MV) identity  *
 * scheme                                                              *
 *								       *
 ***********************************************************************
 *
 * The Mu-Varadharajan (MV) cryptosystem was originally intended when
 * servers broadcast messages to clients, but clients never send
 * messages to servers. There is one encryption key for the server and a
 * separate decryption key for each client. It operated something like a
 * pay-per-view satellite broadcasting system where the session key is
 * encrypted by the broadcaster and the decryption keys are held in a
 * tamperproof set-top box.
 *
 * The MV parameters and private encryption key hide in a DSA cuckoo
 * structure which uses the same parameters, but generated in a
 * different way. The values are used in an encryption scheme similar to
 * El Gamal cryptography and a polynomial formed from the expansion of
 * product terms (x - x[j]), as described in Mu, Y., and V.
 * Varadharajan: Robust and Secure Broadcasting, Proc. Indocrypt 2001,
 * 223-231. The paper has significant errors and serious omissions.
 *
 * Let q be the product of n distinct primes s1[j] (j = 1...n), where
 * each s1[j] has m significant bits. Let p be a prime p = 2 * q + 1, so
 * that q and each s1[j] divide p - 1 and p has M = n * m + 1
 * significant bits. Let g be a generator of Zp; that is, gcd(g, p - 1)
 * = 1 and g^q = 1 mod p. We do modular arithmetic over Zq and then
 * project into Zp* as exponents of g. Sometimes we have to compute an
 * inverse b^-1 of random b in Zq, but for that purpose we require
 * gcd(b, q) = 1. We expect M to be in the 500-bit range and n
 * relatively small, like 30. These are the parameters of the scheme and
 * they are expensive to compute.
 *
 * We set up an instance of the scheme as follows. A set of random
 * values x[j] mod q (j = 1...n), are generated as the zeros of a
 * polynomial of order n. The product terms (x - x[j]) are expanded to
 * form coefficients a[i] mod q (i = 0...n) in powers of x. These are
 * used as exponents of the generator g mod p to generate the private
 * encryption key A. The pair (gbar, ghat) of public server keys and the
 * pairs (xbar[j], xhat[j]) (j = 1...n) of private client keys are used
 * to construct the decryption keys. The devil is in the details.
 *
 * This routine generates a private server encryption file including the
 * private encryption key E and partial decryption keys gbar and ghat.
 * It then generates public client decryption files including the public
 * keys xbar[j] and xhat[j] for each client j. The partial decryption
 * files are used to compute the inverse of E. These values are suitably
 * blinded so secrets are not revealed.
 *
 * The distinguishing characteristic of this scheme is the capability to
 * revoke keys. Included in the calculation of E, gbar and ghat is the
 * product s = prod(s1[j]) (j = 1...n) above. If the factor s1[j] is
 * subsequently removed from the product and E, gbar and ghat
 * recomputed, the jth client will no longer be able to compute E^-1 and
 * thus unable to decrypt the messageblock.
 *
 * How it works
 *
 * The scheme goes like this. Bob has the server values (p, E, q, gbar,
 * ghat) and Alice has the client values (p, xbar, xhat).
 *
 * Alice rolls new random nonce r mod p and sends to Bob in the MV
 * request message. Bob rolls random nonce k mod q, encrypts y = r E^k
 * mod p and sends (y, gbar^k, ghat^k) to Alice.
 * 
 * Alice receives the response and computes the inverse (E^k)^-1 from
 * the partial decryption keys gbar^k, ghat^k, xbar and xhat. She then
 * decrypts y and verifies it matches the original r. The signed
 * response binds this knowledge to Bob's private key and the public key
 * previously received in his certificate.
 *
 * crypto_alice3 - construct Alice's challenge in MV scheme
 *
 * Returns
 * XEVNT_OK	success
 * XEVNT_ID	bad or missing group key
 * XEVNT_PUB	bad or missing public key
 */
static int
crypto_alice3(
	struct peer *peer,	/* peer pointer */
	struct value *vp	/* value pointer */
	)
{
	DSA	*dsa;		/* MV parameters */
	BN_CTX	*bctx;		/* BIGNUM context */
	EVP_MD_CTX ctx;		/* signature context */
	tstamp_t tstamp;
	u_int	len;

	/*
	 * The identity parameters must have correct format and content.
	 */
	if (peer->ident_pkey == NULL)
		return (XEVNT_ID);

	if ((dsa = peer->ident_pkey->pkey->pkey.dsa) == NULL) {
		msyslog(LOG_NOTICE, "crypto_alice3: defective key");
		return (XEVNT_PUB);
	}

	/*
	 * Roll new random r (0 < r < q).
	 */
	if (peer->iffval != NULL)
		BN_free(peer->iffval);
	peer->iffval = BN_new();
	len = BN_num_bytes(dsa->p);
	BN_rand(peer->iffval, len * 8, -1, 1);	/* r mod p */
	bctx = BN_CTX_new();
	BN_mod(peer->iffval, peer->iffval, dsa->p, bctx);
	BN_CTX_free(bctx);

	/*
	 * Sign and send to Bob. The filestamp is from the local file.
	 */
	memset(vp, 0, sizeof(struct value));
	tstamp = crypto_time();
	vp->tstamp = htonl(tstamp);
	vp->fstamp = htonl(peer->ident_pkey->fstamp);
	vp->vallen = htonl(len);
	vp->ptr = emalloc(len);
	BN_bn2bin(peer->iffval, vp->ptr);
	if (tstamp == 0)
		return (XEVNT_OK);

	vp->sig = emalloc(sign_siglen);
	EVP_SignInit(&ctx, sign_digest);
	EVP_SignUpdate(&ctx, (u_char *)&vp->tstamp, 12);
	EVP_SignUpdate(&ctx, vp->ptr, len);
	if (EVP_SignFinal(&ctx, vp->sig, &len, sign_pkey))
		vp->siglen = htonl(len);
	return (XEVNT_OK);
}


/*
 * crypto_bob3 - construct Bob's response to Alice's challenge
 *
 * Returns
 * XEVNT_OK	success
 * XEVNT_ERR	protocol error
 */
static int
crypto_bob3(
	struct exten *ep,	/* extension pointer */
	struct value *vp	/* value pointer */
	)
{
	DSA	*dsa;		/* MV parameters */
	DSA	*sdsa;		/* DSA signature context fake */
	BN_CTX	*bctx;		/* BIGNUM context */
	EVP_MD_CTX ctx;		/* signature context */
	tstamp_t tstamp;	/* NTP timestamp */
	BIGNUM	*r, *k, *u;
	u_char	*ptr;
	u_int	len;

	/*
	 * If the MV parameters are not valid, something awful
	 * happened or we are being tormented.
	 */
	if (mvkey_info == NULL) {
		msyslog(LOG_NOTICE, "crypto_bob3: scheme unavailable");
		return (XEVNT_ID);
	}
	dsa = mvkey_info->pkey->pkey.dsa;

	/*
	 * Extract r from the challenge.
	 */
	len = ntohl(ep->vallen);
	if ((r = BN_bin2bn((u_char *)ep->pkt, len, NULL)) == NULL) {
		msyslog(LOG_ERR, "crypto_bob3: %s",
		    ERR_error_string(ERR_get_error(), NULL));
		return (XEVNT_ERR);
	}

	/*
	 * Bob rolls random k (0 < k < q), making sure it is not a
	 * factor of q. He then computes y = r A^k and sends (y, gbar^k,
	 * and ghat^k) to Alice.
	 */
	bctx = BN_CTX_new(); k = BN_new(); u = BN_new();
	sdsa = DSA_new();
	sdsa->p = BN_new(); sdsa->q = BN_new(); sdsa->g = BN_new();
	while (1) {
		BN_rand(k, BN_num_bits(dsa->q), 0, 0);
		BN_mod(k, k, dsa->q, bctx);
		BN_gcd(u, k, dsa->q, bctx);
		if (BN_is_one(u))
			break;
	}
	BN_mod_exp(u, dsa->g, k, dsa->p, bctx); /* A^k r */
	BN_mod_mul(sdsa->p, u, r, dsa->p, bctx);
	BN_mod_exp(sdsa->q, dsa->priv_key, k, dsa->p, bctx); /* gbar */
	BN_mod_exp(sdsa->g, dsa->pub_key, k, dsa->p, bctx); /* ghat */
	BN_CTX_free(bctx); BN_free(k); BN_free(r); BN_free(u);
#ifdef DEBUG
	if (debug > 1)
		DSA_print_fp(stdout, sdsa, 0);
#endif

	/*
	 * Encode the values in ASN.1 and sign. The filestamp is from
	 * the local file.
	 */
	memset(vp, 0, sizeof(struct value));
	tstamp = crypto_time();
	vp->tstamp = htonl(tstamp);
	vp->fstamp = htonl(mvkey_info->fstamp);
	len = i2d_DSAparams(sdsa, NULL);
	if (len == 0) {
		msyslog(LOG_ERR, "crypto_bob3: %s",
		    ERR_error_string(ERR_get_error(), NULL));
		DSA_free(sdsa);
		return (XEVNT_ERR);
	}
	vp->vallen = htonl(len);
	ptr = emalloc(len);
	vp->ptr = ptr;
	i2d_DSAparams(sdsa, &ptr);
	DSA_free(sdsa);
	if (tstamp == 0)
		return (XEVNT_OK);

	vp->sig = emalloc(sign_siglen);
	EVP_SignInit(&ctx, sign_digest);
	EVP_SignUpdate(&ctx, (u_char *)&vp->tstamp, 12);
	EVP_SignUpdate(&ctx, vp->ptr, len);
	if (EVP_SignFinal(&ctx, vp->sig, &len, sign_pkey))
		vp->siglen = htonl(len);
	return (XEVNT_OK);
}


/*
 * crypto_mv - verify Bob's response to Alice's challenge
 *
 * Returns
 * XEVNT_OK	success
 * XEVNT_ERR	protocol error
 * XEVNT_FSP	bad filestamp
 * XEVNT_ID	bad or missing group key
 * XEVNT_PUB	bad or missing public key
 */
int
crypto_mv(
	struct exten *ep,	/* extension pointer */
	struct peer *peer	/* peer structure pointer */
	)
{
	DSA	*dsa;		/* MV parameters */
	DSA	*sdsa;		/* DSA parameters */
	BN_CTX	*bctx;		/* BIGNUM context */
	BIGNUM	*k, *u, *v;
	u_int	len;
	const u_char *ptr;
	int	temp;

	/*
	 * If the MV parameters are not valid or no challenge was sent,
	 * something awful happened or we are being tormented.
	 */
	if (peer->ident_pkey == NULL) {
		msyslog(LOG_NOTICE, "crypto_mv: scheme unavailable");
		return (XEVNT_ID);
	}
	if (ntohl(ep->fstamp) != peer->ident_pkey->fstamp) {
		msyslog(LOG_NOTICE, "crypto_mv: invalid filestamp %u",
		    ntohl(ep->fstamp));
		return (XEVNT_FSP);
	}
	if ((dsa = peer->ident_pkey->pkey->pkey.dsa) == NULL) {
		msyslog(LOG_NOTICE, "crypto_mv: defective key");
		return (XEVNT_PUB);
	}
	if (peer->iffval == NULL) {
		msyslog(LOG_NOTICE, "crypto_mv: missing challenge");
		return (XEVNT_ID);
	}

	/*
	 * Extract the y, gbar and ghat values from the response.
	 */
	bctx = BN_CTX_new(); k = BN_new(); u = BN_new(); v = BN_new();
	len = ntohl(ep->vallen);
	ptr = (u_char *)ep->pkt;
	if ((sdsa = d2i_DSAparams(NULL, &ptr, len)) == NULL) {
		msyslog(LOG_ERR, "crypto_mv: %s",
		    ERR_error_string(ERR_get_error(), NULL));
		return (XEVNT_ERR);
	}

	/*
	 * Compute (gbar^xhat ghat^xbar) mod p.
	 */
	BN_mod_exp(u, sdsa->q, dsa->pub_key, dsa->p, bctx);
	BN_mod_exp(v, sdsa->g, dsa->priv_key, dsa->p, bctx);
	BN_mod_mul(u, u, v, dsa->p, bctx);
	BN_mod_mul(u, u, sdsa->p, dsa->p, bctx);

	/*
	 * The result should match r.
	 */
	temp = BN_cmp(u, peer->iffval);
	BN_CTX_free(bctx); BN_free(k); BN_free(u); BN_free(v);
	BN_free(peer->iffval);
	peer->iffval = NULL;
	DSA_free(sdsa);
	if (temp == 0)
		return (XEVNT_OK);

	msyslog(LOG_NOTICE, "crypto_mv: identity not verified");
	return (XEVNT_ID);
}


/*
 ***********************************************************************
 *								       *
 * The following routines are used to manipulate certificates          *
 *								       *
 ***********************************************************************
 */
/*
 * cert_sign - sign x509 certificate equest and update value structure.
 *
 * The certificate request includes a copy of the host certificate,
 * which includes the version number, subject name and public key of the
 * host. The resulting certificate includes these values plus the
 * serial number, issuer name and valid interval of the server. The
 * valid interval extends from the current time to the same time one
 * year hence. This may extend the life of the signed certificate beyond
 * that of the signer certificate.
 *
 * It is convenient to use the NTP seconds of the current time as the
 * serial number. In the value structure the timestamp is the current
 * time and the filestamp is taken from the extension field. Note this
 * routine is called only when the client clock is synchronized to a
 * proventic source, so timestamp comparisons are valid.
 *
 * The host certificate is valid from the time it was generated for a
 * period of one year. A signed certificate is valid from the time of
 * signature for a period of one year, but only the host certificate (or
 * sign certificate if used) is actually used to encrypt and decrypt
 * signatures. The signature trail is built from the client via the
 * intermediate servers to the trusted server. Each signature on the
 * trail must be valid at the time of signature, but it could happen
 * that a signer certificate expire before the signed certificate, which
 * remains valid until its expiration. 
 *
 * Returns
 * XEVNT_CRT	bad or missing certificate
 * XEVNT_OK	success
 * XEVNT_PER	host certificate expired
 * XEVNT_PUB	bad or missing public key
 * XEVNT_VFY	certificate not verified
 */
static int
cert_sign(
	struct exten *ep,	/* extension field pointer */
	struct value *vp	/* value pointer */
	)
{
	X509	*req;		/* X509 certificate request */
	X509	*cert;		/* X509 certificate */
	X509_EXTENSION *ext;	/* certificate extension */
	ASN1_INTEGER *serial;	/* serial number */
	X509_NAME *subj;	/* distinguished (common) name */
	EVP_PKEY *pkey;		/* public key */
	EVP_MD_CTX ctx;		/* message digest context */
	tstamp_t tstamp;	/* NTP timestamp */
	u_int	len;
	u_char	*ptr;
	int	i, temp;

	/*
	 * Decode ASN.1 objects and construct certificate structure.
	 * Make sure the system clock is synchronized to a proventic
	 * source.
	 */
	tstamp = crypto_time();
	if (tstamp == 0)
		return (XEVNT_TSP);

	ptr = (u_char *)ep->pkt;
	if ((req = d2i_X509(NULL, &ptr, ntohl(ep->vallen))) == NULL) {
		msyslog(LOG_ERR, "cert_sign: %s",
		    ERR_error_string(ERR_get_error(), NULL));
		return (XEVNT_CRT);
	}
	/*
	 * Extract public key and check for errors.
	 */
	if ((pkey = X509_get_pubkey(req)) == NULL) {
		msyslog(LOG_ERR, "cert_sign: %s",
		    ERR_error_string(ERR_get_error(), NULL));
		X509_free(req);
		return (XEVNT_PUB);
	}

	/*
	 * Generate X509 certificate signed by this server. If this is a
	 * trusted host, the issuer name is the group name; otherwise,
	 * it is the host name. Also copy any extensions that might be
	 * present.
	 */
	cert = X509_new();
	X509_set_version(cert, X509_get_version(req));
	serial = ASN1_INTEGER_new();
	ASN1_INTEGER_set(serial, tstamp);
	X509_set_serialNumber(cert, serial);
	X509_gmtime_adj(X509_get_notBefore(cert), 0L);
	X509_gmtime_adj(X509_get_notAfter(cert), YEAR);
	subj = X509_get_issuer_name(cert);
	X509_NAME_add_entry_by_txt(subj, "commonName", MBSTRING_ASC,
	    hostval.ptr, strlen(hostval.ptr), -1, 0);
	subj = X509_get_subject_name(req);
	X509_set_subject_name(cert, subj);
	X509_set_pubkey(cert, pkey);
	ext = X509_get_ext(req, 0);
	temp = X509_get_ext_count(req);
	for (i = 0; i < temp; i++) {
		ext = X509_get_ext(req, i);
		X509_add_ext(cert, ext, -1);
	}
	X509_free(req);

	/*
	 * Sign and verify the client certificate, but only if the host
	 * certificate has not expired.
	 */
	if (tstamp < cert_host->first || tstamp > cert_host->last) {
		X509_free(cert);
		return (XEVNT_PER);
	}
	X509_sign(cert, sign_pkey, sign_digest);
	if (!X509_verify(cert, sign_pkey)) {
		msyslog(LOG_ERR, "cert_sign: %s",
		    ERR_error_string(ERR_get_error(), NULL));
		X509_free(cert);
		return (XEVNT_VFY);
	}
	len = i2d_X509(cert, NULL);

	/*
	 * Build and sign the value structure. We have to sign it here,
	 * since the response has to be returned right away. This is a
	 * clogging hazard.
	 */
	memset(vp, 0, sizeof(struct value));
	vp->tstamp = htonl(tstamp);
	vp->fstamp = ep->fstamp;
	vp->vallen = htonl(len);
	vp->ptr = emalloc(len);
	ptr = vp->ptr;
	i2d_X509(cert, &ptr);
	vp->siglen = 0;
	vp->sig = emalloc(sign_siglen);
	EVP_SignInit(&ctx, sign_digest);
	EVP_SignUpdate(&ctx, (u_char *)vp, 12);
	EVP_SignUpdate(&ctx, vp->ptr, len);
	if (EVP_SignFinal(&ctx, vp->sig, &len, sign_pkey))
		vp->siglen = htonl(len);
#ifdef DEBUG
	if (debug > 1)
		X509_print_fp(stdout, cert);
#endif
	X509_free(cert);
	return (XEVNT_OK);
}


/*
 * cert_install - install certificate in certificate cache
 *
 * This routine encodes an extension field into a certificate info/value
 * structure. It searches the certificate list for duplicates and
 * expunges whichever is older. Finally, it inserts this certificate
 * first on the list.
 *
 * Returns certificate info pointer if valid, NULL if not.
 */
struct cert_info *
cert_install(
	struct exten *ep,	/* cert info/value */
	struct peer *peer	/* peer structure */
	)
{
	struct cert_info *cp, *xp, *yp, **zp;

	/*
	 * Parse and validate the signed certificate. If valid,
	 * construct the info/value structure; otherwise, scamper home
	 * empty handed.
	 */
	if ((cp = cert_parse((u_char *)ep->pkt, (long)ntohl(ep->vallen),
	    (tstamp_t)ntohl(ep->fstamp))) == NULL)
		return (NULL);

	/*
	 * Scan certificate list looking for another certificate with
	 * the same subject and issuer. If another is found with the
	 * same or older filestamp, unlink it and return the goodies to
	 * the heap. If another is found with a later filestamp, discard
	 * the new one and leave the building with the old one.
	 *
	 * Make a note to study this issue again. An earlier certificate
	 * with a long lifetime might be overtaken by a later
	 * certificate with a short lifetime, thus invalidating the
	 * earlier signature. However, we gotta find a way to leak old
	 * stuff from the cache, so we do it anyway. 
	 */
	zp = &cinfo;
	for (xp = cinfo; xp != NULL; xp = xp->link) {
		if (strcmp(cp->subject, xp->subject) == 0 &&
		    strcmp(cp->issuer, xp->issuer) == 0) {
			if (ntohl(cp->cert.fstamp) <=
			    ntohl(xp->cert.fstamp)) {
				cert_free(cp);
				cp = xp;
				break;
			}
			yp = xp;
			xp = xp->link;
			*zp = xp;
			cert_free(yp);
		} else {
			zp = &xp->link;
		}
	}
	if (xp == NULL) {
		cp->link = cinfo;
		cinfo = cp;
	}
	cp->flags |= CERT_VALID;
	crypto_update();
	return (cp);
}


/*
 * cert_hike - verify the signature using the issuer public key
 *
 * Returns
 * XEVNT_OK	success
 * XEVNT_CRT	bad or missing certificate
 * XEVNT_PER	host certificate expired
 * XEVNT_VFY	certificate not verified
 */
int
cert_hike(
	struct peer *peer,	/* peer structure pointer */
	struct cert_info *yp	/* issuer certificate */
	)
{
	struct cert_info *xp;	/* subject certificate */
	X509	*cert;		/* X509 certificate */
	u_char	*ptr;

	/*
	 * Save the issuer on the new certificate, but remember the old
	 * one.
	 */
	if (peer->issuer != NULL)
		free(peer->issuer);
	peer->issuer = emalloc(strlen(yp->issuer) + 1);
	strcpy(peer->issuer, yp->issuer);
	xp = peer->xinfo;
	peer->xinfo = yp;

	/*
	 * If subject Y matches issuer Y, then the certificate trail is
	 * complete. If Y is not trusted, the server certificate has yet
	 * been signed, so keep trying. Otherwise, save the group key
	 * and light the valid bit. If the host certificate is trusted,
	 * do not execute a sign exchange. If no identity scheme is in
	 * use, light the identity and proventic bits.
	 */
	if (strcmp(yp->subject, yp->issuer) == 0) {
		if (!(yp->flags & CERT_TRUST))
			return (XEVNT_OK);

		peer->grpkey = yp->grpkey;
		peer->crypto |= CRYPTO_FLAG_CERT;
		if (!(peer->crypto & CRYPTO_FLAG_MASK))
			peer->crypto |= CRYPTO_FLAG_VRFY |
			    CRYPTO_FLAG_PROV;

		/*
		 * If the server has an an identity scheme, fetch the
		 * identity credentials. If not, the identity is
		 * verified only by the trusted certificate. The next
		 * signature will set the server proventic.
		 */
		if (!(peer->crypto & CRYPTO_FLAG_MASK) ||
		    sys_groupname == NULL)
			peer->crypto |= CRYPTO_FLAG_VRFY;
	}

	/*
	 * If X exists, verify signature X using public key Y.
	 */
	if (xp == NULL)
		return (XEVNT_OK);

	ptr = (u_char *)xp->cert.ptr;
	cert = d2i_X509(NULL, &ptr, ntohl(xp->cert.vallen));
	if (cert == NULL) {
		xp->flags |= CERT_ERROR;
		return (XEVNT_CRT);
	}
	if (!X509_verify(cert, yp->pkey)) {
		X509_free(cert);
		xp->flags |= CERT_ERROR;
		return (XEVNT_VFY);
	}
	X509_free(cert);

	/*
	 * Signature X is valid only if it begins during the
	 * lifetime of Y. 
	 */
	if (xp->first < yp->first || xp->first > yp->last) {
		xp->flags |= CERT_ERROR;
		return (XEVNT_PER);
	}
	xp->flags |= CERT_SIGN;
	return (XEVNT_OK);
}


/*
 * cert_parse - parse x509 certificate and create info/value structures.
 *
 * The server certificate includes the version number, issuer name,
 * subject name, public key and valid date interval. If the issuer name
 * is the same as the subject name, the certificate is self signed and
 * valid only if the server is configured as trustable. If the names are
 * different, another issuer has signed the server certificate and
 * vouched for it. In this case the server certificate is valid if
 * verified by the issuer public key.
 *
 * Returns certificate info/value pointer if valid, NULL if not.
 */
struct cert_info *		/* certificate information structure */
cert_parse(
	u_char	*asn1cert,	/* X509 certificate */
	long	len,		/* certificate length */
	tstamp_t fstamp		/* filestamp */
	)
{
	X509	*cert;		/* X509 certificate */
	X509_EXTENSION *ext;	/* X509v3 extension */
	struct cert_info *ret;	/* certificate info/value */
	BIO	*bp;
	char	pathbuf[MAXFILENAME];
	u_char	*ptr;
	int	temp, cnt, i;

	/*
	 * Decode ASN.1 objects and construct certificate structure.
	 */
	ptr = asn1cert;
	if ((cert = d2i_X509(NULL, &ptr, len)) == NULL) {
		msyslog(LOG_ERR, "cert_parse: %s",
		    ERR_error_string(ERR_get_error(), NULL));
		return (NULL);
	}
#ifdef DEBUG
	if (debug > 1)
		X509_print_fp(stdout, cert);
#endif

	/*
	 * Extract version, subject name and public key.
	 */
	ret = emalloc(sizeof(struct cert_info));
	memset(ret, 0, sizeof(struct cert_info));
	if ((ret->pkey = X509_get_pubkey(cert)) == NULL) {
		msyslog(LOG_ERR, "cert_parse: %s",
		    ERR_error_string(ERR_get_error(), NULL));
		cert_free(ret);
		X509_free(cert);
		return (NULL);
	}
	ret->version = X509_get_version(cert);
	X509_NAME_oneline(X509_get_subject_name(cert), pathbuf,
	    MAXFILENAME);
	ptr = strstr(pathbuf, "CN=");
	if (ptr == NULL) {
		msyslog(LOG_NOTICE, "cert_parse: invalid subject %s",
		    pathbuf);
		cert_free(ret);
		X509_free(cert);
		return (NULL);
	}
	ret->subject = emalloc(strlen(ptr) + 1);
	strcpy(ret->subject, ptr + 3);

	/*
	 * Extract remaining objects. Note that the NTP serial number is
	 * the NTP seconds at the time of signing, but this might not be
	 * the case for other authority. We don't bother to check the
	 * objects at this time, since the real crunch can happen only
	 * when the time is valid but not yet certificated.
	 */
	ret->nid = OBJ_obj2nid(cert->cert_info->signature->algorithm);
	ret->digest = (const EVP_MD *)EVP_get_digestbynid(ret->nid);
	ret->serial =
	    (u_long)ASN1_INTEGER_get(X509_get_serialNumber(cert));
	X509_NAME_oneline(X509_get_issuer_name(cert), pathbuf,
	    MAXFILENAME);
	if ((ptr = strstr(pathbuf, "CN=")) == NULL) {
		msyslog(LOG_NOTICE, "cert_parse: invalid issuer %s",
		    pathbuf);
		cert_free(ret);
		X509_free(cert);
		return (NULL);
	}
	ret->issuer = emalloc(strlen(ptr) + 1);
	strcpy(ret->issuer, ptr + 3);
	ret->first = asn2ntp(X509_get_notBefore(cert));
	ret->last = asn2ntp(X509_get_notAfter(cert));

	/*
	 * Extract extension fields. These are ad hoc ripoffs of
	 * currently assigned functions and will certainly be changed
	 * before prime time.
	 */
	cnt = X509_get_ext_count(cert);
	for (i = 0; i < cnt; i++) {
		ext = X509_get_ext(cert, i);
		temp = OBJ_obj2nid(ext->object);
		switch (temp) {

		/*
		 * If a key_usage field is present, we decode whether
		 * this is a trusted or private certificate. This is
		 * dorky; all we want is to compare NIDs, but OpenSSL
		 * insists on BIO text strings.
		 */
		case NID_ext_key_usage:
			bp = BIO_new(BIO_s_mem());
			X509V3_EXT_print(bp, ext, 0, 0);
			BIO_gets(bp, pathbuf, MAXFILENAME);
			BIO_free(bp);
			if (strcmp(pathbuf, "Trust Root") == 0)
				ret->flags |= CERT_TRUST;
			else if (strcmp(pathbuf, "Private") == 0)
				ret->flags |= CERT_PRIV;
#if DEBUG
			if (debug)
				printf("cert_parse: %s: %s\n",
				    OBJ_nid2ln(temp), pathbuf);
#endif
			break;

		/*
		 * If a NID_subject_key_identifier field is present, it
		 * contains the GQ public key.
		 */
		case NID_subject_key_identifier:
			ret->grpkey = BN_bin2bn(&ext->value->data[2],
			    ext->value->length - 2, NULL);
			/* fall through */
#if DEBUG
		default:
			if (debug)
				printf("cert_parse: %s\n",
				    OBJ_nid2ln(temp));
#endif
		}
	}
	if (strcmp(ret->subject, ret->issuer) == 0) {

		/*
		 * If certificate is self signed, verify signature.
		 */
		if (!X509_verify(cert, ret->pkey)) {
			msyslog(LOG_NOTICE,
			    "cert_parse: signature not verified %s",
			    ret->subject);
			cert_free(ret);
			X509_free(cert);
			return (NULL);
		}
	} else {

		/*
		 * Check for a certificate loop.
		 */
		if (strcmp(hostval.ptr, ret->issuer) == 0) {
			msyslog(LOG_NOTICE,
			    "cert_parse: certificate trail loop %s",
			    ret->subject);
			cert_free(ret);
			X509_free(cert);
			return (NULL);
		}
	}

	/*
	 * Verify certificate valid times. Note that certificates cannot
	 * be retroactive.
	 */
	if (ret->first > ret->last || ret->first < fstamp) {
		msyslog(LOG_NOTICE,
		    "cert_parse: invalid times %s first %u last %u fstamp %u",
		    ret->subject, ret->first, ret->last, fstamp);
		cert_free(ret);
		X509_free(cert);
		return (NULL);
	}

	/*
	 * Build the value structure to sign and send later.
	 */
	ret->cert.fstamp = htonl(fstamp);
	ret->cert.vallen = htonl(len);
	ret->cert.ptr = emalloc(len);
	memcpy(ret->cert.ptr, asn1cert, len);
	X509_free(cert);
	return (ret);
}


/*
 * cert_free - free certificate information structure
 */
void
cert_free(
	struct cert_info *cinf	/* certificate info/value structure */ 
	)
{
	if (cinf->pkey != NULL)
		EVP_PKEY_free(cinf->pkey);
	if (cinf->subject != NULL)
		free(cinf->subject);
	if (cinf->issuer != NULL)
		free(cinf->issuer);
	if (cinf->grpkey != NULL)
		BN_free(cinf->grpkey);
	value_free(&cinf->cert);
	free(cinf);
}


/*
 * crypto_key - load cryptographic parameters and keys
 *
 * This routine searches the key cache for matching name in the form
 * ntpkey_<key>_<name>, where <key> is one of host, sign, iff, gq, mv,
 * and <name> is the host/group name. If not found, it tries to load a
 * PEM-encoded file of the same name and extracts the filestamp from
 * the first line of the file name. It returns the key pointer if valid,
 * NULL if not.
 */
static struct pkey_info *
crypto_key(
	char	*cp,		/* file name */
	char	*passwd1,	/* password */
	struct sockaddr_storage *addr /* IP address */
	)
{
	FILE	*str;		/* file handle */
	struct pkey_info *pkp;	/* generic key */
	EVP_PKEY *pkey = NULL;	/* public/private key */
	tstamp_t fstamp;
	char	filename[MAXFILENAME]; /* name of key file */
	char	linkname[MAXFILENAME]; /* filestamp buffer) */
	char	statstr[NTP_MAXSTRLEN]; /* statistics for filegen */
	char	*ptr;

	/*
	 * Search the key cache for matching key and name.
	 */
	for (pkp = pkinfo; pkp != NULL; pkp = pkp->link) {
		if (strcmp(cp, pkp->name) == 0)
			return (pkp);
	}  

	/*
	 * Open the key file. If the first character of the file name is
	 * not '/', prepend the keys directory string. If something goes
	 * wrong, abandon ship.
	 */
	if (*cp == '/')
		strcpy(filename, cp);
	else
		snprintf(filename, MAXFILENAME, "%s/%s", keysdir, cp);
	str = fopen(filename, "r");
	if (str == NULL)
		return (NULL);

	/*
	 * Read the filestamp, which is contained in the first line.
	 */
	if ((ptr = fgets(linkname, MAXFILENAME, str)) == NULL) {
		msyslog(LOG_ERR, "crypto_key: empty file %s",
		    filename);
		fclose(str);
		return (NULL);
	}
	if ((ptr = strrchr(ptr, '.')) == NULL) {
		msyslog(LOG_ERR, "crypto_key: no filestamp %s",
		    filename);
		fclose(str);
		return (NULL);
	}
	if (sscanf(++ptr, "%u", &fstamp) != 1) {
		msyslog(LOG_ERR, "crypto_key: invalid filestamp %s",
		    filename);
		fclose(str);
		return (NULL);
	}

	/*
	 * Read and decrypt PEM-encoded private key. If it fails to
	 * decrypt, game over.
	 */
	pkey = PEM_read_PrivateKey(str, NULL, NULL, passwd1);
	fclose(str);
	if (pkey == NULL) {
		msyslog(LOG_ERR, "crypto_key: %s",
		    ERR_error_string(ERR_get_error(), NULL));
		exit (-1);
	}

	/*
	 * Make a new entry in the key cache.
	 */
	pkp = emalloc(sizeof(struct pkey_info));
	pkp->link = pkinfo;
	pkinfo = pkp;
	pkp->pkey = pkey;
	pkp->name = emalloc(strlen(cp) + 1);
	pkp->fstamp = fstamp;
	strcpy(pkp->name, cp);

	/*
	 * Leave tracks in the cryptostats.
	 */
	if ((ptr = strrchr(linkname, '\n')) != NULL)
		*ptr = '\0'; 
	snprintf(statstr, NTP_MAXSTRLEN, "%s mod %d", &linkname[2],
	    EVP_PKEY_size(pkey) * 8);
	record_crypto_stats(addr, statstr);
#ifdef DEBUG
	if (debug)
		printf("crypto_key: %s\n", statstr);
	if (debug > 1) {
		if (pkey->type == EVP_PKEY_DSA)
			DSA_print_fp(stdout, pkey->pkey.dsa, 0);
		else if (pkey->type == EVP_PKEY_RSA)
			RSA_print_fp(stdout, pkey->pkey.rsa, 0);
	}
#endif
	return (pkp);
}


/*
 ***********************************************************************
 *								       *
 * The following routines are used only at initialization time         *
 *								       *
 ***********************************************************************
 */
/*
 * crypto_cert - load certificate from file
 *
 * This routine loads an X.509 RSA or DSA certificate from a file and
 * constructs a info/cert value structure for this machine. The
 * structure includes a filestamp extracted from the file name. Later
 * the certificate can be sent to another machine on request.
 *
 * Returns certificate info/value pointer if valid, NULL if not.
 */
static struct cert_info *	/* certificate information */
crypto_cert(
	char	*cp		/* file name */
	)
{
	struct cert_info *ret; /* certificate information */
	FILE	*str;		/* file handle */
	char	filename[MAXFILENAME]; /* name of certificate file */
	char	linkname[MAXFILENAME]; /* filestamp buffer */
	char	statstr[NTP_MAXSTRLEN]; /* statistics for filegen */
	tstamp_t fstamp;	/* filestamp */
	long	len;
	char	*ptr;
	char	*name, *header;
	u_char	*data;

	/*
	 * Open the certificate file. If the first character of the file
	 * name is not '/', prepend the keys directory string. If
	 * something goes wrong, abandon ship.
	 */
	if (*cp == '/')
		strcpy(filename, cp);
	else
		snprintf(filename, MAXFILENAME, "%s/%s", keysdir, cp);
	str = fopen(filename, "r");
	if (str == NULL)
		return (NULL);

	/*
	 * Read the filestamp, which is contained in the first line.
	 */
	if ((ptr = fgets(linkname, MAXFILENAME, str)) == NULL) {
		msyslog(LOG_ERR, "crypto_cert: empty file %s",
		    filename);
		fclose(str);
		return (NULL);
	}
	if ((ptr = strrchr(ptr, '.')) == NULL) {
		msyslog(LOG_ERR, "crypto_cert: no filestamp %s\n",
		    filename);
		fclose(str);
		return (NULL);
	}
	if (sscanf(++ptr, "%u", &fstamp) != 1) {
		msyslog(LOG_ERR, "crypto_cert: invalid filestamp %s\n",
		    filename);
		fclose(str);
		return (NULL);
	}

	/*
	 * Read PEM-encoded certificate and install.
	 */
	if (!PEM_read(str, &name, &header, &data, &len)) {
		msyslog(LOG_ERR, "crypto_cert: %s\n",
		    ERR_error_string(ERR_get_error(), NULL));
		fclose(str);
		return (NULL);
	}
	fclose(str);
	free(header);
	if (strcmp(name, "CERTIFICATE") != 0) {
		msyslog(LOG_NOTICE, "crypto_cert: wrong PEM type %s",
		    name);
		free(name);
		free(data);
		return (NULL);
	}
	free(name);

	/*
	 * Parse certificate and generate info/value structure. The
	 * pointer and copy nonsense is due something broken in Solaris.
	 */
	ret = cert_parse(data, len, fstamp);
	free(data);
	if (ret == NULL)
		return (NULL);

	if ((ptr = strrchr(linkname, '\n')) != NULL)
		*ptr = '\0'; 
	snprintf(statstr, NTP_MAXSTRLEN, "%s 0x%x len %lu",
	    &linkname[2], ret->flags, len);
	record_crypto_stats(NULL, statstr);
#ifdef DEBUG
	if (debug)
		printf("crypto_cert: %s\n", statstr);
#endif
	return (ret);
}


/*
 * crypto_setup - load keys, certificate and identity parameters
 *
 * This routine loads the public/private host key and certificate. If
 * available, it loads the public/private sign key, which defaults to
 * the host key. The host key must be RSA, but the sign key can be
 * either RSA or DSA. If a trusted certificate, it loads the identity
 * parameters. In either case, the public key on the certificate must
 * agree with the sign key.
 *
 * Required but missing files and inconsistent data and errors are
 * fatal. Allowing configuration to continue would be hazardous and
 * require really messy error checks.
 */
void
crypto_setup(void)
{
	struct pkey_info *pinfo; /* private/public key */
	char	filename[MAXFILENAME]; /* file name buffer */
	char	statstr[NTP_MAXSTRLEN]; /* statistics for filegen */
	l_fp	seed;		/* crypto PRNG seed as NTP timestamp */
	u_int	len;
	int	bytes;
	u_char	*ptr;

	/*
	 * Check for correct OpenSSL version and avoid initialization in
	 * the case of multiple crypto commands.
	 */
	if (crypto_flags & CRYPTO_FLAG_ENAB) {
		msyslog(LOG_NOTICE,
		    "crypto_setup: spurious crypto command");
		return;
	}
	if ((SSLeay() ^ OPENSSL_VERSION_NUMBER) & ~0xff0L) {
		msyslog(LOG_ERR,
		    "crypto_setup: OpenSSL version mismatch. Built against %lx, you have %lx",
		    OPENSSL_VERSION_NUMBER, SSLeay());
                exit (-1);
        }
	ERR_load_crypto_strings();
	OpenSSL_add_all_algorithms();

	/*
	 * Load required random seed file and seed the random number
	 * generator. Be default, it is found as .rnd in the user home
	 * directory. The root home directory may be / or /root,
	 * depending on the system. Wiggle the contents a bit and write
	 * it back so the sequence does not repeat when we next restart.
	 */
	if (!RAND_status()) {
		if (rand_file == NULL)
			RAND_file_name(filename, MAXFILENAME);
		else if (*rand_file == '/')
			strcpy(filename, rand_file);
		else
			snprintf(filename, MAXFILENAME, "%s/%s",
			    keysdir, rand_file);
		if (filename == NULL) {
			msyslog(LOG_ERR,
			    "crypto_setup: seed file unknown name");
			exit (-1);
		}
		if ((bytes = RAND_load_file(filename, -1)) == 0) {
			msyslog(LOG_ERR,
			    "crypto_setup: random seed file %s missing",
			    filename);
			exit (-1);
		}
		get_systime(&seed);
		RAND_seed(&seed, sizeof(l_fp));
		RAND_write_file(filename);
#ifdef DEBUG
		if (debug)
			printf(
			    "crypto_setup: OpenSSL version %lx random seed file %s bytes read %d\n",
			    SSLeay(), filename, bytes);
#endif
	}

	/*
	 * Initialize structures.
	 */
	if (host_file != NULL) {
		sys_hostname = host_file;
	} else {
		gethostname(filename, MAXFILENAME);
		sys_hostname = emalloc(strlen(filename) + 1);
		strcpy(sys_hostname, filename);
	}
	if (passwd == NULL)
		passwd = sys_hostname;
	memset(&hostval, 0, sizeof(hostval));
	memset(&pubkey, 0, sizeof(pubkey));
	memset(&tai_leap, 0, sizeof(tai_leap));

	/*
	 * Load required host key from file "ntpkey_host_<hostname>". If
	 * no host key file is not found or has invalid password, life
	 * as we know it ends. The host key also becomes the default
	 * sign key. 
	 */
	snprintf(filename, MAXFILENAME, "ntpkey_host_%s", sys_hostname);
	pinfo = crypto_key(filename, passwd, NULL);
	if (pinfo == NULL) {
		msyslog(LOG_ERR,
		    "crypto_setup: host key file %s not found or corrupt",
		    filename);
		exit (-1);
	}
	if (pinfo->pkey->type != EVP_PKEY_RSA) {
		msyslog(LOG_ERR,
		    "crypto_setup: host key is not RSA key type");
		exit (-1);
	}
	host_pkey = pinfo->pkey;
	sign_pkey = host_pkey;
	hostval.fstamp = htonl(pinfo->fstamp);
	
	/*
	 * Construct public key extension field for agreement scheme.
	 */
	len = i2d_PublicKey(host_pkey, NULL);
	ptr = emalloc(len);
	pubkey.ptr = ptr;
	i2d_PublicKey(host_pkey, &ptr);
	pubkey.fstamp = hostval.fstamp;
	pubkey.vallen = htonl(len);

	/*
	 * Load optional sign key from file "ntpkey_sign_<hostname>". If
	 * available, it becomes the sign key.
	 */
	if (sign_file != NULL) {
		snprintf(filename, MAXFILENAME, "ntpkey_sign_%s",
		    sign_file);
		pinfo = crypto_key(filename, passwd, NULL);
		if (pinfo != NULL)
		 	sign_pkey = pinfo->pkey;
	}
	sign_siglen = EVP_PKEY_size(sign_pkey);

	/*
	 * Load required certificate from file "ntpkey_cert_<hostname>".
	 */
	snprintf(filename, MAXFILENAME, "ntpkey_cert_%s", sys_hostname);
	cinfo = crypto_cert(filename);
	if (cinfo == NULL) {
		msyslog(LOG_ERR,
		    "crypto_setup: certificate file %s not found or corrupt",
		    filename);
		exit (-1);
	}
	cert_host = cinfo;
	sign_digest = cinfo->digest;
	if (cinfo->flags & CERT_PRIV)
		crypto_flags |= CRYPTO_FLAG_PRIV;

	/*
	 * The certificate must be self-signed.
	 */
	if (strcmp(cinfo->subject, cinfo->issuer) != 0) {
		msyslog(LOG_ERR,
		    "crypto_setup: certificate %s is not self-signed",
		    filename);
		exit (-1);
	}
	hostval.vallen = htonl(strlen(cinfo->subject));
	hostval.ptr = cinfo->subject;

	/*
	 * If trusted certificate, the subject name must match the group
	 * name.
	 */
	if (cinfo->flags & CERT_TRUST) {
		if (sys_groupname == NULL) {
			sys_groupname = hostval.ptr;
		} else if (strcmp(hostval.ptr, sys_groupname) != 0) {
			msyslog(LOG_ERR,
			    "crypto_setup: trusted certificate name %s does not match group name %s",
			    hostval.ptr, sys_groupname);
			exit (-1);
		}
	}
	if (sys_groupname != NULL) {

		/*
		 * Load optional IFF parameters from file
		 * "ntpkey_iffkey_<groupname>".
		 */
		snprintf(filename, MAXFILENAME, "ntpkey_iffkey_%s",
		    sys_groupname);
		iffkey_info = crypto_key(filename, passwd, NULL);
		if (iffkey_info != NULL)
			crypto_flags |= CRYPTO_FLAG_IFF;

		/*
		 * Load optional GQ parameters from file
		 * "ntpkey_gqkey_<groupname>".
		 */
		snprintf(filename, MAXFILENAME, "ntpkey_gqkey_%s",
		    sys_groupname);
		gqkey_info = crypto_key(filename, passwd, NULL);
		if (gqkey_info != NULL)
			crypto_flags |= CRYPTO_FLAG_GQ;

		/*
		 * Load optional MV parameters from file
		 * "ntpkey_mvkey_<groupname>".
		 */
		snprintf(filename, MAXFILENAME, "ntpkey_mvkey_%s",
		    sys_groupname);
		mvkey_info = crypto_key(filename, passwd, NULL);
		if (mvkey_info != NULL)
			crypto_flags |= CRYPTO_FLAG_MV;
	}

	/*
	 * We met the enemy and he is us. Now strike up the dance.
	 */
	crypto_flags |= CRYPTO_FLAG_ENAB | (cinfo->nid << 16);
	snprintf(statstr, NTP_MAXSTRLEN,
	    "setup 0x%x host %s %s\n", crypto_flags, sys_hostname,
	    OBJ_nid2ln(cinfo->nid));
	record_crypto_stats(NULL, statstr);
#ifdef DEBUG
	if (debug)
		printf("crypto_setup: %s\n", statstr);
#endif
}


/*
 * crypto_config - configure data from the crypto command.
 */
void
crypto_config(
	int	item,		/* configuration item */
	char	*cp		/* item name */
	)
{

#ifdef DEBUG
	if (debug > 1)
		printf("crypto_config: item %d %s\n", item, cp);
#endif
	switch (item) {

	/*
	 * Set host name (host).
	 */
	case CRYPTO_CONF_PRIV:
		host_file = emalloc(strlen(cp) + 1);
		strcpy(host_file, cp);
		break;

	/*
	 * Set group name (ident).
	 */
	case CRYPTO_CONF_IDENT:
		sys_groupname = emalloc(strlen(cp) + 1);
		strcpy(sys_groupname, cp);
		break;

	/*
	 * Set private key password (pw).
	 */
	case CRYPTO_CONF_PW:
		passwd = emalloc(strlen(cp) + 1);
		strcpy(passwd, cp);
		break;

	/*
	 * Set random seed file name (randfile).
	 */
	case CRYPTO_CONF_RAND:
		rand_file = emalloc(strlen(cp) + 1);
		strcpy(rand_file, cp);
		break;

	/*
	 * Set sign key file name (sign).
	 */
	case CRYPTO_CONF_SIGN:
		sign_file = emalloc(strlen(cp) + 1);
		strcpy(sign_file, cp);
		break;
	}
}
# else
int ntp_crypto_bs_pubkey;
# endif /* OPENSSL */
