/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef INCLUDED_GR_BLOCK_H
#define INCLUDED_GR_BLOCK_H

#include <gr_runtime.h>
#include <string>

/*!
 * \brief The abstract base class for all signal processing blocks.
 * \ingroup block
 *
 * Blocks have a set of input streams and output streams.  The
 * input_signature and output_signature define the number of input
 * streams and output streams respectively, and the type of the data
 * items in each stream.
 *
 * Although blocks may consume data on each input stream at a
 * different rate, all outputs streams must produce data at the same
 * rate.  That rate may be different from any of the input rates.
 *
 * User derived blocks override two methods, forecast and general_work,
 * to implement their signal processing behavior. forecast is called
 * by the system scheduler to determine how many items are required on
 * each input stream in order to produce a given number of output
 * items.
 *
 * general_work is called to perform the signal processing in the block.
 * It reads the input items and writes the output items.
 */

class gr_block {

 public:
  
  virtual ~gr_block ();
  
  std::string name () const { return d_name; }
  gr_io_signature_sptr input_signature () const  { return d_input_signature; }
  gr_io_signature_sptr output_signature () const { return d_output_signature; }
  long unique_id () const { return d_unique_id; }

  // ----------------------------------------------------------------
  //		override these to define your behavior
  // ----------------------------------------------------------------

  /*!
   * \brief  Estimate input requirements given output request
   *
   * Given a request to product noutput_items, estimate the number of
   * data items required on each input stream.  The estimate doesn't have
   * to be exact, but should be close.
   */
  virtual void forecast (int noutput_items, gr_vector_int &ninput_items_required);

  /*!
   * \brief compute output items from input items
   *
   * \param noutput_items	number of output items to write on each output stream
   * \param ninput_items	number of input items available on each input stream
   * \param input_items		vector of pointers to the input items, one entry per input stream
   * \param output_items	vector of pointers to the output items, one entry per output stream
   *
   * \returns number of items actually written to each output stream, or -1 on EOF.
   * It is OK to return a value less than noutput_items.  -1 <= return value <= noutput_items
   *
   * general_work must call consume or consume_each to indicate how many items
   * were consumed on each input stream.
   */
  virtual int general_work (int noutput_items,
			    gr_vector_int &ninput_items,
			    gr_vector_const_void_star &input_items,
			    gr_vector_void_star &output_items) = 0;

  /*!
   * \brief Confirm that ninputs and noutputs is an acceptable combination.
   *
   * \param ninputs	number of input streams connected
   * \param noutputs	number of output streams connected
   *
   * \returns true if this is a valid configuration for this block.
   *
   * This function is called by the runtime system whenever the
   * topology changes.  Most classes do not need to override this.
   * This check is in addition to the constraints specified by the input
   * and output gr_io_signatures.
   */
  virtual bool check_topology (int ninputs, int noutputs);

  // ----------------------------------------------------------------

  /*!
   * \brief Constrain the noutput_items argument passed to forecast and general_work
   *
   * set_output_multiple causes the scheduler to ensure that the noutput_items
   * argument passed to forecast and general_work will be an integer multiple
   * of \param multiple  The default value of output multiple is 1.
   */
  void set_output_multiple (int multiple);
  int  output_multiple () const { return d_output_multiple; }

  /*!
   * \brief Tell the scheduler \p how_many_items of input stream \p which_input were consumed.
   */
  void consume (int which_input, int how_many_items);

  /*!
   * \brief Tell the scheduler \p how_many_items were consumed on each input stream.
   */
  void consume_each (int how_many_items);

  /*!
   * \brief Set the approximate output rate / input rate
   *
   * Provide a hint to the buffer allocator and scheduler.
   * The default relative_rate is 1.0
   *
   * decimators have relative_rates < 1.0
   * interpolators have relative_rates > 1.0
   */
  void  set_relative_rate (double relative_rate);

  /*!
   * \brief return the approximate output rate / input rate
   */
  double relative_rate () const	{ return d_relative_rate; }


  // ----------------------------------------------------------------------------

 private:

  std::string		d_name;
  gr_io_signature_sptr	d_input_signature;
  gr_io_signature_sptr	d_output_signature;
  int			d_output_multiple;
  double		d_relative_rate;	// approx output_rate / input_rate
  gr_block_detail_sptr	d_detail;		// implementation details
  long			d_unique_id;		// convenient for debugging
  
 protected:

  gr_block (const std::string &name,
	     gr_io_signature_sptr input_signature,
	     gr_io_signature_sptr output_signature);

  // These are really only for internal use, but leaving them public avoids
  // having to work up an ever-varying list of friends

 public:
  gr_block_detail_sptr detail () const { return d_detail; }
  void set_detail (gr_block_detail_sptr detail) { d_detail = detail; }
};

long gr_block_ncurrently_allocated ();

#endif /* INCLUDED_GR_BLOCK_H */
