#!/usr/bin/env python
#
# Copyright 2004 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
# 

from gnuradio import gr
from gnuradio import usrp
from gnuradio import eng_notation
from gnuradio.eng_option import eng_option
from gnuradio.wxgui import stdgui, fftsink, scopesink, slider
from gnuradio import tv_rx
from optparse import OptionParser
import wx

class app_flow_graph (stdgui.gui_flow_graph):
    def __init__(self, frame, panel, vbox, argv):
        stdgui.gui_flow_graph.__init__ (self, frame, panel, vbox, argv)

        self.frame = frame
        self.panel = panel
        
        parser = OptionParser (option_class=eng_option)
        parser.add_option ("-d", "--decim", type="int", default=16,
                           help="set fgpa decimation rate to DECIM")
        parser.add_option ("-c", "--ddc-freq", type="eng_float", default=-5.75e6,
                           help="set Rx DDC frequency to FREQ", metavar="FREQ")
        parser.add_option ("-f", "--freq", type="eng_float", default=97.7e6,
                           help="set DBoard downconverter frequency to FREQ", metavar="FREQ")
        parser.add_option ("-m", "--mux", type="intx", default=0xf0f0f0f0,
                           help="set fpga FR_RX_MUX register to MUX")
        parser.add_option ("-g", "--gain", type="eng_float", default=0,
                           help="set Rx PGA gain in dB (default 0 dB)")
        (options, args) = parser.parse_args ()

        self.u = usrp.source_c (0, options.decim, 1, options.mux, 0)
        self.u.set_rx_freq (0, options.ddc_freq)

        self.which_side = 0;        # 0 -> RX_A; 1 -> RX_B

        dboard = tv_rx.tv_rx (self.u, self.which_side)
        self.dboard = dboard
        
        RF_freq = options.freq
        print "RF freq ", RF_freq
        print dboard.set_freq(RF_freq)
        
        self.u.set_verbose (0)
        
        input_rate = self.u.adc_freq () / self.u.decim_rate ()

        block, fft_win = fftsink.make_fft_sink_c (self, panel, "", 1024, input_rate,0,100)
        self.connect (self.u, block)
        vbox.Add (fft_win, 10, wx.EXPAND)

        # c2f_1 = gr.complex_to_float ()

        #scope, scopewin = scopesink.make_scope_sink_f (self, panel, "Rx Data", input_rate)
        #vbox.Add (scopewin, 8, wx.EXPAND)

        # self.connect (self.u,c2f_1)
            
        #self.connect ((c2f_1, 0), (scope, 0))
        #self.connect ((c2f_1, 1), (scope, 1))

        if 0:
            rms_complex = gr.rms_cf(.0001)
            rms_i = gr.rms_ff(.0001)
            rms_q = gr.rms_ff(.0001)

            self.connect(self.u,rms_complex)
            self.connect((c2f_1,0),rms_i)
            self.connect((c2f_1,1),rms_q)

            ns1 = gr.null_sink(4)
            ns2 = gr.null_sink(4)
            ns3 = gr.null_sink(4)

            self.connect(rms_complex,ns1)
            self.connect(rms_i,ns2)
            self.connect(rms_q,ns3)

        # sliders

        vbox.Add(slider.slider(panel, 0, 115, self.set_gain), 1, wx.ALIGN_CENTER)

        #vbox.Add(slider.slider(panel, 0, 61, self.set_rfagc), 1, wx.ALIGN_CENTER)
        #vbox.Add(slider.slider(panel, 0, 36, self.set_ifagc), 1, wx.ALIGN_CENTER)
        #vbox.Add(slider.slider(panel, 0, 20, self.set_pga), 1, wx.ALIGN_CENTER)

        vbox.Add(slider.slider(panel, -200, 200, self.set_offset_i), 1, wx.ALIGN_CENTER)
        #vbox.Add(slider.slider(panel, -200, 200, self.set_offset_q), 1, wx.ALIGN_CENTER)

        # build small control area at bottom
        hbox = wx.BoxSizer (wx.HORIZONTAL)
        hbox.Add ((1, 1), 1, wx.EXPAND)
        hbox.Add (wx.StaticText (panel, -1, "Set ddc freq: "), 0, wx.ALIGN_CENTER)
        self.tc_freq = wx.TextCtrl (panel, -1, "", style=wx.TE_PROCESS_ENTER)
        hbox.Add (self.tc_freq, 0, wx.ALIGN_CENTER)
        wx.EVT_TEXT_ENTER (self.tc_freq, self.tc_freq.GetId(), self.handle_text_enter)
        hbox.Add ((1, 1), 1, wx.EXPAND)
        # add it to the main vbox
        vbox.Add (hbox, 0, wx.EXPAND)

        self.update_status_bar ()

    def set_gain (self,gain):
        self.dboard.set_gain(gain)

    def set_rfagc (self,gain):
        self.dboard._set_rfagc(gain)
    def set_ifagc (self,gain):
        self.dboard._set_ifagc(gain)

    def set_pga (self,gain):
        self.dboard._set_pga(gain)

    def set_offset_i(self, offset):
        self.u.set_adc_offset(self.which_side*2+0, offset)

    def set_offset_q(self,offset):
        self.u.set_adc_offset(self.which_side*2+1, offset)

    def handle_text_enter (self, event):
        str = event.GetString ()
        self.tc_freq.Clear ()
        self.u.set_rx_freq (0, eng_notation.str_to_num (str))
        self.update_status_bar ()

    def update_status_bar (self):
        ddc_freq = self.u.rx_freq (0)
        decim_rate = self.u.decim_rate ()
        sample_rate = self.u.adc_freq () / decim_rate
        msg = "decim: %d  %sS/s  DDC: %s" % (
            decim_rate,
            eng_notation.num_to_str (sample_rate),
            eng_notation.num_to_str (ddc_freq))
            
        self.frame.GetStatusBar().SetStatusText (msg, 1)

        

def main ():
    app = stdgui.stdapp (app_flow_graph, "USRP FFT")
    app.MainLoop ()

if __name__ == '__main__':
    main ()


