;;; GNU Guix --- Functional package management for GNU
;;; Copyright © 2021 Nicolò Balzarotti <nicolo@nixo.xyz>
;;; Copyright © 2021 Simon Tournier <zimon.toutoune@gmail.com>
;;; Copyright © 2021, 2023 Efraim Flashner <efraim@flashner.co.il>
;;; Copyright © 2021 Jean-Baptiste Volatier <jbv@pm.me>
;;; Copyright © 2023 Sharlatan Hellseher <sharlatanus@gmail.com>
;;;
;;; This file is part of GNU Guix.
;;;
;;; GNU Guix is free software; you can redistribute it and/or modify it
;;; under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 3 of the License, or (at
;;; your option) any later version.
;;;
;;; GNU Guix is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Guix.  If not, see <http://www.gnu.org/licenses/>.

(define-module (gnu packages julia-jll)
  #:use-module ((guix licenses) #:prefix license:)
  #:use-module (guix packages)
  #:use-module (guix git-download)
  #:use-module (guix utils)
  #:use-module (guix build-system julia)
  #:use-module (gnu packages)
  #:use-module (gnu packages astronomy)
  #:use-module (gnu packages audio)
  #:use-module (gnu packages base)
  #:use-module (gnu packages compression)
  #:use-module (gnu packages fontutils)
  #:use-module (gnu packages freedesktop)
  #:use-module (gnu packages fribidi)
  #:use-module (gnu packages gcc)
  #:use-module (gnu packages gettext)
  #:use-module (gnu packages gl)
  #:use-module (gnu packages glib)
  #:use-module (gnu packages gnupg)
  #:use-module (gnu packages graphics)
  #:use-module (gnu packages gtk)
  #:use-module (gnu packages image)
  #:use-module (gnu packages imagemagick)
  #:use-module (gnu packages julia)
  #:use-module (gnu packages julia-xyz)
  #:use-module (gnu packages libffi)
  #:use-module (gnu packages linux)
  #:use-module (gnu packages maths)
  #:use-module (gnu packages mp3)
  #:use-module (gnu packages pcre)
  #:use-module (gnu packages qt)
  #:use-module (gnu packages tls)
  #:use-module (gnu packages video)
  #:use-module (gnu packages web)
  #:use-module (gnu packages xdisorg)
  #:use-module (gnu packages xiph)
  #:use-module (gnu packages xml)
  #:use-module (gnu packages xorg))

;;; TODO: Remove this autogenerated source package
;;; and build it from realse source using <https://github.com/JuliaPackaging/Yggdrasil/>

(define-public julia-bzip2-jll
  (package
    (name "julia-bzip2-jll")
    (version "1.0.7+0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/JuliaBinaryWrappers/Bzip2_jll.jl")
             (commit (string-append "Bzip2-v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "188ngx4axdqr55ahv2xssl3cf699l1sxi41j336m9sw8plf4hhk8"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f                      ; No runtests.jl
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
              (lambda (wrapper)
                (substitute* wrapper
                  (("generate_wrapper_header.*")
                   (string-append
                    "generate_wrapper_header(\"Bzip2\", \""
                    (assoc-ref inputs "bzip2") "\")\n"))))
              ;; There's a Julia file for each platform, override them all
              (find-files "src/wrappers/" "\\.jl$"))
             #t)))))
    (inputs
     (list bzip2))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/Bzip2_jll.jl")
    (synopsis "Bzip2 binary wrappers")
    (description "This Julia module provides a wrapper for the @code{Bzip2}
compression program.")
    (license license:expat)))

(define-public julia-cairo-jll
  (package
    (name "julia-cairo-jll")
    (version "1.16.0+5")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Cairo_jll.jl")
               (commit (string-append "Cairo-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1766k8f63qb4bgyl4g29qrcddwypqx0c4kdg3lji8bsimw3f1bay"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Cairo\"")
                    (string-append "\"" (assoc-ref inputs "cairo") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list cairo))
    (propagated-inputs
     (list julia-jllwrappers
           julia-bzip2-jll
           julia-fontconfig-jll
           julia-freetype2-jll
           julia-glib-jll
           julia-libpng-jll
           julia-lzo-jll
           julia-pixman-jll
           julia-xorg-libxext-jll
           julia-xorg-libxrender-jll
           julia-zlib-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Cairo_jll.jl")
    (synopsis "Cairo library wrappers")
    (description "This package provides a wrapper for the cairo library.")
    (license license:expat)))

(define-public julia-cfitsio-jll
  (package
    (name "julia-cfitsio-jll")
    (version "4.0.0+0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/JuliaBinaryWrappers/CFITSIO_jll.jl")
             (commit (string-append "CFITSIO-v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1k0mqmpyfjr3ibcmda08llw8m166zw0n3lh5y5aj81q37lrxw990"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
      #:phases
      (modify-phases %standard-phases
          (add-after 'link-depot 'override-binary-path
            (lambda* (#:key inputs #:allow-other-keys)
              (map
               (lambda (wrapper)
                 (substitute* wrapper
                   ;; We're not downloading and unpacking cfitsio.
                   (("using LibCURL_jll") "")
                   (("using Zlib_jll") "")
                   (("generate_init_header.*") "generate_init_header()\n" )
                   (("generate_wrapper_header.*")
                    (string-append
                     "generate_wrapper_header(\"CFITSIO\", \""
                     (assoc-ref inputs "cfitsio") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list cfitsio))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/CFITSIO_jll.jl")
    (synopsis "CFITSIO library wrappers")
    (description "This package provides a wrapper for the cfitsio library.")
    (license license:expat)))

(define-public julia-compilersupportlibraries-jll
  (package
    (name "julia-compilersupportlibraries-jll")
    (version "0.4.0+1")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/JuliaBinaryWrappers/CompilerSupportLibraries_jll.jl")
             (commit (string-append "CompilerSupportLibraries-v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "03j6xdvps259mhdzpjqf41l65w2l9sahvxg4wrp34hcf69wkrzpy"))))
    (build-system julia-build-system)
    (arguments
     `(#:tests? #f                      ; no runtests.jl
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
              (lambda (wrapper)
                (substitute* wrapper
                  (("generate_wrapper_header.*")
                   (string-append
                    "generate_wrapper_header(\"CompilerSupportLibraries\", \""
                    (assoc-ref inputs "gfortran:lib") "\")\n"))))
              ;; There's a Julia file for each platform, override them all
              (find-files "src/wrappers/" "\\.jl$"))
             #t)))))
    (inputs                             ;required by artifacts
     `(("gfortran:lib" ,gfortran "lib")))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/CompilerSupportLibraries_jll.jl")
    (synopsis "Internal wrappers")
    (description "This package provides compiler support for libraries.  It is
an autogenerated source package constructed using @code{BinaryBuilder.jl}. The
originating @code{build_tarballs.jl} script can be found on the community
build tree Yggdrasil.")
    (license license:expat)))

(define-public julia-erfa-jll
  (package
    (name "julia-erfa-jll")
    (version "2.0.0+0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/JuliaBinaryWrappers/ERFA_jll.jl")
             (commit (string-append "ERFA-v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32
         "0knlck3vqr19g9z8zgjr7lj0qf1lisji5s2lm00y3ymv9bkj59sl"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f ;no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map (lambda (wrapper)
                    (substitute* wrapper
                      (("generate_wrapper_header.*")
                       (string-append
                        "generate_wrapper_header(\"ERFA\", \""
                        (assoc-ref inputs "erfa") "\")\n"))))
                  ;; There's a Julia file for each platform, override them all
                  (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs (list erfa))
    (propagated-inputs (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/ERFA_jll.jl")
    (synopsis "ERFA library wrappers")
    (description "This package provides a wrapper for the erfa library.")
    (license license:expat)))

(define-public julia-expat-jll
  (package
    (name "julia-expat-jll")
    (version "2.2.10+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Expat_jll.jl")
               (commit (string-append "Expat-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0lkhkh0067lns35njpc1bqbx6653r99lrjcbgrihlln9a7k9qj1s"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("generate_wrapper_header.*")
                    (string-append
                      "generate_wrapper_header(\"Expat\", \""
                      (assoc-ref inputs "expat") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list expat))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/Expat_jll.jl")
    (synopsis "Expat library wrappers")
    (description "This package provides a wrapper for the expat library.")
    (license license:expat)))

(define-public julia-ffmpeg-jll
  (package
    (name "julia-ffmpeg-jll")
    (version "4.3.1+2")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/FFMPEG_jll.jl")
               (commit (string-append "FFMPEG-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1f8pq9nxiiqay9qcsly1wkfpyz9nbxakf78ryi2c7g8p5v80k6d3"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"FFMPEG\"")
                    (string-append "\"" (assoc-ref inputs "ffmpeg") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list ffmpeg-4))
    (propagated-inputs
     (list julia-jllwrappers
           julia-bzip2-jll
           julia-freetype2-jll
           julia-fribidi-jll
           julia-lame-jll
           julia-libass-jll
           julia-libfdk-aac-jll
           julia-libvorbis-jll
           julia-openssl-jll
           julia-opus-jll
           julia-ogg-jll
           julia-x264-jll
           julia-x265-jll
           julia-zlib-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/FFMPEG_jll.jl")
    (synopsis "Library wrappers for ffmpeg")
    (description "This package provides a wrapper for ffmpeg.")
    (license license:expat)))

(define-public julia-fontconfig-jll
  (package
    (name "julia-fontconfig-jll")
    (version "2.13.93+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Fontconfig_jll.jl")
               (commit (string-append "Fontconfig-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1p6mfci77bp31w3xzlwhjccln40g6g2rdpfp38v5xf8sz2aybpw2"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("generate_wrapper_header.*")
                    (string-append
                      "generate_wrapper_header(\"Fontconfig\", \""
                      (assoc-ref inputs "fontconfig") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     `(("fontconfig" ,fontconfig)))
    (propagated-inputs
     (list julia-jllwrappers
           julia-bzip2-jll
           julia-expat-jll
           julia-freetype2-jll
           julia-libuuid-jll
           julia-zlib-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Fontconfig_jll.jl")
    (synopsis "Fontconfig library wrappers")
    (description "This package provides a wrapper for the fontconfig library.")
    (license license:expat)))

(define-public julia-freetype2-jll
  (package
    (name "julia-freetype2-jll")
    (version "2.13.1+0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/JuliaBinaryWrappers/FreeType2_jll.jl")
             (commit (string-append "FreeType2-v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "1kngkk3kkf6dam8g4g2ajvzvydzxbgpgxlcpsg7nm7q6r5b4b4db"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f                      ; No runtests.jl
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
              (lambda (wrapper)
                (substitute* wrapper
                  (("generate_wrapper_header.*")
                   (string-append
                    "generate_wrapper_header(\"FreeType2\", \""
                    (assoc-ref inputs "freetype") "\")\n"))))
              ;; There's a Julia file for each platform, override them all
              (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list freetype))
    (propagated-inputs
     (list julia-jllwrappers julia-bzip2-jll julia-zlib-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/FreeType2_jll.jl")
    (synopsis "Freetype2 binary wrappers")
    (description "This Julia module provides a wrapper for the freetype font
rendering library.")
    (license license:expat)))

(define-public julia-fribidi-jll
  (package
    (name "julia-fribidi-jll")
    (version "1.0.5+5")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/FriBidi_jll.jl")
               (commit (string-append "FriBidi-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1nmaqhsnm51hyvnbr9riqfp3f636xyxn2ybych598xpkhhjqldwh"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"FriBidi\"")
                    (string-append "\"" (assoc-ref inputs "fribidi") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list fribidi))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/FriBidi_jll.jl")
    (synopsis "Fribidi library wrappers")
    (description "This package provides a wrapper for the fribidi library.")
    (license license:expat)))

(define-public julia-gettext-jll
  (package
    (name "julia-gettext-jll")
    (version "0.21.0+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Gettext_jll.jl")
               (commit (string-append "Gettext-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0ghwkwv970q6j9ggkl0zlyrflhx8rd39y5i2zkc6p26084kyrxmf"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("generate_wrapper_header.*")
                    (string-append
                      "generate_wrapper_header(\"Gettext\", \""
                      (assoc-ref inputs "gettext") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     `(("gettext" ,gettext-minimal)))
    (propagated-inputs
     (list julia-jllwrappers julia-compilersupportlibraries-jll
           julia-libiconv-jll julia-xml2-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Gettext_jll.jl")
    (synopsis "Gettext library wrappers")
    (description "This package provides a wrapper for the gettext library.")
    (license license:expat)))

(define-public julia-glfw-jll
  (package
    (name "julia-glfw-jll")
    (version "3.3.4+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/GLFW_jll.jl")
               (commit (string-append "GLFW-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "12r4g8x1pgfrx53wq1a2q0rj4p08q352mmci2px1j4bd0pwi8rc4"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                  (("generate_wrapper_header.*")
                   (string-append
                     "generate_wrapper_header(\"GLFW\", \""
                     (assoc-ref inputs "glfw") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list glfw))
    (propagated-inputs
     (list julia-jllwrappers
           julia-libglvnd-jll
           julia-xorg-libxcursor-jll
           julia-xorg-libxinerama-jll
           julia-xorg-libxrandr-jll
           julia-xorg-libxi-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/GLFW_jll.jl")
    (synopsis "Glfw library wrappers")
    (description "This package provides a wrapper for the glfw library.")
    (license license:expat)))

(define-public julia-glib-jll
  (package
    (name "julia-glib-jll")
    (version "2.68.1+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Glib_jll.jl")
               (commit (string-append "Glib-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0l0fm5m4rznljkig7k4cxpi4skm7j4nc6lqx5xr0sb0wfzbr2llv"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("generate_wrapper_header.*")
                    (string-append
                      "generate_wrapper_header(\"Glib\", \""
                      (assoc-ref inputs "glib") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list glib))
    (propagated-inputs
     (list julia-jllwrappers
           julia-gettext-jll
           julia-libffi-jll
           julia-libiconv-jll
           julia-libmount-jll
           julia-pcre-jll
           julia-zlib-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Glib_jll.jl")
    (synopsis "Glib library wrappers")
    (description "This package provides a wrapper for the glib library.")
    (license license:expat)))

(define-public julia-gr-jll
  (package
    (name "julia-gr-jll")
    (version "0.69.1+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/GR_jll.jl")
               (commit (string-append "GR-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1zpa35j9f6288sa9md28g0bqkdhwx0j0dmsyy3kc0zxjlz188a94"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("generate_wrapper_header.*")
                    (string-append
                      "generate_wrapper_header(\"GR\", \""
                      (assoc-ref inputs "gr-framework") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list gr-framework))
    (propagated-inputs
     (list julia-jllwrappers
           julia-bzip2-jll
           julia-cairo-jll
           julia-ffmpeg-jll
           julia-fontconfig-jll
           julia-glfw-jll
           julia-jpegturbo-jll
           julia-libpng-jll
           julia-libtiff-jll
           julia-pixman-jll
           julia-qt5base-jll
           julia-zlib-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/GR_jll.jl")
    (synopsis "GR framework library wrappers")
    (description "This package provides a wrapper for the GR framework.")
    (license license:expat)))

(define-public julia-gumbo-jll
  (package
    (name "julia-gumbo-jll")
    (version "0.10.1+1")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/JuliaBinaryWrappers/Gumbo_jll.jl")
             (commit (string-append "Gumbo-v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "00a182x5hfpjzyvrxdn8wh4h67q899p5dzqp19a5s22si4g41k76"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (let ((gumbo (string-append (assoc-ref inputs "gumbo-parser"))))
               (for-each
                (lambda (wrapper)
                  (substitute* wrapper
                    (("(const libgumbo = )\"(.*)\"" all const libname)
                     (string-append const "\"" gumbo "/lib/" libname "\"\n"))
                    (("(global artifact_dir =).*" all m)
                     (string-append m " \"" gumbo "\""))))
                ;; There's a Julia file for each platform, override them all
                (find-files "src/wrappers/" "\\.jl$"))))))))
    (inputs
     (list gumbo-parser))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/Gumbo_jll.jl")
    (synopsis "Gumbo HTML parsing library wrappers")
    (description "This package provides a wrapper for Gumbo HTML parsing library.")
    (license license:expat)))

(define-public julia-imagemagick-jll
  (package
    (name "julia-imagemagick-jll")
    (version "6.9.10-12+3")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/ImageMagick_jll.jl")
               (commit (string-append "ImageMagick-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32
          "1a0vnnjl52yqi7jgpr0dmx5ah5nlpylv5hc8aw2l89b9f36ff8jg"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   ;; Make sure we match the current library.
                   (("libMagickWand-6.Q16.so.6") "libMagickWand-6.Q16.so.7")
                   (("artifact\"ImageMagick\"")
                    (string-append "\"" (assoc-ref inputs "imagemagick") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (propagated-inputs
     (list julia-jllwrappers julia-jpegturbo-jll julia-libpng-jll
           julia-libtiff-jll julia-zlib-jll))
    (inputs
     (list imagemagick))
    (home-page "https://github.com/JuliaBinaryWrappers/ImageMagick_jll.jl")
    (synopsis "Imagemagick library wrappers")
    (description "This package provides a wrapper for Imagemagick.")
    (license license:expat)))

(define-public julia-jllwrappers
  (package
    (name "julia-jllwrappers")
    (version "1.3.0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/JuliaPackaging/JLLWrappers.jl")
             (commit (string-append "v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "0v7xhsv9z16d657yp47vgc86ggc01i1wigqh3n0d7i1s84z7xa0h"))))
    (arguments
     ;; Wants to download stuff
     '(#:tests? #f
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'custom-override-path
           (lambda* (#:key inputs #:allow-other-keys)
             ;; Make @generate_wrapper_header take an optional argument that
             ;; guix packagers can pass to override the default "override"
             ;; binary path.  This won't be needed when something like
             ;; https://github.com/JuliaPackaging/JLLWrappers.jl/pull/27
             ;; will be merged.
             (substitute* "src/wrapper_generators.jl"
               (("generate_wrapper_header.*")
                "generate_wrapper_header(src_name, override_path = nothing)\n")
               (("pkg_dir = .*" all)
                (string-append
                 all "\n" "override = something(override_path,"
                 "joinpath(dirname(pkg_dir), \"override\"))\n"))
               (("@static if isdir.*") "@static if isdir($override)\n")
               (("return joinpath.*") "return $override\n"))
             #t)))))
    (build-system julia-build-system)
    (propagated-inputs
     (list julia-preferences))
    (home-page "https://github.com/JuliaPackaging/JLLWrappers.jl")
    (synopsis "Julia macros used by JLL packages")
    (description "This package contains Julia macros that enable JLL packages
to generate themselves.  It is not intended to be used by users, but rather is
used in autogenerated packages via @code{BinaryBuilder.jl}.")
    (license license:expat)))

(define-public julia-jpegturbo-jll
  (package
    (name "julia-jpegturbo-jll")
    (version "2.0.1+2")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/JpegTurbo_jll.jl")
               (commit (string-append "JpegTurbo-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32
          "1xp1x0hrj337bgwwffwpyq7xg031j2a38fim29lixqa0a0y80x6y"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"JpegTurbo\"")
                    (string-append "\"" (assoc-ref inputs "libjpeg-turbo") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libjpeg-turbo))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/JpegTurbo_jll.jl")
    (synopsis "Libjpeg-turbo library wrappers")
    (description "This package provides a wrapper for the libjpeg-turbo library.")
    (license license:expat)))

(define-public julia-lame-jll
  (package
    (name "julia-lame-jll")
    (version "3.100.0+1")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/LAME_jll.jl")
               (commit (string-append "LAME-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1ck14hwrapcn5wiazf4m2brkqmlpiqpyr0468p467418av837y1c"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"LAME\"")
                    (string-append "\"" (assoc-ref inputs "lame") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list lame))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/LAME_jll.jl")
    (synopsis "Lame library wrappers")
    (description "This package provides a wrapper for the lame audio encoder.")
    (license license:expat)))

(define-public julia-libass-jll
  (package
    (name "julia-libass-jll")
    (version "0.14.0+2")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/libass_jll.jl")
               (commit (string-append "libass-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "05nvis5ifmaq1g0c98hcpnl0ky22qgp64d4lb6g6r4yrrn6kqlsc"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"libass\"")
                    (string-append "\"" (assoc-ref inputs "libass") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libass))
    (propagated-inputs
     (list julia-jllwrappers julia-bzip2-jll julia-freetype2-jll
           julia-fribidi-jll julia-zlib-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/libass_jll.jl")
    (synopsis "Libass library wrappers")
    (description "This package provides a wrapper for libass.")
    (license license:expat)))

(define-public julia-libfdk-aac-jll
  (package
    (name "julia-libfdk-aac-jll")
    (version "0.1.6+2")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/libfdk_aac_jll.jl")
               (commit (string-append "libfdk_aac-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0jinb205dn1yfvl0mx7dsah4xj3r8vc3ig8yl72langjc7vrwdn0"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"libfdk_aac\"")
                    (string-append "\"" (assoc-ref inputs "libfdk") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libfdk))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/libfdk_aac_jll.jl")
    (synopsis "FDK library wrappers")
    (description "This package provides a wrapper for the libfdk audio library.")
    (license license:expat)))

(define-public julia-libffi-jll
  (package
    (name "julia-libffi-jll")
    (version "3.3.0+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Libffi_jll.jl")
               (commit (string-append "Libffi-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "008ajchli77airvibdgqw7gvmhk9w63qrv94f88iz1ixxyrycnm0"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("lib64") "lib")
                   (("artifact\"Libffi\"")
                    (string-append "\"" (assoc-ref inputs "libffi") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libffi))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/Libffi_jll.jl")
    (synopsis "Libffi library wrappers")
    (description "This package provides a wrapper for the libffi library.")
    (license license:expat)))

(define-public julia-libgcrypt-jll
  (package
    (name "julia-libgcrypt-jll")
    (version "1.8.7+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Libgcrypt_jll.jl")
               (commit (string-append "Libgcrypt-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "06fsdwrfw3f4cdbg1ssswznvj8ar3w5w4vxxag1hacs14pxlm5gi"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                  (("generate_wrapper_header.*")
                   (string-append
                     "generate_wrapper_header(\"Libgcrypt\", \""
                     (assoc-ref inputs "libgcrypt") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libgcrypt))
    (propagated-inputs
     (list julia-jllwrappers julia-libgpg-error-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Libgcrypt_jll.jl")
    (synopsis "Library wrappers for libgcrypt")
    (description "This package provides a wrapper for the libgcrypt library.")
    (license license:expat)))

(define-public julia-libglvnd-jll
  (package
    (name "julia-libglvnd-jll")
    (version "1.3.0+2")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Libglvnd_jll.jl")
               (commit (string-append "Libglvnd-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1lrppqj836saryqxj9xrqn0cih513qhijkhgqdww5azw8w917d3w"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Libglvnd\"")
                    (string-append "\"" (assoc-ref inputs "libglvnd") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libglvnd))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-libx11-jll julia-xorg-libxext-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Libglvnd_jll.jl")
    (synopsis "Libglvnd library wrappers")
    (description "This package provides a wrapper for the libglvnd library.")
    (license license:expat)))

(define-public julia-libgpg-error-jll
  (package
    (name "julia-libgpg-error-jll")
    (version "1.42.0+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Libgpg_error_jll.jl")
               (commit (string-append "Libgpg_error-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0jmsn0mhn6b35b5awbrlpjjszknsplr62li574fkgwfxlfixb8iy"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                  (("generate_wrapper_header.*")
                   (string-append
                     "generate_wrapper_header(\"Libgpg_error\", \""
                     (assoc-ref inputs "libgpg-error") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libgpg-error))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/Libgpg_error_jll.jl")
    (synopsis "Library wrappers for libgpg-error")
    (description "This package provides a wrapper for the libgpg-error library.")
    (license license:expat)))

(define-public julia-libiconv-jll
  (package
    (name "julia-libiconv-jll")
    (version "1.16.1+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Libiconv_jll.jl")
               (commit (string-append "Libiconv-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "185f2460ypig1klgi8rdb0s2c8l4hxdbmsz9ihj7bdydl964s2hh"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("generate_wrapper_header.*")
                    (string-append
                      "generate_wrapper_header(\"Libiconv\", \""
                      (assoc-ref inputs "libiconv") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libiconv))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/Libiconv_jll.jl")
    (synopsis "Libiconv library wrappers")
    (description "This package provides a wrapper for the libiconv library.")
    (license license:expat)))

(define-public julia-libmount-jll
  (package
    (name "julia-libmount-jll")
    (version "2.35.0+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Libmount_jll.jl")
               (commit (string-append "Libmount-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "04i4vm5rshz8nfmnxqx8rm3b9jd7b1xdiriac1bpsx3n0qf4pw89"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("generate_wrapper_header.*")
                    (string-append
                      "generate_wrapper_header(\"Libmount\", \""
                      (assoc-ref inputs "util-linux") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list `(,util-linux "lib")))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/Libmount_jll.jl")
    (synopsis "Libmount library wrappers")
    (description "This package provides a wrapper for the libmount library
from util-linux.")
    (license license:expat)))

(define-public julia-libpng-jll
  (package
    (name "julia-libpng-jll")
    (version "1.6.37+5")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/LibPNG_jll.jl")
               (commit (string-append "libpng-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "10azxv26j8r193dg9i9x7ajm3frhwbhj8zzi40fj59fxm81xz6dx"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"libpng\"")
                    (string-append "\"" (assoc-ref inputs "libpng") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libpng))
    (propagated-inputs
     (list julia-jllwrappers julia-zlib-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/LibPNG_jll.jl")
    (synopsis "Libpng library wrappers")
    (description "This package provides a wrapper for the libpng library.")
    (license license:expat)))

(define-public julia-libsass-jll
  (let ((commit "69bf10603aad0ebf1f6df088c5fd7c4a5d1eb0ca"))
    (package
      (name "julia-libsass-jll")
      (version "3.5.5+0")                 ;tag not created upstream
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/libsass_jll.jl")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "1fb6rf422533bsmfslvc20ag1hr50bf9xaj32rvh7nv593sbiygn"))))
      (build-system julia-build-system)
      (arguments
       `(#:tests? #f                      ; no runtests.jl
         #:phases
         (modify-phases %standard-phases
           (add-after 'link-depot 'override-binary-path
             (lambda* (#:key inputs #:allow-other-keys)
               (map
                (lambda (wrapper)
                  (substitute* wrapper
                    (("generate_wrapper_header.*")
                     (string-append
                      "generate_wrapper_header(\"libsass\", \""
                      (assoc-ref inputs "libsass") "\")\n"))))
                ;; There's a Julia file for each platform, override them all
                (find-files "src/wrappers/" "\\.jl$"))
               #t)))))
      (inputs
       (list libsass))
      (propagated-inputs
       (list julia-jllwrappers))
      (home-page "https://github.com/JuliaBinaryWrappers/libsass_jll.jl")
      (synopsis "Julia wrapper for libsass")
      (description "This package provides a wrapper for libsass.  It is an
autogenerated source package constructed using @code{BinaryBuilder.jl}.  The
originating @code{build_tarballs.jl} script can be found on the community
build tree Yggdrasil.")
      (license license:expat))))

(define-public julia-libtiff-jll
  (package
    (name "julia-libtiff-jll")
    (version "4.1.0+1")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Libtiff_jll.jl")
               (commit (string-append "Libtiff-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "07zzhmwmh2g4645ghv76z40hza2ghlb7sw15b1pii7f9kfcsgf45"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Libtiff\"")
                    (string-append "\"" (assoc-ref inputs "libtiff") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libtiff))
    (propagated-inputs
     (list julia-jllwrappers julia-jpegturbo-jll julia-zlib-jll
           julia-zstd-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Libtiff_jll.jl")
    (synopsis "Libtiff library wrappers")
    (description "This package provides a wrapper for libtiff")
    (license license:expat)))

(define-public julia-libuuid-jll
  (package
    (name "julia-libuuid-jll")
    (version "2.36.0+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Libuuid_jll.jl")
               (commit (string-append "Libuuid-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0iz1qrrpl8sp336m1f884sdgwbdlkp9sm371cwcfq4iriaihbmbs"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("generate_wrapper_header.*")
                    (string-append
                      "generate_wrapper_header(\"Libuuid\", \""
                      (assoc-ref inputs "util-linux") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list `(,util-linux "lib")))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/Libmount_jll.jl")
    (synopsis "Libuuid library wrappers")
    (description "This package provides a wrapper for the libuuid library from util-linux.")
    (license license:expat)))

(define-public julia-libvorbis-jll
  (package
    (name "julia-libvorbis-jll")
    (version "1.3.6+4")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/libvorbis_jll.jl")
               (commit (string-append "libvorbis-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1zmwhpjy6gr865psjn600g6ickskp4sq34qf1qg8ji6x4v09pzrh"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"libvorbis\"")
                    (string-append "\"" (assoc-ref inputs "libvorbis") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libvorbis))
    (propagated-inputs
     (list julia-jllwrappers julia-ogg-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/libvorbis_jll.jl")
    (synopsis "Libvorbis library wrappers")
    (description
     "This package provides a wrapper for the libvorbis audio library.")
    (license license:expat)))

(define-public julia-lzo-jll
  (package
    (name "julia-lzo-jll")
    (version "2.10.1+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/LZO_jll.jl")
               (commit (string-append "LZO-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1gy57znz3b6pk902vgdzlrwrxib0bcfl0zr1prinfbr9vfmiv1h0"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                  (("generate_wrapper_header.*")
                   (string-append
                     "generate_wrapper_header(\"LZO\", \""
                     (assoc-ref inputs "lzo") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list lzo))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/LZO_jll.jl")
    (synopsis "LZO library wrappers")
    (description "This package provides a wrapper for the lzo library.")
    (license license:expat)))

(define-public julia-mbedtls-jll
  (package
    (name "julia-mbedtls-jll")
    (version "2.26.0+0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/JuliaBinaryWrappers/MbedTLS_jll.jl")
             (commit (string-append "MbedTLS-v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "171bbz39k3w81h03098aj70vay28354awi9gmhj0l8s72xfdigi6"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f                      ; No runtests.jl
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
              (lambda (wrapper)
                (substitute* wrapper
                  (("generate_wrapper_header.*")
                   (string-append
                    "generate_wrapper_header(\"MbedTLS\", \""
                    (assoc-ref inputs "mbedtls") "\")\n"))))
              ;; There's a Julia file for each platform, override them all
              (find-files "src/wrappers/" "\\.jl$"))
             #t)))))
    (inputs (list mbedtls-lts))
    (propagated-inputs (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/MbedTLS_jll.jl")
    (synopsis "Apache's mbed TLS binary wrappers")
    (description "This Julia module provides @code{mbed TLS} libraries and
wrappers.")
    (license license:expat)))

(define-public julia-ogg-jll
  (package
    (name "julia-ogg-jll")
    (version "1.3.4+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Ogg_jll.jl")
               (commit (string-append "Ogg-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0p27wgy48gfw8g0hzlvcxrp0346nqnyxa88pydm87ll3sfx9b4ww"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Ogg\"")
                    (string-append "\"" (assoc-ref inputs "libogg") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libogg))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/Ogg_jll.jl")
    (synopsis "Libogg library wrappers")
    (description "This package provides a wrapper for the libogg library.")
    (license license:expat)))

(define-public julia-openspecfun-jll
  (let ((commit "6c505cce3bdcd9cd2b15b4f9362ec3a42c4da71c"))
    (package
      (name "julia-openspecfun-jll")
      (version "0.5.3+4")                 ;tag not created upstream
      (source
       (origin
         (method git-fetch)
         (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/OpenSpecFun_jll.jl")
               (commit commit)))
         (file-name (git-file-name name version))
         (sha256
          (base32 "0bl2gcgndsbiwhwy8fl070cjm1fyf9kxj6gkikgirmzgjl29iakn"))))
      (build-system julia-build-system)
      (arguments
       `(#:tests? #f                      ; no runtests.jl
         #:phases
         (modify-phases %standard-phases
           (add-after 'link-depot 'override-binary-path
             (lambda* (#:key inputs #:allow-other-keys)
               (map
                (lambda (wrapper)
                  (substitute* wrapper
                    (("generate_wrapper_header.*")
                     (string-append
                      "generate_wrapper_header(\"OpenSpecFun\", \""
                      (assoc-ref inputs "openspecfun") "\")\n"))))
                ;; There's a Julia file for each platform, override them all
                (find-files "src/wrappers/" "\\.jl$"))
               #t)))))
      (inputs
       (list openspecfun))
      (propagated-inputs
       (list julia-jllwrappers julia-compilersupportlibraries-jll))
      (home-page "https://github.com/JuliaBinaryWrappers/OpenSpecFun_jll.jl")
      (synopsis "Internal wrappers")
      (description "This package provides a wrapper for OpenSpecFun.  It is an
autogenerated source package constructed using @code{BinaryBuilder.jl}. The
originating @code{build_tarballs.jl} script can be found on the community
build tree Yggdrasil.")
      (license license:expat))))

(define-public julia-openssl-jll
  (package
    (name "julia-openssl-jll")
    (version "1.1.1+2")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/OpenSSL_jll.jl")
               (commit (string-append "OpenSSL-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0qly9pjhah95jdgvckkj615yfbsavvsygpfq9sqz4716q4zv0d5z"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"OpenSSL\"")
                    (string-append "\"" (assoc-ref inputs "openssl") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list openssl))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/OpenSSL_jll.jl")
    (synopsis "Openssl library wrappers")
    (description "This package provides a wrapper for the openssl library.")
    (license license:expat)))

(define-public julia-opus-jll
  (package
    (name "julia-opus-jll")
    (version "1.3.1+1")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Opus_jll.jl")
               (commit (string-append "Opus-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1zm0rnr3pi83bzza1azlrv8l7l0mjpykc3qz4b5p9zcdzf7aw4vn"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Opus\"")
                    (string-append "\"" (assoc-ref inputs "opus") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list opus))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/Opus_jll.jl")
    (synopsis "Opus library wrappers")
    (description "This package provides a wrapper for the opus audio library.")
    (license license:expat)))

(define-public julia-pcre-jll
  (package
    (name "julia-pcre-jll")
    (version "8.44.0+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/PCRE_jll.jl")
               (commit (string-append "PCRE-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0fcn8m16vy92v9dj437hzlknq9zl6sw6rckmr5gmh07sf0n3wszs"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("generate_wrapper_header.*")
                    (string-append
                      "generate_wrapper_header(\"PCRE\", \""
                      (assoc-ref inputs "pcre") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list pcre))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/PCRE_jll.jl")
    (synopsis "PCRE library wrappers")
    (description "This package provides a wrapper for the pcre library.")
    (license license:expat)))

(define-public julia-pixman-jll
  (package
    (name "julia-pixman-jll")
    (version "0.40.1+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Pixman_jll.jl")
               (commit (string-append "Pixman-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0ahawpgsiccmpa7gyxny7hq058igqvpza7ybqa44vl2nynnry2g7"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("generate_wrapper_header.*")
                    (string-append
                      "generate_wrapper_header(\"Pixman\", \""
                      (assoc-ref inputs "pixman") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list pixman))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/Pixman_jll.jl")
    (synopsis "Pixman library wrappers")
    (description "This package provides a wrapper for the pixman library.")
    (license license:expat)))

(define-public julia-qt5base-jll
  (package
    (name "julia-qt5base-jll")
    (version "5.15.2+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Qt5Base_jll.jl")
               (commit (string-append "Qt5Base-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1hhs316dl2jy56y2j4809vwpfj6ffbjchl1a27x44mmh9bj7vxzy"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                  (("generate_wrapper_header.*")
                   (string-append
                     "generate_wrapper_header(\"Qt5Base\", \""
                     (assoc-ref inputs "qtbase") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list qtbase-5))
    (propagated-inputs
     (list julia-jllwrappers
           julia-libglvnd-jll
           julia-compilersupportlibraries-jll
           julia-fontconfig-jll
           julia-glib-jll
           julia-openssl-jll
           julia-xkbcommon-jll
           julia-xorg-libxcb-jll
           julia-xorg-libxext-jll
           julia-xorg-xcb-util-image-jll
           julia-xorg-xcb-util-keysyms-jll
           julia-xorg-xcb-util-renderutil-jll
           julia-xorg-xcb-util-wm-jll
           julia-zlib-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Qt5Base_jll.jl")
    (synopsis "Qtbase library wrappers")
    (description "This package provides a wrapper for the qtbase library.")
    (license license:expat)))

(define-public julia-wayland-jll
  (package
    (name "julia-wayland-jll")
    (version "1.17.0+3")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Wayland_jll.jl")
               (commit (string-append "Wayland-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1w53inz62va3f022pgw3rfw5z5vgiv8z9dg3lfzpjrdb0lcd6ab6"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Wayland\"")
                    (string-append "\"" (assoc-ref inputs "wayland") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list wayland))
    (propagated-inputs
     (list julia-jllwrappers julia-expat-jll julia-libffi-jll
           julia-xml2-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Wayland_jll.jl")
    (synopsis "Wayland library wrappers")
    (description "This package provides a wrapper for the wayland library.")
    (license license:expat)))

(define-public julia-wayland-protocols-jll
  (package
    (name "julia-wayland-protocols-jll")
    (version "1.18.0+3")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Wayland_protocols_jll.jl")
               (commit (string-append "Wayland_protocols-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1dc9d2wzgybqjlg8l7f4ridkv2d66dg3lb3zihnl0k64psibn4x9"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Wayland_protocols\"")
                    (string-append "\"" (assoc-ref inputs "wayland-protocols") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list wayland-protocols))
    (propagated-inputs
     (list julia-jllwrappers julia-wayland-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Wayland_protocols_jll.jl")
    (synopsis "Wayland-protocols library wrappers")
    (description "This package provides a wrapper for the wayland-protocols library.")
    (license license:expat)))

(define-public julia-wcs-jll
  (package
    (name "julia-wcs-jll")
    (version "7.7.0+0")
    (source
     (origin
       (method git-fetch)
       (uri (git-reference
             (url "https://github.com/JuliaBinaryWrappers/WCS_jll.jl")
             (commit (string-append "WCS-v" version))))
       (file-name (git-file-name name version))
       (sha256
        (base32 "16i9899jwcp5i9mh88rn4b83v3i5v8g1jygixrr0grjnvf5qfvpk"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
              (lambda (wrapper)
                (substitute* wrapper
                  (("generate_wrapper_header.*")
                   (string-append
                    "generate_wrapper_header(\"WCS\", \""
                    (assoc-ref inputs "wcslib") "\")\n"))))
              ;; There's a Julia file for each platform, override them all
              (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list wcslib-7.12))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/WCS_jll.jl")
    (synopsis "WCS library wrappers")
    (description "This package provides a wrapper for the wcs library.")
    (license license:expat)))

(define-public julia-x264-jll
  (package
    (name "julia-x264-jll")
    (version "2020.7.14+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/x264_jll.jl")
               (commit (string-append "x264-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "131srmmcwhp9f2x4dq3dw4pzv2z0428mdrb923yzzlm7a89nf28p"))))
    (build-system julia-build-system)
    (arguments
     `(#:tests? #f ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (let ((libx264 (assoc-ref inputs "libx264")))
              (map
               (lambda (wrapper)
                 (substitute* wrapper
                   ;; Make sure we match the current library.
                   (("libx264.so.157")
                    (string-append "libx264.so."
                                   ,(version-major (package-version libx264))))
                   (("artifact\"x264\"")
                    (string-append "\"" libx264 "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$"))))))))
    (inputs
     (list libx264))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/x264_jll.jl")
    (synopsis "Library wrappers for x264")
    (description "This package provides a wrapper for the x264 video library.")
    (license license:expat)))

(define-public julia-x265-jll
  (package
    (name "julia-x265-jll")
    (version "3.0.0+1")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/x265_jll.jl")
               (commit (string-append "x265-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "082jgjlc2zm9rzg8p7l9nd4hlg17ziwp2b8rrcpicpb6fxb7sjh4"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"x265\"")
                    (string-append "\"" (assoc-ref inputs "x265") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list x265))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/x265_jll.jl")
    (synopsis "Library wrappers for x265")
    (description "This package provides a wrapper for the x265 video library.")
    (license license:expat)))

(define-public julia-xkbcommon-jll
  (package
    (name "julia-xkbcommon-jll")
    (version "0.9.1+3")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/xkbcommon_jll.jl")
               (commit (string-append "xkbcommon-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1g2gmd3mj1p369kzvrd02ldgr9s712vs9774v1phb59jxlshc0zc"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"xkbcommon\"")
                    (string-append "\"" (assoc-ref inputs "libxkbcommon") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libxkbcommon))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-libxcb-jll
           julia-xorg-xkeyboard-config-jll julia-wayland-jll
           julia-wayland-protocols-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/xkbcommon_jll.jl")
    (synopsis "Libxkbcommon library wrappers")
    (description "This package provides a wrapper for the libxkbcommon library.")
    (license license:expat)))

(define-public julia-xml2-jll
  (package
    (name "julia-xml2-jll")
    (version "2.9.12+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/XML2_jll.jl")
               (commit (string-append "XML2-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1g6wf5r5v2qx6cwv05kd9amcsmv65vkajk43m9r1c35jqs9m8fnm"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("generate_wrapper_header.*")
                    (string-append
                      "generate_wrapper_header(\"XML2\", \""
                      (assoc-ref inputs "libxml2") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libxml2))
    (propagated-inputs
     (list julia-jllwrappers julia-libiconv-jll julia-zlib-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/XML2_jll.jl")
    (synopsis "XML2 library wrappers")
    (description "This package provides a wrapper for the libxml2 library.")
    (license license:expat)))

(define-public julia-xorg-libpthread-stubs-jll
  (package
    (name "julia-xorg-libpthread-stubs-jll")
    (version "0.1.0+2")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_libpthread_stubs_jll.jl")
               (commit (string-append "Xorg_libpthread_stubs-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "161f4111xsb8xq4zs59jw95s94xfn1yxpii0p0dhn3yqgligggvx"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_libpthread_stubs\"")
                    (string-append "\"" (assoc-ref inputs "libpthread-stubs") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libpthread-stubs))
    (propagated-inputs
     (list julia-jllwrappers julia-xslt-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_libpthread_stubs_jll.jl")
    (synopsis "Libpthread-stubs library wrappers")
    (description "This package provides a wrapper for the libpthread-stubs library.")
    (license license:expat)))

(define-public julia-xorg-libx11-jll
  (package
    (name "julia-xorg-libx11-jll")
    (version "1.6.9+2")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_libX11_jll.jl")
               (commit (string-append "Xorg_libX11-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1fw2dmmw04jmyss43g66q80w3j7wshmxgjccir1hh4c5d8x2zs39"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_libX11\"")
                    (string-append "\"" (assoc-ref inputs "libx11") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libx11))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-libxcb-jll julia-xorg-xtrans-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_libX11_jll.jl")
    (synopsis "LibX11 library wrappers")
    (description "This package provides a wrapper for the libx11 library.")
    (license license:expat)))

(define-public julia-xorg-libxau-jll
  (package
    (name "julia-xorg-libxau-jll")
    (version "1.0.9+3")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_libXau_jll.jl")
               (commit (string-append "Xorg_libXau-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1bc59hzg4jgdy0lwykp6avxsb87fq9j79c30yxprwjvxq8xm3p8z"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_libXau\"")
                    (string-append "\"" (assoc-ref inputs "libxau") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libxau))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_libXau_jll.jl")
    (synopsis "Libxau library wrappers")
    (description "This package provides a wrapper for the libxau library.")
    (license license:expat)))

(define-public julia-xorg-libxcb-jll
  (package
    (name "julia-xorg-libxcb-jll")
    (version "1.13.0+2")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_libxcb_jll.jl")
               (commit (string-append "Xorg_libxcb-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "109m4r4v6ww31rq0klyqd3rf3j1yiycvld82d514d040w5027ssk"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_libxcb\"")
                    (string-append "\"" (assoc-ref inputs "libxcb") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libxcb))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-libxau-jll
           julia-xorg-libpthread-stubs-jll julia-xorg-libxdmcp-jll
           julia-xslt-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_libxcb_jll.jl")
    (synopsis "Libxcb library wrappers")
    (description "This package provides a wrapper for the libxcb library.")
    (license license:expat)))

(define-public julia-xorg-libxcursor-jll
  (package
    (name "julia-xorg-libxcursor-jll")
    (version "1.2.0+3")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_libXcursor_jll.jl")
               (commit (string-append "Xorg_libXcursor-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0hxhpsjw1zk30qphrp90g1wvqfs1hr47qifn1gqgx73ci5nmq0y7"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_libXcursor\"")
                    (string-append "\"" (assoc-ref inputs "libxcursor") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libxcursor))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-libxfixes-jll
           julia-xorg-libxrender-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_libXcursor_jll.jl")
    (synopsis "Libxcursor library wrappers")
    (description "This package provides a wrapper for the libxcursor library.")
    (license license:expat)))

(define-public julia-xorg-libxdmcp-jll
  (package
    (name "julia-xorg-libxdmcp-jll")
    (version "1.1.3+3")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_libXdmcp_jll.jl")
               (commit (string-append "Xorg_libXdmcp-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1ghw8c7ibkm2hv4h38qyhbxfmyizxszqv6fv8qzlb5031dmshnap"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_libXdmcp\"")
                    (string-append "\"" (assoc-ref inputs "libxdmcp") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libxdmcp))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_libXdmcp_jll.jl")
    (synopsis "Libxdmcp library wrappers")
    (description "This package provides a wrapper for the libxdmcp library.")
    (license license:expat)))

(define-public julia-xorg-libxext-jll
  (package
    (name "julia-xorg-libxext-jll")
    (version "1.3.4+2")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_libXext_jll.jl")
               (commit (string-append "Xorg_libXext-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1xmx86l54apvqv0xwy0rha7knjl4x5crllqra56nhi0arhw8ywfc"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_libXext\"")
                    (string-append "\"" (assoc-ref inputs "libxext") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libxext))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-libx11-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_libXext_jll.jl")
    (synopsis "LibXext library wrappers")
    (description "This package provides a wrapper for the libxext library.")
    (license license:expat)))

(define-public julia-xorg-libxfixes-jll
  (package
    (name "julia-xorg-libxfixes-jll")
    (version "5.0.3+3")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_libXfixes_jll.jl")
               (commit (string-append "Xorg_libXfixes-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0sjpclxinbcq3msnaqdfqlpfhnlvl15qn7dam968i4qwrpyv43dv"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_libXfixes\"")
                    (string-append "\"" (assoc-ref inputs "libxfixes") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libxfixes))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-libx11-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_libXfixes_jll.jl")
    (synopsis "Libxfixes library wrappers")
    (description "This package provides a wrapper for the libxfixes library.")
    (license license:expat)))

(define-public julia-xorg-libxi-jll
  (package
    (name "julia-xorg-libxi-jll")
    (version "1.7.10+3")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_libXi_jll.jl")
               (commit (string-append "Xorg_libXi-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1jhrng5sf44880x3pnw1gpb88z21c7nznfyzhs4a5z910ndrynd7"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_libXi\"")
                    (string-append "\"" (assoc-ref inputs "libxi") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libxi))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-libxext-jll
           julia-xorg-libxfixes-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_libXi_jll.jl")
    (synopsis "Libxi library wrappers")
    (description "This package provides a wrapper for the libxi library.")
    (license license:expat)))

(define-public julia-xorg-libxinerama-jll
  (package
    (name "julia-xorg-libxinerama-jll")
    (version "1.1.4+3")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_libXinerama_jll.jl")
               (commit (string-append "Xorg_libXinerama-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0jybxbqxd4jc9ka3rk3v5yh8ps2fapdibldr7bymllzw1w2i25rn"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_libXinerama\"")
                    (string-append "\"" (assoc-ref inputs "libxinerama") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libxinerama))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-libxext-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_libXinerama_jll.jl")
    (synopsis "Libxinerama library wrappers")
    (description "This package provides a wrapper for the libxinerama library.")
    (license license:expat)))

(define-public julia-xorg-libxkbfile-jll
  (package
    (name "julia-xorg-libxkbfile-jll")
    (version "1.1.0+3")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_libxkbfile_jll.jl")
               (commit (string-append "Xorg_libxkbfile-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0zrnrixz34h54n0c06ziaxcajvndydzgxxh5jbvqx1xrij5rw5gy"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_libxkbfile\"")
                    (string-append "\"" (assoc-ref inputs "libxkbfile") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libxkbfile))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-libx11-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_libxkbfile_jll.jl")
    (synopsis "Libxkbfile library wrappers")
    (description "This package provides a wrapper for the libxkbfile library.")
    (license license:expat)))

(define-public julia-xorg-libxrandr-jll
  (package
    (name "julia-xorg-libxrandr-jll")
    (version "1.5.2+3")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_libXrandr_jll.jl")
               (commit (string-append "Xorg_libXrandr-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0y4gsw5x643qdamf932agzdlyayzp63pn8j7j0lckd6dzqfdz46g"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_libXrandr\"")
                    (string-append "\"" (assoc-ref inputs "libxrandr") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libxrandr))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-libxext-jll
           julia-xorg-libxrender-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_libXrandr_jll.jl")
    (synopsis "Libxrandr library wrappers")
    (description "This package provides a wrapper for the libxrandr library.")
    (license license:expat)))

(define-public julia-xorg-libxrender-jll
  (package
    (name "julia-xorg-libxrender-jll")
    (version "0.9.10+3")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_libXrender_jll.jl")
               (commit (string-append "Xorg_libXrender-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "041kjqpkgcjf72msg4zm4wja623wfsy9gmkqjvsj46lj885qizz7"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_libXrender\"")
                    (string-append "\"" (assoc-ref inputs "libxrender") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libxrender))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-libx11-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_libXrender_jll.jl")
    (synopsis "Library wrappers libXrender")
    (description "This package provides a wrapper for the libXrender library.")
    (license license:expat)))

(define-public julia-xorg-xcb-util-jll
  (package
    (name "julia-xorg-xcb-util-jll")
    (version "0.4.0+1")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_xcb_util_jll.jl")
               (commit (string-append "Xorg_xcb_util-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0jywbxwf9x2naqsh9hh231bqpphh15v7cdhijcspjfggwkyq1npi"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_xcb_util\"")
                    (string-append "\"" (assoc-ref inputs "xcb-util") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list xcb-util))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-libxcb-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_xcb_util_jll.jl")
    (synopsis "Xcb-util library wrappers")
    (description "This package provides a wrapper for the xcb-util library.")
    (license license:expat)))

(define-public julia-xorg-xcb-util-image-jll
  (package
    (name "julia-xorg-xcb-util-image-jll")
    (version "0.4.0+1")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_xcb_util_image_jll.jl")
               (commit (string-append "Xorg_xcb_util_image-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1f9xx094nylg7dcfxm0qmph4xy492rd3yxa8arijqyi6rs8zrgxz"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_xcb_util_image\"")
                    (string-append "\"" (assoc-ref inputs "xcb-util-image") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list xcb-util-image))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-xcb-util-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_xcb_util_image_jll.jl")
    (synopsis "Xcb-util-image library wrappers")
    (description "This package provides a wrapper for the xcb-util-image library.")
    (license license:expat)))

(define-public julia-xorg-xcb-util-keysyms-jll
  (package
    (name "julia-xorg-xcb-util-keysyms-jll")
    (version "0.4.0+1")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_xcb_util_keysyms_jll.jl")
               (commit (string-append "Xorg_xcb_util_keysyms-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "03i3fw9p16rpjnki80w4rhmaiqvjlfsr94bf9yizndqsw1lcq42l"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_xcb_util_keysyms\"")
                    (string-append "\"" (assoc-ref inputs "xcb-util-keysyms") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list xcb-util-keysyms))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-xcb-util-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_xcb_util_keysyms_jll.jl")
    (synopsis "Xcb-util-keysyms library wrappers")
    (description "This package provides a wrapper for the xcb-util-keysyms library.")
    (license license:expat)))

(define-public julia-xorg-xcb-util-renderutil-jll
  (package
    (name "julia-xorg-xcb-util-renderutil-jll")
    (version "0.3.9+1")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_xcb_util_renderutil_jll.jl")
               (commit (string-append "Xorg_xcb_util_renderutil-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1zxz459sxma7cv32x2y8fnvwdz1f37fq0xhkihdsnkfdl761gn1a"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_xcb_util_renderutil\"")
                    (string-append "\"" (assoc-ref inputs "xcb-util-renderutil") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list xcb-util-renderutil))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-xcb-util-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_xcb_util_renderutil_jll.jl")
    (synopsis "Xcb-util-renderutil library wrappers")
    (description "This package provides a wrapper for the xcb-util-renderutil library.")
    (license license:expat)))

(define-public julia-xorg-xcb-util-wm-jll
  (package
    (name "julia-xorg-xcb-util-wm-jll")
    (version "0.4.1+1")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_xcb_util_wm_jll.jl")
               (commit (string-append "Xorg_xcb_util_wm-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0g6jhfb3l04lrx3cm3b8wc0pp5271dpncwin5pg94nljdj4mgj53"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_xcb_util_wm\"")
                    (string-append "\"" (assoc-ref inputs "xcb-util-wm") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list xcb-util-wm))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-xcb-util-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_xcb_util_wm_jll.jl")
    (synopsis "Xcb-util-wm library wrappers")
    (description "This package provides a wrapper for the xcb-util-wm library.")
    (license license:expat)))

(define-public julia-xorg-xkbcomp-jll
  (package
    (name "julia-xorg-xkbcomp-jll")
    (version "1.4.2+3")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_xkbcomp_jll.jl")
               (commit (string-append "Xorg_xkbcomp-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1rkb9525iq0jjpq8v333b728kfillgixxsim37mqdplad85l36wl"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_xkbcomp\"")
                    (string-append "\"" (assoc-ref inputs "xkbcomp") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list xkbcomp))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-libxkbfile-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_xkbcomp_jll.jl")
    (synopsis "Library wrappers for xkbcomp")
    (description "This package provides a wrapper for the xkbcomp library.")
    (license license:expat)))

(define-public julia-xorg-xkeyboard-config-jll
  (package
    (name "julia-xorg-xkeyboard-config-jll")
    (version "2.27.0+3")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_xkeyboard_config_jll.jl")
               (commit (string-append "Xorg_xkeyboard_config-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "1lgynzxd0mn64zbf0njqkd1hz1illqnl3p7hi9abwh5vbdf4pwhw"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_xkeyboard_config\"")
                    (string-append "\"" (assoc-ref inputs "xkeyboard-config") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list xkeyboard-config))
    (propagated-inputs
     (list julia-jllwrappers julia-xorg-xkbcomp-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_xkeyboard_config_jll.jl")
    (synopsis "Xkeyboard-config library wrappers")
    (description "This package provides a wrapper for the xkeyboard-config library.")
    (license license:expat)))

(define-public julia-xorg-xtrans-jll
  (package
    (name "julia-xorg-xtrans-jll")
    (version "1.4.0+2")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Xorg_xtrans_jll.jl")
               (commit (string-append "Xorg_xtrans-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "16rzkgc7l1j57l43v5ffrak164bdff7h1amm0y3mcnwjqmkig9dn"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("artifact\"Xorg_xtrans\"")
                    (string-append "\"" (assoc-ref inputs "xtrans") "\""))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list xtrans))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/Xorg_xtrans_jll.jl")
    (synopsis "Library wrappers for xtrans")
    (description "This package provides a wrapper for the xtrans library.")
    (license license:expat)))

(define-public julia-xslt-jll
  (package
    (name "julia-xslt-jll")
    (version "1.1.34+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/XSLT_jll.jl")
               (commit (string-append "XSLT-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0wjkfnrgpd7c6i4ga5xzsqqmfrxcdkr1kjsxmd9bff8cqvyknnhq"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                  (("generate_wrapper_header.*")
                   (string-append
                     "generate_wrapper_header(\"XSLT\", \""
                     (assoc-ref inputs "libxslt") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list libxslt))
    (propagated-inputs
     (list julia-jllwrappers
           julia-libgcrypt-jll
           julia-libgpg-error-jll
           julia-libiconv-jll
           julia-xml2-jll
           julia-zlib-jll))
    (home-page "https://github.com/JuliaBinaryWrappers/XSLT_jll.jl")
    (synopsis "Xslt library wrappers")
    (description "This package provides a wrapper for the libxslt library.")
    (license license:expat)))

(define-public julia-zlib-jll
  (package
    (name "julia-zlib-jll")
    (version "1.2.12+1")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Zlib_jll.jl")
               (commit (string-append "Zlib-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "05ih0haqapkzr40swvq63cafnqlc4yp6yfa1wvdyq8v3n4kxhfqa"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("generate_wrapper_header.*")
                    (string-append
                      "generate_wrapper_header(\"Zlib\", \""
                      (assoc-ref inputs "zlib") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     (list zlib))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/Zlib_jll.jl")
    (synopsis "Zlib library wrappers")
    (description "This package provides a wrapper for Zlib.")
    (license license:expat)))

(define-public julia-zstd-jll
  (package
    (name "julia-zstd-jll")
    (version "1.5.0+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Zstd_jll.jl")
               (commit (string-append "Zstd-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "15g5wsvga4p9bjmx97xqwqdnfzfbwfl6c4a9iaphcncwgcrnw6y6"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("generate_wrapper_header.*")
                    (string-append
                      "generate_wrapper_header(\"Zstd\", \""
                      (assoc-ref inputs "zstd:lib") "\")\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     `(("zstd:lib" ,zstd "lib")))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/Zstd_jll.jl")
    (synopsis "Zstd library wrappers")
    (description "This package provides a wrapper for the zstd library.")
    (license license:expat)))

(define-public julia-sundials-jll
  (package
    (name "julia-sundials-jll")
    (version "5.2.1+0")
    (source
      (origin
        (method git-fetch)
        (uri (git-reference
               (url "https://github.com/JuliaBinaryWrappers/Sundials_jll.jl")
               (commit (string-append "Sundials-v" version))))
        (file-name (git-file-name name version))
        (sha256
         (base32 "0cijb9frq8gj8bjpqf2lr5d0jxlj262y6h6xi4z3536dingrvffc"))))
    (build-system julia-build-system)
    (arguments
     '(#:tests? #f  ; no runtests
       #:phases
       (modify-phases %standard-phases
         (add-after 'link-depot 'override-binary-path
           (lambda* (#:key inputs #:allow-other-keys)
             (map
               (lambda (wrapper)
                 (substitute* wrapper
                   (("global artifact_dir.*")
                    (string-append
                     "global artifact_dir = \""
                      (assoc-ref inputs "sundials") "\"\n"))))
               ;; There's a Julia file for each platform, override them all
               (find-files "src/wrappers/" "\\.jl$")))))))
    (inputs
     `(("sundials" ,sundials-julia)))
    (propagated-inputs
     (list julia-jllwrappers))
    (home-page "https://github.com/JuliaBinaryWrappers/Sundials_jll.jl")
    (synopsis "SUndials library wrappers")
    (description "This package provides a wrapper for the sundials library.")
    (license license:expat)))
