=pod

=head1 NAME

XML::Handler::EasyTree::Traversal - Visit each node of an EasyTree

=head1 SYNOPSIS

  use XML::Parser;
  use XML::Handler::EasyTree;
  use XML::Handler::EasyTree::Traversal;
  use XML::Handler::EasyTree::Visitor;

  #make an EasyTree
  my $p=         new XML::Parser(Style=>'EasyTree');
  my $easy_tree= $p->parsefile('something.xml');

  #traverse the EasyTree
  my $traversal= new XML::Handler::EasyTree::Traversal;
  my $visitor=   new XML::Handler::EasyTree::Visitor;

  #visit each node of easytree $tree with $visitor
  $lTraversal->traverse($tree,$visitor);

=head1 DESCRIPTION

XML::Handler::EasyTree::Traversal and XML::Handler::EasyTree::Visitor
permit simple visitation of all nodes of an EasyTree. The current application 
of that is the XML::Handler::EasyTree::Writer;

C<traverse> will visit each element in the content of each node of the
EasyTree, in a depth first in-order search. On each visited node $n of
the EasyTree, C<$visitor->start_visit($n)> will be called. If a
non-zero value is returned, and $n is not a leaf node, then the
traversal will go deeper into the EasyTree, visiting each node contained 
in $n->{content}. If C<0> is returned, then $n will be treated like a leaf node,
i.e. $n->end_visit() will be called before returning.

=head1 EXAMPLE

Parse a tree and as XML and write it out to a file "xml-test-outfile.xml"

  #!perl -w
  use strict;
  use XML::Parser;
  use XML::Handler::EasyTree::WriterVisitor;
  use IO::File;
  
  $XML::Handler::EasyTree::Noempty=1;
  my $xml=<<'EOF';
  <foo>
    <head id="a">Hello <em>there</em>
    </head>
    <bar>Howdy<ref/>
    </bar>
    do
  </foo>
  EOF
  my $p=new XML::Parser(Style=>'EasyTree');
  my $tree=$p->parse($xml);


  $visitor=new XML::Handler::EasyTree::WriteVisitor(new IO::Stream('xml-test-outfile.xml','w');

  my $traversal=new XML::Handler::EasyTree::Traversal();

  $traversal->traverse($tree,$visitor);

  
=head1 KNOWN WEAKNESSES



=head1 AUTHOR

Wolfgang M&uuml;ller <wolfgang.mueller@cui.unige.ch>

Copyright (c) 2001 Wolfgang Mller. All rights reserved. This program
is free software; you can redistribute it and/or modify it under the same
terms as Perl itself.

=head1 SEE ALSO

  XML::Handler::EasyTree, XML::Handler::EasyTree::Visitor

=cut


package XML::Handler::EasyTree::Traversal;
use strict;
use vars qw(@ISA @EXPORT_OK $VERSION);
$VERSION='0.01';
@ISA=qw(Exporter);
@EXPORT_OK=qw(new
	      writeToStream);

sub new(){
  my $class = shift;
  my $self = {};
  bless $self, $class;
  $self->initialize(@_);
  return $self;
}

sub initialize(){
  my$self=shift;
}

sub traverse( $$ ){

  my$self=shift;

  my $lContent=shift;
  my $lVisitor=shift;

  my $lRoot;
  foreach $lRoot (@{$lContent}){
    my $lDeeper=$lVisitor->start_visit($lRoot);
    if($lDeeper && defined $lRoot->{content}){
      $self->traverse($lRoot->{content},$lVisitor);
    }
    $lVisitor->end_visit($lRoot);
  }
  return;
}


