/* GNU Mailutils -- a suite of utilities for electronic mail
   Copyright (C) 1999-2020 Free Software Foundation, Inc.

   GNU Mailutils is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3, or (at your option)
   any later version.

   GNU Mailutils is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with GNU Mailutils.  If not, see <http://www.gnu.org/licenses/>. */

#include <config.h>
#include <stdlib.h>
#include <mailutils/mailutils.h>
#include <mailutils/sys/envelope.h>
#include <muaux.h>
#include <sysexits.h>

int truncate_opt;
int from_filter;
int recode_charset;
char *charset;

static struct mu_option decodemail_options[] = 
{
  { "truncate", 't', NULL, MU_OPTION_DEFAULT,
    N_("truncate the output mailbox, if it exists"),
    mu_c_bool, &truncate_opt },
  { "charset", 'c', N_("CHARSET"), MU_OPTION_DEFAULT,
    N_("recode output to this charset"),
    mu_c_string, &charset },
  { "recode", 'R', NULL, MU_OPTION_DEFAULT,
    N_("recode text parts to the current charset"),
    mu_c_bool, &recode_charset },
  MU_OPTION_END
}, *options[] = { decodemail_options, NULL };

struct mu_cli_setup cli = {
  .optv = options,
  .prog_doc = N_("GNU decodemail -- decode messages."),
  .prog_args = N_("[INBOX] [OUTBOX]")
};

static char *decodemail_capa[] = {
  "debug",
  "mailbox",
  "locking",
  "mime",
  NULL
};

char *charset;

static void
define_charset (void)
{
  struct mu_lc_all lc_all = { .flags = 0 };
  char *ep = getenv ("LC_ALL");
  if (!ep)
    ep = getenv ("LANG");
  
  if (ep && mu_parse_lc_all (ep, &lc_all, MU_LC_CSET) == 0)
    {
      charset = mu_strdup (lc_all.charset);
      mu_lc_all_free (&lc_all);
    }
  else
    charset = mu_strdup ("us-ascii");
}

static mu_message_t message_decode (mu_message_t, mu_coord_t *, size_t);

static void message_store_mbox (mu_message_t, mu_mailbox_t, mu_coord_t);
static void message_store_stdout (mu_message_t, mu_mailbox_t, mu_coord_t);
static void crd_error (mu_coord_t crd, size_t n, char const *fmt, ...);

int
main (int argc, char **argv)
{
  int rc;
  mu_mailbox_t imbox, ombox = NULL;
  char *imbox_name = NULL, *ombox_name = NULL;
  void (*message_store) (mu_message_t, mu_mailbox_t, mu_coord_t);
  mu_iterator_t itr;
  unsigned long i;
  int err = 0;
  mu_coord_t crd;
  
  /* Native Language Support */
  MU_APP_INIT_NLS ();

  /* register the formats.  */
  mu_register_all_mbox_formats ();
  mu_register_extra_formats ();
  mu_auth_register_module (&mu_auth_tls_module);
  mu_cli_capa_register (&mu_cli_capa_mime);
  
  mu_cli (argc, argv, &cli, decodemail_capa, NULL, &argc, &argv);

  switch (argc)
    {
    case 2:
      ombox_name = argv[1];
    case 1:
      imbox_name = argv[0];
      break;
    case 0:
      break;
    default:
      mu_error (_("too many arguments; try %s --help for help"),
		mu_program_name);
      exit (EX_USAGE);
    }

  if (!charset && recode_charset)
    define_charset ();
  
  /* Open input mailbox */
  rc = mu_mailbox_create_default (&imbox, imbox_name);
  if (rc != 0)
    {
      if (imbox_name)
	mu_error (_("could not create mailbox `%s': %s"),
		  imbox_name,
		  mu_strerror (rc));
      else
	mu_error (_("could not create default mailbox: %s"),
		  mu_strerror (rc));
      exit (EX_OSERR);
    }

  rc = mu_mailbox_open (imbox, MU_STREAM_READ);
  if (rc)
    {
      mu_url_t url = NULL;

      mu_mailbox_get_url (imbox, &url);
      mu_error (_("could not open input mailbox `%s': %s"),
		mu_url_to_string (url), mu_strerror (rc));
      exit (EX_NOINPUT);
    }

  /* Create output mailbox */
  if (ombox_name)
    {
      mu_property_t prop;
      const char *type;
      
      rc = mu_mailbox_create_default (&ombox, ombox_name);
      if (rc != 0)
	{
	  mu_error (_("could not create output mailbox `%s': %s"),
		    ombox_name,
		    mu_strerror (rc));
	  exit (EX_OSERR);
	}
      rc = mu_mailbox_open (ombox, MU_STREAM_RDWR|MU_STREAM_CREAT);
      if (rc)
	{
	  mu_error (_("could not open mailbox `%s': %s"),
		    ombox_name, mu_strerror (rc));
	  exit (EX_CANTCREAT);
	}

      if (mu_mailbox_get_property (ombox, &prop) == 0 &&
	  mu_property_sget_value (prop, "TYPE", &type) == 0 &&
	  strcmp (type, "MBOX") == 0)
	from_filter = 1;
      
      if (truncate_opt)
	{
	  mu_mailbox_get_iterator (ombox, &itr);
	  for (mu_iterator_first (itr), i = 1; !mu_iterator_is_done (itr);
	       mu_iterator_next (itr), i++)
	    {
	      mu_message_t m;
	      mu_attribute_t a;
	      
	      rc = mu_iterator_current (itr, (void **)&m);
	      mu_message_get_attribute (m, &a);
	      mu_attribute_set_deleted (a);
	    }
	  mu_iterator_destroy (&itr);
	}
      message_store = message_store_mbox;
    }
  else
    {
      message_store = message_store_stdout;
      from_filter = 1;
    }
  
  rc = mu_mailbox_get_iterator (imbox, &itr);
  if (rc)
    {
      mu_diag_funcall (MU_DIAG_ERROR, "mu_mailbox_get_iterator", NULL, rc);
      exit (EX_SOFTWARE);
    }

  rc = mu_coord_alloc (&crd, 1);
  if (rc)
    mu_alloc_die ();
  
  for (mu_iterator_first (itr), i = 1; !mu_iterator_is_done (itr);
       mu_iterator_next (itr), i++)
    {
      mu_message_t msg, newmsg;

      rc = mu_iterator_current (itr, (void **)&msg);
      if (rc)
	{
	  mu_error (_("cannot read message %lu: %s"),
		    i, mu_strerror (rc));
	  err = 1;
	  continue;
	}
      crd[1] = i;
      newmsg = message_decode (msg, &crd, 1);
      message_store (newmsg, ombox, crd);
      mu_message_unref (newmsg);
    }
  mu_mailbox_destroy (&imbox);
  (truncate_opt ? mu_mailbox_expunge : mu_mailbox_sync) (ombox);
  mu_mailbox_destroy (&ombox);

  if (err)
    exit (EX_UNAVAILABLE);
  exit (EX_OK);
}

static void
message_store_mbox (mu_message_t msg, mu_mailbox_t mbx, mu_coord_t crd)
{
  int rc = mu_mailbox_append_message (mbx, msg);
  if (rc)
    {
      crd_error (crd, 1, _("cannot store message: %s"), mu_strerror (rc));
      switch (rc)
	{
	case MU_ERR_INVALID_EMAIL:
	case MU_ERR_EMPTY_ADDRESS:
	  break;

	default:
	  exit (EX_IOERR);
	}
    }
}

static void
env_print (mu_message_t msg)
{
  mu_envelope_t env;
  char const *buf;
  size_t len;
  
  mu_message_get_envelope (msg, &env);
  if (mu_envelope_sget_sender (env, &buf))
    buf = "UNKNOWN";
  mu_printf ("From %s ", buf);
  
  if (mu_envelope_sget_date (env, &buf))
    { 
      char datebuf[MU_DATETIME_FROM_LENGTH+1];
      time_t t;
      struct tm *tm;

      t = time (NULL);
      tm = gmtime (&t);
      mu_strftime (datebuf, sizeof datebuf, MU_DATETIME_FROM, tm);
      buf = datebuf;
    }

  mu_printf ("%s", buf);
  len = strlen (buf);
  if (len > 1 && buf[len-1] != '\n')
    mu_printf ("\n");
}

static void
hdr_print (mu_message_t msg)
{
  mu_header_t hdr;
  mu_stream_t str;
  
  mu_message_get_header (msg, &hdr);
  mu_header_get_streamref (hdr, &str);
  mu_stream_copy (mu_strout, str, 0, NULL);
  mu_stream_destroy (&str);
}

static void
body_print (mu_message_t msg)
{
  int rc;
  mu_body_t body;
  mu_stream_t str;
  
  mu_message_get_body (msg, &body);
  rc = mu_body_get_streamref (body, &str);
  if (rc)
    {
      mu_error (_("cannot get body stream: %s"), mu_strerror (rc));
      exit (EX_OSERR);
    }
  mu_stream_copy (mu_strout, str, 0, NULL);
  mu_stream_destroy (&str);
}

static void
message_store_stdout (mu_message_t msg, mu_mailbox_t mbx, mu_coord_t crd)
{
  env_print (msg);
  hdr_print (msg);
  body_print (msg);
  mu_printf ("\n");
}

/*
 * Display diagnostics with the location of the message part it
 * belongs to.
 * 
 * Arguments:
 *   crd  -  A mu_coord_t object describing the location.
 *   dim  -  Number of significant positions in crd.
 *   fmt  -  Format string
 */
static void
crd_error (mu_coord_t crd, size_t n, char const *fmt, ...)
{
  va_list ap;
  char *pfx = mu_coord_part_string (crd, n);

  mu_diag_printf (MU_DIAG_ERROR, "%s: ", pfx);
  free (pfx);

  va_start (ap, fmt);
  mu_diag_cont_vprintf (fmt, ap);
  va_end (ap);
  mu_stream_write (mu_strerr, "\n", 1, NULL);
}

static inline int
is_address_header (char const *name)
{
  return !mu_c_strcasecmp (name, MU_HEADER_FROM) ||
    !mu_c_strcasecmp (name, MU_HEADER_TO) ||
    !mu_c_strcasecmp (name, MU_HEADER_CC) ||
    !mu_c_strcasecmp (name, MU_HEADER_BCC);
}

static int
qstring_needed (char const *s)
{
  for (; *s; s++)
    {
      if (mu_isascii (*s) && !mu_istspec (*s))
	continue;
      return 1;
    }
  return 0;
}

static void
qstring_format (mu_stream_t stream, char const *s)
{
  if (!s)
    return;
  if (qstring_needed (s))
    {
      char const *cp;
  
      mu_stream_write (stream, "\"", 1, NULL);
      while (*(cp = mu_str_skip_cset_comp (s, "\\\"")))
	{
	  mu_stream_write (stream, s, cp - s, NULL);
	  mu_stream_write (stream, "\\", 1, NULL);
	  mu_stream_write (stream, cp, 1, NULL);
	  s = cp + 1;
	}
      if (*s)
	mu_stream_write (stream, s, strlen (s), NULL);
      mu_stream_write (stream, "\"", 1, NULL);
    }
  else
    mu_stream_write (stream, s, strlen (s), NULL);
}
  
static int
address_decode (char const *name, char const *value, char const *charset,
		mu_header_t newhdr)
{
  int rc;
  mu_address_t addr;
  mu_stream_t mstr;
  mu_transport_t trans[2];
  
  rc = mu_memory_stream_create (&mstr, MU_STREAM_RDWR);
  if (rc)
    return rc;
  
  rc = mu_address_create (&addr, value);
  if (rc == 0)
    {
      mu_address_t cur;
      for (cur = addr; cur; cur = cur->next)
	{
	  char *s;
	  
	  rc = mu_rfc2047_decode (charset, cur->personal, &s);
	  if (rc == 0)
	    {
	      qstring_format (mstr, s);
	      free (s);
	    }
	  else
	    qstring_format (mstr, cur->personal);
	  mu_stream_printf (mstr, " <%s>", cur->email);
	  if (cur->next)
	    mu_stream_write (mstr, ", ", 2, NULL);
	}
      mu_stream_write (mstr, "", 1, NULL);
      rc = mu_stream_err (mstr);
      if (rc == 0)
	{
	  mu_stream_ioctl (mstr, MU_IOCTL_TRANSPORT,
			   MU_IOCTL_OP_GET,
			   trans);
	  mu_header_append (newhdr, name, (char*)trans[0]);
	}
      mu_stream_destroy (&mstr);
      mu_address_destroy (&addr);
    }
  return rc;
}

/*
 * Decode a single message or message part.
 *
 * Arguments:
 *   msg  -  Message or message part.
 *   crd  -  Pointer to mu_coord_t object that keeps its location.
 *   dim  -  Number of significant positions in crd.  If it is 1,
 *           msg is the message.  If it is greater than 1, msg is
 *           part of a MIME message.
 *
 * The function can reallocate crd to increase its actual dimension.
 * It can modify the coordinate positions starting from dim+1 (inclusive).
 */
static mu_message_t
message_decode (mu_message_t msg, mu_coord_t *crd, size_t dim)
{
  mu_message_t newmsg;
  int ismime = 0;
  int rc;

  mu_message_is_multipart (msg, &ismime);
  if (!ismime)
    {
      mu_stream_t str;
      
      rc = message_body_stream (msg, from_filter, charset, &str);
      if (rc)
	{
	  mu_message_ref (msg);
	  return msg;
	}
      else
	{
	  mu_body_t body;
	  mu_stream_t bstr;
	  mu_header_t hdr, newhdr;
	  mu_iterator_t itr;
	  size_t i;
	  char *content_type = NULL;
	  mu_stream_stat_buffer stat;
	  
	  mu_message_create (&newmsg, NULL);
	  mu_message_get_body (newmsg, &body);
	  mu_body_get_streamref (body, &bstr);
	  mu_stream_set_stat (bstr,
			      MU_STREAM_STAT_MASK (MU_STREAM_STAT_IN8BIT),
			      stat);
	  rc = mu_stream_copy (bstr, str, 0, NULL);
	  if (rc)
	    {
	      crd_error (*crd, dim, "mu_stream_copy: %s", mu_strerror (rc));
	      if (mu_stream_err (bstr))
		{
		  exit (EX_IOERR);
		}
	      else
		{
		  mu_stream_printf (bstr,
				    "\n[decodemail: content decoding failed: %s]\n",
				    mu_strerror (rc));
		}
	    }
	  mu_stream_unref (bstr);
	  mu_stream_unref (str);

	  mu_message_get_header (msg, &hdr);
	  mu_message_get_header (newmsg, &newhdr);
	  mu_header_get_iterator (hdr, &itr);
	  
	  for (mu_iterator_first (itr), i = 1; !mu_iterator_is_done (itr);
	       mu_iterator_next (itr), i++)
	    {
	      const char *name;
	      const char *value;
	      char *s;
	      
	      rc = mu_iterator_current_kv (itr, (void const **) &name,
					   (void**)&value);

	      if (!mu_c_strcasecmp (name, MU_HEADER_CONTENT_TYPE))
		{
		  if (charset)
		    {
		      mu_content_type_t ct;
		      struct mu_mime_param **pparam;
		      char *vc = mu_strdup (value);
		      size_t len;
		      mu_string_unfold (vc, &len);
		      rc = mu_content_type_parse_ext (vc, NULL,
						      MU_CONTENT_TYPE_RELAXED |
						      MU_CONTENT_TYPE_PARAM,
						      &ct);
		      if (rc)
			{
			  crd_error (*crd, dim,
				     "mu_content_type_parse_ext(%s): %s",
				     vc, mu_strerror (rc));
			  free (vc);
			  continue;
			}
		      free (vc);
		      rc = mu_assoc_install_ref (ct->param, "charset", &pparam);
		      switch (rc)
			{
			case 0:
			  *pparam = mu_zalloc (sizeof **pparam);
			  break;

			case MU_ERR_EXISTS:
			  free ((*pparam)->value);
			  break;

			default:
			  crd_error (*crd, dim, "mu_assoc_install_ref: %s",
				     mu_strerror (rc));
			  exit (EX_IOERR);
			}
		      (*pparam)->value = mu_strdup (charset);
		      mu_content_type_format (ct, &content_type);
		      mu_content_type_destroy (&ct);
		      continue;
		    }
		}
	      else if (!mu_c_strcasecmp (name,
					 MU_HEADER_CONTENT_TRANSFER_ENCODING))
		continue;
	      else if (is_address_header (name))
		{
		  if (address_decode (name, value, charset, newhdr))
		    mu_header_append (newhdr, name, value);
		  continue;
		}
	      
	      rc = mu_rfc2047_decode (charset, value, &s);
	      if (rc == 0)
		{
		  mu_header_append (newhdr, name, s);
		  free (s);
		}
	      else
		mu_header_append (newhdr, name, value);
	    }
	  mu_iterator_destroy (&itr);

	  mu_header_set_value (newhdr,
			       MU_HEADER_CONTENT_TRANSFER_ENCODING,
			       stat[MU_STREAM_STAT_IN8BIT] ? "8bit" : "7bit",
			       1);
	  if (charset)
	    {
	      if (!content_type)
		mu_asprintf (&content_type, "text/plain; charset=%s", charset);
	      mu_header_set_value (newhdr,
				   MU_HEADER_CONTENT_TYPE,
				   content_type,
				   1);
	      free (content_type);
	    }
	}
    }
  else
    {
      size_t nparts, i;
      mu_mime_t mime;
      mu_header_t hdr, newhdr;
      mu_iterator_t itr;
      char *s;
      mu_content_type_t ct;
      
      /* FIXME: The following could be simplified if we could obtain
	 a mime object from the message */
      mu_message_get_header (msg, &hdr);
      rc = mu_header_aget_value_unfold (hdr, MU_HEADER_CONTENT_TYPE, &s);
      if (rc)
	{
	  crd_error (*crd, dim, "mu_header_aget_value_unfold(%s): %s",
		     MU_HEADER_CONTENT_TYPE, mu_strerror (rc));
	  mu_message_ref (msg);
	  return msg;
	}
      rc = mu_content_type_parse_ext (s, NULL,
				      MU_CONTENT_TYPE_RELAXED |
				      MU_CONTENT_TYPE_PARAM, &ct);
      if (rc)
	{
	  crd_error (*crd, dim, "mu_content_type_parse_ext(%s): %s",
		     s, mu_strerror (rc));
	  free (s);
	  mu_message_ref (msg);
	  return msg;
	}
      free (s);

      if (!ct->subtype)
	{
	  mu_content_type_destroy (&ct);
	  mu_message_ref (msg);
	  return msg;
	}
      
      ++dim;
      if (dim > mu_coord_length (*crd))
	{
	  rc = mu_coord_realloc (crd, dim);
	  if (rc)
	    mu_alloc_die ();
	}
      
      mu_mime_create_multipart (&mime, ct->subtype, ct->param);
      mu_content_type_destroy (&ct);
      mu_message_get_num_parts (msg, &nparts);

      for (i = 1; i <= nparts; i++)
	{
	  mu_message_t msgpart, msgdec;
	  
	  (*crd)[dim] = i;
	  mu_message_get_part (msg, i, &msgpart);
	  
	  msgdec = message_decode (msgpart, crd, dim);
	  mu_mime_add_part (mime, msgdec);
	  mu_message_unref (msgdec);
	}
      --dim;
      
      mu_mime_to_message (mime, &newmsg);
      mu_mime_unref (mime);
      
      /* Copy headers */
      mu_message_get_header (newmsg, &newhdr);
      mu_header_get_iterator (hdr, &itr);

      for (mu_iterator_first (itr), i = 1; !mu_iterator_is_done (itr);
	   mu_iterator_next (itr), i++)
	{
	  const char *name;
	  const char *value;
	  char *s;
      
	  rc = mu_iterator_current_kv (itr, (void const **) &name,
				       (void**)&value);

	  if (mu_c_strcasecmp (name, MU_HEADER_MIME_VERSION) == 0 ||
	      mu_c_strcasecmp (name, MU_HEADER_CONTENT_TYPE) == 0)
	    continue;
	  else if (is_address_header (name))
	    {
	      if (address_decode (name, value, charset, newhdr))
		mu_header_append (newhdr, name, value);
	      continue;
	    }
	  rc = mu_rfc2047_decode (charset, value, &s);
	  if (rc == 0)
	    {
	      mu_header_append (newhdr, name, s);
	      free (s);
	    }
	  else
	    mu_header_append (newhdr, name, value);
	}
      mu_iterator_destroy (&itr);
    }

  if (dim == 1)
    {
      /* Copy envelope */
      mu_envelope_t env, newenv;

      rc = mu_message_get_envelope (msg, &env);
      if (rc == 0)
	{
	  char *sender = NULL, *date = NULL;
	  if ((rc = mu_envelope_aget_sender (env, &sender)) != 0)
	    {
	      crd_error (*crd, dim, "mu_envelope_aget_sender: %s",
			 mu_strerror (rc));
	    }
	  else if ((rc = mu_envelope_aget_date (env, &date)) != 0)
	    {
	      free (sender);
	      sender = NULL;
	      crd_error (*crd, dim, "mu_envelope_aget_date: %s",
			 mu_strerror (rc));
	    }
	  
	  if (sender)
	    {
	      if ((rc = mu_envelope_create (&newenv, newmsg)) == 0)
		{
		  newenv->sender = sender;
		  newenv->date = date;
		  mu_message_set_envelope (newmsg, newenv,
					   mu_message_get_owner (newmsg));
		}
	      else
		{
		  free (sender);
		  free (date);
		  crd_error (*crd, dim, "mu_envelope_create: %s",
			     mu_strerror (rc));
		}
	    }
	}
      else
	mu_diag_funcall (MU_DIAG_ERROR, "mu_message_get_envelope",
			 NULL, rc);
    }
  
  return newmsg;
}

