/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#include <config.h>
#include <assert.h>
#include <stdlib.h>
#include "common.h"
#include "dfm.h"
#include "error.h"
#include "expr.h"
#include "file-handle.h"
#include "misc.h"
#include "lexer.h"
#include "str.h"
#include "var.h"
#include "vfm.h"
#include "inpt-pgm.h"

#undef DEBUGGING
/*#define DEBUGGING 1*/
#include "debug-print.h"

/* A bit-vector of two-bit entries.  The array tells INPUT PROGRAM how
   to initialize each `value'.  Modified by envector(), devector(), which
   are called by create_variable(), also by LEAVE, COMPUTE(!).  */
unsigned char *inp_init;

/* Number of bytes allocated for inp_init. */
size_t inp_init_size;

/* Number of `values' created inside INPUT PROGRAM. */
static int inp_nval;

static int end_case_trns_proc (any_trns *, ccase *);
static int end_file_trns_proc (any_trns * t, ccase * c);
static int reread_trns_proc (any_trns *, ccase *);
static void reread_trns_free (any_trns *);

int
cmd_input_program (void)
{
  match_id (INPUT);
  match_id (PROGRAM);
  discard_variables ();

  vfm_source = &input_program_source;

  inp_init = NULL;
  inp_init_size = 0;

  if (token != '.')
    syntax_error (_("expecting end of command"));
  return 1;
}

int
cmd_end_input_program (void)
{
  match_id (END);
  match_id (INPUT);
  match_id (PROGRAM);

  if (vfm_source != &input_program_source)
    return msg (SE, _("No matching INPUT PROGRAM command."));
  if (default_dict.nval == 0)
    msg (SW, _("No data-input or transformation commands specified "
	 "between INPUT PROGRAM and END INPUT PROGRAM."));

  /* Mark the boundary between INPUT PROGRAM and more-mundane
     transformations. */
  f_trns = n_trns;

  /* Mark the boundary between input program `values' and
     later-created `values'. */
  inp_nval = default_dict.nval;

  if (token != '.')
    syntax_error (_("expecting end of command"));
  return 1;
}

/* Initializes temp_case.  Called before the first case is read. */
static void
init_case (void)
{
  value *val = temp_case->data;
  unsigned char *cp = inp_init;
  unsigned char c;
  int i, j;

  /* This code is 2-3X the complexity it might be, but I felt like
     it.  It initializes temp_case values to 0, or SYSMIS, or
     blanks, as appropriate. */
  for (i = 0; i < inp_nval / 4; i++)
    {
      c = *cp++;
      for (j = 0; j < 4; j++)
	{
	  switch (c & INP_MASK)
	    {
	    case INP_NUMERIC | INP_RIGHT:
	      val++->f = SYSMIS;
	      break;
	    case INP_NUMERIC | INP_LEFT:
	      val++->f = 0.0;
	      break;
	    case INP_STRING | INP_RIGHT:
	    case INP_STRING | INP_LEFT:
	      memset (val++->s, ' ', MAX_SHORT_STRING);
	      break;
	    }
	  c >>= 2;
	}
    }
  if (inp_nval % 4)
    {
      c = *cp;
      for (j = 0; j < inp_nval % 4; j++)
	{
	  switch (c & INP_MASK)
	    {
	    case INP_NUMERIC | INP_RIGHT:
	      val++->f = SYSMIS;
	      break;
	    case INP_NUMERIC | INP_LEFT:
	      val++->f = 0.0;
	      break;
	    case INP_STRING | INP_RIGHT:
	    case INP_STRING | INP_LEFT:
	      memset (val++->s, ' ', MAX_SHORT_STRING);
	      break;
	    }
	  c >>= 2;
	}
    }
}

/* Clears temp_case.  Called between reading successive records. */
static void
clear_case (void)
{
  value *val = temp_case->data;
  unsigned char *cp = inp_init;
  unsigned char c;
  int i, j;

  /* This code is 2-3X the complexity it might be, but I felt like
     it.  It initializes temp_case values to SYSMIS, or
     blanks, or does nothing, as appropriate. */
  for (i = 0; i < inp_nval / 4; i++)
    {
      c = *cp++;
      for (j = 0; j < 4; j++)
	{
	  if (!(c & INP_LEFT))
	    {
	      if (c & INP_STRING)
		memset (val->s, ' ', MAX_SHORT_STRING);
	      else
		val->f = SYSMIS;
	    }
	  val++;
	  c >>= 2;
	}
    }
  
  if (inp_nval % 4)
    {
      c = *cp;
      for (j = 0; j < inp_nval % 4; j++)
	{
	  if (!(c & INP_LEFT))
	    {
	      if (c & INP_STRING)
		memset (val->s, ' ', MAX_SHORT_STRING);
	      else
		val->f = SYSMIS;
	    }
	  val++;
	  c >>= 2;
	}
    }
}

/* Executes each transformation in turn on a `blank' case.  When a
   transformation fails, returning -2, then that's the end of the
   file.  -1 means go on to the next transformation.  Otherwise the
   return value is the index of the transformation to go to next. */
void
input_program_source_read (void)
{
  int i;

  /* Nonzero if there were any END CASE commands in the set of
     transformations. */
  int end_case = 0;

  /* We don't automatically write out cases if the user took over
     that prerogative.  */
  for (i = 0; i < f_trns; i++)
    if (t_trns[i]->proc == end_case_trns_proc)
      end_case = 1;

  init_case ();
  while (1)
    {
      /* Index of current transformation. */
      int i;

      /* Return value of last-called transformation. */
      int code;

      debug_printf (("input-program: "));

      /* Perform transformations on `blank' case. */
      for (i = 0; i < f_trns;)
	{
#if DEBUGGING
	  printf ("/%d", i);
	  if (t_trns[i]->proc == end_case_trns_proc)
	    printf ("\n");
#endif
	  code = t_trns[i]->proc (t_trns[i], temp_case);
	  switch (code)
	    {
	    case -1:
	      i++;
	      break;
	    case -2:
	      return;
	    case -3:
	      goto next_case;
	    default:
	      i = code;
	      break;
	    }
	}

#if DEBUGGING
      if (!end_case)
	printf ("\n");
#endif

      /* Write the case if appropriate. */
      if (!end_case)
	if (!write_case ())
	  return;

      /* Blank out the case for the next iteration. */
    next_case:
      clear_case ();
    }
}

static void
input_program_source_destroy_source (void)
{
  cancel_transformations ();
  free (inp_init);
  inp_init = NULL;
}

case_stream input_program_source =
  {
    NULL,
    input_program_source_read,
    NULL,
    NULL,
    input_program_source_destroy_source,
    NULL,
    "INPUT PROGRAM",
  };

int
cmd_end_case (void)
{
  trns_header *t;

  match_id (END);
  match_id (CASE);
  if (token != '.')
    return syntax_error (_("expecting end of command"));

  if (vfm_source != &input_program_source)
    return msg (SE, _("This command may only be executed between INPUT PROGRAM "
		"and END INPUT PROGRAM."));

  t = xmalloc (sizeof (trns_header));
  t->proc = end_case_trns_proc;
  t->free = NULL;
  add_transformation ((any_trns *) t);

  return 1;
}

int
end_case_trns_proc (unused any_trns * t, unused ccase * c)
{
#if DEBUGGING
  printf ("END CASE\n");
#endif
  if (!write_case ())
    return -2;
  clear_case ();
  return -1;
}

/* REREAD transformation. */
typedef struct
  {
    trns_header h;

    file_handle *handle;	/* File to move file pointer back on. */
    struct expression *column;	/* Column to reset file pointer to. */
  }
reread_trns;

/* Parses REREAD command. */
int
cmd_reread (void)
{
  /* File to be re-read. */
  file_handle *h;
  
  /* Expression for column to set file pointer to. */
  expression *e;

  /* Created transformation. */
  reread_trns *t;

  match_id (REREAD);

  h = default_handle;
  e = NULL;
  while (token != '.')
    {
      if (match_id (COLUMN))
	{
	  match_tok ('=');
	  
	  if (e)
	    {
	      msg (SE, _("COLUMN subcommand multiply specified."));
	      free_expression (e);
	      return 0;
	    }
	  
	  e = parse_expression (PXP_NUMERIC);
	  if (!e)
	    return 0;
	}
      else if (match_id (FILE))
	{
	  match_tok ('=');
	  if (token != ID)
	    {
	      syntax_error (_("expecting file handle name"));
	      free_expression (e);
	      return 0;
	    }
	  h = fh_get_handle_by_name (tokstr);
	  if (!h)
	    {
	      free_expression (e);
	      return 0;
	    }
	  get_token ();
	}
      else
	{
	  syntax_error (NULL);
	  free_expression (e);
	}
    }

  t = xmalloc (sizeof (reread_trns));
  t->h.proc = reread_trns_proc;
  t->h.free = reread_trns_free;
  t->handle = h;
  t->column = e;
  add_transformation ((any_trns *) t);

  return 1;
}

static int
reread_trns_proc (any_trns * _t, ccase * c)
{
  reread_trns *t = (reread_trns *) _t;

  if (t->column == NULL)
    dfm_bkwd_record (t->handle, 1);
  else
    {
      value column;

      evaluate_expression (t->column, c, &column);
      if (!finite (column.f) || column.f < 1)
	{
	  msg (SE, _("REREAD: Column numbers must be positive finite "
	       "numbers.  Column set to 1."));
	  dfm_bkwd_record (t->handle, 1);
	}
      else
	dfm_bkwd_record (t->handle, column.f);
    }
  return -1;
}

static void
reread_trns_free (any_trns * t)
{
  free_expression (((reread_trns *) t)->column);
}

/* Parses END FILE command. */
int
cmd_end_file (void)
{
  trns_header *t;

  match_id (END);
  match_id (FILE);
  if (token != '.')
    return syntax_error (_("expecting end of command"));

  if (vfm_source != &input_program_source)
    return msg (SE, _("This command may only be executed between INPUT PROGRAM "
		"and END INPUT PROGRAM."));

  t = xmalloc (sizeof (trns_header));
  t->proc = end_file_trns_proc;
  t->free = NULL;
  add_transformation ((any_trns *) t);

  return 1;
}

static int
end_file_trns_proc (unused any_trns * t, unused ccase * c)
{
#if DEBUGGING
  printf ("END FILE\n");
#endif
  return -2;
}
