
;;; jtw-build-jtw.el --- A system for translating *.jtw -> *.java

;; Copyright (C) 2006-2011 Davin Pearson

;; Author/Maintainer: Davin Pearson <http://davin.50webs.com>
;; Keywords: J.T.W. Java
;; Package: dlisp (Davin's version of elisp)
;; Note: in files in the dlisp package, the d- prefix stands for _D_avin's customizations
;; Version: 2.0

;; This file is part of GNU Java Training Wheels.

;; Limitation of Warranty

;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or (at
;; your option) any later version.
;;
;; This program is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs, see the file COPYING.  If not, see:
;;
;; <http://www.gnu.org/licenses/gpl-3.0.txt>.


;;; Commentary:

;; This code compiles a *.jtw file into a *.java file, also giving
;; useful diagnostics on problematic *.jtw code.

;;; Known Bugs:

;; none so far!

;;; Code:

(require 'cl)

(if (not (boundp '*prefix*))
    (setq *prefix* "/usr"))

(assert (and (boundp '*prefix*) *prefix*))

(defmacro d-quote (&rest rest)
  )

;;(message "1")

;; (setq *def-dir* "~/jtw-tutorials/")
(setq *def-dir* default-directory)

(setq load-path (cons (concat *prefix* "/share/emacs/site-lisp/dlisp")
                      load-path))

;;(require 'emergency-bindings)
(require 'diagnose)
(require 'jtw-mode)
(require 'directory-files-deep)

;;(message "3")

(setq backup-inhibited t)

(defvar *pp-namespace* "#foomatic"
  "Namespace for pre-processor"
  )
(defvar *list-namespace* "#transalpine"
  "Namespace for file include stack")

(defvar *drive-spec* "\\([a-zA-Z]:/\\|~[-a-zA-Z0-9_]*/\\|~/\\|/\\|\\)"
  "Regexp for drive letters in Windows and GNU/Linux")

;; (checkpoint "foo %s" 123)
;; (setq msg "abc%s")
;; (setq rest '("abc" "def"))
(defun checkpoint (msg &rest rest)
  (message (apply 'format (format "*** checkpoint %s" msg) rest))
  )

(defadvice message (around jtw-cpp activate)
  (if (and format-string
           (not (string-match "^Mark set"                      format-string))
           (not (string-match "^Indenting region"              format-string))
           (not (string-match "^Loading "                      format-string))
           (not (string-match "^directory-files-deep scanning" format-string))
           (not (string-match "^Fontifying"                    format-string))
           )
      ad-do-it)
  )

;;(message "4")

;; (2- 10)
(defun 2- (&rest args)
  (apply '+ -2 args))

(defun n- (&rest args)
  (apply '+ -2 args))

(defun warn--find-enclosing-class ()
  (save-excursion
    (save-match-data
      (let (class-or-interface name)
        (when (re-search-backward (concat "^\\(\\(abstract\\|final\\|public\\)[ \t]+\\)*"
                                          "\\(class\\|interface\\)[ \t]+"
                                          "\\([A-Za-z0-9_]*\\)") nil t)
          (setq class-or-interface (buffer-substring-no-properties (match-beginning 3) (match-end 3)))
          (setq name               (buffer-substring-no-properties (match-beginning 4) (match-end 4)))
          ;;(debug "Survey said")
          (cons class-or-interface name))
        ))))

;; (warn--get-location)
;; (switch-to-buffer *current-buffer*)
(defun warn--get-location ()
  "Gets the location of the #foomatic declaration"
  (save-match-data
    (save-excursion
      (let (l p file line str-to-search)
        ;;(set-buffer *buf*)
        (beginning-of-line)
        (setq p (point))
        ;;(debug "We are a star")
        (setq str-to-search (concat "^[ \t]*//+ " *pp-namespace* " #location[0-9] (\\(" *drive-spec* "[-a-zA-Z0-9_+./]+\\):\\([0-9]+\\))"))
        ;;(debug "Raspberry Beret")
        (assert (and 1 (or (looking-at         str-to-search)
                           (re-search-backward str-to-search nil t))))
        (assert (and 2 (match-beginning 1)))
        (assert (and 3 (match-end 1)))
        (assert (and 4 (match-beginning 3)))
        (assert (and 5 (match-end 3)))
        (setq file (buffer-substring-no-properties           (match-beginning 1) (match-end 1)))
        (setq line (read-str (buffer-substring-no-properties (match-beginning 3) (match-end 3))))
        (assert (stringp  file))
        (assert (integerp line))
        (beginning-of-line)
        (setq l (n- (count-lines (point) p)))
        ;;(debug "Hopelessly devoted to you")
        (cons file (+ l line))))))

(setq warn--err-msgs nil)

(defun warn--log-message (msg &optional file line)
  (save-excursion
    (if (and file line)
        (setq warn--err-msgs (cons (list file line (format "*** Error (%s:%d) %s\n" file line msg))
                                   warn--err-msgs))
      (let ((n (warn--get-location)))
        ;;(debug "Purple Rain")
        (setq warn--err-msgs (cons (list (car n) (cdr n) (format "*** Error (%s:%d) %s\n" (car n) (cdr n) msg))
                                   warn--err-msgs))
        ) ;; END LET!
      ) ;; END IF!
    ))

(defun warn-equals (a b)
  (and (string= (nth 0 a) (nth 0 b))
       (= (nth 1 a) (nth 1 b))
       (string= (nth 2 a) (nth 2 b))))

(defun warn--pred (a b)
  ;;(debug "Two hearts beat as one")
  (let ((case-fold-search t))
    (if (string=-ignore-case (nth 0 a) (nth 0 b))
        (< (nth 1 a) (nth 1 b))
      (string<-ignore-case (nth 0 a) (nth 0 b)))))

;;; (warn--list-to-string '((12 . "12 abc") (22 . "22 abc")))
(defun warn--list-to-string (list)
  (let ((ptr list) (str ""))
    (while ptr
      ;;(debug "The Velvet Underground: Look into her false coloured eyes")
      (setq str (concat str (nth 2 (car ptr))))
      (setq ptr (cdr ptr)))
    str))

(defun warn--looking-at (&optional peek-amount)
  (if (not peek-amount)
      (setq peek-amount 20))
  (buffer-substring-no-properties (point) (warn--clamp-point (+ (point) peek-amount))))

(defmacro warn--safe-command (&rest cmd)
  `(condition-case err
       ,(cons 'progn cmd)
     (error )))

;; (warn--clamp-point -2)
(defun warn--clamp-point (newpoint)
  (max (min (point-max) newpoint) (point-min)))

(defun warn--inside-comment-or-string ()
  (save-match-data
    (let ((p (get-char-property (warn--clamp-point (1- (point))) 'face)))
      (or (eq p 'font-lock-string-face)
          (eq p 'font-lock-comment-face)
          (eq p 'font-lock-doc-face)
          (eq p 'font-lock-doc-string-face)
          (eq p 'd-face-super-comment)
          )))
  )

;; (warn--make-list 3)
(defun warn--make-list (n)
  (let ((list nil))
    (while (> n 0)
      (setq list (cons (cons n t) list))
      (decf n))
    list))

(defvar warn--list nil "Changed by warn--process-list")

(defun warn--goto-line (line)
  (goto-char (point-min))
  (forward-line (1- line)))

;; (warn--process-list)
(defun warn--process-list ()
  (save-excursion
    (let (ptr)
      (goto-char (point-max))
      (setq maxline (warn--what-line))
      (setq warn--list (warn--make-list maxline))
      ;;(setq warn--list nil)
      (setq ptr warn--list)
      ;;(debug "warn-list=%s" warn--list)
      (while ptr
        (when (car ptr)
          (warn--goto-line (caar ptr))
          (beginning-of-line)
          ;;(checkpoint "*** line=%d" (warn--what-line))
          (when (looking-at (concat "^   \\(\\(abstract\\|final\\|private\\|protected\\|public\\)"
                                    "[ \t]+\\)*"
                                    "\\(function\\|method\\|constructor\\)"))
            (setq abstract (save-excursion (beginning-of-line)
                                           (warn--re-search-forward--no-strings-no-comments
                                            "\\<abstract\\>"
                                            (point-at-eol) t)))
            (setcdr (car ptr) nil)
            (setq ptr (cdr ptr))
            (warn--goto-line (caar ptr))
            (when ptr
              ;;(checkpoint "*** smeg")
              ;;(checkpoint "*** abstract=%s" abstract)
              (when (not abstract)
                (while (and (not (looking-at "^   begin\\>")) ptr)
                  (setcdr (car ptr) nil)
                  (setq ptr (cdr ptr))
                  (when ptr
                    ;;(checkpoint "*** culling inner line=%d" (warn--what-line))
                    (warn--goto-line (caar ptr)))
                  )))))
        (setq ptr (cdr ptr)))
      (setq ptr warn--list)
      (while ptr
        (when (car ptr)
          (warn--goto-line (caar ptr))
          (beginning-of-line)
          (if (looking-at "^   \\(\\(final\\|private\\|protected\\|public\\)[ \t]+\\)*\\(property\\|classVar\\)\\>")
              (setcdr (car ptr) nil)))
        (setq ptr (cdr ptr)))
      ;; NOTE: Needed for keyword var missing
      (setq ptr warn--list)
      (while ptr
        (when (car ptr)
          (warn--goto-line (caar ptr))
          (beginning-of-line)
          (if (looking-at "^[ \t]*catch[ \t\r\n]*(")
              (setcdr (car ptr) nil)))
        (setq ptr (cdr ptr)))
      (setq ptr warn--list)
      (while ptr
        (when (car ptr)
          (warn--goto-line (caar ptr))
          (beginning-of-line)
          (if (looking-at "^   }")
              (setcdr (car ptr) nil)))
        (setq ptr (cdr ptr)))
      warn--list)))

;;(warn--are-we-editing-p (setq filename "ptr/Beer.jtw"))
(defun warn--are-we-editing-p (filename)
  (setq filename (expand-file-name filename))
  (let ((ptr    (buffer-list))
        (found nil))
    (while (and ptr (not found))
      (if (and os-type--mswindows-p
               (buffer-file-name (car ptr))
               (string=-ignore-case (buffer-file-name (car ptr)) filename))
          (setq found t)
        (if (string= (buffer-file-name (car ptr)) filename)
            (setq found t)))
      (setq ptr (cdr ptr)))
    found)
  )

(defun warn--find-enclosing-package ()
  (save-excursion
    (save-match-data
      (goto-char (point-min))
      (if (warn--re-search-forward--no-strings-no-comments "^package \\([a-z.]+\\);" nil t)
          (setq package (buffer-substring-no-properties (match-beginning 1) (match-end 1)))
        (setq package ""))
      package)
  ))

;; (setq filch (warn--build-method-database))
(defun warn--build-method-database ()
  (save-excursion
    (let (file class str type package method-name list ptr warn--master-list-primary e)
      (setq package "")
      (checkpoint "default-directory #frog=%s" default-directory)
      (checkpoint "*stump*=%s" *stump*)
      (setq list (directory-files-deep *def-dir* t "\\.jtw$"))
      (setq list (mapcar (function
                          (lambda (x)
                            (if (string-match "#" x)
                                nil x))) list))
      (setq list (delq nil list))
      (setq ptr list)
      ;;(checkpoint "**** #1 ptr=%s" ptr)
      ;;(checkpoint "**** #1 default-directory=%s" default-directory)
      (checkpoint "**** #1 *def-dir*=%s" *def-dir*)
      (checkpoint "**** find=%s" (find "pkg/inner/Apple.jtw" ptr :test 'string=))
      (while ptr
        (checkpoint "a")
        ;;(setq file "pkg/inner/Apple.jtw")
        (setq file (car ptr))
        (setq e (d-currently-editing-file file))
        (setq *current-buffer* (current-buffer))
        (switch-to-buffer *current-buffer*)
        ;;(if (string-match "^Beer.jtw$" file) (debug "#2b Found Beer.jtw"))
        ;;(if (string-match (regexp-quote "pkg/Beer.jtw") file)
        ;;    (debug "found Beer class"))
        ;;(debug "Nightsbridge")
        (checkpoint "b")
        ;;(checkpoint "file=%s" file)
        ;;(debug "file" file)
        (checkpoint "**** #2 default-directory=%s" default-directory)
        (if (string-match "pkg.inner.pkg" file) (debug "I can't dance#1"))
        (if (string-match "pkg.pkg"       file) (debug "I can't dance#2"))
        (if (string-match "pkg.pkg" file) (error "#13 file=%s" file))
        (find-file file)
        ;;(switch-to-buffer (find-file file))
        ;;(debug "Hellhound on my trail")
        (checkpoint "post default-directory=%s" default-directory)
        (checkpoint "post file=%s" file)
        (checkpoint "post *stump*=%s" *stump*)
        ;;(checkpoint "post (concat default-directory file)=%s" (concat default-directory file))
        ;;(debug "Carmina Burana")
        ;;(if (string= "pkg/inner/Apple.jtw" file)
        ;;    (debug "matched pkg/inner/Apple.jtw"))
        (goto-char (point-min))
        (beginning-of-line)
        (while (warn--re-search-forward--no-strings-no-comments
                (concat
                 "^   \\(\\(abstract\\|final\\|private\\|protected\\|public\\)"
                 "[ \t]+\\)*\\(method\\|property\\|function\\|classVar\\)\\>") nil t)
          (setq type (buffer-substring-no-properties (match-beginning 3) (match-end 3)))
          (skip-chars-forward " \t")
          (skip-chars-forward "a-zA-Z0-9")
          (skip-chars-forward " \t")
          (setq p (point))
          (skip-chars-forward "a-zA-Z0-9_")
          (if (looking-at "[ \t\r\n]*(")
              (setq was-bra t)
            (setq was-bra nil))
          ;;(checkpoint "5")
          (setq name (buffer-substring-no-properties p (point)))
          ;;(checkpoint "**** name=%s" name)
          ;;(checkpoint "2")
          (setq package (warn--find-enclosing-package))
          (setq class   (cdr (warn--find-enclosing-class)))

          ;;(checkpoint "**** class=%s" class)
          (if (string-match "^\\.[A-Z][a-zA-Z0-9_]*" class)
              (setq class (substring class 1)))
          ;;(checkpoint "3")
          (when (and (not (string-match "\\.$" package)) (not (string= package "")))
            (setq package (concat package ".")))
          (when (and package (stringp package) (not (string= package "")) (not (string= package ".")))
            (setq class (concat package class))
            )

          ;;(if (string= file "pkg/inner/Apple.jtw") (debug "Shake it up"))

          ;;(when (string-match "Apple" class)
          ;;  (d-kill-jtw-buffers)
          ;;  (debug "Thunderstruck"))
          (when (or (string= "pkg.inner" package) (string= "pkg.pkg.inner" package))
            ;;(d-kill-jtw-buffers)
            ;;(debug "Hotel California")
            ;;(debug "Just what I needed")
            )
          ;;(debug (format "That's fairly interesting name=%s" name))

          (when (not (assoc class warn--master-list-primary))
            (setq warn--master-list-primary (cons (list class '(functions) '(classVars) '(methods) '(properties)) warn--master-list-primary))
            ;;(setq warn--master-list-primary (cons (cons class name) warn--master-list-primary))
            ;;(setq warn--master-list-primary (cons (list class '(functions) '(classVars)) warn--master-list-primary))
            (assert warn--master-list-primary)
            )
          (when (string= class "Turtle")
            ;;(if (string= name "getName")  (debug "Wrong location"))
            ;;(if (string= name "insert")   (debug "Wrong location"))
            ;;(if (string= name "toString") (debug "Wrong location"))
            ;;(if (string= name "sayPile") (debug "Wrong location"))
            )
          (checkpoint "**** class=%s, name=%s" class name)
          ;;(if (string= class "pkg.inner.Apple")
          ;;    (debug "Thunderstruck"))
          (when name
            (checkpoint "1")
            ;;(setq class "Main")
            ;;(debug "I Can't Help Falling in Love With You")
            (setq cons (assoc class warn--master-list-primary))
            (setq cdr (cdr cons))
            (checkpoint "2")
            ;;(debug "A fallen angel")
            ;;(if (string= class "pkg.Beer") (debug "Teriyaki"))
            (setq alist (assoc class warn--master-list-primary))
            (when (or (string= type "function") (string= type "classVar"))
              (if was-bra
                  (setcdr (last (assq 'functions alist)) (cons name nil))
                (setcdr (last (assq 'classVars alist)) (cons name nil))))
            (when (or (string= type "method") (string= type "property"))
              (if was-bra
                  (setcdr (last (assq 'methods alist)) (cons name nil))
                (setcdr (last (assq 'properties alist)) (cons name nil))))
            )
          (forward-line 1)
          (beginning-of-line)
          )
        ;;(checkpoint "d")
        (if (not e) (kill-buffer))
        (setq ptr (cdr ptr)))
      ;;(debug "Calamansi")
      ;;(checkpoint "e")
      warn--master-list-primary
      )
    )
  )

;; (setq zeg (warn--cull-package-spec (setq file "pkg/Beer.jtw") zog))
;; (setq file "MyFirstProgram.jtw")
;; (setq warn--master-list nil)
(defun warn--cull-package-spec (file warn--master-list)
  (save-excursion
    (let ((e (warn--are-we-editing-p file))
          imported-packages package count
          imported-decl
          imported-packages
          ptr ptr2)
      ;; -------------------------------------------------------------
      (setq warn--master-list--secondary warn--master-list)
      (if (string-match "pkg.pkg" file) (error "#14 file=%s" file))
      (find-file file)
      (checkpoint "1 file=%s" file)
      (setq *current-buffer* (current-buffer))
      (assert (string-match "\\.jtw$" file))
      (setq file (substring file 0 (match-beginning 0)))
      (checkpoint "file=%s" file)
      (goto-char (point-min))
      (setq package (warn--find-enclosing-package))
      (checkpoint "2")
      ;;(setq package "pkg")
      (checkpoint "#1 package=%s" package)
      ;;(assert warn--master-list--secondary)
      (checkpoint "warn--master-list--secondary=%s" warn--master-list--secondary)
      ;;(debug "This could be love")
      (checkpoint "3")
      ;;(if (string-match "pkg/Wine" file) (debug "file matches pkg/Wine"))
      ;;(if (string= file "pkg/Beer") (debug "You make me feel like a natural woman"))
      ;;(if (not (string= "TurtleTest2" file)) (debug "calamansi"))
      (when (not (string= package ""))
        (setq ptr2 warn--master-list--secondary)
        (while ptr2
          (checkpoint "4")
          ;;(setq ptr2 (cdr ptr2))
          (setq class (caar ptr2))
          ;;(setq class "Wine")
          (when (string-match (concat package ".\\([A-Z][a-zA-Z0-9_]*\\)") class)
            ;;(debug "Axis: bold as love")
            (checkpoint "5")
            (setq new-class (substring class (match-beginning 1)))
            (setq old-class class)
            (setq old-list (car ptr2))
            (setq new-list (copy-list old-list))
            (setf (car new-list) new-class)
            (setq warn--master-list--secondary (cons new-list
                                                     warn--master-list--secondary))
            ;;(debug "Mary from Arkansaw")
            )
          (checkpoint "6")
          (setq ptr2 (cdr ptr2))))
      (checkpoint "7")
      ;;(debug "Frog and toad")
      ;;(debug "foo")
      ;;(sit-for 5)
      (goto-char (point-min))
      (setq str "^import[ \t]+\\([a-z.]+\\)\\.\\*;")
      (setq imported-packages nil)
      (beginning-of-line)
      (setq count 0)
      (checkpoint "8")
      (while (or (d-quote save-match-data
                   (beginning-of-line)
                   (if (looking-at str)
                       (setq import-decl (buffer-substring-no-properties
                                          (match-beginning 1)
                                          (match-end 1)))))
                 (save-match-data
                   (if (warn--re-search-forward--no-strings-no-comments str nil t)
                       (setq import-decl (buffer-substring-no-properties
                                          (match-beginning 1)
                                          (match-end 1))))))
        (setq imported-packages (cons import-decl imported-packages))
        (checkpoint "Dawn Upshaw %s" count)
        (incf count)
        )
      (checkpoint "9")
      ;;(debug "Oxygene")
      ;;(if imported-packages (debug (format "Sultans of swing imported-packages=%s" imported-packages)))
      (setq count 0)
      (setq ptr2 warn--master-list--secondary)
      ;;(debug "Why stop there?")
      (while ptr2
        (checkpoint "10")
        (setq class (caar ptr2))
        (if (not warn--master-list--secondary)
            (setq warn--master-list--secondary (cons (cons class nil) warn--master-list--secondary)))
        (if (not (assoc class warn--master-list--secondary))
            (setq warn--master-list--secondary (cons (list class) warn--master-list--secondary)))
        (setq ptr2 (cdr ptr2)))
      ;; -------------------------------------------------------------
      (checkpoint "11")
      (setq ptr2 warn--master-list--secondary)
      (while ptr2
        (checkpoint "12")
        (setq class (caar ptr2))
        ;;(if (string-match "Apple" class) (debug "Penelope"))
        (setq ptr imported-packages)
        (while ptr
          (when (and (caar ptr2) (car ptr))
            (setq import-package (car ptr))
            ;;(if (string= class "pkg.Beer") (debug "Beer class found"))
            (when (string-match (concat "^"
                                        (regexp-quote import-package)
                                        "\\.\\([A-Z][a-zA-Z0-9_]*\\)")
                                class)
              ;;(debug "(string-match")
              (if (and (match-beginning 1) (match-end 1))
                  (progn
                    (setq class-less-package-spec (substring class (match-beginning 1) (match-end 1)))
                    (setf old-list (car ptr2))
                    (setq new-list (copy-list old-list))
                    (setf (car new-list) class-less-package-spec)
                    (setq warn--master-list--secondary (cons new-list warn--master-list--secondary))
                    ;;(debug "Match for class")
                    )
                (debug "not (and (match-beginning 1) (match-end 1))")))
            )
          (setq ptr (cdr ptr)))
        (setq ptr2 (cdr ptr2)))
      (checkpoint "13")
      (checkpoint "#2 package=%s" package)
      (if (not e) (kill-buffer nil))
      (checkpoint "14")
      warn--master-list--secondary
      )))

;; (setq zog (warn--build))
(defun warn--build ()
  (message "*** beg warn--build-method-database")
  (let (file ptr list count (case-fold-search nil))
    ;;(debug "Mass in B minor")
    ;;(switch-to-buffer *buf*)
    (save-excursion
      (setq file (concat *stump* ".jtw"))
      (setq warn--master-list-thirdly (warn--build-method-database))
      ;;(debug "Moving into the street")
      (if (string= "Beer.jtw" file) (debug "#1b Found Beer.jtw"))
      (setq warn--master-list-thirdly (warn--cull-package-spec file warn--master-list-thirdly))
      ;;(debug "I am a lonesome hobo")
      ;;(assert warn--master-list-thirdly)
      ;;(debug (format "warn--master-list-thirdly=%s" warn--master-list-thirdly))
      ;;(debug "Friday on my mind")
      ;;(assert (string= (concat default-directory file) "c:/home/hairy-lemon/src/50webs-com/J.T.W/jtw-tutorials-here/MyFirstProgram.jtw"))
      (if (string-match "pkg.pkg" file) (error "#15 file=%s" file))
      (find-file file)
      (goto-char (point-min))
      (checkpoint "a")
      ;;(font-lock-fontify-buffer)
      ;;(if (not (string-match "Turtle" file))
      ;;    (debug "Nothing's gonna stop us now"))
      (while (re-search-forward "\\(^\\(public\\|private\\|protected\\|final\\|abstract\\)[ \t]+\\)*class \\([A-Z][a-zA-Z0-9_]*\\)" nil t)
        (save-excursion
          (checkpoint "b")
          (setq class (buffer-substring-no-properties (match-beginning 3) (match-end 3)))
          (checkpoint "c")
          (setq ptr2 warn--master-list-thirdly)
          (checkpoint "d")
          (setq count 0)
          (while ptr2
            (incf count)
            (checkpoint (format "e count=%d" count))
            (goto-char (point-min))
            (checkpoint "f")
            (font-lock-fontify-buffer)
            (while (warn--re-search-forward--no-strings-no-comments
                    "\\<\\([a-z.]+\\.\\)?\\([A-Z][a-zA-Z0-9_]*\\)\\(\\.[a-z][a-zA-Z0-9_]*\\)?\\((\\)?" nil t)
              ;;(checkpoint "g")
              (if (and (match-beginning 1) (match-end 1))
                  (setq package (match-string 1))
                (setq package ""))
              ;; (setq function-call "notHere")
              (if (and (match-beginning 4) (match-end 4))
                  (setq open-bra t)
                (setq open-bra nil))
              (if (and (match-beginning 3) (match-end 3))
                  (progn
                    (setq name (match-string 3))
                    (setq name (substring name 1))
                    ;;(debug "Newport Jazz Festival")
                    )
                (setq name ""))
              (setq class-internal (buffer-substring-no-properties
                                    (match-beginning 2)
                                    (match-end 2)))
              ;;(setq package "pkg.inner")
              ;;(setq class-internal "Apple")
              (if (and (not (string= "" package))
                       (not (string-match "^[a-z.]+\\.$" package)))
                  (setq package (concat package ".")))
              (if (string= package "")
                  (setq package-and-class class-internal)
                (setq package-and-class (concat package class-internal)))
              (when (and (not (string-match "^RuntimeException$" package-and-class))
                         (not (string= package-and-class      "System"))
                         (not (string= package-and-class      class-internal))
                         (not (string-match "^java\\.lang\\." package))
                         (not (string-match "^java\\.io\\."   package))
                         (not (string-match "^java\\.util\\." package))
                         (not (string-match "^javax\\."       package))
                         )
                (if (and (string= package-and-class "pkg.Wine")
                         (string= name "foo"))
                    t;;(debug "Unit Structures")
                  )
                (if (and (not (string= "pkg.inner.Apple" package-and-class))
                         (not (string= "pkg.Beer" package-and-class)))
                         ;;(not (string= "Apple" package-and-class))
                         ;;(not (string= "Error4" package-and-class)))
                         t;;(debug "Mr State Trooper")
                  )
                (when (not (assoc package-and-class warn--master-list-thirdly))
                  (warn--log-message (format "Error 43: Illegal package name %s" package-and-class) file (warn--what-line))
                  )
                ;;(debug "Hells Bells")
                ;;(setq package-and-class "pkg.Beer")
                ;;(setq function-call "sexy")
                ;;(if (string-match "sexy" function-call) (debug "Wagner's Ring Cycle"))
                (when (and name (not (string= name "")) (assoc package-and-class warn--master-list-thirdly))
                  ;;(if (string= name "noSuchMethod")
                  ;;    (debug "Diminuendo in blue"))
                  (setq f-func (find name (cdr (assq 'functions (assoc package-and-class warn--master-list-thirdly))) :test 'string=))
                  (setq f-vrbl (find name (cdr (assq 'classVars (assoc package-and-class warn--master-list-thirdly))) :test 'string=))
                  (if (and was-bra (not f-func))
                      (warn--log-message (format "Error 44: Class %s has no function called %s" package-and-class name) file (warn--what-line)))
                  (if (and (not was-bra) (not f-vrbl))
                      (warn--log-message (format "Error 45: Class %s has no classVar called %s" package-and-class name) file (warn--what-line)))
                  )))
            ;;(debug "Newport Up")
            (checkpoint "h")
            (goto-char (point-min))
            ;;(debug "Don't play that song")
            (setq ptr2 (cdr ptr2)))))
      ;; NOTE: end of ptr list...
      (checkpoint "k")
      (message "*** end warn--build-method-database")
      ;;(setq warn--master-list-thirdly (delete-duplicates warn--err-msgs :test 'warn-equals))
      ;;(debug "Jesus he knows me")
      warn--master-list-thirdly
      )))

(defun warn--skip-whitespace ()
  (let ((done nil))
    (while (not done)
      (cond
       ((looking-at "[\t\r\n ]")
        (skip-chars-forward "\t\r\n "))
       ((looking-at "//")   (re-search-forward "$"    nil t))
       ((looking-at "/\\*") (re-search-forward "\\*/" nil t))
       (t
        (setq done t))))))

(defun warn--splat-quest (str)
  (save-match-data
    (when (not (eq nil str))
      (while (string-match "\\<QUEST\\>" str)
        (setq str (concat (substring str 0 (match-beginning 0))
                          "?"
                          (substring str (match-end 0)))))
      (while (string-match "\\\\" str)
        ;;(debug "Radiohead: Lucky")
        (setq str (concat (substring str 0 (match-beginning 0))
                          (substring str (match-end 0)))
              ))
      )
    str))

;; (warn--splat-quest (setq str "\"abc\""))
;; (warn--cull-quotes (setq str "\"abc\""))
(defun warn--cull-quotes (str)
  (when (and str (stringp str) (> (length str) 0))
    (when (= (aref str 0) ?\")
      (setq str (substring str 1)))
    (when (= (aref str (1- (length str))) ?\")
      (setq str (substring str 0 (1- (length str)))))
    )
  str)

(defun warn--buffer-to-string ()
  (buffer-substring-no-properties (point-min) (point-max)))

(defun warn--current-line-as-string ()
  (buffer-substring (point-at-bol) (point-at-eol)))

(defun warn--what-line ()
  "Different from what-line in that it returns the result rather
than printing it"
  (save-excursion
    (beginning-of-line)
    (1+ (count-lines 1 (point)))))

(defun warn--re-search-backward--no-strings-no-comments (str &optional bound noerror)
  (let (done get (found t) (p (point)))
    ;;(checkpoint "* sit-for 0.1")
    ;;(sit-for 0.1)
    (while (not done)
      (if (not (re-search-backward str bound noerror))
          (progn
            (if (eq noerror t)
                (goto-char p)
              (goto-char (point-min)))
            (setq found nil)
            (setq done t))
        (setq get (get-text-property (warn--clamp-point (1- (point))) 'face))
        (if (and (not (eq get 'font-lock-string-face))
                 (not (eq get 'font-lock-comment-face))
                 (not (eq get 'font-lock-doc-face))
                 (not (eq get 'font-lock-doc-string-face))
                 (not (eq get 'd-face-super-comment)))
            (setq done t)
          (setq done nil))))
    found))

;; (re-search-forward "amber" nil nil)
;; (warn--re-search-forward--no-strings-no-comments "use")
(defun warn--re-search-forward--no-strings-no-comments (str &optional bound noerror)
  "TODO: add this code to ~/c2java/patch-c2j.el
I am a comment on how to use this function"
  (let (done get (found 'sexy) (p (point)))
    ;;(checkpoint "* sit-for 0.1")
    ;;(sit-for 0.1)
    (while (not done)
      (if (not (re-search-forward str bound noerror))
          (progn
            (if (eq noerror t)
                (goto-char p)
              (goto-char (point-max)))
            (setq found nil)
            (setq done t))
        (setq get (get-text-property (warn--clamp-point (1- (point))) 'face))
        (if (and (not (eq get 'font-lock-string-face))
                 (not (eq get 'd-face-super-comment))
                 (not (eq get 'font-lock-comment-face))
                 (not (eq get 'font-lock-doc-face))
                 (not (eq get 'font-lock-doc-string-face)))
            (progn
              (setq found t)
              (setq done t))
          )))
    (assert (not (eq found 'sexy)))
    ;; NOTE: no save-match-data
    found))



;; (d-read-line-pre)
(defun d-read-line-pre ()
  (let (cb)
    (setq cb (current-buffer))
    (unwind-protect
        (progn
          (setq buf-name "*jtw-cpp*")
          ;;(checkpoint "Vampire Weekend")
          (if (get-buffer buf-name)
              (kill-buffer buf-name))
          (setq *buf-read-char* (generate-new-buffer buf-name))
          (set-buffer *buf-read-char*)
          (insert-file-contents (concat *def-dir* *stump* ".jtw"))
          (goto-char (point-min)))
      (set-buffer cb))))

;; (warn--read-line)
(defun d-read-line ()
  (let (cb line)
    (setq cb (current-buffer))
    (unwind-protect
        (progn
          (set-buffer *buf-read-char*)
          (if (eobp)
              (error "Eobp")
            (setq line (warn--current-line-as-string))
            (forward-line 1)
            ))
      (set-buffer cb))
    line))

(defadvice d-indent-new-comment-line (around jtw2java activate)
  ;;(debug "splat")
  (save-match-data
    (let (file line)
      (if (save-excursion
            (beginning-of-line)
            ;;(debug "Split for the city")
            (warn--re-search-forward--no-strings-no-comments (concat "^[ \t]*//+ " *pp-namespace* "#location[0-9] "
                                                               "(\\(" *drive-spec* "[-a-zA-Z0-9_./+]*\\):\\([0-9]+\\))")
                                                       (point-at-eol) t))
          (progn
            ;;(debug "Famous last words")
            (setq file (buffer-substring-no-properties (match-beginning 1) (match-end 1)))
            (setq line (read-str (buffer-substring-no-properties (match-beginning 2) (match-end 2))))
            (if (string-match "pkg.pkg" file) (error "#16 file=%s" file))
            (find-file file)
            (warn--goto-line line))
        ;;(debug "Frogger")
        ad-do-it))))

;; (setq file "Code.java")
(defadvice newline (around jtw2java activate)
  ;;(d-beeps "foo")
  (save-match-data
    (let (file line)
      (if (save-excursion
            (beginning-of-line)
            (warn--re-search-forward--no-strings-no-comments "^\\*\\*\\* Error (\\([a-zA-Z_][-a-zA-Z0-9_.]*\\):\\([0-9]+\\))"
                                                       (point-at-eol) t))
          (progn
            (setq file (buffer-substring-no-properties (match-beginning 1) (match-end 1)))
            (setq line (read-str (buffer-substring-no-properties (match-beginning 2) (match-end 2))))
            ;;(debug "Swords and Knives")
            (if (get-buffer file)
                (set-buffer (get-buffer file))
              (if (string-match "pkg.pkg" file) (error "#17 file=%s" file))
              (find-file file))
            (warn--goto-line line))
        ad-do-it))))

(defun warn--get-transalpine ()
  (save-excursion
    (assert (re-search-backward *list-namespace* nil t))
    (skip-chars-forward "#")
    (skip-chars-forward "a-z")
    (setq *current-buffer* (current-buffer))
    (eval (read (current-buffer)))))

;;(message "pre doit")

(defun doit ()
  (interactive)
  (let (new-file
        beg end class m-call c-call include-file buf buf2 count str
        was-error buf-name file-stack file line ptr ptr2)
    (checkpoint "1")
    (progn
      ;;(checkpoint "*stump*=%s" *stump*)
      ;;(checkpoint "(concat *def-dir* *stump*)=%s" (concat *def-dir* *stump*))
      (setq d-face-cc-global 'd-face-cc-global)
      (setq d-face-cc-digits 'd-face-cc-digits)
      (message "#68 d-face-cc-global=%s" d-face-cc-global)
      (message "#69 *def-dir*=%s" *def-dir*)
      (setq *buf* (find-file (concat *def-dir* *stump* ".java")))
      ;;(switch-to-buffer *buf*)
      ;;(debug "Malted Milk")
      (set-buffer *buf*)
      (checkpoint "**** #4 *def-dir*=%s"         *def-dir*)
      (checkpoint "**** #4 *def-dir*=%s" *def-dir*)
      (read-only-mode -1)
      (erase-buffer)
      (setq *buf-old* (find-file (concat *def-dir* *stump* ".jtw")))
      (set-buffer *buf-old*)
      ;;(switch-to-buffer *buf-old*)
      ;;(debug "Milkcow's Calf Blues")
      (setq str (buffer-substring-no-properties (point-min) (point-max)))
      ;;(checkpoint "str=%s" str)
      ;;(checkpoint "(buffer-file-name *buf-old*)=%s" (buffer-file-name *buf-old*))
      ;;(checkpoint "(buffer-file-name *buf*)=%s" (buffer-file-name *buf*))
      ;;(message "sit-for 1")
      ;;(sit-for 1)
      ;;(set-buffer-modified-p nil)
      ;;(kill-buffer *buf-old*)
      (set-buffer *buf*)
      (insert str)
      (goto-char (point-min))
      (setq file-stack (list (cons (concat *stump* ".jtw") 1)))
      (insert "/** Automatically generated file. Do not edit! */\n")
      (insert (format "// %s (setq file-stack '%s)\n" *list-namespace* (prin1-to-string file-stack)))
      (insert (format "// %s #location0 (%s:%d)\n"    *pp-namespace* (concat *stump* ".jtw") 1))
      )
    ;;(debug "I would die for you")
    (checkpoint "2")
    ;;(error "Franz Schubert")
    (progn
      (set-buffer *buf*)
      ;;(switch-to-buffer *buf*)
      (goto-char (point-min))
      (setq file-stack (list (cons (concat *stump* ".jtw") 1)))
      (checkpoint "3")
      (while (warn--re-search-forward--no-strings-no-comments "^[ \t]*include \"\\([^\"\t\r\n]+\\)\";" nil t)
        ;;(error "Found include")
        (checkpoint "4")
        (save-excursion
          (checkpoint "5")
          (setq file-to-include (buffer-substring-no-properties (match-beginning 1) (match-end 1)))
          (setq line-to-include 1)
          (warn--get-transalpine)
          ;;(setq file-stack nil)
          ;;(debug "Sniffer dog")
          (setq file-stack (cons (cons file-to-include line-to-include) file-stack))
          (beginning-of-line)
          (if (warn--inside-comment-or-string)
              (debug (format "file=%s" file)))
          (delete-region (point-at-bol) (point-at-eol))
          (setq p (point))
          ;;(setq line-old (d-what-line))
          ;;(setq file-old (file-name-nondirectory (buffer-file-name)))
          ;;(assert (string-match "\\.java$" file-old))
          ;;(setq file-old (concat (substring file-old 0 (match-beginning 0)) ".jtw"))
          ;;(error (format "file-old=%s" file-old))
          ;;(insert (format "// %s #location0 (%s:%d)\n" *pp-namespace* file-old line-old))
          (setq n (warn--get-location))
          (setq file-old (car n))
          (setq line-old (cdr n))
          (setq p-prior (save-excursion (warn--goto-line line-old) (point)))
          (setq dif (n- (count-lines p-prior (point))))
          (setq line-new (+ line-old dif))
          (checkpoint "6")
          ;;(if (= (length file-stack) 3) (debug "Ain't got the power anymore"))
          (progn
            (setq file (caar file-stack))
            (setq line (cdar file-stack))
            (assert (string= file file-to-include))
            )
          (progn
            (checkpoint "7")
            (setq ptr (assoc file file-stack))
            (setq ptr2 file-stack)
            (while ptr2
              (when (eq (car ptr2) ptr)
                (setq ptr2 (cdr ptr2))
                ;;(debug "If the devil made me here")
                (setf (cdar ptr2) line-new)
                (setq ptr2 nil))
              (setq ptr2 (cdr ptr2))))
          ;;(debug "I would do time for you")
          ;; ---------------------------------------------------------
          (checkpoint "8")
          (insert (format "// #1%s (setq file-stack '%s)\n" *list-namespace* (prin1-to-string file-stack)))
          (insert (format "// %s #location1 (%s:%d)\n" *pp-namespace* file line))
          ;;(assert (string= file-old (caar (last file-stack))))
          ;;(assert (string= file-old (concat *stump* ".jtw")))
          (setq m (make-marker))
          (set-marker m (1+ (point)))
          (insert-file-contents file-to-include)
          (save-excursion
            ;;(debug "Carmina Burana")
            (goto-char m)
            (setq file-stack (cdr file-stack))
            (if file-stack
                (progn
                  (setq file (caar file-stack))
                  (setq line (cdar file-stack))
                  (assert (string= file (caar file-stack)))
                  (assert (= line (cdar file-stack)))
                  (assert (stringp file))
                  (assert (numberp line)))
              (setq file-stack (list (cons (concat *stump* ".jtw") 1)))
              (setq file (caar file-stack))
              (setq line (cdar file-stack))
              ;;(debug "Cecil Taylor")
              )
            ;;(setq line (+ line (count-lines p (point))))
            ;;(error "Cassandra Wilson")
            ;;(debug "Aladdin Sane")
            (assert (string= file (caar file-stack)))
            ;;(if (not (= line (cdar file-stack))) (error "Schmu"))
            (insert (format "// #2%s (setq file-stack '%s)\n" *list-namespace* (prin1-to-string file-stack)))
            (insert (format "// %s #location2 (%s:%d)\n" *pp-namespace* file line))
            (set-marker m nil)
            )))
        )

    ;;(debug "Gangster Rapp")

    (checkpoint "**** #9 *def-dir*=%s" *def-dir*)
    (set-buffer *buf*)
    ;;(switch-to-buffer *buf*)
    (checkpoint "**** #8 *def-dir*=%s" *def-dir*)

    (checkpoint "doit:-6")

    (progn
      ;;(switch-to-buffer *buf*)
      (set-buffer *buf*)
      (goto-char (point-min))
      (font-lock-mode 1)
      ;;(font-lock-fontify-buffer)
      ;;(error "Come as you are")
      )

    ;;(debug "The kind you find in a second hand store")

    (if (fboundp 'jtw-mode)
        (jtw-mode)
      (java-mode))
    (if (fboundp 'jtw--meta-control-backslash)
        (jtw--meta-control-backslash)
      (c-set-offset 'substatement-open 0)
      (setq c-basic-offset 3)
      (indent-region (point-min) (point-max)))
    (setq warn--err-msgs nil)
    (setq warn--outside-of-class-error nil)
    (setq font-lock-keywords-case-fold-search nil)

    (setq *current-buffer* (current-buffer))
    ;;(switch-to-buffer *current-buffer*)

    ;;(debug "Grandmaster Flash")

    (checkpoint "**** #7 *def-dir*=%s" *def-dir*)
    (progn
      (setq warn--master-list-fourthly (warn--build))
      ;;(warn--kill-jtw-buffers)
      )
    (checkpoint "warn--master-list-fourthly=%s" warn--master-list-fourthly)
    ;;(debug "No son of mine")
    (progn
      (goto-char (point-min))
      (while (warn--re-search-forward--no-strings-no-comments
              "^\\(\\(public\\|private\\|protected\\|final\\|abstract\\)[ \t]+\\)*function" nil t)
        ;;(debug "Van Morrison")
        (setq warn--outside-of-class-error t)
        (warn--log-message "Error 38: Function outside of class")))

    (checkpoint "doit:-4")

    (progn
      (goto-char (point-min))
      (while (warn--re-search-forward--no-strings-no-comments "^\\(\\(public\\|private\\|protected\\|final\\|abstract\\)[ \t]+\\)*method" nil t)
        ;;(debug "Ronan Keating")
        (setq warn--outside-of-class-error t)
        (warn--log-message "Error 39: Method outside of class")))

    (checkpoint "doit:-3")

    (progn
      (goto-char (point-min))
      (while (warn--re-search-forward--no-strings-no-comments "^\\(\\(public\\|private\\|protected\\|final\\|abstract\\)[ \t]+\\)*property" nil t)
        (setq warn--outside-of-class-error t)
        (warn--log-message "Error 40: Property outside of class")))

    (checkpoint "doit:-2")

    (progn
      (goto-char (point-min))
      (while (warn--re-search-forward--no-strings-no-comments "^\\(\\(public\\|private\\|protected\\|final\\|abstract\\)[ \t]+\\)*classVar" nil t)
        (setq warn--outside-of-class-error t)
        (warn--log-message "Error 41: classVar outside of class")))

    (checkpoint "doit:-1")

    ;;(error "Texas radio and the big beat")

    (progn
      (set-buffer *buf*)
      (warn--process-list))

    (checkpoint "doit:0")

    (save-excursion
      (set-buffer *buf*)
      (goto-char (point-min))
      (while (warn--re-search-forward--no-strings-no-comments "\\<begin\\>" nil t)
        (save-excursion
          (forward-char -1)
          (when (not (warn--inside-comment-or-string))
            (delete-region (match-beginning 0) (match-end 0))
            (insert "{")))))

    (checkpoint "doit:1")

    (save-excursion
      (goto-char (point-min))
      (while (warn--re-search-forward--no-strings-no-comments "elseif" nil t)
        (replace-match "else if" 'fixedcase)))

    (save-excursion
      (goto-char (point-min))
      (while (warn--re-search-forward--no-strings-no-comments "\\<beginMain\\>" nil t)
        (save-excursion
          (forward-char -1)
          (when (not (warn--inside-comment-or-string))
            (delete-region (match-beginning 0) (match-end 0))
            (insert "public static void main(String[] args)\n   {")))))

    (checkpoint "doit:2")

    (save-excursion
      (goto-char (point-min))
      (while (warn--re-search-forward--no-strings-no-comments "\\<end\\>" nil t)
        (save-excursion
          (forward-char -1)
          (when (not (warn--inside-comment-or-string))
            (delete-region (match-beginning 0) (match-end 0))
            (insert "}")))))

    (checkpoint "doit:3")

    (save-excursion
      (goto-char (point-min))
      (while (warn--re-search-forward--no-strings-no-comments "\\<endMain\\>" nil t)
        (save-excursion
          (forward-char -1)
          (when (not (warn--inside-comment-or-string))
            (delete-region (match-beginning 0) (match-end 0))
            (insert "}")))))

    (checkpoint "doit:4")
    ;;(checkpoint "buffer-name=%s" (buffer-name))

    

    (setq indent-tabs-mode nil)
    (java-mode)
    (setq c-basic-offset 3)
    (c-set-offset 'substatement-open 0)
    ;;(switch-to-buffer (current-buffer))
    (setq c-basic-offset 3)
    (setq indent-tabs-mode nil)
    (font-lock-fontify-buffer)
    ;;(debug "Walkin' Blues")
    (indent-region (point-min) (point-max))

    ;;(debug "Friday on my mind")

    (let (p1 p2 str form type variable T var start stop step-size step-size-2
             this_start this_stop this_step this_step_size file line p-prior
             beg0 end0)
      (setq strobe nil)
      (checkpoint "2")
      (save-excursion
        (goto-char (point-min))
        (setq *superfor* 0)
        (while (re-search-forward "\\<superfor\\>" nil t)
          (setq beg0 (match-beginning 0))
          (setq end0 (match-end 0))
          ;;(checkpoint "sitting for 1 seconds...")
          (font-lock-fontify-buffer)
          ;;(sit-for 1)
          (when (save-excursion
                  (save-match-data
                    (re-search-forward "(" (point-at-eol) t)
                    (forward-char -1)
                    (re-search-forward "\\<var\\>" nil t)
                    (not (warn--inside-comment-or-string))))
            ;;(error "Smelly cat")
            (setq *current-buffer* (current-buffer))
            ;;(switch-to-buffer *current-buffer*)
            (setq p1 beg0)
            (assert (save-match-data
                      (looking-at "[ \t\r\n]*(")))
            (setq p2 (save-excursion
                       (forward-sexp 1)
                       (point)))
            (setq str (buffer-substring-no-properties end0 p2))
            ;;(checkpoint "str=%s" str)
            (setq form (read-str str))
            ;;(checkpoint "form=%s" form)
            ;;(debug "form")
            (assert (consp form))
            (message "*** form=%s" form)
            ;;(error "Rolling Stones plays Cuba")
            (delete-region p1 p2)
            (incf *superfor*)
            (setq this (format "superfor_%d_" *superfor*))
            (when (not (eq (nth 0 form) 'var))
              (warn--log-message "Error 35: Keyword var missing from superfor construct")
              (when (and (not (eq (nth 1 form) 'char))
                         (not (eq (nth 1 form) 'short))
                         (not (eq (nth 1 form) 'int))
                         (not (eq (nth 1 form) 'long))
                         (not (eq (nth 1 form) 'float))
                         (not (eq (nth 1 form) 'double)))
                (warn--log-message (concat
                                    "Error 37:#1 argument type to superfor macro must be"
                                    " one of char/short/int/long/float/double"))))
            (when (eq (nth 0 form) 'var)
              (if (and (not (eq (nth 1 form) 'char))
                       (not (eq (nth 1 form) 'short))
                       (not (eq (nth 1 form) 'int))
                       (not (eq (nth 1 form) 'long))
                       (not (eq (nth 1 form) 'float))
                       (not (eq (nth 1 form) 'double)))
                  (warn--log-message (concat
                                      "Error 37:#2 argument type to superfor macro must be"
                                      " one of char/short/int/long/float/double")))
              (progn
                ;;(debug "Radiohead: Let Down")
                ;; (setq form '(var int i=0 to stop))
                ;; (setq form '(var int i =0 to stop))
                ;; (setq form '(var int i = 0 to stop))
                (setq type     (nth 1 form))
                (setq T        (prin1-to-string type))
                (setq variable (prin1-to-string (nth 2 form)))
                (if (string-match "=" variable)
                    (progn
                      (setq pre-red-str--variable (substring variable 0 (match-beginning 0)))
                      ;;(string-match "=" variable-equals)
                      (setq pre-red-str--start    (substring variable (match-end 0)))
                      ;;(debug "Radiohead: Fitter Happier")
                      ;;(debug "Jean Jarre: Equinoxe Part III")
                      (setq variable (and (not (string= "" pre-red-str--variable))
                                          (prin1-to-string (read-str pre-red-str--variable))))
                      (setq start (and (not (string= "" pre-red-str--start))
                                       (prin1-to-string (read-str pre-red-str--start))))
                      (cond
                       ((eq (nth 3 form) nil)
                        (setq start (concat start "()"))
                        (setq strobe t)
                        (debug "Queen: One Vision")
                        (cond
                         ((eq (nth 4 form) 'to)
                          (setq to 'to))
                         ((eq (nth 4 form) 'downto)
                          (setq to 'downto))
                         (t
                          (debug "Pretenders: Space Invader")))
                        (setq stop (prin1-to-string (nth 5 form)))
                        (cond
                         ((and (eq (nth 6 form) nil) (> (length form) 6))
                          (setq stop (concat stop "()"))
                          ;;(debug "The Pretenders: Kid")
                          (if (eq (nth 7 form) 'step)
                              (setq step-size (nth 8 form))))
                         ((string-match "(" (prin1-to-string (nth 6 form)))
                          (setq stop (concat stop (and (nth 6 form) (prin1-to-string (nth 6 form)))))
                          (if (eq (nth 7 form) 'step)
                              (setq step-size (nth 8 form))))
                         ((eq (nth 6 form) 'step)
                          (debug "Pretenders: Private Life")
                          (setq step-size (nth 7 form)))
                         (t
                          ;;(debug "Queen: It's a Kind of Magic")
                          )
                         ) ;; END COND!
                        (if strobe (debug "Queen: One Year of Love"))
                        )
                       ((string-match "(" (prin1-to-string (nth 3 form)))
                        ;;(debug "Public Enemy: Letter to the New York Post")
                        (setq start (and (nth 3 form) (concat start (prin1-to-string (nth 3 form)))))
                        (cond
                         ((eq (nth 4 form) 'to)
                          (setq to 'to))
                         ((eq (nth 4 form) 'downto)
                          (setq to 'downto))
                         (t
                          (debug "Elvis: King Creole")))
                        (setq stop (prin1-to-string (nth 5 form)))
                        (cond
                         ((and (eq (nth 6 form) nil) (> (length form) 6))
                          (setq stop (concat stop "()"))
                          ;;(debug "The Pretenders: Kid")
                          (if (eq (nth 7 form) 'step)
                              (setq step-size (nth 8 form))))
                         ;; ------------------------------------------
                         ((string-match "(" (prin1-to-string (nth 6 form)))
                          (setq stop (concat stop (prin1-to-string (nth 6 form))))
                          (if (eq (nth 7 form) 'step)
                              (setq step-size (nth 8 form))))
                         ;; ------------------------------------------
                         ((eq (nth 6 form) 'step)
                          (debug "Pretenders: Private Life")
                          (setq step-size (nth 7 form)))
                         (t
                          ;;(debug "Queen: It's a Kind of Magic")
                          )
                         )
                        ((eq (nth 3 form) 'to)
                         (setq to 'to)
                         (setq stop (nth 4 form))
                         (if (eq (nth 5 form) 'step)
                             (setq step-size (nth 6 form))
                           )
                         ;;(debug "aaa")
                         )
                        ((eq (nth 3 form) 'downto)
                         (setq to 'downto)
                         (setq stop (nth 4 form))
                         (if (eq (nth 5 form) 'step)
                             (setq step-size (nth 6 form))
                           )
                         (debug "bbb")
                         )
                        (t
                         (debug "Dire Straits: The Bug")))
                       )
                      ;;(debug "Bach's Mass in B Minor: Et in terra pax")
                      )
                  (if (eq (nth 3 form) '=)
                      (setq start (and (nth 4 form) (prin1-to-string (nth 4 form))))
                    (setq   start (and (nth 3 form) (prin1-to-string (nth 3 form))))
                    )) ;; END IF!
                ;;(debug "Bach's Mass in B Minor: Kyrie eleison")
                (cond
                 ((eq (nth 3 form) 'to)
                  (setq to 'to)
                  (setq stop (prin1-to-string (nth 4 form)))
                  (if (and (eq (nth 5 form) nil) (> (length form) 5))
                      (setq stop (concat stop "()"))
                    (if (string-match "(" (prin1-to-string (nth 5 form)))
                        (setq stop (concat stop (prin1-to-string (nth 5 form))))
                      (if (eq (nth 5 form) 'step)
                          (setq step-size (prin1-to-string (nth 6 form))))))
                  )
                 ((eq (nth 3 form) 'downto)
                  (setq to 'downto)
                  (setq stop (prin1-to-string (nth 4 form)))
                  (if (and (eq (nth 5 form) nil) (> (length form) 5))
                      (setq stop (concat stop "()"))
                    (if (string-match "(" (prin1-to-string (nth 5 form)))
                        (setq stop (concat stop (prin1-to-string (nth 5 form))))
                      (if (eq (nth 5 form) 'step)
                          (setq step-size (prin1-to-string (nth 6 form))))))
                  )
                 (t
                  ;;(debug "Jean Michel Jarre: Oxygene III")))
                  ;;(debug "Joaquin Rodrigo")
                  (setq start (prin1-to-string (nth 4 form)))
                  (cond
                   ((eq (nth 5 form) 'to)
                    (setq to 'to))
                   ((eq (nth 5 form) 'downto)
                    (setq to 'downto))
                   (t
                    (debug "Dire Straits: Planet of New Orleans")))
                  (setq stop (prin1-to-string (nth 6 form)))
                  (if (eq (nth 7 form) 'step)
                      (progn
                        (setq step-size (and (nth 8 form) (prin1-to-string (nth 8 form))))
                        (assert (numberp (nth 8 form)))
                        )
                    )
                  )
                 )
                )
              ;;(debug "Rod Stewart's Maggie May")
              (progn
                (setq var         variable)
                (setq start-2     (warn--cull-quotes (warn--splat-quest start)))
                (setq stop-2      (warn--cull-quotes (warn--splat-quest stop)))
                (setq step-size-2 (warn--splat-quest (and step-size (prin1-to-string step-size))))
                ) ;; END PROGN!
              ;; -----------------------------------------------------
              ;;(debug "The Pretenders: Precious")
              (setq this_start     (concat this "start"))
              (setq this_stop      (concat this "stop"))
              (setq this_step      (concat this "step"))
              (setq this_step_size (concat this "step_size"))
              ;;(debug "My Parties")
              (insert (concat (concat "var " T " " this_start " = " start-2 "; ")
                              (concat "var " T " " this_stop  " = " stop-2  "; ")
                              (if step-size
                                  (concat "var " T " " this_step " = " step-size-2 "; "
                                          "var " T " " this_step_size " = "
                                          (cond
                                           ((eq to 'to)
                                            (concat "Math.abs(" this_step ")"))
                                           ((eq to 'downto)
                                            (concat "-Math.abs(" this_step ")"))
                                           (t
                                            (debug "Dire Straits: Heavy Fuel")))
                                          ";\n")
                                (concat "var " T " " this_step_size " = "
                                        (cond
                                         ((eq to 'to)
                                          "1")
                                         ((eq to 'downto)
                                          "-1")
                                         (t
                                          (debug "Dire Straits: Ticket to Heaven")))
                                        ) ;; END CONCAT!
                                )         ;; END IF!
                              ";\n")      ;; END CONCAT!
                      )                   ;; END INSERT!
              ;;(debug "Rod Stewart: Hot Legs")
              (setq line 0)
              (setq p-prior
                    (save-excursion
                      (beginning-of-line)
                      (setq str (concat "^[ \t]*//+ " *pp-namespace* " #location[0-9]"
                                        " (\\("       *drive-spec*   "[-a-zA-Z0-9_./]+\\):\\([0-9]+\\))"))
                      (if (or (looking-at str) (re-search-backward str nil t))
                          (progn
                            ;;(debug "Antonio Vivaldi")
                            (setq file (buffer-substring-no-properties           (match-beginning 1)
                                                                                 (match-end 1)))
                            (assert (stringp file))
                            (setq line (read-str (buffer-substring-no-properties (match-beginning 3)
                                                                                 (match-end 3))))
                            (assert (integerp line))
                            (point)
                            )
                        (setq file (concat *def-dir* *stump* ".jtw"))
                        (setq line 1)
                        (goto-char (point-min))
                        (forward-line 2)
                        (point)
                        )))
              (setq line (+ line (count-lines p-prior (point))))
              (decf line)
              (decf line)
              (insert (format "// %s (setq file-stack '%s)\n" *list-namespace* (prin1-to-string file-stack)))
              (insert (format "// %s #location3 (%s:%d)\n"    *pp-namespace*   file line))
              (insert (concat "for (var " T " " var " = " this_start ";"
                              " ((" this_step_size " > 0) ? " var " <= "
                              this_stop " : " var " >= " this_stop "); "
                              var " += " this_step_size ")"))
              (if strobe (debug "Pretenders: The Wait"))
              ;;(debug "Yehudi Menuhin")
              )))))

    ;;(debug "Gareth Farr's Owhiro")

    (progn

      (checkpoint "3")
      (save-excursion
        ;;(font-lock-fontify-buffer)
        (goto-char (point-min))
        (while (warn--re-search-forward--no-strings-no-comments (concat "^   \\(\\(abstract\\|final\\|"
                                                                        "private\\|protected\\|public\\)"
                                                                  "[ \t]+\\)*method\\>") nil t)
          (if (not (looking-at "[ \t]+[A-Za-z_][A-Za-z0-9_]*[ \t]+[A-Za-z_][A-Za-z0-9_]*("))
              (warn--log-message "Error 1: Method needs a return type"))))

      (checkpoint "4")

      (save-excursion
        ;;(font-lock-fontify-buffer)
        (goto-char (point-min))
        (while (warn--re-search-forward--no-strings-no-comments "^[ \t]*\\([a-z]+\\.\\)*[A-Z][a-zA-Z0-9_]*[ \t]+[a-z][a-zA-Z0-9_]*" nil t)
          (save-excursion
            (forward-line -1)
            (warn--log-message "Error 35: Keyword var missing"))))

      (checkpoint "doit:6")

      (save-excursion
        ;;(font-lock-fontify-buffer)
        (goto-char (point-min))
        (while (warn--re-search-forward--no-strings-no-comments (concat "^   \\(\\(abstract\\|final\\|"
                                                                  "private\\|protected\\|public\\)"
                                                                  "[ \t]+\\)*function\\>") nil t)
          (if (string= "interface" (car (warn--find-enclosing-class)))
              (warn--log-message "Error 42: Function cannot be present in an interface")
            (if (not (looking-at "[ \t]+[A-Za-z_][A-Za-z0-9_]*[ \t]+[A-Za-z_][A-Za-z0-9_]*("))
                (warn--log-message "Error 2: Function needs a return type")))))

      (checkpoint "doit:7")

      (save-excursion
        ;;(font-lock-fontify-buffer)
        (goto-char (point-min))
        (while (warn--re-search-forward--no-strings-no-comments "constructor[ \t]+\\([A-Za-z0-9]*\\)" nil t)
          (when (not (warn--inside-comment-or-string))
            (setq ctor-class (buffer-substring-no-properties (match-beginning 1) (match-end 1)))
            (setq encl-class (cdr (warn--find-enclosing-class)))
            (if (not (string= ctor-class encl-class))
                (warn--log-message (format "Error 3: Constructor class should be %s" encl-class)))
            ;;(debug "Vanilla")
            )))

      ;;(checkpoint "doit:8")

      (save-excursion
        ;;(checkpoint "sitting for 2 seconds...")
        ;;(font-lock-fontify-buffer)
        (goto-char (point-min))
        (while (< (point) (point-max))
          (beginning-of-line)
          (font-lock-fontify-buffer)
          (when (warn--re-search-forward--no-strings-no-comments "\\<constructor\\>" (point-at-eol) t)
            (when (not (warn--inside-comment-or-string))
              ;;(checkpoint "sitting for 1 second...")
              ;;(sit-for 1)
              (beginning-of-line)
              (if (warn--re-search-forward--no-strings-no-comments "\\<property\\>" (point-at-eol) t)
                  (if (not (warn--inside-comment-or-string))
                      (warn--log-message (concat "Error 4: One line cannot have"
                                                 " constructor and property"
                                                 " at the same time"))))
              (beginning-of-line)
              (if (warn--re-search-forward--no-strings-no-comments "\\<classVar\\>" (point-at-eol) t)
                  (if (not (warn--inside-comment-or-string))
                      (warn--log-message (concat "Error 5: One line cannot have"
                                                 " constructor and classVar"
                                                 " at the same time"))))
              (beginning-of-line)
              (if (warn--re-search-forward--no-strings-no-comments "\\<method\\>" (point-at-eol) t)
                  (if (not (warn--inside-comment-or-string))
                      (warn--log-message (concat "Error 6: One line cannot have"
                                                 " constructor and method"
                                                 " at the same time"))))
              (beginning-of-line)
              (if (warn--re-search-forward--no-strings-no-comments "\\<function\\>" (point-at-eol) t)
                  (if (not (warn--inside-comment-or-string))
                      (warn--log-message (concat "Error 7: One line cannot have"
                                                 " constructor and function at"
                                                 " the same time"))))))
          ;;(font-lock-fontify-buffer)
          (when (warn--re-search-forward--no-strings-no-comments "\\<property\\>" (point-at-eol) t)
            ;;(checkpoint "sitting for 1 second...")
            ;;(sit-for 1)
            (when (not (warn--inside-comment-or-string))
              (beginning-of-line)
              (if (warn--re-search-forward--no-strings-no-comments "\\<classVar\\>" (point-at-eol) t)
                  (if (not (warn--inside-comment-or-string))
                      (warn--log-message (concat "Error 8: One line cannot have"
                                                 " property and classVar at the"
                                                 " same time"))))
              (progn
                ;;(font-lock-fontify-buffer)
                (beginning-of-line)
                (if (warn--re-search-forward--no-strings-no-comments "\\<method\\>" (point-at-eol) t)
                    (if (not (warn--inside-comment-or-string))
                        (warn--log-message (concat "Error 9: One line cannot have"
                                                   " property and method at the"
                                                   " same time")))))
              (progn
                ;;(font-lock-fontify-buffer)
                (beginning-of-line)
                (if (warn--re-search-forward--no-strings-no-comments "\\<function\\>" (point-at-eol) t)
                    (if (not (warn--inside-comment-or-string))
                        (warn--log-message (concat "Error 10: One line cannot have"
                                                   " property and function at the"
                                                   " same time")))))))
          (beginning-of-line)
          ;;(font-lock-fontify-buffer)
          (when (warn--re-search-forward--no-strings-no-comments "\\<classVar\\>" (point-at-eol) t)
            (when (not (warn--inside-comment-or-string))
              (beginning-of-line)
              (if (warn--re-search-forward--no-strings-no-comments "\\<method\\>" (point-at-eol) t)
                  (if (not (warn--inside-comment-or-string))
                      (warn--log-message (concat "Error 11: One line cannot have"
                                                 " classVar and method at the"
                                                 " same time"))))
              (beginning-of-line)
              (if (warn--re-search-forward--no-strings-no-comments "\\<function\\>" (point-at-eol) t)
                  (if (not (warn--inside-comment-or-string))
                      (warn--log-message (concat "Error 12: One line cannot have"
                                                 " classVar and function at the"
                                                 " same time"))))))
          ;;(font-lock-fontify-buffer)
          (beginning-of-line)
          (when (warn--re-search-forward--no-strings-no-comments "\\<method\\>" (point-at-eol) t)
            (when (not (warn--inside-comment-or-string))
              (beginning-of-line)
              (if (warn--re-search-forward--no-strings-no-comments "\\<function\\>" (point-at-eol) t)
                  (if (not (warn--inside-comment-or-string))
                      (warn--log-message (concat "Error 13: One line cannot have"
                                                 " method and function at the"
                                                 " same time"))))))
          (forward-line 1)))

      (checkpoint "doit:9")

      (save-excursion
        ;;(font-lock-fontify-buffer)
        (goto-char (point-min))
        (while (warn--re-search-forward--no-strings-no-comments "^   \\([a-z]+\\)" nil t)
          (when (and (not (save-excursion
                            (beginning-of-line)
                            (and (warn--re-search-forward--no-strings-no-comments "\\<function\\>" (point-at-eol) t)
                                 (not (warn--inside-comment-or-string)))))
                     (not (save-excursion
                            (beginning-of-line)
                            (and (warn--re-search-forward--no-strings-no-comments "\\<method\\>" (point-at-eol) t)
                                 (not (warn--inside-comment-or-string)))))
                     (not (save-excursion
                            (beginning-of-line)
                            (and (warn--re-search-forward--no-strings-no-comments "\\<classVar\\>" (point-at-eol) t)
                                 (not (warn--inside-comment-or-string)))))
                     (not (save-excursion
                            (beginning-of-line)
                            (and (warn--re-search-forward--no-strings-no-comments "\\<property\\>" (point-at-eol) t)
                                 (not (warn--inside-comment-or-string)))))
                     (not (save-excursion
                            (beginning-of-line)
                            (and (warn--re-search-forward--no-strings-no-comments "\\<constructor\\>" (point-at-eol) t)
                                 (not (warn--inside-comment-or-string)))))
                     (not (save-excursion
                            (beginning-of-line)
                            (looking-at (concat "^   public static void main("
                                                "String\\[\\] [a-z][a-zA-Z_]+)")))))
            (save-excursion
              (beginning-of-line)
              (checkpoint "doit:9b")
              (warn--log-message (concat "Error 14: This line needs one"
                                         " of the following keywords: "
                                         "function/method/classVar/"
                                         "property/constructor"))))))

      (when (not warn--outside-of-class-error)
        (progn
          (goto-char (point-min))
          ;;(checkpoint "sit-for 1")
          ;;(font-lock-fontify-buffer)
          ;;(sit-for 1)
          (while (warn--re-search-forward--no-strings-no-comments "\\<function\\>" nil t)
            ;;(beginning-of-line)
            (setq *current-buffer* (current-buffer))
            ;;(switch-to-buffer *current-buffer*)
            ;;(error "Elevate me mama peek=%s" (warn--looking-at 50))
            (save-excursion
              (when (and (not (save-excursion
                                (forward-char -2)
                                (warn--inside-comment-or-string)))
                         (string= "class" (car (warn--find-enclosing-class))))
                (beginning-of-line)
                (when (warn--re-search-forward--no-strings-no-comments "(" (point-at-eol) t)
                  ;;(checkpoint "looking-at=%s" (warn--looking-at 50))
                  (backward-char 1)
                  (assert (and 'where-ever-i-may-roam (looking-at "(")))
                  (warn--safe-command (forward-sexp))
                  (beginning-of-line)
                  (forward-line 1)
                  ;;(checkpoint "sit-for 1")
                  ;;(sit-for 1)
                  (if (not (looking-at "^   {"))
                      (if (looking-at "^[ \t]+{")
                          (warn--log-message (concat "Error 15: Functions cannot"
                                                     " reside inside functions/"
                                                     "methods/constructors"))
                        (warn--log-message (concat "Error 16: Function must have"
                                                   " begin on the following line"))))))))))

      (when (not warn--outside-of-class-error)
        ;;(font-lock-fontify-buffer)
        (save-excursion
          (goto-char (point-min))
          (while (warn--re-search-forward--no-strings-no-comments "\\<constructor\\>" nil t)
            (save-excursion
              (when (not (warn--inside-comment-or-string))
                (beginning-of-line)
                (warn--re-search-forward--no-strings-no-comments "(" nil t)
                (backward-char 1)
                ;;(setq *current-buffer* (current-buffer))
                ;;(switch-to-buffer *current-buffer*)
                (assert (and 1 (looking-at "(")))
                (warn--safe-command (forward-sexp))
                (beginning-of-line)
                (forward-line 1)
                (if (not (looking-at "^   {"))
                    (if (looking-at "^[ \t]+{")
                        (warn--log-message (concat "Error 17: Constructors cannot"
                                                   " reside inside functions/"
                                                   "methods/constructors"))
                      (warn--log-message (concat "Error 18: Constructor must"
                                                 " have begin on the following"
                                                 " line")))))))))

      (checkpoint "doit:12")

      ;;(font-lock-fontify-buffer)

      (when (not warn--outside-of-class-error)
        (save-excursion
          (goto-char (point-min))
          (while (warn--re-search-forward--no-strings-no-comments "\\<method\\>" nil t)
            ;;(switch-to-buffer *current-buffer*)
            ;; ---------------------------------------------------
            ;;(setq barf-buf (current-buffer))
            ;;(assert (warn--inside-comment-or-string))
            ;;(debug "All Apologies")
            ;; ---------------------------------------------------
            (save-excursion
              (when (not (warn--inside-comment-or-string))
                (setq cons (warn--find-enclosing-class))
                (when (and (string= (car cons) "class")
                           (not (save-excursion
                                  (beginning-of-line)
                                  (warn--re-search-forward--no-strings-no-comments "\\<abstract\\>"
                                                                             (point-at-eol) t))))
                  (beginning-of-line)
                  (warn--re-search-forward--no-strings-no-comments "(" nil t)
                  (backward-char 1)

                  (assert (and 2 (looking-at "(")))
                  (warn--safe-command (forward-sexp))
                  (beginning-of-line)
                  (forward-line 1)
                  (if (not (looking-at "^   {"))
                      (if (looking-at "^   [ \t]+{")
                          (warn--log-message (concat "Error 19: Methods cannot"
                                                     " reside inside functions/"
                                                     "methods/constructors"))
                        (warn--log-message (concat "Error 20: Method must have"
                                                   " begin on the following line")
                                           )))))))))

      ;;(checkpoint "doit:13")

      (save-excursion
        ;;(font-lock-fontify-buffer)
        (goto-char (point-min))
        (while (warn--re-search-forward--no-strings-no-comments "\\<property\\>" nil t)
          (when (not (warn--inside-comment-or-string))
            (beginning-of-line)
            (forward-line 1)
            (if (looking-at "^[ \t]+{")
                (warn--log-message (concat "Error 21: Property must not have"
                                           " begin on the following line"))))))

      ;;(checkpoint "doit:14")

      (save-excursion
        (goto-char (point-min))
        (while (warn--re-search-forward--no-strings-no-comments "\\<classVar\\>" nil t)
          (when (not (warn--inside-comment-or-string))
            (beginning-of-line)
            (forward-line 1)
            (if (looking-at "^[ \t]+{")
                (warn--log-message (concat "Error 22: Class variable must not"
                                           " have begin on the following line"))))))

      ;;(checkpoint "doit:15")

      (save-excursion
        ;;(font-lock-fontify-buffer)
        (goto-char (point-min))
        (while (warn--re-search-forward--no-strings-no-comments "if[ \t]+" nil t)
          (when (not (warn--inside-comment-or-string))
            (if (not (looking-at "("))
                (warn--log-message "Error 23: Expecting ( after if statement"))
            (condition-case err
                (forward-sexp 1)
              (error (warn--log-message (concat "Error 24: Unbalanced"
                                                " parentheses after if"
                                                " statement"))))
            ;;(skip-chars-forward " \t\r\n")
            (warn--skip-whitespace)
            (if (not (looking-at "\\<then\\>"))
                (warn--log-message (concat "Error 25: Expecting then keyword"
                                           " after if statement"))))))

      ;;(checkpoint "doit:15")

      (save-excursion
        ;;(font-lock-fontify-buffer)
        (goto-char (point-min))
        (setq count 0)
        (while (warn--re-search-forward--no-strings-no-comments "[{}]" nil t)
          (save-excursion
            ;;(debug "Got my mojo working")
            ;;(forward-char -1)
            (if (warn--inside-comment-or-string)
                (error "warn--inside-comment-or-string")
              (setq str (buffer-substring-no-properties (match-beginning 0) (match-end 0)))
              ;;(sit-for 1)
              (if (string= str "{") (incf count))
              (if (string= str "}") (decf count))
              (if (< count 0)
                  (warn--log-message "Error 26: More ends than begins")))
            ))
        (when (> count 0)
          (goto-char (point-max))
          (warn--log-message "Error 27: Missing ends at the end of the file")
          ;;(debug "Sunday bloody Sunday")
          )
        )

      ;;(checkpoint "doit:16")

      (save-excursion
        ;;(font-lock-fontify-buffer)
        (goto-char (point-min))
        (while (warn--re-search-forward--no-strings-no-comments (concat "^   \\(\\(abstract\\|final\\|"
                                                                  "private\\|protected\\|public\\)"
                                                                  "[ \t]+\\)*method\\>") nil t)
          (setq end (save-excursion
                      (warn--re-search-forward--no-strings-no-comments ")" nil t)
                      (end-of-line)
                      (point)))
          (if (and (string= (car (warn--find-enclosing-class)) "class")
                   (save-excursion
                     (beginning-of-line)
                     (not (warn--re-search-forward--no-strings-no-comments "\\<abstract\\>" (point-at-eol) t)))
                   (save-excursion
                     (beginning-of-line)
                     (warn--re-search-forward--no-strings-no-comments ";" end t)))
              (warn--log-message "Error 28: Spurious semicolon at the end of the line"))))

      ;;(checkpoint "doit:17")

      (save-excursion
        ;;(font-lock-fontify-buffer)
        (goto-char (point-min))
        (when (warn--re-search-forward--no-strings-no-comments "\\<public static void main(String\\[\\] args) {" nil t)
          ;;(checkpoint "doit::17")
          (beginning-of-line)
          (setq beg (point))
          (setq end (save-excursion
                      (warn--re-search-forward--no-strings-no-comments "{" nil t)
                      (condition-case err
                          (forward-sexp 1)
                        (error ))
                      (point)))
          ;;(checkpoint "doit::18")
          (setq class (cdr (warn--find-enclosing-class)))
          ;;(checkpoint "doit::19")
          (while (warn--re-search-forward--no-strings-no-comments "[^.a-zA-Z0-9_]\\([a-z][A-Za-z0-9_]*\\)[ \t]*(" end t)
            ;;(checkpoint "doit::20")
            (when (not (warn--inside-comment-or-string))
              ;;(checkpoint "doit::21")
              (setq m-call (buffer-substring-no-properties (match-beginning 1) (match-end 1)))
              (debug "Ying Yang")
              (checkpoint "*** m-call=%s" m-call)
              (checkpoint "*** class=%s" class)
              (checkpoint "*** warn--master-list-fourthly=%s" (prin1-to-string warn--master-list-fourthly))
              (checkpoint "*** assoc=%s" (assoc class warn--master-list-fourthly))
              (checkpoint "*** find=%s" (find m-call (cdr
                                                   (assoc class
                                                          warn--master-list-fourthly))
                                           :test 'string=))
              ;;(checkpoint "doit:22")
              (debug "Darlington County")
              (if (find m-call (cdr (assoc class warn--master-list-fourthly)) :test 'string=)
                  (progn
                    (warn--log-message (concat "Error 29: Cannot call a method"
                                               " from the main function"))
                    ))
              (forward-char -1)
              ))
          (goto-char beg)
          ;;(checkpoint "doit:23")
          (while (warn--re-search-forward--no-strings-no-comments (concat "\\<\\([A-Z]+[a-z][A-Za-z0-9]*\\)"
                                                                      "\\.\\([a-z][A-Za-z0-9]*\\)[ \t]*(") end t)
            ;;(checkpoint "doit:24")
            (when (not (warn--inside-comment-or-string))
              ;;(checkpoint "doit:25")
              (setq c-call (buffer-substring-no-properties (match-beginning 1) (match-end 1)))
              ;;(checkpoint "doit:26")
              (setq m-call (buffer-substring-no-properties (match-beginning 2) (match-end 2)))
              ;;(debug "Rock Shop")
              (debug "Sonny Rollins")
              (when (find m-call (cdr (assoc c-call warn--master-list-fourthly)) :test 'string=)
                (warn--log-message (concat "Error 30: Cannot call a method"
                                           " with a class name prefix from"
                                           " the main function"))
                ;;(debug "Sonny Rollins")
                )
              ;;(checkpoint "doit:27")
              ))))

      ;;(checkpoint "doit:28")

      (save-excursion
        ;;(font-lock-fontify-buffer)
        (goto-char (point-min))
        (while (warn--re-search-forward--no-strings-no-comments (concat "^   \\(\\(abstract\\|final\\|"
                                                                  "private\\|protected\\|public\\)"
                                                                  "[ \t]+\\)*function\\>") nil t)
          (beginning-of-line)
          (forward-line 1)
          (when (looking-at "^   {")
            (setq beg (point))
            (setq end (save-excursion
                        (warn--re-search-forward--no-strings-no-comments "^   }" nil t)
                        (point)))
            (setq class (cdr (warn--find-enclosing-class)))
            (while (warn--re-search-forward--no-strings-no-comments (concat "[^.a-zA-Z0-9_]"
                                                                            "\\([a-z][A-Za-z0-9_]*\\)"
                                                                            "[ \t]*(") end t)
              (when (not (warn--inside-comment-or-string))
                (setq m-call (buffer-substring-no-properties
                              (match-beginning 1)
                              (match-end 1)))
                ;;(debug "Cover me")
                (when (find m-call (cdr (nth 3 (assoc class warn--master-list-fourthly))) :test 'string=)
                  (debug "R.E.M.: Everybody Hurts")
                  (warn--log-message "Error 31: Cannot call a method from a function"))))
            (goto-char beg)
            (while (warn--re-search-forward--no-strings-no-comments (concat "\\<\\([A-Z]+[a-z][A-Za-z0-9]*\\)"
                                                                      "\\.\\([a-z][A-Za-z0-9]*\\)") end t)
              (when (not (warn--inside-comment-or-string))
                (setq c-call (buffer-substring-no-properties
                              (match-beginning 1)
                              (match-end 1)))
                (setq m-call (buffer-substring-no-properties
                              (match-beginning 2)
                              (match-end 2)))
                (if (find m-call (cadr (assoc c-call warn--master-list-fourthly)) :test 'string=)
                    (warn--log-message (concat "Error 32: Cannot call a method"
                                               " with a class name prefix"
                                               " from a function"))))))))
      ;;(checkpoint "doit:29")
      (save-excursion
        ;;(font-lock-fontify-buffer)
        (goto-char (point-min))
        (while (warn--re-search-forward--no-strings-no-comments (concat "^   \\(\\(abstract\\|final\\|"
                                                                  "private\\|protected\\|public\\)"
                                                                  "[ \t]+\\)*method\\>") nil t)
          ;;(checkpoint "doit:30")
          (beginning-of-line)
          (forward-line 1)
          (when (looking-at "^   {")
            ;;(checkpoint "doit:31")
            (setq beg (point))
            (setq end (save-excursion
                        (warn--re-search-forward--no-strings-no-comments "^   }" nil t)
                        (point)))
            ;;(checkpoint "doit:32")
            (setq class (cdr (warn--find-enclosing-class)))
            (while (warn--re-search-forward--no-strings-no-comments (concat "\\<\\([A-Z]+[a-z][A-Za-z0-9]*\\)"
                                                                      "\\.\\([a-z][A-Za-z0-9]*\\)") end t)
              ;;(checkpoint "doit:33")
              (when (not (warn--inside-comment-or-string))
                ;;(checkpoint "doit:34")
                (setq c-call (buffer-substring-no-properties
                              (match-beginning 1)
                              (match-end 1)))
                (setq m-call (buffer-substring-no-properties
                              (match-beginning 2)
                              (match-end 2)))
                (if (find m-call (cadr (assoc c-call warn--master-list-fourthly)) :test 'string=)
                    (warn--log-message (concat "Error 33: Cannot call a method"
                                               " without an object from a method"))))))))
      ;;(checkpoint "doit:35")
      (save-excursion
        ;;(font-lock-fontify-buffer)
        (goto-char (point-min))
        ;;(checkpoint "doit:36")
        (while (warn--re-search-forward--no-strings-no-comments (concat "^   \\(\\(abstract\\|final\\|"
                                                                  "private\\|protected\\|public\\)"
                                                                  "[ \t]+\\)*constructor\\>") nil t)
          ;;(checkpoint "doit:37")
          (beginning-of-line)
          (forward-line 1)
          (when (looking-at "^   {")
            (setq beg (point))
            (setq end (save-excursion
                        (warn--re-search-forward--no-strings-no-comments "^   }" nil t)
                        (point)))
            ;;(checkpoint "doit:38")
            (setq class (cdr (warn--find-enclosing-class)))
            (while (warn--re-search-forward--no-strings-no-comments (concat "\\<\\([A-Z]+[a-z][A-Za-z0-9]*\\)"
                                                                      "\\.\\([a-z][A-Za-z0-9]*\\)") end t)
              ;;(checkpoint "doit:39")
              (setq c-call (buffer-substring-no-properties
                            (match-beginning 1)
                            (match-end 1)))
              (setq m-call (buffer-substring-no-properties
                            (match-beginning 2)
                            (match-end 2)))
              (end-of-line)
              (forward-char -1)
              (prin1-to-string (warn--current-line-as-string))
              ;;(char-after (point))
              ;;(get-char-property (1- (point)) 'face)
              ;;(checkpoint "doit:40")
              (when (not (warn--inside-comment-or-string))
                ;;(checkpoint "doit:41")
                (if (find m-call (cdr (assoc c-call warn--master-list-fourthly)) :test 'string=)
                    (warn--log-message (concat "Error 34: Cannot call a method"
                                               " without an object from a constructor"))))))))
      ;;(checkpoint "buffer-name=%s" (buffer-name))
      ;;(checkpoint "doit:42")
      (setq stage 1)
      (when (= 0 (length warn--err-msgs))
        ;;(debug "Viagra")
        (save-excursion ;;
          ;;(font-lock-fontify-buffer)
          (goto-char (point-min))
          ;;(warn--log-message (format "case-fold-search=%s" case-fold-search))
          (setq case-fold-search nil)
          (while (warn--re-search-forward--no-strings-no-comments (concat "^[ ][ ][ ][ ][ ][ ][ ]*\\("
                                                                    "\\(superfor\\|for\\|while\\)[ \t]*(\\)?"
                                                                    "\\(boolean\\|char\\|float\\|"
                                                                    "double\\|int\\|long\\|short\\|"
                                                                    "[A-Z]\\|[A-Z]+[a-z][a-zA-Z0-9]*\\)"
                                                                    "\\([][]*\\)[ \t]+"
                                                                    "\\([a-z][A-Za-z0-9]*\\)") nil t)
            (save-excursion
              (setq n3 (buffer-substring-no-properties (match-beginning 3) (match-end 3)))
              (setq n5 (buffer-substring-no-properties (match-beginning 5) (match-end 5)))
              (setq peek (save-excursion
                           (beginning-of-line)
                           (warn--looking-at)))
              (setq post (warn--looking-at))
              (beginning-of-line)
              (forward-line -1)
              (setq stage 2)
              (warn--log-message "Error 35: Keyword var missing")
              )))

        (save-excursion
          ;;(font-lock-fontify-buffer)
          (goto-char (point-min))
          (while (warn--re-search-forward--no-strings-no-comments "\\<var\\>" nil t)
            (when (not (warn--inside-comment-or-string))
              (when (and (save-excursion
                           (beginning-of-line) ;;; for (var int i=0;)
                           (not (looking-at "^[ ][ ][ ][ ][ ][ ][ ]*var")))
                         (save-excursion
                           (skip-chars-backward "a-z") ;; var
                           (skip-chars-backward "(")   ;; open bracket
                           (skip-chars-backward " \t") ;; spaces
                           (skip-chars-backward "a-z") ;; for or while
                           (not (looking-at "\\(superfor\\|for\\|while\\)"))
                           ;;(debug 123)
                           ))
                ;;(when (cdr (assq (warn--what-line) warn--list))
                (save-excursion
                  ;;(forward-line -1)
                  (setq stage 2)
                  ;;(error "Foomatic")
                  (warn--log-message "Error 36: Keyword var does not belong here"))))))

        )
      ;;(checkpoint "buffer-name=%s" (buffer-name))

      )

     ;;; first point of splats

    (progn
      ;;(font-lock-fontify-buffer)
      (goto-char (point-min))
      (while (warn--re-search-forward--no-strings-no-comments "\\<\\(function\\|classVar\\)\\> " nil t)
        (when (not (warn--inside-comment-or-string))
          (delete-region (match-beginning 0) (match-end 0))
          (insert " static "))))
    ;;(checkpoint "doit:43")
    (progn
      ;;(font-lock-fontify-buffer)
      (goto-char (point-min))
      (while (warn--re-search-forward--no-strings-no-comments (concat "\\<\\(method\\|property\\|constructor\\|then\\)[ \t\r\n]") nil t)
        (save-excursion
          (forward-char -2)
          (when (not (warn--inside-comment-or-string))
            (delete-region (match-beginning 0) (match-end 0))))))
    ;;(checkpoint "doit:43b")
    (progn
      (goto-char (point-min))
      (while (warn--re-search-forward--no-strings-no-comments "\\<var[ \t\r\n]+" nil t)
        (save-excursion
          (forward-char -2)
          (when (not (warn--inside-comment-or-string))
            (delete-region (match-beginning 0) (match-end 0))))))
    ;;(checkpoint "doit:44")
    (progn
      ;;(font-lock-fontify-buffer)
      (goto-char (point-min))
      (while (warn--re-search-forward--no-strings-no-comments "\\<and\\>" nil t)
        (when (not (warn--inside-comment-or-string))
          (delete-region (match-beginning 0) (match-end 0))
          (insert "&&"))))
    ;;(checkpoint "doit:45")
    (progn
      ;;(font-lock-fontify-buffer)
      (goto-char (point-min))
      (while (warn--re-search-forward--no-strings-no-comments "\\<or\\>" nil t)
        (when (not (warn--inside-comment-or-string))
          (delete-region (match-beginning 0) (match-end 0))
          (insert "||"))))
    (checkpoint "doit:46")
    (setq was-error nil)
    (checkpoint "buffer-name=%s" (buffer-name))
    (if warn--err-msgs
        (progn
          (checkpoint "doit:47")
          (setq was-error t)
          (setq warn--err-msgs (sort warn--err-msgs 'warn--pred))
          (setq warn--err-msgs (cons (list "" 0 "** Stage 1: SJS Debugger Stage 1\n")
                                     warn--err-msgs))
          (setq warn--err-msgs (delete-duplicates warn--err-msgs :test 'warn-equals))
          (setq str (warn--list-to-string warn--err-msgs))
          (message str)
          ;;(error "*** Error compiling SJS code (smeg)")
          ))
    (progn
      (checkpoint "doit:48")
      ;;(checkpoint "doit:50")
      ;;(checkpoint "buffer-name=%s" (buffer-name))
      ;;(checkpoint "doit:51")
      ;;(checkpoint "doit:52")
      ;;(assert (string= (buffer-name) "Error.java"))
      ;;(checkpoint "doit:53")
      (java-mode)
      (assert (eq major-mode 'java-mode))
      ;;(sit-for 5)
      ;;(checkpoint "doit:35a")
      (widen)
      (setq c-basic-offset 3)
      (c-set-offset 'substatement-open 0)
      (indent-region (point-min) (point-max))
      ;;(checkpoint "doit:37 indented region!")
      ;;(insert (format "was-error=%s\n" was-error))
      ;;(if (string-match "pkg/inner/A" *stump*)
      ;;    (debug "Toilet seat"))
      (if was-error
          (progn
            (setq buf-name (buffer-name))
            (checkpoint "Big Dipper")
            (set-buffer-modified-p nil)
            (setq debug-on-error nil)
            (error "Error building %s" buf-name))
        (and 'dr-beat (save-some-buffers 'NO-QUESTIONS))
        )
      )
    )
  )

(when (not (and (boundp '*stump*) *stump*))
  (setq *stump* "Main")
  (setq *stump* "Error2")
  (setq *stump* "Error")
  (setq *stump* "pkg/inner/A")
  (setq *stump* "Error4")
  (setq *stump* "Error")
  (setq *stump* "pkg/Beer")
  (setq *stump* "SuperFor")
  (setq *stump* "PowerOf2")
  (setq *stump* "MyFirstProgram")
  (setq *stump* "pkg/Beer")
  (setq *stump* "AnimalTest")
  )

(setq debug-on-error (if noninteractive nil t))
(setq debug-on-error t)

(provide 'jtw-build-jtw)

;;; (round (/ (warn--what-line) 58.0)) 33 pages
;;; (warn--what-line)
;;; end of jtw-build-jtw.el

