#!perl

use v5.10;
use Test::More;
use Data::Dumper;

use CPAN::DistnameInfo;

use URI::PackageURL;
use URI::PackageURL::Util qw(purl_to_urls);

while (my $distpath = <DATA>) {

    chomp $distpath;

    subtest "$distpath" => sub {

        my $d = CPAN::DistnameInfo->new($distpath);

        my $purl1 = PURL->new(
            type       => 'cpan',
            namespace  => $d->cpanid,
            name       => $d->dist,
            version    => $d->version,
            qualifiers => {distpath => $distpath}
        );

        my $urls1 = purl_to_urls($purl1);
        my $like1 = quotemeta($distpath);

        like $urls1->{download}, qr{$like1},
            sprintf("PURL(1): %s\n\t - Download URL(1): %s", $purl1, $urls1->{download});

        my $purl2 = $purl1->clone;

        # Alternative distname
        my $distpath2 = (split '/', $distpath, 3)[2];
        $purl2->qualifiers({distpath => $distpath2});

        my $urls2 = purl_to_urls($purl2);
        my $like2 = quotemeta($distpath2);

        like $urls2->{download}, qr{$like2},
            sprintf("PURL(2): %s\n\t - Download URL(2): %s", $purl2, $urls2->{download});

    };

}

done_testing();

# distribution path in __DATA__ are taken from 02packages.details.txt snapshot ~2025-09-13 00:00:00)

# Algorithm/Script

# use CPAN::PackageDetails;
# use URI::PackageURL;
#
# my $package_details = CPAN::PackageDetails->read('02packages.details.txt');
# my $records         = $package_details->entries->get_hash;
#
# foreach my $package (keys %{$records}) {
#     foreach my $version (keys %{$records->{$package}}) {
#
#         my $record = $records->{$package}{$version};
#
#         my $d = CPAN::DistnameInfo->new($record->path);
#
#         next unless $d->dist;
#         next unless $d->version;
#
#         my $qualifiers = {};
#
#         if ($d->extension ne 'tar.gz') {
#             $qualifiers->{ext} = $d->extension;
#         }
#
#         my $purl = URI::PackageURL->new(
#             type       => 'cpan',
#             namespace  => $d->cpanid,
#             name       => $d->dist,
#             version    => $d->version,
#             qualifiers => $qualifiers
#         );
#
#         my $download_url = $purl->to_urls->{download};
#         $download_url =~ s{https://www.cpan.org/authors/id/}{};
#
#         if ($download_url ne $d->pathname) {
#             say $d->pathname;
#         }
#
#     }
# }

__DATA__
A/AA/AAMEND/RADIUS/RADIUS-XMLParser-2.30.tar.gz
A/AA/AARDEN/Tk-ActivityBar/Tk-ActivityBar-0.01.tar.gz
A/AA/AARDO/combine_4.005.tar.gz
A/AA/AAU/MRIM/Net-MRIM-1.11.tar.gz
A/AB/ABERNDT/Win32-Snarl/win32-snarl-1.01.tar.gz
A/AE/AECOOPER/monotone/Monotone-AutomateStdio-1.10.tar.gz
A/AF/AFRYER/TUXEDO_1_00.zip
A/AH/AHCHORN/Math/ES/Math-ES-0.08.tar.gz
A/AK/AKOLB/Solaris/Solaris-Lgrp-0.1.4.tar.gz
A/AL/ALEXEVIL/evilboss/EvilBoss-1.01.tar.gz
A/AL/ALEXMASS/bioinfo/InSilicoSpectro-0.9.7.tar.gz
A/AL/ALEXMASS/bioinfo/InSilicoSpectro-0.9.9.tar.gz
A/AM/AMBS/Biblio/Biblio-Thesaurus-0.43.tar.gz
A/AM/AMBS/Bundle/Bundle-SDK-SIMON-1.02.tar.gz
A/AM/AMBS/Dancer/Dancer2-Plugin-Emailesque-0.03.tar.gz
A/AM/AMBS/Finance/Finance-Kitko-Charts-0.01.tar.gz
A/AM/AMBS/Lingua/Lingua-Flags-0.07.tar.gz
A/AM/AMBS/Lingua/Lingua-FreeLing3-0.09.tar.gz
A/AM/AMBS/Lingua/Lingua-FreeLing3-Utils-0.09.tar.gz
A/AM/AMBS/Lingua/Lingua-Identify-0.56.tar.gz
A/AM/AMBS/Lingua/Lingua-PT-ProperNames-0.10.tar.gz
A/AM/AMBS/Lingua/Lingua-PT-Speaker-0.12.tar.gz
A/AM/AMBS/Lingua/Task-Lingua-PT-0.02.tar.gz
A/AM/AMBS/POSIX/POSIX-Open3-0.01.tar.gz
A/AM/AMBS/Search/Search-Glimpse-0.04.tar.gz
A/AM/AMBS/String/String-Tokeniser-0.05.tar.gz
A/AM/AMBS/Text/Text-Ngram-0.15.tar.gz
A/AM/AMBS/Text/Text-RewriteRules-0.25.tar.gz
A/AM/AMBS/Text/Text-WordGrams-0.07.tar.gz
A/AM/AMBS/Tie/Tie-Ispell-0.08.tar.gz
A/AM/AMBS/XML/XML-XCES-0.01.tar.gz
A/AM/AMITSIDES/WWW/WWW-BitTorrent-0.01.tar.gz
A/AN/ANDREWN/Parse-Plain/Parse-Plain-3.03.tar.gz
A/AN/ANK/FSWatcher/Win32-FileSystem-Watcher-0.1.0.tar.gz
A/AN/ANK/pngwriter/Image-PNGwriter-0.01.tar.gz
A/AN/ANPARKER/mikrotik/MikroTik-Client-v0.530.tar.gz
A/AN/ANURADHA/Usergrid/Usergrid-Client-0.11.tar.gz
A/AN/ANURADHA/Usergrid/Usergrid-Client-0.22.tar.gz
A/AS/ASARIH/HPUX-SDUX/HPUX-SDUX-0.03.tar.gz
A/AS/ASARIH/IPC-Door/IPC-Door-0.11.tar.gz
A/AS/ASCOPE/Net-ITE.pm-0.05.tar.gz
A/AW/AWKAY/Apache/Apache-SiteControl-1.01.tar.gz
A/AW/AWKAY/Apache2/Apache2-SiteControl-1.05.tar.gz
A/AY/AYATES/data/Data-Predicate-2.1.1.tar.gz
A/AZ/AZS/graph/Graph-ModularDecomposition-0.15.tar.gz
B/BA/BALDR/BioGraph/BioGraph-1.0.0.tar.gz
B/BA/BALDR/BioGraph/BioGraph-1.0.1.tar.gz
B/BA/BARRYP/Term-GetKey/LibTerm-GetKey-1.02.tar.gz
B/BB/BBENNETT/collate/Unicode-Collate-Standard-V3_1_1-0.1.tar.gz
B/BB/BBENNETT/dt-fmt-datemanip/DateTime-Format-DateManip-0.04.tar.gz
B/BB/BBENNETT/nameday/DateTime-Event-NameDay-0.02.tar.gz
B/BE/BERA/Text/Text-Stripper-1.16.tar.gz
B/BH/BHUVAN/bhuvan/Dblink-1.0.tar.gz
B/BJ/BJEPS/RDBMS.pm-0.93.tar.gz
B/BL/BLADE/Class-CompiledC2.21.tgz
B/BL/BLAKEW/Games-Tetris-Complete-0.03-withoutworldwriteables.tar.gz
B/BL/BLUECURIO/SeqDiff/SeqDiff-v0.1.tar.gz
B/BL/BLX/XSDSQL_20130120.tar.gz
B/BL/BLX/XSDSQL_20130411.tar.gz
B/BM/BMIDD/perl5.00402-bindist04-msvcAlpha.tar.gz
B/BO/BOFTX/Fiscal5253/DateTimeX-Fiscal-Fiscal5253-2.01.tar.gz
B/BO/BOFTX/Games-Dukedom/Games-Dukedom-v0.1.3.tar.gz
B/BO/BOFTX/MooX/MooX-Role-Reconstruct-v0.1.2.tar.gz
B/BO/BOFTX/Retail454/DateTime-Fiscal-Retail454-0.03.tar.gz
B/BR/BRACETA/AcmeCPANAuthorsPortuguese/Acme-CPANAuthors-Portuguese-0.04.tar.gz
B/BR/BRACETA/XML-Hash/XML-Hash-0.95.tar.gz
B/BR/BRIAN/psilambdallc/KappaCUDA-1.5.0.tar.gz
B/BU/BULKDD/Win32/Acme-Win32-PEPM-0.02.tar.gz
B/BU/BULKDD/Win32/Win32-API-0.84.tar.gz
B/BU/BULKDD/Win32/Win32-ExeAsDll-0.01.tar.gz
B/BU/BULKDD/benchmark/Acme-Shining-0.01.tar.gz
B/BU/BULKDD/benchmark/Benchmark-Perl-CoreHashes-0.02.tar.gz
B/BU/BULKDD/examples/Example-XS-FasterHashes-0.01.tar.gz
B/BW/BWARFIELD/NRGN/Test-AutoLoader-0.03.tar.gz
B/BY/BYRNE/SOAP/SOAP-Lite-0.60a.tar.gz
B/BY/BYRNE/SOAP/SOAP-MIME-0.55-7.tar.gz
C/CA/CAFFIEND/font_ft2_0.1.0.tgz
C/CA/CARLOSFB/RT/RT-Authen-Federation-1.2.2.tar.gz
C/CA/CARPENTER/Geo-GNUPlot_0.01.tar.gz
C/CA/CARPENTER/Storm-Tracker_0.01.tar.gz
C/CA/CARPENTER/Storm-Tracker_0.02.tar.gz
C/CC/CCOLBOURN/Data-Sync/Data-Sync-0.08.tar.gz
C/CC/CCOLBOURN/Text-StripAccents/Text-StripAccents-0.11.tar.gz
C/CC/CCOLBOURN/ai-nnflex/AI-NNFlex-0.24.tar.gz
C/CD/CDENT/Kwiki/Kwiki-Backlinks-0.10.tar.gz
C/CD/CDENT/Kwiki/Kwiki-Blog-0.10.tar.gz
C/CD/CDENT/Kwiki/Kwiki-DatedAnnounce-0.01.tar.gz
C/CD/CDENT/Kwiki/Kwiki-FetchRSS-0.08.tar.gz
C/CD/CDENT/Kwiki/Kwiki-Keywords-0.14.tar.gz
C/CD/CDENT/Kwiki/Kwiki-Orphans-0.11.tar.gz
C/CD/CDENT/Kwiki/Kwiki-PPerl-0.01.tar.gz
C/CD/CDENT/Kwiki/Kwiki-PageStats-0.06.tar.gz
C/CD/CDENT/Kwiki/Kwiki-PageTemperature-0.01.tar.gz
C/CD/CDENT/Kwiki/Kwiki-Purple-0.04.tar.gz
C/CD/CDENT/Kwiki/Kwiki-Raw-0.02.tar.gz
C/CD/CDENT/Kwiki/Kwiki-Soap-0.05.tar.gz
C/CD/CDENT/Kwiki/Kwiki-Soap-Google-0.04.tar.gz
C/CD/CDENT/Kwiki/Kwiki-Technorati-0.04.tar.gz
C/CD/CDENT/Kwiki/Kwiki-Test-0.03.tar.gz
C/CD/CDENT/Kwiki/Kwiki-Theme-CPB-0.10.tar.gz
C/CD/CDENT/Kwiki/Kwiki-Trackback-0.01.tar.gz
C/CD/CDENT/Kwiki/Kwiki-Yahoo-0.04.tar.gz
C/CD/CDENT/Purple/Purple-1.4.tar.gz
C/CD/CDENT/PurpleWiki/PurpleWiki-0.94.tar.gz
C/CD/CDENT/TiddlyWeb/TiddlyWeb-Wikrad-0.9.tar.gz
C/CG/CGLASS/webservicejamendo/WebService-Jamendo_2.tar.gz
C/CH/CHENGYU/circle-chain/Circle-Block-0.05.tar.gz
C/CH/CHENGYU/circle-chain/Circle-Chain-0.07.tar.gz
C/CH/CHENGYU/circle-chain/Circle-Miner-0.02.tar.gz
C/CH/CHENGYU/circle-chain/Circle-Node-0.05.tar.gz
C/CH/CHENGYU/circle-chain/Circle-User-0.08.tar.gz
C/CH/CHENGYU/circle-chain/Circle-Wallet-0.05.tar.gz
C/CH/CHENRYN/NG/NG-0.001.tar.gz
C/CH/CHRWIN/acme-pia-export/Acme-PIA-Export-0.019.tar.gz
C/CH/CHRWIN/convert-xtext/Convert-XText-0.01.tar.gz
C/CH/CHRWIN/date-spoken-german/Date-Spoken-German-0.05.tar.gz
C/CJ/CJCOLLIER/AMPRRip44/AMPR-Rip44-0.03.tar.gz
C/CJ/CJCOLLIER/Earth-USA-Washington-Cascadia/Earth-USA-Washington-Cascadia-0.01.tar.gz
C/CJ/CJCOLLIER/HPSG/HPSG-Mason-0.04.tar.gz
C/CJ/CJCOLLIER/Lingua-HPSG/Lingua-HPSG-0.02.tar.gz
C/CJ/CJCOLLIER/Lingua-HPSG/Lingua-HPSG-0.03.tar.gz
C/CJ/CJCOLLIER/NIST-NVD/NIST-NVD-1.02.01.tar.gz
C/CJ/CJCOLLIER/NIST-NVD/NIST-NVD-Store-SQLite3-1.02.00.tar.gz
C/CJ/CJCOLLIER/WordPress-Plugin/WordPress-Plugin-WallFlower-0.0.3.tar.gz
C/CL/CLADI/SmarTalk_v10.tar.gz
C/CL/CLADI/ios_get_interfaces_1.7.tar.gz
C/CM/CML/Sendmail_M4.0.29a.tar.gz
C/CO/CORLISS/Class-EHierarchy/Class-EHierarchy-2.01.tar.gz
C/CO/CORLISS/Net-ICAP/Net-ICAP-0.04.tar.gz
C/CO/CORLISS/Paranoid-BerkeleyDB/Paranoid-BerkeleyDB-2.02.tar.gz
C/CO/CORLISS/Paranoid-BerkeleyDB/Paranoid-BerkeleyDB-2.06.tar.gz
C/CO/CORLISS/Paranoid-Log-Email/Paranoid-Log-Email-2.00.tar.gz
C/CO/CORLISS/Paranoid-Log-Syslog/Paranoid-Log-Syslog-2.00.tar.gz
C/CO/CORLISS/Paranoid/Paranoid-0.36.tar.gz
C/CO/CORLISS/Paranoid/Paranoid-2.10.tar.gz
C/CO/CORLISS/Parse-PlainConfig/Parse-PlainConfig-3.07.tar.gz
C/CP/CPANPERL/frames/WWW-Mechanize-Frames-0.03.tar.gz
C/CP/CPKOIS/ACH/ACH-0.01.tar.gz
C/CP/CPKOIS/ACHGenerator/ACH-Generator-0.01.tar.gz
C/CP/CPKOIS/ACHParser/ACH-Parser-0.01.tar.gz
C/CP/CPKOIS/BluepayBluepay20DailyReport/Bluepay-Bluepay20DailyReport-0.15.tar.gz
C/CP/CPKOIS/BluepayBluepay20Post/Bluepay-Bluepay20Post-0.10.tar.gz
C/CW/CWELCH/O/B/ObjectRowMap-0.11.tar.gz
C/CW/CWHITE/HPUX-FS_0.05.tar.gz
C/CW/CWHITE/HPUX-LVM_0.06.tar.gz
D/DA/DAROLD/Apache2/ModProxyPerlHtml/Apache2-ModProxyPerlHtml-4.1.tar.gz
D/DA/DAVEOLA/Image/Image-BMP-1.26.tar.gz
D/DA/DAVEROTH/Win32_AdminMisc_Source_980511.zip
D/DA/DAVEROTH/Win32_Tie_Ini_Source_980311.zip
D/DA/DAVIDCYL/Syntax-Highlight-Perl-Improved-101/Syntax-Highlight-Perl-Improved-1.01.tar.gz
D/DA/DAVIDNICO/AIS/AIS-client-0.07.tar.gz
D/DA/DAVIDNICO/Acme/Acme-SvGROW-0.03.tar.gz
D/DA/DAVIDNICO/Acme/Acme-landmine-1.00.tar.gz
D/DA/DAVIDNICO/Acme/Acme-please-0.01.tar.gz
D/DA/DAVIDNICO/Acme/Array-Frugal-0.01.tar.gz
D/DA/DAVIDNICO/DBI/DBIx-bind_param_inline-0.03.tar.gz
D/DA/DAVIDNICO/DirDB/DirDB-0.12.tar.gz
D/DA/DAVIDNICO/DirDB/DirDB-FTP-0.03.tar.gz
D/DA/DAVIDNICO/DirDB/DirDB-Storable-0.07.tar.gz
D/DA/DAVIDNICO/ParsingPerl/Macrame-0.13.tar.gz
D/DA/DAVIDNICO/SMTP/TipJar-MTA-0.34.tar.gz
D/DA/DAVIDNICO/SMTP/dateheader-1.0.tar.gz
D/DA/DAVIDNICO/Singlethreaded/HTTP-Server-Singlethreaded-0.12.tar.gz
D/DA/DAVIDNICO/StateMachines/Graph-Easy-StateMachine-0.07.tar.gz
D/DA/DAVIDNICO/handy_tied_functions/Scope-local_OnExit-0.01.tar.gz
D/DA/DAVIDNICO/handy_tied_functions/Tie-Alias-1.01.tar.gz
D/DA/DAVIDNICO/handy_tied_functions/Tie-Function-0.02.tar.gz
D/DA/DAVIDNICO/handy_tied_functions/Tie-Hash-Cache-MRU-0.02.tar.gz
D/DA/DAVIDNICO/handy_tied_functions/Tie-LazyFunction-0.01.tar.gz
D/DA/DAVIDNICO/handy_tied_functions/Tie-Quicksort-Lazy-0.04.tar.gz
D/DA/DAVIDNICO/handy_tied_functions/Tie-UrlEncoder-0.02.tar.gz
D/DA/DAVIDNICO/handy_tied_functions/TipJar-sparse-array-perl-hashbased-0.01.tar.gz
D/DA/DAVOR/Statistics-R-IO/Statistics-R-IO-0.101.tar.gz
D/DB/DBIALAC/SDP/SDP-0.1.tar.gz
D/DB/DBURDICK/BoostGraph/Boost-Graph-1.4.tar.gz
D/DB/DBURDICK/ConnectDots/Bio-ConnectDots-1.0.2.tar.gz
D/DE/DETI/Net/Net-ENUM-0.3.tar.gz
D/DE/DETI/STUN/STUN-RFC_5389-0.1.tar.gz
D/DG/DGRAHAM/simpleXMLParse/simplexmlparse_v3.1.tar.gz
D/DI/DISSENT/VCSLibCVS/VCS-LibCVS-1.tar.gz
D/DM/DMALONE/App-Diskd/App-Diskd-0.01.tar.gz
D/DM/DMALONE/Crypt-IDA/Crypt-IDA-0.03.tar.gz
D/DM/DMALONE/Math-FastGF2/Math-FastGF2-0.07.tar.gz
D/DM/DMALONE/Net-OnlineCode-004/Net-OnlineCode-0.04.tar.gz
D/DM/DMITRYSOL/SystemAdministration/Logging/Net-Dev-Tools-SyslogMessage-1.01.tar.gz
D/DO/DONANGEL/Chemistry/Chemistry-Harmonia-0.118.tar.gz
D/DO/DONANGEL/LaTeX/LaTeX-Replicase-0.031.tar.gz
D/DO/DONANGEL/Lingua-RU/Lingua-RU-Numeral-0.071.tar.gz
D/DO/DONANGEL/Math/Math-Assistant-0.05.tar.gz
D/DO/DOTTK/Domain-Register-TK.2.1.tar.gz
D/DR/DRULENSOT/UPS/perl-Hardware-UPS-Perl-0.43.tar.gz
D/DU/DUMB/HTML-Form-XSS/HTML-Form-XSS-0.31.tar.gz
D/DU/DUMB/HTML-Form-XSS/HTML-Form-XSS-1.tar.gz
D/DU/DUMB/Mungo/CGI-Mungo-1.701.tar.gz
D/DU/DUMB/Mungo/CGI-Mungo-1.9.tar.gz
D/DU/DUMB/Net-GNUDBSearch/Net-GNUDBSearch-1.01.tar.gz
D/DU/DUMB/PSGI-Hector/PSGI-Hector-2.0.tar.gz
D/DU/DUMB/Plack-App-WebMySQL/Plack-App-WebMySQL-4.01.tar.gz
D/DU/DUMB/RT-Extension-MenubarUserTickets/RT-Extension-MenubarUserTickets-1.1.tar.gz
D/DV/DVDPOL/Math-Geometry-Planar-1.18-withoutworldwriteables.tar.gz
E/EA/EASR/ONTO-PERL/ONTO-PERL-1.35.tar.gz
E/EA/EASR/ONTO-PERL/ONTO-PERL-1.45.tar.gz
E/ED/EDD/beancounter_0.8.10.tar.gz
E/EF/EFORGEOT/HTML-WikiConverter-Txt2tags-004/HTML-WikiConverter-Txt2tags-0.04.tar.gz
E/EG/EGERMAN/GPS_Poi_0.01.tar.gz
E/EH/EHASTINGS/EBI-FGPT-FuzzyRecogniser_0.09.tar.gz
E/EH/EHOOD/perlSGML.1997Sep18.tar.gz
E/EI/EIJABB/MARC-Lint_1.53.tar.gz
E/EI/EIM/WWW-SMS/WWW-SMS-0.09.tar.gz
E/EI/EISELEKD/sparse-decpp/C-sparse-0.05.tar.gz
E/EI/EISELEKD/sparse-decpp/C-sparse-0.07.tar.gz
E/EM/EMORISSE/Statistics-SPC.v0.1.tar.gz
E/EN/ENOOR/TaskForest/TaskForest-1.37.tar.gz
E/ET/ETWOECMA/MRS/promessaging-MRS-1.01.tar.gz
E/EW/EWILHELM/Drawing/CAD-Drawing-0.10.tar.gz
E/EW/EWILHELM/Math/Math-Geometry-Planar-GPC-Polygon-0.05.tar.gz
F/FA/FANSIPANS/CouchDB-ExternalProcess/CouchDB-ExternalProcess-0.02.tar.gz
F/FA/FANTASMA/PDL-Dims/PDL-Dims-0.019.tar.gz
F/FA/FANTASMA/PDL-IO-Dcm/PDL-IO-Dcm-1.011.tar.gz
F/FA/FANTASMA/PDL-IO-Nifti/PDL-IO-Nifti-0.73.tar.gz
F/FC/FCO/App-Rad-Plugin-TT/App-Rad-Plugin-TT-0.2.tar.gz
F/FE/FELIXK/Bio/RNA/BarMap/Bio-RNA-BarMap-0.04.tar.gz
F/FE/FELIXK/Bio/RNA/BarMap/Bio-RNA-Treekin-0.05.tar.gz
F/FE/FELIXK/Bio/RNA/Barriers/Bio-RNA-Barriers-0.03.tar.gz
F/FE/FELIXL/vptk/vptk_w_2_42b.tgz
F/FE/FERZ/Commercial_Software_Interfaces/Conductrics-Client-0.003.tar.gz
F/FL/FLOWERYSO/SmugMug/WWW-SmugMug-API-1.03.tar.gz
F/FO/FOOF/libao-perl_0.03-1.tar.gz
G/GA/GAAS/GAAS/Unicode-String-2.10.tar.gz
G/GA/GABOR/Text-Format0.52+NWrap0.11.tar.gz
G/GA/GABY/EasyDB.0.5b2.tgz
G/GA/GARAFOLA/Solr/Solr-0.03.tar.gz
G/GB/GBOSS/perl_archie.1.5.tar.gz
G/GB/GBOUGARD/FusionInventory-Agent_2.6.tar.gz
G/GL/GLENSMALL/Perl56/Win32-PerfMon.0.07-Perl5.6.zip
G/GO/GODE/Crypt/Crypt-Present-0.02.tar.gz
G/GO/GONZUS/JavaScript-Duktape-XS-0000081targz/JavaScript-Duktape-XS-0.000081.tar.gz
G/GO/GOOZBACH/asterisk-store/Asterisk-Store-Queue-0.1.tar.gz
G/GO/GOOZBACH/asterisk-store/Asterisk-Store-Queue-Member-0.1.tar.gz
G/GO/GOYALI/SMS_API_4.01.1.tar.gz
G/GR/GRANDPA/AI-ParticleSwarmOptimization/AI-ParticleSwarmOptimization-1.006.tar.gz
G/GR/GRANDPA/HTML-Normalize/HTML-Normalize-1.0004.tar.gz
G/GR/GRANDPA/Video-Dumper-QuickTime/Video-Dumper-QuickTime-1.0005.tar.gz
G/GR/GRANDPA/Win32-MSI-HighLevel/Win32-MSI-HighLevel-1.0008.tar.gz
G/GR/GRANDPA/Win32-PEFile/Win32-PEFile-0.7008.tar.gz
G/GR/GRM/CamelPKI/App-CamelPKI-0.07.tar.gz
H/HA/HANENKAMP/Class-DBI-DDL/Class-DBI-DDL-1.01.tar.gz
H/HA/HANENKAMP/Class-Hash/Class-Hash-1.01.tar.gz
H/HA/HANENKAMP/SPOPS/SPOPSx-Tool-DateTime-0.02.tar.gz
H/HA/HANENKAMP/SPOPS/SPOPSx-Tool-YAML-0.03.tar.gz
H/HA/HANENKAMP/Tie-Filter/Tie-Filter-1.02.tar.gz
H/HA/HANENKAMP/Tree-BPTree/Tree-BPTree-1.07.tar.gz
H/HA/HARPREET/XMS-MotifSetv1.0.tar.gz
H/HC/HCAMP/ESplit1.00.zip
H/HE/HESSU/POCSAG/POCSAG-Encode-1.00.tar.gz
H/HI/HINNERK/PerlIDS/CGI-IDS-1.0217.tar.gz
H/HI/HINNERK/StripTags/HTML-StripTags-1.01.tar.gz
H/HO/HOREA/File-RoundRobin/File-RoundRobin-0.07.tar.gz
H/HO/HOREA/Log-Log4perl/Log-Log4perl-Appender-File-FixedSize-0.01.tar.gz
H/HO/HOREA/Mojolicious/Mojolicious-Plugin-WWWSession-0.06.tar.gz
H/HO/HOREA/RPi-PIGPIO/RPi-PIGPIO-0.017.tar.gz
H/HO/HOREA/WWW-Session/WWW-Session-0.12.tar.gz
I/IK/IKALLEN/WebService-Technorati/WebService-Technorati-0.02.tar.gz
I/IK/IKALLEN/WebService-Technorati/WebService-Technorati-0.04.tar.gz
I/IK/IKALLEN/XMLXOXO/XML-XOXO-0.03.tar.gz
I/IL/ILYAZ/modules/Audio-FindChunks-2.03.zip
I/IL/ILYAZ/modules/Bundle-OS2_default-1.07.tar.gz
I/IL/ILYAZ/modules/Data-Flow-1.02.tar.gz
I/IL/ILYAZ/modules/Devel-RegExp-0.1.tar.gz
I/IL/ILYAZ/modules/DoubleBlind-0.01.tar.gz
I/IL/ILYAZ/modules/FileSystem-LL-FAT-0.05.tar.gz
I/IL/ILYAZ/modules/FreezeThaw-0.5001.tar.gz
I/IL/ILYAZ/modules/Image-Bitmap2Paths-0.01.tar.gz
I/IL/ILYAZ/modules/MP3-Splitter-0.04.tar.gz
I/IL/ILYAZ/modules/MP3-Tag-0.9713.tar.gz
I/IL/ILYAZ/modules/MP3-Tag-1.13.tar.gz
I/IL/ILYAZ/modules/MP3-Tag-1.16.zip
I/IL/ILYAZ/modules/Math-Pari-2.030528.tar.gz
I/IL/ILYAZ/modules/Net-ChooseFName-0.01.tar.gz
I/IL/ILYAZ/modules/Numeric-LL_Array-0.1504.tar.gz
I/IL/ILYAZ/modules/Run-0.03.tar.gz
I/IL/ILYAZ/modules/SOM-0.0601.zip
I/IL/ILYAZ/modules/SelfUnzip-0.01.zip
I/IL/ILYAZ/modules/Softref-0.02.tar.gz
I/IL/ILYAZ/modules/Term-Gnuplot-0.90380906.zip
I/IL/ILYAZ/modules/Term-ReadLine-Perl-1.0303.tar.gz
I/IL/ILYAZ/modules/Text-Trie-0.21.tar.gz
I/IL/ILYAZ/modules/UI-KeyboardLayout-0.7901-try1.zip
I/IL/ILYAZ/modules/c_plus_plus-0.1.tar.gz
I/IL/ILYAZ/modules/etext/etext.1.6.3.zip
I/IL/ILYAZ/os2/OS2-FTP-0_10.zip
I/IL/ILYAZ/os2/OS2-Proc-0.02.tar.gz
I/IL/ILYAZ/os2/OS2-SoftInstaller-0.09.tar.gz
I/IL/ILYAZ/os2/OS2-UPM-0_10.zip
I/IL/ILYAZ/os2/OS2-WinObject-0.02.tar.gz
I/IL/ILYAZ/os2/tk/Bundle-Tk_OS2src-1.00.tar.gz
I/IL/ILYAZ/os2/tk/Tk-OS2src-1.04.zip
I/IN/INA/App/GhostWork/App-GhostWork-0.06.tar.gz
I/IN/INA/App/japerl/App-japerl-0.15.tar.gz
I/IN/INA/Arabic/Arabic-1.22.tar.gz
I/IN/INA/Big5/Big5-1.22.tar.gz
I/IN/INA/Big5HKSCS/Big5HKSCS-1.22.tar.gz
I/IN/INA/CP932/R2/CP932-R2-0.07.tar.gz
I/IN/INA/CP932IBM/R2/CP932IBM-R2-0.07.tar.gz
I/IN/INA/CP932NEC/R2/CP932NEC-R2-0.07.tar.gz
I/IN/INA/CP932X/R2/CP932X-R2-0.07.tar.gz
I/IN/INA/Char/Arabic/Char-Arabic-1.02.tar.gz
I/IN/INA/Char/Arabic/Char-Arabic-1.15.tar.gz
I/IN/INA/Char/Big5HKSCS/Char-Big5HKSCS-1.02.tar.gz
I/IN/INA/Char/Big5HKSCS/Char-Big5HKSCS-1.15.tar.gz
I/IN/INA/Char/Big5Plus/Char-Big5Plus-1.02.tar.gz
I/IN/INA/Char/Big5Plus/Char-Big5Plus-1.15.tar.gz
I/IN/INA/Char/Cyrillic/Char-Cyrillic-1.02.tar.gz
I/IN/INA/Char/Cyrillic/Char-Cyrillic-1.15.tar.gz
I/IN/INA/Char/EUCJP/Char-EUCJP-1.02.tar.gz
I/IN/INA/Char/EUCJP/Char-EUCJP-1.15.tar.gz
I/IN/INA/Char/EUCTW/Char-EUCTW-1.02.tar.gz
I/IN/INA/Char/EUCTW/Char-EUCTW-1.15.tar.gz
I/IN/INA/Char/GB18030/Char-GB18030-1.02.tar.gz
I/IN/INA/Char/GB18030/Char-GB18030-1.15.tar.gz
I/IN/INA/Char/GBK/Char-GBK-1.02.tar.gz
I/IN/INA/Char/GBK/Char-GBK-1.15.tar.gz
I/IN/INA/Char/Greek/Char-Greek-1.02.tar.gz
I/IN/INA/Char/Greek/Char-Greek-1.15.tar.gz
I/IN/INA/Char/HP15/Char-HP15-1.02.tar.gz
I/IN/INA/Char/HP15/Char-HP15-1.15.tar.gz
I/IN/INA/Char/Hebrew/Char-Hebrew-1.02.tar.gz
I/IN/INA/Char/Hebrew/Char-Hebrew-1.15.tar.gz
I/IN/INA/Char/INFORMIXV6ALS/Char-INFORMIXV6ALS-1.02.tar.gz
I/IN/INA/Char/INFORMIXV6ALS/Char-INFORMIXV6ALS-1.15.tar.gz
I/IN/INA/Char/JIS8/Char-JIS8-1.02.tar.gz
I/IN/INA/Char/JIS8/Char-JIS8-1.15.tar.gz
I/IN/INA/Char/KOI8R/Char-KOI8R-1.02.tar.gz
I/IN/INA/Char/KOI8R/Char-KOI8R-1.15.tar.gz
I/IN/INA/Char/KOI8U/Char-KOI8U-1.02.tar.gz
I/IN/INA/Char/KOI8U/Char-KOI8U-1.15.tar.gz
I/IN/INA/Char/KPS9566/Char-KPS9566-1.02.tar.gz
I/IN/INA/Char/KPS9566/Char-KPS9566-1.15.tar.gz
I/IN/INA/Char/Latin1/Char-Latin1-1.02.tar.gz
I/IN/INA/Char/Latin1/Char-Latin1-1.15.tar.gz
I/IN/INA/Char/Latin10/Char-Latin10-1.02.tar.gz
I/IN/INA/Char/Latin10/Char-Latin10-1.15.tar.gz
I/IN/INA/Char/Latin2/Char-Latin2-1.02.tar.gz
I/IN/INA/Char/Latin2/Char-Latin2-1.15.tar.gz
I/IN/INA/Char/Latin3/Char-Latin3-1.02.tar.gz
I/IN/INA/Char/Latin3/Char-Latin3-1.15.tar.gz
I/IN/INA/Char/Latin4/Char-Latin4-1.02.tar.gz
I/IN/INA/Char/Latin4/Char-Latin4-1.15.tar.gz
I/IN/INA/Char/Latin5/Char-Latin5-1.02.tar.gz
I/IN/INA/Char/Latin5/Char-Latin5-1.15.tar.gz
I/IN/INA/Char/Latin6/Char-Latin6-1.02.tar.gz
I/IN/INA/Char/Latin6/Char-Latin6-1.15.tar.gz
I/IN/INA/Char/Latin7/Char-Latin7-1.02.tar.gz
I/IN/INA/Char/Latin7/Char-Latin7-1.15.tar.gz
I/IN/INA/Char/Latin8/Char-Latin8-1.02.tar.gz
I/IN/INA/Char/Latin8/Char-Latin8-1.15.tar.gz
I/IN/INA/Char/Latin9/Char-Latin9-1.02.tar.gz
I/IN/INA/Char/Latin9/Char-Latin9-1.15.tar.gz
I/IN/INA/Char/OldUTF8/Char-OldUTF8-1.02.tar.gz
I/IN/INA/Char/OldUTF8/Char-OldUTF8-1.15.tar.gz
I/IN/INA/Char/Sjis/Char-Sjis-1.02.tar.gz
I/IN/INA/Char/Sjis/Char-Sjis-1.15.tar.gz
I/IN/INA/Char/TIS620/Char-TIS620-1.02.tar.gz
I/IN/INA/Char/TIS620/Char-TIS620-1.15.tar.gz
I/IN/INA/Char/UHC/Char-UHC-1.02.tar.gz
I/IN/INA/Char/UHC/Char-UHC-1.15.tar.gz
I/IN/INA/Char/USASCII/Char-USASCII-1.02.tar.gz
I/IN/INA/Char/USASCII/Char-USASCII-1.15.tar.gz
I/IN/INA/Char/UTF2/Char-UTF2-1.02.tar.gz
I/IN/INA/Char/UTF2/Char-UTF2-1.15.tar.gz
I/IN/INA/Char/Windows1252/Char-Windows1252-1.02.tar.gz
I/IN/INA/Char/Windows1252/Char-Windows1252-1.15.tar.gz
I/IN/INA/Char/Windows1258/Char-Windows1258-1.02.tar.gz
I/IN/INA/Char/Windows1258/Char-Windows1258-1.15.tar.gz
I/IN/INA/Cyrillic/Cyrillic-1.22.tar.gz
I/IN/INA/DBD/PgPPSjis/DBD-PgPPSjis-0.0801008.tar.gz
I/IN/INA/DBD/mysqlPPrawSjis/DBD-mysqlPPrawSjis-0.16.tar.gz
I/IN/INA/Fake/Encode/Fake-Encode-0.11.tar.gz
I/IN/INA/Fake/Our/Fake-Our-0.12.tar.gz
I/IN/INA/Fake/Our/Fake-Our-0.17.tar.gz
I/IN/INA/GBK/GBK-1.22.tar.gz
I/IN/INA/Greek/Greek-1.22.tar.gz
I/IN/INA/IOas/CP932/IOas-CP932-0.10.tar.gz
I/IN/INA/IOas/CP932IBM/IOas-CP932IBM-0.10.tar.gz
I/IN/INA/IOas/CP932NEC/IOas-CP932NEC-0.10.tar.gz
I/IN/INA/IOas/CP932X/IOas-CP932X-0.10.tar.gz
I/IN/INA/IOas/SJIS2004/IOas-SJIS2004-0.10.tar.gz
I/IN/INA/Jacode/Jacode-2.13.4.31.tar.gz
I/IN/INA/Jacode4e/Jacode4e-2.13.6.21.tar.gz
I/IN/INA/Jacode4e/RoundTrip/Jacode4e-RoundTrip-2.13.81.14.tar.gz
I/IN/INA/KOI8R/KOI8R-1.22.tar.gz
I/IN/INA/KOI8U/KOI8U-1.22.tar.gz
I/IN/INA/KPS9566/KPS9566-1.22.tar.gz
I/IN/INA/KSC5601/KSC5601-1.22.tar.gz
I/IN/INA/Latin1/Latin1-1.22.tar.gz
I/IN/INA/Latin10/Latin10-1.22.tar.gz
I/IN/INA/Latin2/Latin2-1.22.tar.gz
I/IN/INA/Latin3/Latin3-1.22.tar.gz
I/IN/INA/Latin4/Latin4-1.22.tar.gz
I/IN/INA/Latin5/Latin5-1.22.tar.gz
I/IN/INA/Latin6/Latin6-1.22.tar.gz
I/IN/INA/Latin7/Latin7-1.22.tar.gz
I/IN/INA/Latin8/Latin8-1.22.tar.gz
I/IN/INA/Latin9/Latin9-1.22.tar.gz
I/IN/INA/Modern/Open/Modern-Open-0.14.tar.gz
I/IN/INA/Perl/Exactly/Perl-Exactly-0.07.tar.gz
I/IN/INA/Perl7/Handy/Perl7-Handy-0.11.tar.gz
I/IN/INA/SJIS2004/R2/SJIS2004-R2-0.07.tar.gz
I/IN/INA/See/See-0.01.tar.gz
I/IN/INA/Sjis/Sjis-1.22.tar.gz
I/IN/INA/Stable/Module/Stable-Module-0.11.tar.gz
I/IN/INA/Strict/Perl/Strict-Perl-2023.03.tar.gz
I/IN/INA/UTF2/UTF2-1.22.tar.gz
I/IN/INA/UTF8/R2/UTF8-R2-0.28.tar.gz
I/IN/INA/Windows1250/Windows1250-1.23.tar.gz
I/IN/INA/Windows1252/Windows1252-1.22.tar.gz
I/IN/INA/Windows1254/Windows1254-1.23.tar.gz
I/IN/INA/Windows1257/Windows1257-1.23.tar.gz
I/IN/INA/Windows1258/Windows1258-1.22.tar.gz
I/IN/INA/mb/Encode/mb-Encode-0.04.tar.gz
I/IN/INA/mb/JSON/mb-JSON-0.03.tar.gz
I/IN/INA/mb/mb-0.58.tar.gz
I/IN/INFOFLEX/DBD_SQLFLEX_8.2.tar.gz
I/IS/ISJOUNG/Proc-Branch-001/Proc-Branch-0.01.tar.gz
I/IT/ITDEV/Tk-PlotDataset/Tk-PlotDataset-2.05.tar.gz
I/IV/IVORRACH/Chorus/Chorus-Expert-1.03.tar.gz
J/JA/JAE/perlipse/Perlipse-0.02.zip
J/JA/JAMES/Class-DBI-Plugin-Calendar-0.18-withoutworldwriteables.tar.gz
J/JA/JAMESP/payroll/business-payroll-1.3.tar.gz
J/JA/JANESKIL/Tools/Daje-0.05.tar.gz
J/JA/JANESKIL/Tools/Daje-Config-0.06.tar.gz
J/JA/JANESKIL/Tools/Daje-Database-Migrator-0.01.tar.gz
J/JA/JANESKIL/Tools/Daje-Database-Model-0.03.tar.gz
J/JA/JANESKIL/Tools/Daje-Generate-Templates-1.04.tar.gz
J/JA/JANESKIL/Tools/Daje-Plugin-Apploader-0.02.tar.gz
J/JA/JANESKIL/Tools/Daje-Plugin-Login-0.02.tar.gz
J/JA/JANESKIL/Tools/Daje-Plugin-Signup-0.02.tar.gz
J/JA/JANESKIL/Tools/Daje-Plugin-Tools-0.03.tar.gz
J/JA/JANESKIL/Tools/Daje-Plugin-workflow-0.04.tar.gz
J/JA/JANESKIL/Tools/Daje-Tools-Datasections-0.11.tar.gz
J/JA/JANESKIL/Tools/Daje-Tools-Filechanged-0.02.tar.gz
J/JA/JANESKIL/Tools/Daje-Tools-JWT-0.01.tar.gz
J/JA/JANESKIL/Tools/Daje-Workflow-0.02.tar.gz
J/JA/JANESKIL/Tools/Daje-Workflow-0.10.tar.gz
J/JA/JANESKIL/Tools/Daje-Workflow-Client-Activity-0.03.tar.gz
J/JA/JANESKIL/Tools/Daje-Workflow-Common-Base-0.03.tar.gz
J/JA/JANESKIL/Tools/Daje-Workflow-Config-0.02.tar.gz
J/JA/JANESKIL/Tools/Daje-Workflow-Database-0.04.tar.gz
J/JA/JANESKIL/Tools/Daje-Workflow-Database-0.19.tar.gz
J/JA/JANESKIL/Tools/Daje-Workflow-Database-Model-0.16.tar.gz
J/JA/JANESKIL/Tools/Daje-Workflow-FileChanged-Activity-0.03.tar.gz
J/JA/JANESKIL/Tools/Daje-Workflow-GeneratePerl-Activity-0.12.tar.gz
J/JA/JANESKIL/Tools/Daje-Workflow-GenerateSQL-Activity-0.03.tar.gz
J/JA/JANESKIL/Tools/Daje-Workflow-GenerateSQL-Activity-1.07.tar.gz
J/JA/JANESKIL/Tools/Daje-Workflow-GenerateSchema-Activity-0.16.tar.gz
J/JA/JANESKIL/Tools/Daje-Workflow-Loader-0.04.tar.gz
J/JA/JANESKIL/Tools/Daje-Workflow-Loader-0.13.tar.gz
J/JA/JANESKIL/Tools/Daje-Workflow-Savefile-Activity-1.10.tar.gz
J/JA/JANESKIL/Tools/Daje-Workflow-Templates-0.02.tar.gz
J/JA/JASKA/IRC/Crypt/IRC-Crypt-0.01.tar.gz
J/JC/JCB/WARC/WARC-v0.0.1.tar.gz
J/JC/JCLINE/Robotics-023/Robotics-0.23.tar.bz2
J/JC/JCNORTON/LanguageLogo/Language-Logo-1.000.tar.gz
J/JE/JEFFMOCK/GPIB_0_30.tgz
J/JH/JHALLOCK/Gapp-0.26-withoutworldwriteables.tar.gz
J/JH/JHALLOCK/Gtk2-Ex-FormFactory-TimeEntry-0.02-withoutworldwriteables.tar.gz
J/JH/JHALLOCK/Storm-0.03-withoutworldwriteables.tar.gz
J/JH/JHELBERG/lanman.1.0.10.0.zip
J/JH/JHIVER/Unicode-Transliterate.0.3.tgz
J/JK/JKING/thrift/Thrift-0.22.0.tar.gz
J/JK/JKING/thrift/Thrift-0.9.3.tar.gz
J/JM/JMAHAN/iodbc_ext_0_1.tar.gz
J/JM/JMCADA/dbix_mypassword/DBIx-MyPassword-1.02.tar.gz
J/JM/JMCADA/net_bittorrent/Net-BitTorrent-PeerPacket-1.2.tar.gz
J/JM/JMCADA/weather/Weather-NWS-0.02.tar.gz
J/JM/JMCADA/xml_rai_enclosure/XML-RAI-Enclosure-1.02.tar.gz
J/JM/JMEHNLE/apache-auth-userdb/Apache-Auth-UserDB-v0.120.tar.gz
J/JM/JMEHNLE/clamav-client/ClamAV-Client-0.11.tar.gz
J/JM/JMEHNLE/courier-filter/Courier-Filter-v0.200.tar.gz
J/JM/JMEHNLE/mail-spf-query/Mail-SPF-Query-1.999.1.tar.gz
J/JM/JMEHNLE/mail-spf-test/Mail-SPF-Test-v1.001.tar.gz
J/JM/JMEHNLE/net-address-ip-local/Net-Address-IP-Local-0.1.2.tar.gz
J/JM/JMEHNLE/net-address-ipv4-local/Net-Address-IPv4-Local-0.12.tar.gz
J/JM/JMEHNLE/www-restaurant-menu/WWW-Restaurant-Menu-0.11.tar.gz
J/JM/JMS/EdgeExpressDB_0.953h.tar.gz
J/JM/JMS/MQdb_0.954.tar.gz
J/JO/JOEYATES/Database/Wrapper/Database-Wrapper-1.04.tar.gz
J/JO/JONG/Bioinf_V2.0.tar.gz
J/JO/JONIONS/PANT/PANT-0.17.tar.gz
J/JO/JORDAN/Doxygen-Filter-Perl/doxygen-filter-perl-1.73-2.tar.gz
J/JO/JORDAN/Term-RouterCLI/Term-RouterCLI-1.00.tar.gz
J/JO/JORVIS/Log-Cabin/Log-Cabin-0.06.tar.gz
J/JO/JORVIS/TIGR/TIGR-0.03.tar.gz
J/JU/JULVR/Parse/Preprocessor/App-UnifdefPlus-0.005.005.tar.gz
J/JW/JWHITE/SlideMap/SlideMap-1.4.tar.gz
J/JW/JWU/File-Mirror/File-Mirror-0.01.tar.gz
J/JW/JWU/File-Mirror/File-Mirror-0.10.tar.gz
J/JW/JWU/RPC-Object/RPC-Object-0.04.tar.gz
J/JW/JWU/RPC-Object/RPC-Object-0.23.tar.gz
J/JW/JWU/RPC-Object/RPC-Object-0.31.tar.gz
J/JW/JWU/Thread-Pool-Simple/Thread-Pool-Simple-0.25.tar.gz
K/KA/KARB/dataware/Tie-PagedArray-0.0.3.tar.gz
K/KA/KARJALA/Pavatar/Net-Pavatar-1.01.tar.gz
K/KA/KARTHIKU/Number-Convert.1.0.2.tar.gz
K/KE/KEVIN/AuthenRadius/Apache2-AuthenRadius-0.9.tar.gz
K/KE/KEVIN/Net-Autoconfig/Net-Autoconfig-v1.13.2.tar.gz
K/KI/KIAVASH/kateb/kateb-1.1.0.tar.gz
K/KI/KIAVASH/mira/Mira-00.07.56.tar.gz
K/KI/KIAVASH/text/Text-Markmoredown-0.000036.tar.gz
K/KI/KIWIROY/tmp/Alien-KentSrc-0.4.tar.gz
K/KI/KIWIROY/tmp/Alien-MUSCLE-0.01.tar.gz
K/KI/KIWIROY/tmp/Devel-IPerl-Plugin-EnvironmentModules-0.03.tar.gz
K/KI/KIWIROY/tmp/Devel-IPerl-Plugin-Perlbrew-0.03.tar.gz
K/KI/KIWIROY/tmp/Mojo-File-Role-Extension-0.02.tar.gz
K/KI/KIWIROY/tmp/Mojo-Path-Role-Relative-0.01.tar.gz
K/KI/KIWIROY/tmp/Mojo-Transaction-HTTP-Role-Mechanize-0.06.tar.gz
K/KI/KIWIROY/tmp/Test-Applify-0.02.tar.gz
K/KI/KIWIROY/tmp/Test-Applify-0.06.tar.gz
K/KM/KMCGRAIL/SpamAssassin/Mail-SpamAssassin-3.4.1.tar.gz
K/KO/KOBOLDWIZ/Perl5/Bio-Epithelium-0.1.4.tar.gz
K/KO/KOBOLDWIZ/Perl5/Cobol-Compiler-0.1.19.tar.gz
K/KO/KOBOLDWIZ/Perl5/DuckCurses-0.1.3.tar.gz
K/KO/KOBOLDWIZ/Perl5/DuckCurses-0.1.4.tar.gz
K/KO/KOBOLDWIZ/Perl5/Evo-EA-0.1.5.tar.gz
K/KO/KOBOLDWIZ/Perl5/Evo-GA-0.1.13.tar.gz
K/KO/KOBOLDWIZ/Perl5/Evo-Immune-0.1.11.tar.gz
K/KO/KOBOLDWIZ/Perl5/Evo-Inference-0.1.12.tar.gz
K/KO/KOBOLDWIZ/Perl5/Evo-Maple-0.1.1.tar.gz
K/KO/KOBOLDWIZ/Perl5/Evo-Parser-0.1.24.tar.gz
K/KO/KOBOLDWIZ/Perl5/Evo-QUIP-0.1.1.tar.gz
K/KO/KOBOLDWIZ/Perl5/GFX-Enhancer-0.1.22.tar.gz
K/KO/KOBOLDWIZ/Perl5/Game-NURBS-0.1.13.tar.gz
K/KO/KOBOLDWIZ/Perl5/Game-Ninja-0.1.6.tar.gz
K/KO/KOBOLDWIZ/Perl5/Game-SDL-0.1.1.tar.gz
K/KO/KOBOLDWIZ/Perl5/Lagrange-0.1.6.tar.gz
K/KO/KOBOLDWIZ/Perl5/Lagrange-0.1.9.tar.gz
K/KO/KOBOLDWIZ/Perl5/Perl-Oberon-0.1.3.tar.gz
K/KO/KOBOLDWIZ/Perl5/RogueCurses-0.0.6.tar.gz
K/KO/KOBOLDWIZ/Perl5/RogueQuest-0.1.5.tar.gz
K/KO/KOBOLDWIZ/Perl5/Stats-Fermi-0.1.2.tar.gz
K/KO/KOBOLDWIZ/Perl5/Stats-Fermi-0.1.7.tar.gz
K/KO/KOBOLDWIZ/Perl5/Stats-Hawking-0.1.3.tar.gz
K/KT/KTORP/Interval.0.01.tar.gz
K/KT/KTORP/KTORP/Interval.0.03.tar.gz
K/KU/KUNGFUFTR/Match-Any_0.03.tar.gz
K/KW/KWMAK/Chart/Gnuplot/Chart-Gnuplot-0.23.tar.gz
K/KW/KWMAK/Chart/Gnuplot/Pie/Chart-Gnuplot-Pie-0.04.tar.gz
K/KW/KWMAK/PBS/Client/PBS-Client-0.11.tar.gz
K/KY/KYLED/ObjectivePerl/ObjectivePerl-0.04.tar.gz
L/LA/LABZERONE/tmojo/HTML-Tmojo-0.300.tar.gz
L/LE/LEGART/Apache-Gallery-1.0.2-withoutworldwriteables.tar.gz
L/LE/LESPAUL/ClearCase/Region/ClearCase-Region-1.01.tar.gz
L/LG/LGODDARD/List-Member-0.044-withoutworldwriteables.tar.gz
L/LI/LINC/Tie/Tie-ScalarFile-1.12.tar.gz
L/LI/LINKDD/Lemonldap/NG/Lemonldap-NG-Cli-0.2.tgz
L/LI/LINSALROB/SGE/ScheduleSGE-0.02.tgz
L/LS/LSFISV/LSF_PERL_API/LSF_Batch_0.06.tar.gz
L/LT/LTBOOTS/test/genomics.01.tar.gz
M/MA/MALETIN/Teamspeak/Teamspeak-0.6.tar.gz
M/MA/MALUKU/dotiac/Bundle-Dotiac-0.1.tar.gz
M/MA/MALUKU/dotiac/Dotiac-0.8.tar.gz
M/MA/MALUKU/dotiac/Dotiac-addon-case-insensitive-0.4.tar.gz
M/MA/MALUKU/dotiac/Dotiac-addon-html_template-0.4.tar.gz
M/MA/MALUKU/dotiac/Dotiac-addon-importloop-0.2.tar.gz
M/MA/MALUKU/dotiac/Dotiac-addon-json-0.1.1.tar.gz
M/MA/MALUKU/dotiac/Dotiac-addon-jsonify-0.1.tar.gz
M/MA/MALUKU/dotiac/Dotiac-addon-markup-0.2.tar.gz
M/MA/MALUKU/dotiac/Dotiac-addon-unparsed-0.2.tar.gz
M/MA/MALUKU/games-freelancer-utf/FLUTF-1.001.tar.gz
M/MA/MALUKU/sofu-config/sofu-config-0.5.tar.gz
M/MA/MALUKU/sofu/Sofu-0.3.tar.gz
M/MA/MALUKU/sofu/sofu-0.2.9.tar.gz
M/MA/MALUKU/tie-multikeyinsertorderhash/tie-multikeyinsertorderhash-0.1.tar.gz
M/MA/MALUKU/tie-proxy-changes/Tie-Proxy-Changes-0.2.tar.gz
M/MA/MARCB/Net-SMTP-Pipelining/Net-SMTP-Pipelining-0.0.4.tar.gz
M/MA/MARCB/POSIX-SchedYield/POSIX-SchedYield-0.0.2.tar.gz
M/MA/MARCB/Unix-Lsof/Unix-Lsof-0.1.0.tar.gz
M/MA/MARIAB/WWW-TMDB-API0.03.tar.gz
M/MA/MARIAB/WWW-TMDB-API0.04.tar.gz
M/MA/MARKLE/A2C/Apache2-Controller-v1.1.1.tar.gz
M/MA/MARKPF/cisco/Cisco-ShowIPRoute-Parser-1.02.tar.gz
M/MA/MARROTTE/LWP-UserAgent-ExponentialBackoff-004/LWP-UserAgent-ExponentialBackoff-0.04.tar.gz
M/MA/MATHIAS/App-ArduinoBuilder/App-ArduinoBuilder-0.04.tar.gz
M/MA/MATHIAS/App-Pfind/App-Pfind-1.05.tar.gz
M/MA/MATHIAS/DataStructure/DataStructure-0.02.tar.gz
M/MA/MATHIAS/Dist-Setup/App-ArduinoBuilder-0.08.tar.gz
M/MA/MATHIAS/Dist-Setup/App-MediaPi-0.01.tar.gz
M/MA/MATHIAS/Dist-Setup/App-PTP-1.13.tar.gz
M/MA/MATHIAS/Dist-Setup/App-PTP-1.16.tar.gz
M/MA/MATHIAS/Dist-Setup/Dist-Setup-0.16.tar.gz
M/MA/MATHIAS/Dist-Setup/Future-IO-Impl-AnyEvent-0.02.tar.gz
M/MA/MATHIAS/Dist-Setup/Google-Protobuf-Loader-0.01.tar.gz
M/MA/MATHIAS/Dist-Setup/Log-Any-Simple-0.06.tar.gz
M/MA/MATHIAS/Dist-Setup/Markdown-Perl-1.10.tar.gz
M/MA/MATHIAS/Dist-Setup/Markdown-Perl-1.11.tar.gz
M/MA/MATHIAS/Dist-Setup/Music-Harmonica-TabsCreator-1.05.tar.gz
M/MA/MATHIAS/Dist-Setup/Parallel-TaskExecutor-0.05.tar.gz
M/MA/MATHIAS/Dist-Setup/UserAgent-Any-0.05.tar.gz
M/MA/MATHIAS/Dist-Setup/UserAgent-Any-JSON-0.05.tar.gz
M/MA/MATHIAS/Eval-Safe/Eval-Safe-0.02.tar.gz
M/MA/MATHIAS/Math-Numerical/Math-Numerical-0.05.tar.gz
M/MA/MATHIAS/OAuth-Consumer/OAuth-Consumer-0.03.tar.gz
M/MA/MATHIAS/SQL-Exec/SQL-Exec-0.09.tar.gz
M/MA/MATHIAS/SQL-Exec/SQL-Exec-0.10.tar.gz
M/MA/MATHIAS/String-Replace/String-Replace-0.02.tar.gz
M/MA/MATHIAS/Test-Subs/Test-Subs-0.08.tar.gz
M/MB/MBSTE/MultiCat_0.04.tar.gz
M/MB/MBV/mailboxvalidator/MailboxValidator-SingleValidation-2.00.tar.gz
M/ME/MELONMAN/PDF-EasyPDF_0_04.tgz
M/ME/MEWSOFT/mewsoft/Config-IniMan-1.20.tar.gz
M/ME/MEWSOFT/mewsoft/Date-HijriDate-1.01.tar.gz
M/ME/MEWSOFT/mewsoft/File-BufferedInput-1.03.tar.gz
M/ME/MEWSOFT/mewsoft/MP3-Merge-1.01.tar.gz
M/ME/MEWSOFT/mewsoft/Religion-Islam-PrayTime-1.06.tar.gz
M/ME/MEWSOFT/mewsoft/Religion-Islam-Quran-2.01.tar.gz
M/MF/MFENTON/TimeConvert0.5.zip
M/MI/MIKEM/Authen/Authen-ACE4-1.4.tar.gz
M/MI/MIKEM/DigestMD4/Digest-MD4-1.9.tar.gz
M/MI/MIKI/Apache2/Apache2-BalanceLogic-0.0.1.tar.gz
M/MJ/MJEWELL/citeparser/Biblio-Citation-Parser-1.10.tar.gz
M/MJ/MJEWELL/docparser/Biblio-Document-Parser-1.10.tar.gz
M/MJ/MJFS/CGI/Base64-Serializer_0.1.tar.gz
M/MM/MML/Test-HTTP/Test-HTTP-0.22.tar.gz
M/MN/MNOONING/PlRPC/PlRPC-0.2020.tar.gz
M/MO/MODULO/AppConfig-Exporter/AppConfig-Exporter-1.5.tar.gz
M/MO/MODULO/Polyglot/MasonX-Resolver-Polyglot-0.95.tgz
M/MO/MODULO/Regexp-Log-Helix/Regexp-Log-Helix-1.04.tar.gz
M/MP/MPREWITT/listgroup/Net-NIS-Listgroup-1.0.0.tar.gz
M/MP/MPREWITT/net-smtp-retryable/Net-SMTP-Retryable-0.0.2.tar.gz
M/MT/MTADEL/Getopt-FileConfig/Getopt-FileConfig-1.0001.tar.gz
M/MU/MUIR/modules/BSD-Ipfwgen-1.5.tar.gz
M/MU/MUIR/modules/CGI-Out-2006.0215.tar.gz
M/MU/MUIR/modules/Callback-1.07.tar.gz
M/MU/MUIR/modules/Cisco-Reconfig-0.912.tar.gz
M/MU/MUIR/modules/Config-Checker-0.42.tar.gz
M/MU/MUIR/modules/Daemon-Generic-0.85.tar.gz
M/MU/MUIR/modules/Devel-FileProfile-0.22.tar.gz
M/MU/MUIR/modules/File-BasicFlock-98.1202.tar.gz
M/MU/MUIR/modules/File-Flock-2014.01.tar.gz
M/MU/MUIR/modules/File-OldSlurp-2004.0430.tar.gz
M/MU/MUIR/modules/File-RdistByRsync-0.3.tar.gz
M/MU/MUIR/modules/File-Slurp-Remote-0.44.tar.gz
M/MU/MUIR/modules/Getopt-Tiny-1.02.tar.gz
M/MU/MUIR/modules/HTML-Transmorgify-0.12.tar.gz
M/MU/MUIR/modules/IO-Event-0.704.tar.gz
M/MU/MUIR/modules/IO-Event-0.813.tar.gz
M/MU/MUIR/modules/List-EvenMore-0.11.tar.gz
M/MU/MUIR/modules/Log-Parallel-0.303.tar.gz
M/MU/MUIR/modules/Mail-SendVarious-0.4.tar.gz
M/MU/MUIR/modules/Net-SMTP-Receive-0.301.tar.gz
M/MU/MUIR/modules/OOPS-0.1008.tar.gz
M/MU/MUIR/modules/OOPS-0.2005.tar.gz
M/MU/MUIR/modules/Object-Dependency-0.41.tar.gz
M/MU/MUIR/modules/Object-Transaction-1.01.tar.gz
M/MU/MUIR/modules/Plugins-0.41.tar.gz
M/MU/MUIR/modules/Proc-Forkfunc-96.042201.tar.gz
M/MU/MUIR/modules/Proc-JobQueue-0.903.tar.gz
M/MU/MUIR/modules/Proc-Parallel-0.601.tar.gz
M/MU/MUIR/modules/Qpsmtpd-Plugin-Quarantine-1.02.tar.gz
M/MU/MUIR/modules/Sort-MergeSort-0.31.tar.gz
M/MU/MUIR/modules/Stream-Aggregate-0.406.tar.gz
M/MU/MUIR/modules/String-CRC-1.0.tar.gz
M/MU/MUIR/modules/SyslgScnDamn-Blacklist-0.44.tar.gz
M/MU/MUIR/modules/SyslogScan-Daemon-0.41.tar.gz
M/MU/MUIR/modules/SyslogScan-Daemon-SpamDetector-0.56.tar.gz
M/MU/MUIR/modules/Test-MultiFork-0.6.tar.gz
M/MU/MUIR/modules/Time-ParseDate-2015.103.tar.gz
M/MU/MUIR/modules/Twiki-Upload-0.3.tar.gz
M/MU/MUNROER/kif/kif-1.02.tar.gz
M/MU/MUNROER/kif/kif-2.01.tar.gz
M/MU/MUNROER/stanzafile/StanzaFile-1.05.tar.gz
M/MV/MVOELKER/FTDI/FTDI-D2XX-0.06.tar.gz
N/NA/NANARDON/VFSsimple/VFSsimple-0.03.tar.gz
N/NA/NANARDON/VFSsimple/VFSsimple-Drv-File-0.03.tar.gz
N/NA/NANARDON/VFSsimple/VFSsimple-Drv-Ftp-0.03.tar.gz
N/NA/NANARDON/VFSsimple/VFSsimple-Drv-Http-0.03.tar.gz
N/NA/NANARDON/VFSsimple/VFSsimple-Drv-Iso-0.03.tar.gz
N/NA/NANARDON/VFSsimple/VFSsimple-Drv-Rsync-0.03.tar.gz
N/NA/NARAZAKA/Config-Column/Config-Column-1.00.tar.gz
N/NA/NARAZAKA/SQL-Object-Interp/SQL-Object-Interp-0.04.tar.gz
N/NC/NCICB/CaCORE/CaCORE-3.2.1_r1.tar.gz
N/NE/NEVES/text-histogram/Text-Histogram-0.01.tar.gz
N/NH/NHARALE/File-ReadSimple.1.1.tar.gz
O/OL/OLEGS/Perl-Data-Enumerable-Lazy/Data-Enumerable-Lazy-0.032.tar.gz
O/OL/OLIVIERT/LogValidator/W3C-LogValidator-0.1.tar.gz
O/OL/OLIVIERT/LogValidator/W3C-LogValidator-1.5.tar.gz
O/OL/OLIVIERT/WebService/WebService-Validator-Feed-W3C-0.8.tar.gz
O/OL/OLIVIERT/XHTML/W3C-XHTML-HTMLCompatChecker-0.10.tar.gz
P/PA/PARDUS/Term-ReadLine-Zoid/Term-ReadLine-Zoid-0.07.tar.gz
P/PA/PAULDOOM/Apache-AppSamurai/Apache-AppSamurai-1.01.tar.gz
P/PE/PEASWORTH/CayleyDickson/CayleyDickson-0.01.tgz
P/PE/PEASWORTH/FinanceBitStampAPI/Finance-BitStamp-API-0.01.tar.gz
P/PE/PEASWORTH/FinanceBitStampAPI/Finance-BitStamp-API-0.02.tar.gz
P/PE/PEASWORTH/FinanceBitStampSocket/Finance-BitStamp-Socket-0.02.tar.gz
P/PE/PEASWORTH/FinanceCaVirtexAPI/Finance-CaVirtex-API-0.03.tar.gz
P/PE/PEASWORTH/FinanceLocalBitcoinsAPI/Finance-LocalBitcoins-API-0.02.tar.gz
P/PE/PEASWORTH/Tangle/Tangle-0.03.tar.gz
P/PE/PETERGAL/PGForth1.0.tar.gz
P/PE/PETERGAL/PGForth1.3.tar.gz
P/PE/PETERHI/AI-ExpertSystem-Simple.1.2.tgz
P/PE/PETERSHAW/Told-Client/Told-Client-0.001.tar.gz
P/PH/PHILOU/NCAR/NCAR-0.01.tar.gz
P/PH/PHOENIXL/expr_eval_1_0.zip
P/PH/PHOENIXL/extensible_report_generator_1.13.zip
P/PJ/PJFL/class-usul/Class-Usul-0.84.1.tar.gz
P/PJ/PJFL/data-cloudweights/Data-CloudWeights-0.15.1.tar.gz
P/PJ/PJFL/data-validation/Data-Validation-0.28.1.tar.gz
P/PJ/PJFL/devel-cover-report-ownserver/Devel-Cover-Report-OwnServer-0.4.1.tar.gz
P/PJ/PJFL/dist-zilla-plugin-abstractfrompod/Dist-Zilla-Plugin-AbstractFromPOD-0.3.1.tar.gz
P/PJ/PJFL/dist-zilla-plugin-manifestinroot/Dist-Zilla-Plugin-ManifestInRoot-0.16.1.tar.gz
P/PJ/PJFL/dist-zilla-plugin-test-reportmetadata/Dist-Zilla-Plugin-Test-ReportMetadata-0.5.1.tar.gz
P/PJ/PJFL/file-dataclass/File-DataClass-0.73.1.tar.gz
P/PJ/PJFL/file-gettext/File-Gettext-0.33.1.tar.gz
P/PJ/PJFL/ipc-srlock/IPC-SRLock-0.31.1.tar.gz
P/PJ/PJFL/module-provision/Module-Provision-0.42.1.tar.gz
P/PJ/PJFL/unexpected/Unexpected-1.0.1.tar.gz
P/PJ/PJFL/web-components-role-email/Web-Components-Role-Email-0.3.1.tar.gz
P/PJ/PJFL/web-components-role-tt/Web-Components-Role-TT-0.8.1.tar.gz
P/PJ/PJFL/web-components/Web-Components-0.13.1.tar.gz
P/PJ/PJFL/web-composablerequest/Web-ComposableRequest-0.22.1.tar.gz
P/PJ/PJFL/yakuake-sessions/Yakuake-Sessions-0.15.1.tar.gz
P/PK/PKALUSKI/LRpt_0.16.tar.gz
P/PM/PMAREK/DigestManberHash/Digest-ManberHash-0.7.tar.gz
P/PM/PMAREK/Games/Hack/Patch/Games-Hack-Patch-i686-0.52.tar.gz
P/PM/PMAREK/Games/Hack/Patch/Games-Hack-Patch-x86_64-0.13.tar.gz
P/PM/PMAREK/Win32/MSI/Win32-MSI-DB-1.06.tar.gz
P/PO/PORTON/DBI/DBI-Transaction-0.101.tar.gz
P/PT/PTANDLER/PBib/Bundle-PBib-2.08.01.tar.gz
P/PT/PTANDLER/PBib/Bundle-PBib-2.08.tar.gz
P/PT/PTHOMSEN/BoxBackup/BBConfig-0.03.tar.gz
R/RA/RAYNERLUC/Finance-Bank-Halifax-Sharedealing/Finance-Bank-Halifax-Sharedealing-0.03.tar.gz
R/RC/RCALEY/speech_pm_1.0.tgz
R/RC/RCHIBOIS/Pots/Pots-0.01.tar.gz
R/RE/REENDERSF/App-NetdiscoX-Web-Plugin-GraphLinkSwitch_0.1.tar.gz
R/RE/RETOH/CMS/CMS-MediaWiki-0.8014.tar.gz
R/RE/RETOH/Geo/Geo-Query-0.04.tar.gz
R/RE/RETOH/Geo/Geo-Query-LatLong-0.8011.tar.gz
R/RE/RETOH/Template/Website-1.14.01.tar.gz
R/RH/RHASE/Tivoli_0.01.tar.gz
R/RI/RINGERC/RT/Extension/PushoverNotify/RT-Extension-PushoverNotify-0.01.tar.gz
R/RI/RINGERC/RT/Extension/SMSNotify/RT-Extension-SMSNotify-1.04.tar.gz
R/RI/RINGERC/RT/Extension/ShiftPlanning/RT-Extension-ShiftPlanning-0.01.tar.gz
R/RI/RINGERC/SMS/Send/RedOxygen/SMS-Send-RedOxygen-1.06.tar.gz
R/RI/RINGERC/WebService/ShiftPlanning/WebService-ShiftPlanning-0.01.tar.gz
R/RJ/RJB/SMcli/SANtricity-SMcli-0.02.tar.gz
R/RK/RKACKLEY/Tk-Pgplot/Tk-Pgplot-0.40.tar.gz
R/RM/RMCFARLA/AI-LibNeural/AI-LibNeural-0.02.tar.gz
R/RM/RMCFARLA/Gtk2-Perl-Ex/Gtk2-Ex-Simple-List-0.50.tar.gz
R/RM/RMCFARLA/Gtk2-Perl-Ex/Gtk2-Ex-Simple-Menu-0.7.tar.gz
R/RM/RMCFARLA/Gtk2-Perl-Ex/Gtk2-Ex-Simple-Tree-0.50.tar.gz
R/RM/RMCFARLA/Gtk2-Perl/Gnome2-Print-0.94.tar.gz
R/RM/RMCFARLA/Gtk2-Perl/Gtk2-1.023.tar.gz
R/RO/ROBERTMAY/Win32-GUI-HyperLink/Win32-GUI-HyperLink-0.14.tar.gz
R/RO/ROBERTMAY/Win32-GUI-OPenGLFrame/Win32-GUI-OpenGLFrame-0.02.tar.gz
R/RO/ROBERTMAY/Win32-GUI-SplashScreen/Win32-GUI-SplashScreen-0.04.tar.gz
R/RO/ROBERTMAY/Win32-GUI/Win32-GUI-1.06.tar.gz
R/RO/ROBERTMAY/Win32-VisualStyles/Win32-VisualStyles-0.02.tar.gz
R/RO/ROHITBASU/OTRS/OTRS-Kernel-Module-TicketAutoAssignment-1.0.1.zip
R/RO/ROHITBASU/OTRS/OTRS-Kernel-System-MultilevelEscalation-1.0.1.zip
R/RO/ROOT/Biblio/Biblio-Thesaurus-SQLite-0.10.tar.gz
R/RS/RSMITH/vuser/VUser-Email-0.3.2.tar.gz
R/RS/RSMITH/vuser/VUser-Email-Postfix-SQL-0.1.1.tar.gz
R/RS/RSMITH/vuser/VUser-ExtLib-SQL-0.1.1.tar.gz
R/RS/RSMITH/vuser/VUser-Firewall-0.1.0.tar.gz
R/RS/RSMITH/vuser/VUser-Google-Api-1.0.1.tar.gz
R/RS/RSMITH/vuser/VUser-Google-Apps-0.2.0.tar.gz
R/RS/RSMITH/vuser/VUser-Install-0.1.0.tar.gz
R/RS/RSMITH/vuser/VUser-Radius-0.1.1.tar.gz
R/RS/RSMITH/vuser/VUser-Radius-Acct-0.1.0.tar.gz
R/RS/RSMITH/vuser/VUser-Radius-Acct-SQL-0.1.0.tar.gz
R/RS/RSMITH/vuser/VUser-Radius-SQL-0.1.1.tar.gz
R/RS/RSMITH/vuser/VUser-SpamAssassin-0.3.0.tar.gz
R/RS/RSMITH/vuser/VUser-SpamAssassin-SQL-0.1.0.tar.gz
R/RS/RSMITH/vuser/VUser-SquirrelMail-0.1.1.tar.gz
R/RS/RSMITH/vuser/vsoapd-0.4.1.tar.gz
R/RS/RSMITH/vuser/vuser-0.5.0.tar.gz
R/RS/RST/FrameMaker-MifTree/FrameMaker-MifTree-0.075.tar.gz
R/RW/RWP/oo_sub/oo_sub-v1.1.0.tar.gz
R/RY/RYOCHIN/Acme-Void/Acme-Void-0.02.tar.gz
R/RY/RYOCHIN/Log-Dispatch-MacGrowl/Log-Dispatch-MacGrowl-0.04.tar.gz
S/SA/SANDEEPV/GuiBuilder_v0_3.zip
S/SA/SAXJAZMAN/RT/RT-CIFMinimal-0.01.tar.gz
S/SA/SAXJAZMAN/RT/RT-IODEF-0.08.tar.gz
S/SA/SAXJAZMAN/Snort/Snort-Rule-1.07.tar.gz
S/SA/SAXJAZMAN/cif/RINO-Client-0.03.tar.gz
S/SA/SAXJAZMAN/database/Database-Schema-Config-.02.tar.gz
S/SA/SAXJAZMAN/iodef/Iodef-Pb-Simple-0.01.tar.gz
S/SA/SAXJAZMAN/iodef/Iodef-Pb-Simple-0.21.tar.gz
S/SA/SAXJAZMAN/iodef/XML-IODEF-0.11.tar.gz
S/SA/SAXJAZMAN/iodef/XML-IODEF-PhraudReport-0.01.tar.gz
S/SA/SAXJAZMAN/iodef/XML-IODEF-Simple-0.02.tar.gz
S/SA/SAXJAZMAN/lwp/LWPx-ParanoidAgent-1.12.tar.gz
S/SA/SAXJAZMAN/malware/Malware-1.02.tar.gz
S/SA/SAXJAZMAN/malware/XML-Malware-0.01.tar.gz
S/SA/SAXJAZMAN/net/Net-Abuse-Utils-Spamhaus-0.09.tar.gz
S/SA/SAXJAZMAN/net/Net-Connection-Simple-1.02.tar.gz
S/SA/SAXJAZMAN/net/Net-DNS-Match-0.05.tar.gz
S/SA/SAXJAZMAN/protobuf/Google-ProtocolBuffers-0.12.tar.gz
S/SC/SCHROEER/ChainMake/ChainMake-0.9.0.tar.gz
S/SC/SCHWIGON/clarity/Clarity-XOG-Merge-1.001.tar.gz
S/SC/SCHWIGON/class-methodmaker/Class-MethodMaker-2.25.tar.gz
S/SC/SCHWIGON/idempotent/App-Prove-Plugin-Idempotent-0.01.tar.gz
S/SC/SCHWIGON/linux-personality/Linux-Personality-0.01.tar.gz
S/SC/SCHWIGON/lockfile-simple/LockFile-Simple-0.208.tar.gz
S/SC/SCHWIGON/object-tiny-rw/Object-Tiny-RW-1.07.tar.gz
S/SC/SCHWIGON/pdf-fdf-simple/PDF-FDF-Simple-0.21.tar.gz
S/SC/SCHWIGON/perl-formance/Benchmark-Perl-Formance-0.53.tar.gz
S/SC/SCHWIGON/perl-formance/Benchmark-Perl-Formance-0.55.tar.gz
S/SC/SCHWIGON/perl-formance/Benchmark-Perl-Formance-Cargo-0.07.tar.gz
S/SC/SCHWIGON/test/Test-Mock-Test-0.01.tar.gz
S/SC/SCHWIGON/twikiclient/WWW-TWikiClient-0.11.tar.gz
S/SC/SCILLEY/POE/Component/IRC/Plugin/IRCDHelp-0.02.tar.gz
S/SE/SENGER/NET-IPFilterSimple_V1.1.tar.gz
S/SE/SENGER/NET-IPFilter_V1.1.2.tar.gz
S/SF/SFLEX/SF_form_secure/SF_form_secure_v4.0.tar.gz
S/SH/SHIMI/Win32-Watir-0.06-withoutworldwriteables.tar.gz
S/SH/SHIRIRU/Data/Data-NestedSet-1.03.tar.gz
S/SH/SHOGE/video-info/Video-Info-Quicktime_PL-0.07.tar.gz
S/SH/SHY/Wifi/Wifi-0.01a.tar.gz
S/SH/SHY/acpi/Acpi-0.1.tar.gz
S/SI/SIMATIKA/subclustv1_0.tar.gz
S/SK/SKATTOOR/Device-Modem-GSM/Device-Modem-GSM-0.3.tar.gz
S/SK/SKNPP/basis/basis-0.05003.tar.gz
S/SK/SKUPSY/OpenSSL/Crypt-OpenSSL-PBKDF2-0.04.tar.gz
S/SK/SKUPSY/VisionDB/VisionDB-Read-0.04.tar.gz
S/SL/SLENK/MCPrimers/Bio-MCPrimers-2.5.tar.gz
S/SM/SMANROSS/Win32-Exchange_v0.046a.tar.gz
S/SM/SMCNABB/Drupal-Module-Starter/Drupal-Module-Starter-0.06.tar.gz
S/SM/SMCNABB/POE-Framework-MIDI/POE-Framework-MIDI-0.09.tar.gz
S/SM/SMCNABB/POE-Framework-MIDI/POE-Framework-MIDI-0.1.tar.gz
S/SN/SNEVINE/WWW-Bleep-092/WWW-Bleep-0.92.tgz
S/SO/SOMIAN/experimental/unregistered/Filesys-CygwinPaths-0.04.tar.gz
S/SO/SOMMAR/Win32-SqlServer-2.015-withoutworldwriteables.tar.gz
S/SO/SORBS/C/C-Mlock-1.12.tar.gz
S/SO/SORBS/Mail/Pegasus/Mail-Pegasus-1.06.tgz
S/SO/SORBS/WebService/WebService-VirusTotal-1.0.3165.tar.gz
S/SP/SPARSONS/tools/Net-Dev-MIBLoadOrder-1.1.0.tar.gz
S/SP/SPARSONS/tools/Net-Dev-Tools-Syslog-1.0.0.tar.gz
S/SU/SUMUS/DWH_File.0.03.tar.gz
S/SU/SUNILS/PDF/Xtract/008/PDF-Xtract-0.08.tar.gz
S/SW/SWORDSMAN/TSM_0.60.tar.gz
T/TB/TBONECA/BSD/Jail/BSD-Jail-0.01.tar.gz
T/TD/TDWINTER/Hash/KeyMorpher/Hash-KeyMorpher-0.09.tar.gz
T/TD/TDWINTER/Net/Das/Net-DAS-0.19.tar.gz
T/TE/TEDK/Win32/SimpleProcess/SimpleProcess_1.0.zip
T/TE/TEEJAY/Data/Iterator/Data-Iterator-EasyObj-0.01.tar.gz
T/TE/TEEJAY/GraphViz/GraphViz-DBI-0.02.tar.gz
T/TE/TEEJAY/Imager/Chart/Imager-Chart-Radial-0.1.tar.gz
T/TE/TELS/audio/Audio-Audiere-0.05.tar.gz
T/TE/TELS/convert/Convert-Morse-0.06.tar.gz
T/TE/TELS/convert/Convert-Wiki-0.05.tar.gz
T/TE/TELS/devel/Devel-Graph-0.12.tar.gz
T/TE/TELS/devel/Devel-Size-Report-0.13.tar.gz
T/TE/TELS/games/Games-3D-0.10.tar.gz
T/TE/TELS/games/Games-Console-0.04.tar.gz
T/TE/TELS/games/Games-Irrlicht-0.04.tar.gz
T/TE/TELS/games/Games-OpenGL-Font-2D-0.07.tar.gz
T/TE/TELS/graph/Graph-Convert-0.09.tar.gz
T/TE/TELS/graph/Graph-Dependency-0.02.tar.gz
T/TE/TELS/graph/Graph-Easy-Manual-0.41.tar.gz
T/TE/TELS/graph/Graph-Flowchart-0.11.tar.gz
T/TE/TELS/graph/Graph-Regexp-0.05.tar.gz
T/TE/TELS/graph/Graph-Usage-0.12.tar.gz
T/TE/TELS/math/Math-Fractal-Mandelbrot-0.04.tar.gz
T/TE/TELS/sdl/SDL-App-FPS-0.21.tar.gz
T/TE/TELS/wiki/Mediawiki-POD-0.06.tar.gz
T/TH/THALJEF/criticism/criticism-1.02.tar.gz
T/TI/TIMB/perl5.004_04.tar.gz
T/TO/TOMO/src/Net-IP-Match-Bin-0.14.tar.gz
T/TO/TOMO/src/Net-POP3S-0.12.tar.gz
T/TO/TOMO/src/Net-SMTPS-0.10.tar.gz
T/TP/TPABA/Config-ReadAndCheck/Config-ReadAndCheck-0.04.tar.gz
T/TP/TPABA/File-DirList/File-DirList-0.05.tar.gz
T/TP/TPABA/Getopt-Std-WithCheck/Getopt-Std-WithCheck-0.04.tar.gz
T/TP/TPABA/Net-Socket-NonBlock/Net-Socket-NonBlock-0.15.tar.gz
T/TP/TPABA/Nginx/Nginx-ReadBody-0.07.tar.gz
T/TP/TPABA/Term-Screen/Term-Screen-Uni-0.04.tar.gz
T/TP/TPABA/Term-Screen/Term-Screen-Win32-0.03.tar.gz
T/TP/TPABA/UniLog/UniLog-0.14.tar.gz
T/TR/TRIZOR/dists/Test-AskAnExpert-0.03.tar.gz
T/TR/TRIZOR/dists/Test-Smart-0.02.tar.gz
T/TS/TSCHULZ/TSCHULZ/Lingua-DE-Tagger-0.16.tar.gz
T/TT/TTG/Google-GeoCoder-Smart/Google-GeoCoder-Smart-1.18.tar.gz
T/TU/TURNERMM/HTML-WikiConverter-DokuWikiFCK-0.32_Feb-27-2013.tgz
T/TU/TUSHAR/Log-SelfHistory_0.1.tar.gz
T/TU/TUSHAR/Text-CSV-UniqueColumns_0.4.tar.gz
T/TW/TWITTEK/konstrukt/Konstrukt-0.5-beta13.tar.gz
T/TW/TWITTEK/konstrukt/Konstrukt-0.5-beta6.tar.gz
T/TW/TWYLIE/modules/BLASTaid-v0.0.3.tar.gz
T/TW/TWYLIE/modules/DateStamp-v1.0.4.tar.gz
T/TW/TWYLIE/modules/EOL-v0.0.2.tar.gz
T/TW/TWYLIE/modules/FASTAParse-0.0.3.tar.gz
U/UH/UHANDA/contribs/CMD-Colors-0.1.tar.gz
U/UN/UNICOLET/Win32-TaskScheduler2.0.3.zip
V/VA/VAYDE/Lazy/CGI-Lazy-1.10.tar.gz
V/VA/VAYDE/Lazy/CGI-Lazy-ModPerl-0.04.tar.gz
V/VG/VGIRISH/Net/Net-Ifstat-0.01.tar.gz
V/VG/VGIRISH/Net/Net-Netcat-0.05.tar.gz
V/VG/VGIRISH/Progress/Progress-PV-0.02.tar.gz
V/VR/VRAG/Log-Any-Adapter-Multiplexor-003/Log-Any-Adapter-Multiplexor-0.03.tar.gz
V/VT/VTFRVL/Plack-Middleware-DNSBL/Plack-Middleware-DNSBL-0.0304.tar.gz
W/WA/WAYNEDAV/DBZ_File_1.1.tar.gz
W/WD/WDS/Dreamer/VM-Dreamer-0.851.tar.gz
W/WI/WIHAA/Graphics/Graphics-EPS-3.00.tar.gz
W/WI/WITHERING/HOI-Comprehensions-0045/HOI-Comprehensions-0.045.tar.gz
W/WI/WITHERING/HOIMatch-0072/HOI-Match-0.072.tar.gz
W/WI/WITHERING/Wraith-012/Wraith-0.12.tar.gz
W/WI/WITTEN/Net_TCLink.pm-3.4.tar.gz
W/WN/WNEESSEN/OpenVAS-OMP_0.04.tar.gz
W/WO/WOLFTOWER/Algorithm/TrunkClassifier_v101/Algorithm-TrunkClassifier-v1.0.1.tar.gz
W/WO/WORR/Formspring/WWW-Formspring-0.03.tar.gz
W/WY/WYCHUNG/wychung/pamlyn-0.01.tar.gz
X/XI/XIONG/developer-tools/Acme-Teddy-1.002003.tar.gz
X/XI/XIONG/developer-tools/Cheat-Meta-v0.0.5.tar.gz
X/XI/XIONG/developer-tools/Class-Lite-v0.1.0.tar.gz
X/XI/XIONG/developer-tools/Devel-Comments-v1.1.4.tar.gz
X/XI/XIONG/developer-tools/Error-Base-v1.0.2.tar.gz
X/XI/XIONG/developer-tools/Test-Ranger-v0.0.4.tar.gz
X/XX/XXGHOSTXX/Facebook/FB-Statify-1.04.zip
X/XX/XXGHOSTXX/Tinder/Tinder-API-0.03.zip
Y/YU/YUMPY/Namegame/Lingua-EN-Namegame-0.05.tar.gz
Z/ZA/ZACS/Mail/Mail-Queue-DB-0.03.tar.gz
Z/ZA/ZAPAD/Log-Sentry/Log-Sentry-0.01.tar.gz
Z/ZE/ZEUS/App-Regather/App-Regather-0.85.00.tar.gz
Z/ZE/ZEUS/App-gqmt/App-gqmt-1.02.tar.gz
Z/ZM/ZMIJ/Array/Array-Utils-0.5.tar.gz
Z/ZM/ZMIJ/Cache/Memcached/Cache-Memcached-Semaphore-0.3.tar.gz
