use 5.014;
use warnings;
use Test::More 0.98;

sub is_close {
    my ($x, $y, $desc) = @_;

    if (defined($y)) {
        ok(abs($x - $y) < 1e-9, $desc);
    }
}

sub example01 {
    my $bspline;
    my $u;
    my $s;
    my $template;
    my $D;

    $bspline = Math::BSpline::Basis->new(
        degree         => 2,
        knot_vector    => [0, 0, 0, 1, 2, 3, 4, 4, 5, 5, 5],
    );

    $u = 5/2;
    $s = $bspline->find_knot_span($u);
    is($s, 4, 'knot span index 4');
    # TODO: calculate remaining values
    $template = [
        [1/8, 3/4, 1/8],
        [undef, undef, 1/2],
        [undef, undef, 1],
    ];
    $D = $bspline->evaluate_basis_derivatives($s, $u, 2);
    is(
        scalar(@$D),
        scalar(@$template),
        sprintf(
            '%d derivatives',
            scalar(@$template),
        )
    );    
    for (my $k=0;$k<@$template;$k++) {
        my $this_D        = $D->[$k];
        my $this_template = $template->[$k];
        is(
            scalar(@$this_D),
            scalar(@$this_template),
            sprintf(
                '%d evaluated basis functions',
                scalar(@$this_template),
            )
        );
        for (my $i=0;$i<@$this_template;$i++) {
            is_close(
                $this_D->[$i],
                $this_template->[$i],
                sprintf(
                    'derivative %d, value %d (%.3f) is close to %.3f',
                    $k,
                    $i,
                    $this_D->[$i],
                    $this_template->[$i] // 0,
                ),
            );
        }
    }
}

sub gsl {
    my $bspline;
    my $d;

    # This function reads in blocks of data for testing. Blocks are
    # separated by blank lines and each block has the following
    # structure (see example below in the DATA section):
    # p d knot_vector
    # u d+1
    # [d+1 lines of derivatives]
    # u d+1
    # [d+1 lines of derivatives]
    # ...
    #
    # This structure is printed out by evaluate_basis_derivatives.c.

    my $state = 0;
    while (my $line = <DATA>) {
        $line =~ s/^\s+//;
        $line =~ s/\s+$//;
        if ($line eq '') {
            $state = 0;
            next;
        }
        elsif ($state == 0) {
            my $pieces = [split(/\s+/, $line)];
            my $p      = shift(@$pieces);
            $d = shift(@$pieces);
            $bspline = Math::BSpline::Basis->new(
                degree      => $p,
                knot_vector => $pieces,
            );
            $state = 1;
            next;
        }
        elsif ($state == 1) {
            my $pieces   = [split(/\s+/, $line)];
            my $u        = $pieces->[0];
            my $n_rows   = $pieces->[1];
            my $template = [];
            for (my $i=0;$i<$n_rows;$i++) {
                $line = <DATA>;
                $line =~ s/^\s+//;
                $line =~ s/\s+$//;
                push(
                    @$template,
                    [split(/\s+/, $line)],
                );
            }

            my $s = $bspline->find_knot_span($u);
            my $D = $bspline->evaluate_basis_derivatives($s, $u, $d);
            is(
                scalar(@$D),
                scalar(@$template),
                sprintf(
                    '%d derivatives at %.3f',
                    scalar(@$template),
                    $u,
                )
            );
            for (my $k=0;$k<@$template;$k++) {
                my $this_D = [
                    (map { 0 } (1..($s-$bspline->degree))),
                    @{$D->[$k]},
                ];
                my $this_template = $template->[$k];
                ok(
                    scalar(@$this_D) <= scalar(@$this_template),
                    sprintf(
                        'at most %d basis functions for %d. derivative',
                        scalar(@$this_template),
                        $k,
                    ),
                );
                for (my $i=0;$i<@$this_template;$i++) {
                    is_close(
                        $this_D->[$i] // 0,
                        $this_template->[$i],
                        sprintf(
                            'derivative %d, value %d (%.3f) is close to %.3f',
                            $k,
                            $i,
                            $this_D->[$i] // 0,
                            $this_template->[$i],
                        ),
                    );
                }
            }
        }
    }
}

use_ok('Math::BSpline::Basis');
example01;
gsl;
done_testing;

__END__
3 3 0 2 4 6 8
-0.1 4
0 0 0 0 0 0 0
0 0 0 0 0 0 0
0 0 0 0 0 0 0
0 0 0 0 0 0 0
8.1 4
0 0 0 0 0 0 0
0 0 0 0 0 0 0
0 0 0 0 0 0 0
0 0 0 0 0 0 0

3 2 0.0000000000 2.0000000000 4.0000000000 6.0000000000 8.0000000000
0.000000000000000 3
1.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-1.5000000000 1.5000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000
1.5000000000 -2.2500000000 0.7500000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000
0.200000000000000 3
0.7290000000 0.2567500000 0.0140833333 0.0001666667 0.0000000000 0.0000000000 0.0000000000
-1.2150000000 1.0762500000 0.1362500000 0.0025000000 0.0000000000 0.0000000000 0.0000000000
1.3500000000 -1.9875000000 0.6125000000 0.0250000000 0.0000000000 0.0000000000 0.0000000000
0.400000000000000 3
0.5120000000 0.4340000000 0.0526666667 0.0013333333 0.0000000000 0.0000000000 0.0000000000
-0.9600000000 0.7050000000 0.2450000000 0.0100000000 0.0000000000 0.0000000000 0.0000000000
1.2000000000 -1.7250000000 0.4750000000 0.0500000000 0.0000000000 0.0000000000 0.0000000000
0.600000000000000 3
0.3430000000 0.5422500000 0.1102500000 0.0045000000 0.0000000000 0.0000000000 0.0000000000
-0.7350000000 0.3862500000 0.3262500000 0.0225000000 0.0000000000 0.0000000000 0.0000000000
1.0500000000 -1.4625000000 0.3375000000 0.0750000000 0.0000000000 0.0000000000 0.0000000000
0.800000000000000 3
0.2160000000 0.5920000000 0.1813333333 0.0106666667 0.0000000000 0.0000000000 0.0000000000
-0.5400000000 0.1200000000 0.3800000000 0.0400000000 0.0000000000 0.0000000000 0.0000000000
0.9000000000 -1.2000000000 0.2000000000 0.1000000000 0.0000000000 0.0000000000 0.0000000000
1.000000000000000 3
0.1250000000 0.5937500000 0.2604166667 0.0208333333 0.0000000000 0.0000000000 0.0000000000
-0.3750000000 -0.0937500000 0.4062500000 0.0625000000 0.0000000000 0.0000000000 0.0000000000
0.7500000000 -0.9375000000 0.0625000000 0.1250000000 0.0000000000 0.0000000000 0.0000000000
1.200000000000000 3
0.0640000000 0.5580000000 0.3420000000 0.0360000000 0.0000000000 0.0000000000 0.0000000000
-0.2400000000 -0.2550000000 0.4050000000 0.0900000000 0.0000000000 0.0000000000 0.0000000000
0.6000000000 -0.6750000000 -0.0750000000 0.1500000000 0.0000000000 0.0000000000 0.0000000000
1.400000000000000 3
0.0270000000 0.4952500000 0.4205833333 0.0571666667 0.0000000000 0.0000000000 0.0000000000
-0.1350000000 -0.3637500000 0.3762500000 0.1225000000 0.0000000000 0.0000000000 0.0000000000
0.4500000000 -0.4125000000 -0.2125000000 0.1750000000 0.0000000000 0.0000000000 0.0000000000
1.600000000000000 3
0.0080000000 0.4160000000 0.4906666667 0.0853333333 0.0000000000 0.0000000000 0.0000000000
-0.0600000000 -0.4200000000 0.3200000000 0.1600000000 0.0000000000 0.0000000000 0.0000000000
0.3000000000 -0.1500000000 -0.3500000000 0.2000000000 0.0000000000 0.0000000000 0.0000000000
1.800000000000000 3
0.0010000000 0.3307500000 0.5467500000 0.1215000000 0.0000000000 0.0000000000 0.0000000000
-0.0150000000 -0.4237500000 0.2362500000 0.2025000000 0.0000000000 0.0000000000 0.0000000000
0.1500000000 0.1125000000 -0.4875000000 0.2250000000 0.0000000000 0.0000000000 0.0000000000
2.000000000000000 3
0.0000000000 0.2500000000 0.5833333333 0.1666666667 0.0000000000 0.0000000000 0.0000000000
-0.0000000000 -0.3750000000 0.1250000000 0.2500000000 0.0000000000 0.0000000000 0.0000000000
0.0000000000 0.3750000000 -0.6250000000 0.2500000000 0.0000000000 0.0000000000 0.0000000000
2.200000000000000 3
0.0000000000 0.1822500000 0.5964166667 0.2211666667 0.0001666667 0.0000000000 0.0000000000
0.0000000000 -0.3037500000 0.0087500000 0.2925000000 0.0025000000 0.0000000000 0.0000000000
0.0000000000 0.3375000000 -0.5375000000 0.1750000000 0.0250000000 0.0000000000 0.0000000000
2.400000000000000 3
0.0000000000 0.1280000000 0.5880000000 0.2826666667 0.0013333333 0.0000000000 0.0000000000
0.0000000000 -0.2400000000 -0.0900000000 0.3200000000 0.0100000000 0.0000000000 0.0000000000
0.0000000000 0.3000000000 -0.4500000000 0.1000000000 0.0500000000 0.0000000000 0.0000000000
2.600000000000000 3
0.0000000000 0.0857500000 0.5615833333 0.3481666667 0.0045000000 0.0000000000 0.0000000000
0.0000000000 -0.1837500000 -0.1712500000 0.3325000000 0.0225000000 0.0000000000 0.0000000000
0.0000000000 0.2625000000 -0.3625000000 0.0250000000 0.0750000000 0.0000000000 0.0000000000
2.800000000000000 3
0.0000000000 0.0540000000 0.5206666667 0.4146666667 0.0106666667 0.0000000000 0.0000000000
0.0000000000 -0.1350000000 -0.2350000000 0.3300000000 0.0400000000 0.0000000000 0.0000000000
0.0000000000 0.2250000000 -0.2750000000 -0.0500000000 0.1000000000 0.0000000000 0.0000000000
3.000000000000000 3
0.0000000000 0.0312500000 0.4687500000 0.4791666667 0.0208333333 0.0000000000 0.0000000000
0.0000000000 -0.0937500000 -0.2812500000 0.3125000000 0.0625000000 0.0000000000 0.0000000000
0.0000000000 0.1875000000 -0.1875000000 -0.1250000000 0.1250000000 0.0000000000 0.0000000000
3.200000000000001 3
0.0000000000 0.0160000000 0.4093333333 0.5386666667 0.0360000000 0.0000000000 0.0000000000
0.0000000000 -0.0600000000 -0.3100000000 0.2800000000 0.0900000000 0.0000000000 0.0000000000
0.0000000000 0.1500000000 -0.1000000000 -0.2000000000 0.1500000000 0.0000000000 0.0000000000
3.400000000000001 3
0.0000000000 0.0067500000 0.3459166667 0.5901666667 0.0571666667 0.0000000000 0.0000000000
0.0000000000 -0.0337500000 -0.3212500000 0.2325000000 0.1225000000 0.0000000000 0.0000000000
0.0000000000 0.1125000000 -0.0125000000 -0.2750000000 0.1750000000 0.0000000000 0.0000000000
3.600000000000001 3
0.0000000000 0.0020000000 0.2820000000 0.6306666667 0.0853333333 0.0000000000 0.0000000000
0.0000000000 -0.0150000000 -0.3150000000 0.1700000000 0.1600000000 0.0000000000 0.0000000000
0.0000000000 0.0750000000 0.0750000000 -0.3500000000 0.2000000000 0.0000000000 0.0000000000
3.800000000000001 3
0.0000000000 0.0002500000 0.2210833333 0.6571666667 0.1215000000 0.0000000000 0.0000000000
0.0000000000 -0.0037500000 -0.2912500000 0.0925000000 0.2025000000 0.0000000000 0.0000000000
0.0000000000 0.0375000000 0.1625000000 -0.4250000000 0.2250000000 0.0000000000 0.0000000000
4.000000000000001 3
0.0000000000 0.0000000000 0.1666666667 0.6666666667 0.1666666667 0.0000000000 0.0000000000
0.0000000000 0.0000000000 -0.2500000000 -0.0000000000 0.2500000000 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.2500000000 -0.5000000000 0.2500000000 0.0000000000 0.0000000000
4.200000000000001 3
0.0000000000 0.0000000000 0.1215000000 0.6571666667 0.2210833333 0.0002500000 0.0000000000
0.0000000000 0.0000000000 -0.2025000000 -0.0925000000 0.2912500000 0.0037500000 0.0000000000
0.0000000000 0.0000000000 0.2250000000 -0.4250000000 0.1625000000 0.0375000000 0.0000000000
4.400000000000001 3
0.0000000000 0.0000000000 0.0853333333 0.6306666667 0.2820000000 0.0020000000 0.0000000000
0.0000000000 0.0000000000 -0.1600000000 -0.1700000000 0.3150000000 0.0150000000 0.0000000000
0.0000000000 0.0000000000 0.2000000000 -0.3500000000 0.0750000000 0.0750000000 0.0000000000
4.600000000000001 3
0.0000000000 0.0000000000 0.0571666667 0.5901666667 0.3459166667 0.0067500000 0.0000000000
0.0000000000 0.0000000000 -0.1225000000 -0.2325000000 0.3212500000 0.0337500000 0.0000000000
0.0000000000 0.0000000000 0.1750000000 -0.2750000000 -0.0125000000 0.1125000000 0.0000000000
4.800000000000002 3
0.0000000000 0.0000000000 0.0360000000 0.5386666667 0.4093333333 0.0160000000 0.0000000000
0.0000000000 0.0000000000 -0.0900000000 -0.2800000000 0.3100000000 0.0600000000 0.0000000000
0.0000000000 0.0000000000 0.1500000000 -0.2000000000 -0.1000000000 0.1500000000 0.0000000000
5.000000000000002 3
0.0000000000 0.0000000000 0.0208333333 0.4791666667 0.4687500000 0.0312500000 0.0000000000
0.0000000000 0.0000000000 -0.0625000000 -0.3125000000 0.2812500000 0.0937500000 0.0000000000
0.0000000000 0.0000000000 0.1250000000 -0.1250000000 -0.1875000000 0.1875000000 0.0000000000
5.200000000000002 3
0.0000000000 0.0000000000 0.0106666667 0.4146666667 0.5206666667 0.0540000000 0.0000000000
0.0000000000 0.0000000000 -0.0400000000 -0.3300000000 0.2350000000 0.1350000000 0.0000000000
0.0000000000 0.0000000000 0.1000000000 -0.0500000000 -0.2750000000 0.2250000000 0.0000000000
5.400000000000002 3
0.0000000000 0.0000000000 0.0045000000 0.3481666667 0.5615833333 0.0857500000 0.0000000000
0.0000000000 0.0000000000 -0.0225000000 -0.3325000000 0.1712500000 0.1837500000 0.0000000000
0.0000000000 0.0000000000 0.0750000000 0.0250000000 -0.3625000000 0.2625000000 0.0000000000
5.600000000000002 3
0.0000000000 0.0000000000 0.0013333333 0.2826666667 0.5880000000 0.1280000000 0.0000000000
0.0000000000 0.0000000000 -0.0100000000 -0.3200000000 0.0900000000 0.2400000000 0.0000000000
0.0000000000 0.0000000000 0.0500000000 0.1000000000 -0.4500000000 0.3000000000 0.0000000000
5.800000000000002 3
0.0000000000 0.0000000000 0.0001666667 0.2211666667 0.5964166667 0.1822500000 0.0000000000
0.0000000000 0.0000000000 -0.0025000000 -0.2925000000 -0.0087500000 0.3037500000 0.0000000000
0.0000000000 0.0000000000 0.0250000000 0.1750000000 -0.5375000000 0.3375000000 0.0000000000
6.000000000000003 3
0.0000000000 0.0000000000 0.0000000000 0.1666666667 0.5833333333 0.2500000000 0.0000000000
0.0000000000 0.0000000000 0.0000000000 -0.2500000000 -0.1250000000 0.3750000000 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.2500000000 -0.6250000000 0.3750000000 0.0000000000
6.200000000000003 3
0.0000000000 0.0000000000 0.0000000000 0.1215000000 0.5467500000 0.3307500000 0.0010000000
0.0000000000 0.0000000000 0.0000000000 -0.2025000000 -0.2362500000 0.4237500000 0.0150000000
0.0000000000 0.0000000000 0.0000000000 0.2250000000 -0.4875000000 0.1125000000 0.1500000000
6.400000000000003 3
0.0000000000 0.0000000000 0.0000000000 0.0853333333 0.4906666667 0.4160000000 0.0080000000
0.0000000000 0.0000000000 0.0000000000 -0.1600000000 -0.3200000000 0.4200000000 0.0600000000
0.0000000000 0.0000000000 0.0000000000 0.2000000000 -0.3500000000 -0.1500000000 0.3000000000
6.600000000000003 3
0.0000000000 0.0000000000 0.0000000000 0.0571666667 0.4205833333 0.4952500000 0.0270000000
0.0000000000 0.0000000000 0.0000000000 -0.1225000000 -0.3762500000 0.3637500000 0.1350000000
0.0000000000 0.0000000000 0.0000000000 0.1750000000 -0.2125000000 -0.4125000000 0.4500000000
6.800000000000003 3
0.0000000000 0.0000000000 0.0000000000 0.0360000000 0.3420000000 0.5580000000 0.0640000000
0.0000000000 0.0000000000 0.0000000000 -0.0900000000 -0.4050000000 0.2550000000 0.2400000000
0.0000000000 0.0000000000 0.0000000000 0.1500000000 -0.0750000000 -0.6750000000 0.6000000000
7.000000000000004 3
0.0000000000 0.0000000000 0.0000000000 0.0208333333 0.2604166667 0.5937500000 0.1250000000
0.0000000000 0.0000000000 0.0000000000 -0.0625000000 -0.4062500000 0.0937500000 0.3750000000
0.0000000000 0.0000000000 0.0000000000 0.1250000000 0.0625000000 -0.9375000000 0.7500000000
7.200000000000004 3
0.0000000000 0.0000000000 0.0000000000 0.0106666667 0.1813333333 0.5920000000 0.2160000000
0.0000000000 0.0000000000 0.0000000000 -0.0400000000 -0.3800000000 -0.1200000000 0.5400000000
0.0000000000 0.0000000000 0.0000000000 0.1000000000 0.2000000000 -1.2000000000 0.9000000000
7.400000000000004 3
0.0000000000 0.0000000000 0.0000000000 0.0045000000 0.1102500000 0.5422500000 0.3430000000
0.0000000000 0.0000000000 0.0000000000 -0.0225000000 -0.3262500000 -0.3862500000 0.7350000000
0.0000000000 0.0000000000 0.0000000000 0.0750000000 0.3375000000 -1.4625000000 1.0500000000
7.600000000000004 3
0.0000000000 0.0000000000 0.0000000000 0.0013333333 0.0526666667 0.4340000000 0.5120000000
0.0000000000 0.0000000000 0.0000000000 -0.0100000000 -0.2450000000 -0.7050000000 0.9600000000
0.0000000000 0.0000000000 0.0000000000 0.0500000000 0.4750000000 -1.7250000000 1.2000000000
7.800000000000004 3
0.0000000000 0.0000000000 0.0000000000 0.0001666667 0.0140833333 0.2567500000 0.7290000000
0.0000000000 0.0000000000 0.0000000000 -0.0025000000 -0.1362500000 -1.0762500000 1.2150000000
0.0000000000 0.0000000000 0.0000000000 0.0250000000 0.6125000000 -1.9875000000 1.3500000000
8.000000000000000 3
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 1.0000000000
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 -1.5000000000 1.5000000000
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.7500000000 -2.2500000000 1.5000000000

5 4 -3.0000000000 -1.6000000000 -0.2000000000 1.2000000000 2.6000000000 4.0000000000
-3.000000000000000 5
1.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-3.5714285714 3.5714285714 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000
10.2040816327 -15.3061224490 5.1020408163 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-21.8658892128 38.2653061224 -20.0437317784 3.6443148688 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000
31.2369845898 -58.5693461058 36.8769956962 -10.8461752048 1.3015410246 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-2.800000000000000 5
0.4626643660 0.4553943297 0.0776951127 0.0041645870 0.0000811087 0.0000004958 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-1.9277681918 1.2002997263 0.6667229979 0.0591392007 0.0015938713 0.0000123956 0.0000000000 0.0000000000 0.0000000000 0.0000000000
6.4258939727 -8.7668084727 1.7912372270 0.5262288751 0.0232004852 0.0002479126 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-16.0647349319 27.4160320105 -13.2622898528 1.6894209093 0.2178531763 0.0037186886 0.0000000000 0.0000000000 0.0000000000 0.0000000000
26.7745582198 -49.9233950140 30.9374235603 -8.7027643905 0.8769907380 0.0371868864 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-2.600000000000000 5
0.1859344321 0.5534598679 0.2311647043 0.0282179603 0.0012071690 0.0000158664 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-0.9296721604 -0.0684238710 0.7989946870 0.1872841783 0.0116188361 0.0001983301 0.0000000000 0.0000000000 0.0000000000 0.0000000000
3.7186886416 -4.2244302969 -0.2820694199 0.7043471746 0.0814806000 0.0019833006 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-11.1560659249 18.2959481169 -7.6687623543 0.1632091126 0.3507962952 0.0148747546 0.0000000000 0.0000000000 0.0000000000 0.0000000000
22.3121318498 -41.2774439222 24.9978514243 -6.5593535762 0.4524404514 0.0743737728 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-2.399999999999999 5
0.0609269947 0.4770445350 0.3766845660 0.0795706402 0.0056527786 0.0001204855 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-0.3807937169 -0.5995455762 0.6205561671 0.3233864610 0.0353926191 0.0010040459 0.0000000000 0.0000000000 0.0000000000 0.0000000000
1.9039685845 -1.3331498780 -1.3554620099 0.6200913310 0.1578583328 0.0066936396 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-7.1398821919 10.9050544416 -3.2631492830 -0.9343205212 0.3988293568 0.0334681978 0.0000000000 0.0000000000 0.0000000000 0.0000000000
17.8497054799 -32.6314928304 19.0582792884 -4.4159427619 0.0278901648 0.1115606592 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-2.199999999999999 5
0.0144582614 0.3429523413 0.4705270511 0.1551381810 0.0164164402 0.0005077250 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-0.1204855120 -0.6887011364 0.3076319611 0.4235448634 0.0748365430 0.0031732810 0.0000000000 0.0000000000 0.0000000000 0.0000000000
0.8032367466 0.2528708276 -1.6665234283 0.3591977769 0.2353516723 0.0158664049 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-4.0161837330 5.2433509847 -0.0454506390 -1.6031679922 0.3619523611 0.0594990183 0.0000000000 0.0000000000 0.0000000000 0.0000000000
13.3872791099 -23.9855417386 13.1187071524 -2.2725319477 -0.3966601218 0.1487475457 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-1.999999999999999 5
0.0019039686 0.2157769084 0.4994577601 0.2447706285 0.0365412807 0.0015494536 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-0.0237996073 -0.5623586898 -0.0110699852 0.4610054866 0.1284755277 0.0077472680 0.0000000000 0.0000000000 0.0000000000 0.0000000000
0.2379960731 0.8794698637 -1.4528365606 0.0074029450 0.2969786068 0.0309890720 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-1.7849705480 1.3108377462 1.9843335779 -1.8433333003 0.2401653081 0.0929672160 0.0000000000 0.0000000000 0.0000000000 0.0000000000
8.9248527399 -15.3395906468 7.1791350165 -0.1291211334 -0.8212104084 0.1859344321 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-1.799999999999999 5
0.0000594990 0.1217349914 0.4712322247 0.3346819777 0.0684357708 0.0038555364 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-0.0014874755 -0.3778187660 -0.2543583031 0.4261617183 0.1914380913 0.0160647349 0.0000000000 0.0000000000 0.0000000000 0.0000000000
0.0297495091 0.8924852740 -0.9519842923 -0.3495567323 0.3257571250 0.0535491164 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-0.4462426370 -0.8924852740 2.8262033676 -1.6548164455 0.0334681978 0.1338727911 0.0000000000 0.0000000000 0.0000000000 0.0000000000
4.4624263700 -6.6936395549 1.2395628805 2.0142896809 -1.2457606949 0.2231213185 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-1.599999999999999 5
0.0000000000 0.0625000000 0.4050925926 0.4108796296 0.1131944444 0.0083333333 0.0000000000 0.0000000000 0.0000000000 0.0000000000
0.0000000000 -0.2232142857 -0.3885582011 0.3265542328 0.2554563492 0.0297619048 0.0000000000 0.0000000000 0.0000000000 0.0000000000
0.0000000000 0.6377551020 -0.4015495087 -0.6259448224 0.3047052154 0.0850340136 0.0000000000 0.0000000000 0.0000000000 0.0000000000
0.0000000000 -1.3666180758 2.4801587302 -1.0376174279 -0.2581389699 0.1822157434 0.0000000000 0.0000000000 0.0000000000 0.0000000000
0.0000000000 1.9523115369 -4.7000092554 4.1577004952 -1.6703109815 0.2603082049 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-1.399999999999999 5
0.0000000000 0.0289165229 0.3223535608 0.4625541827 0.1699310142 0.0162440997 0.0000006198 0.0000000000 0.0000000000 0.0000000000
0.0000000000 -0.1204855120 -0.4252802514 0.1858793404 0.3091759055 0.0506950229 0.0000154945 0.0000000000 0.0000000000 0.0000000000
0.0000000000 0.4016183733 0.0055091684 -0.7558578987 0.2230386810 0.1253817854 0.0003098907 0.0000000000 0.0000000000 0.0000000000
0.0000000000 -1.0040459332 1.6155636210 -0.2892313388 -0.5416889788 0.2147542691 0.0046483608 0.0000000000 0.0000000000 0.0000000000
0.0000000000 1.6734098887 -3.9459418364 3.3261603961 -1.1651891077 0.0650770512 0.0464836080 0.0000000000 0.0000000000 0.0000000000
-1.200000000000000 5
0.0000000000 0.0116209020 0.2393087702 0.4844379078 0.2354337726 0.0291788144 0.0000198330 0.0000000000 0.0000000000 0.0000000000
0.0000000000 -0.0581045100 -0.3968770453 0.0330808344 0.3415646506 0.0800881577 0.0002479126 0.0000000000 0.0000000000 0.0000000000
0.0000000000 0.2324180401 0.2547301720 -0.7527245592 0.0947645822 0.1683326392 0.0024791258 0.0000000000 0.0000000000 0.0000000000
0.0000000000 -0.6972541203 0.9017819956 0.2928467305 -0.7242146130 0.2082465639 0.0185934432 0.0000000000 0.0000000000 0.0000000000
0.0000000000 1.3945082406 -3.1918744174 2.4946202971 -0.6600672339 -0.1301541025 0.0929672160 0.0000000000 0.0000000000 0.0000000000
-1.000000000000000 5
0.0000000000 0.0038079372 0.1660276031 0.4765452666 0.3046391054 0.0488294808 0.0001506069 0.0000000000 0.0000000000 0.0000000000
0.0000000000 -0.0237996073 -0.3318998477 -0.1085581625 0.3453215591 0.1176810010 0.0012550574 0.0000000000 0.0000000000 0.0000000000
0.0000000000 0.1189980365 0.3762761989 -0.6498064078 -0.0599122059 0.2060773289 0.0083670494 0.0000000000 0.0000000000 0.0000000000
0.0000000000 -0.4462426370 0.3388138540 0.7086167800 -0.8057158724 0.1626926281 0.0418352472 0.0000000000 0.0000000000 0.0000000000
0.0000000000 1.1156065925 -2.4378069984 1.6630801981 -0.1549453601 -0.3253852561 0.1394508241 0.0000000000 0.0000000000 0.0000000000
-0.800000000000000 5
0.0000000000 0.0009036413 0.1074724431 0.4428821132 0.3714272905 0.0766798556 0.0006346562 0.0000000000 0.0000000000 0.0000000000
0.0000000000 -0.0075303445 -0.2528673844 -0.2224068482 0.3171865807 0.1616513953 0.0039666012 0.0000000000 0.0000000000 0.0000000000
0.0000000000 0.0502022967 0.4003099458 -0.4803650485 -0.2207868084 0.2308066084 0.0198330061 0.0000000000 0.0000000000 0.0000000000
0.0000000000 -0.2510114833 -0.0733408038 0.9580788098 -0.7861927570 0.0780924615 0.0743737728 0.0000000000 0.0000000000 0.0000000000
0.0000000000 0.8367049444 -1.6837395794 0.8315400990 0.3501765138 -0.5206164098 0.1859344321 0.0000000000 0.0000000000 0.0000000000
-0.600000000000000 5
0.0000000000 0.0001189980 0.0647051824 0.3901152298 0.4294306936 0.1136930793 0.0019368170 0.0000000000 0.0000000000 0.0000000000
0.0000000000 -0.0014874755 -0.1762658416 -0.2984867416 0.2579406398 0.2086153339 0.0096840850 0.0000000000 0.0000000000 0.0000000000
0.0000000000 0.0148747546 0.3569941096 -0.2776620852 -0.3676543504 0.2347112314 0.0387363400 0.0000000000 0.0000000000 0.0000000000
0.0000000000 -0.1115606592 -0.3346819777 1.0412328197 -0.6656452669 -0.0455539359 0.1162090201 0.0000000000 0.0000000000 0.0000000000
0.0000000000 0.5578032962 -0.9296721604 -0.0000000000 0.8552983876 -0.7158475635 0.2324180401 0.0000000000 0.0000000000 0.0000000000
-0.400000000000000 5
0.0000000000 0.0000037187 0.0360937297 0.3262418630 0.4728419623 0.1599993058 0.0048194205 0.0000000000 0.0000000000 0.0000000000
0.0000000000 -0.0000929672 -0.1125488663 -0.3334716823 0.1724056362 0.2536269610 0.0200809187 0.0000000000 0.0000000000 0.0000000000
0.0000000000 0.0018593443 0.2764913870 -0.0749591220 -0.4803099568 0.2099819520 0.0669363955 0.0000000000 0.0000000000 0.0000000000
0.0000000000 -0.0278901648 -0.4452096679 0.9580788098 -0.4440734020 -0.2082465639 0.1673409889 0.0000000000 0.0000000000 0.0000000000
0.0000000000 0.2789016481 -0.1756047414 -0.8315400990 1.3604202614 -0.9110787172 0.2789016481 0.0000000000 0.0000000000 0.0000000000
-0.200000000000000 5
0.0000000000 0.0000000000 0.0185185185 0.2592592593 0.4972222222 0.2145833333 0.0104166667 0.0000000000 0.0000000000 0.0000000000
0.0000000000 0.0000000000 -0.0661375661 -0.3306878307 0.0694444444 0.2901785714 0.0372023810 0.0000000000 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.1889644747 0.0944822373 -0.5385487528 0.1488095238 0.1062925170 0.0000000000 0.0000000000 0.0000000000
0.0000000000 0.0000000000 -0.4049238743 0.7086167800 -0.1214771623 -0.4099854227 0.2277696793 0.0000000000 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.5784626776 -1.6630801981 1.8655421352 -1.1063098709 0.3253852561 0.0000000000 0.0000000000 0.0000000000
0.000000000000000 5
0.0000000000 0.0000000000 0.0085678586 0.1958490758 0.5002968753 0.2749804975 0.0203045909 0.0000011018 0.0000000000 0.0000000000
0.0000000000 0.0000000000 -0.0356994110 -0.2997417991 -0.0383489766 0.3104072047 0.0633554361 0.0000275458 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.1189980365 0.2048377660 -0.5283636778 0.0475165771 0.1564603814 0.0005509168 0.0000000000 0.0000000000
0.0000000000 0.0000000000 -0.2974950913 0.4044073898 0.2091762361 -0.5887923683 0.2644400812 0.0082637525 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.4958251522 -1.3790137046 1.4409918486 -0.6817595843 0.0413187627 0.0826375254 0.0000000000 0.0000000000
0.200000000000000 5
0.0000000000 0.0000000000 0.0034432302 0.1404485345 0.4824291135 0.3371874236 0.0364564396 0.0000352587 0.0000000000 0.0000000000
0.0000000000 0.0000000000 -0.0172161511 -0.2524300942 -0.1380583818 0.3073671767 0.0998967169 0.0004407335 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.0688646045 0.2600327465 -0.4605389289 -0.0810467530 0.2082809962 0.0044073347 0.0000000000 0.0000000000
0.0000000000 0.0000000000 -0.2065938134 0.1570112982 0.4549195772 -0.6826892565 0.2442971844 0.0330550101 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.4131876268 -1.0949472111 1.0164415620 -0.2572092977 -0.2427477308 0.1652750507 0.0000000000 0.0000000000
0.400000000000000 5
0.0000000000 0.0000000000 0.0011282777 0.0953033100 0.4462753201 0.3961181849 0.0609071617 0.0002677456 0.0000000000 0.0000000000
0.0000000000 0.0000000000 -0.0070517355 -0.1986485597 -0.2198540373 0.2773026120 0.1460205073 0.0022312132 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.0352586775 0.2714298385 -0.3520565175 -0.2198984550 0.2503917019 0.0148747546 0.0000000000 0.0000000000
0.0000000000 0.0000000000 -0.1322200406 -0.0335714947 0.6157528609 -0.6916760873 0.1673409889 0.0743737728 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.3305501015 -0.8108807177 0.5918912755 0.1673409889 -0.5268142242 0.2479125761 0.0000000000 0.0000000000
0.600000000000000 5
0.0000000000 0.0000000000 0.0002677456 0.0609071617 0.3961181849 0.4462753201 0.0953033100 0.0011282777 0.0000000000 0.0000000000
0.0000000000 0.0000000000 -0.0022312132 -0.1460205073 -0.2773026120 0.2198540373 0.1986485597 0.0070517355 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.0148747546 0.2503917019 -0.2198984550 -0.3520565175 0.2714298385 0.0352586775 0.0000000000 0.0000000000
0.0000000000 0.0000000000 -0.0743737728 -0.1673409889 0.6916760873 -0.6157528609 0.0335714947 0.1322200406 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.2479125761 -0.5268142242 0.1673409889 0.5918912755 -0.8108807177 0.3305501015 0.0000000000 0.0000000000
0.800000000000000 5
0.0000000000 0.0000000000 0.0000352587 0.0364564396 0.3371874236 0.4824291135 0.1404485345 0.0034432302 0.0000000000 0.0000000000
0.0000000000 0.0000000000 -0.0004407335 -0.0998967169 -0.3073671767 0.1380583818 0.2524300942 0.0172161511 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.0044073347 0.2082809962 -0.0810467530 -0.4605389289 0.2600327465 0.0688646045 0.0000000000 0.0000000000
0.0000000000 0.0000000000 -0.0330550101 -0.2442971844 0.6826892565 -0.4549195772 -0.1570112982 0.2065938134 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.1652750507 -0.2427477308 -0.2572092977 1.0164415620 -1.0949472111 0.4131876268 0.0000000000 0.0000000000
1.000000000000000 5
0.0000000000 0.0000000000 0.0000011018 0.0203045909 0.2749804975 0.5002968753 0.1958490758 0.0085678586 0.0000000000 0.0000000000
0.0000000000 0.0000000000 -0.0000275458 -0.0633554361 -0.3104072047 0.0383489766 0.2997417991 0.0356994110 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.0005509168 0.1564603814 0.0475165771 -0.5283636778 0.2048377660 0.1189980365 0.0000000000 0.0000000000
0.0000000000 0.0000000000 -0.0082637525 -0.2644400812 0.5887923683 -0.2091762361 -0.4044073898 0.2974950913 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.0826375254 0.0413187627 -0.6817595843 1.4409918486 -1.3790137046 0.4958251522 0.0000000000 0.0000000000
1.200000000000000 5
0.0000000000 0.0000000000 0.0000000000 0.0104166667 0.2145833333 0.4972222222 0.2592592593 0.0185185185 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.0000000000 -0.0372023810 -0.2901785714 -0.0694444444 0.3306878307 0.0661375661 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.1062925170 0.1488095238 -0.5385487528 0.0944822373 0.1889644747 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.0000000000 -0.2277696793 0.4099854227 0.1214771623 -0.7086167800 0.4049238743 0.0000000000 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.3253852561 -1.1063098709 1.8655421352 -1.6630801981 0.5784626776 0.0000000000 0.0000000000
1.400000000000000 5
0.0000000000 0.0000000000 0.0000000000 0.0048194205 0.1599993058 0.4728419623 0.3262418630 0.0360937297 0.0000037187 0.0000000000
0.0000000000 0.0000000000 0.0000000000 -0.0200809187 -0.2536269610 -0.1724056362 0.3334716823 0.1125488663 0.0000929672 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.0669363955 0.2099819520 -0.4803099568 -0.0749591220 0.2764913870 0.0018593443 0.0000000000
0.0000000000 0.0000000000 0.0000000000 -0.1673409889 0.2082465639 0.4440734020 -0.9580788098 0.4452096679 0.0278901648 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.2789016481 -0.9110787172 1.3604202614 -0.8315400990 -0.1756047414 0.2789016481 0.0000000000
1.600000000000000 5
0.0000000000 0.0000000000 0.0000000000 0.0019368170 0.1136930793 0.4294306936 0.3901152298 0.0647051824 0.0001189980 0.0000000000
0.0000000000 0.0000000000 0.0000000000 -0.0096840850 -0.2086153339 -0.2579406398 0.2984867416 0.1762658416 0.0014874755 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.0387363400 0.2347112314 -0.3676543504 -0.2776620852 0.3569941096 0.0148747546 0.0000000000
0.0000000000 0.0000000000 0.0000000000 -0.1162090201 0.0455539359 0.6656452669 -1.0412328197 0.3346819777 0.1115606592 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.2324180401 -0.7158475635 0.8552983876 0.0000000000 -0.9296721604 0.5578032962 0.0000000000
1.800000000000000 5
0.0000000000 0.0000000000 0.0000000000 0.0006346562 0.0766798556 0.3714272905 0.4428821132 0.1074724431 0.0009036413 0.0000000000
0.0000000000 0.0000000000 0.0000000000 -0.0039666012 -0.1616513953 -0.3171865807 0.2224068482 0.2528673844 0.0075303445 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.0198330061 0.2308066084 -0.2207868084 -0.4803650485 0.4003099458 0.0502022967 0.0000000000
0.0000000000 0.0000000000 0.0000000000 -0.0743737728 -0.0780924615 0.7861927570 -0.9580788098 0.0733408038 0.2510114833 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.1859344321 -0.5206164098 0.3501765138 0.8315400990 -1.6837395794 0.8367049444 0.0000000000
2.000000000000000 5
0.0000000000 0.0000000000 0.0000000000 0.0001506069 0.0488294808 0.3046391054 0.4765452666 0.1660276031 0.0038079372 0.0000000000
0.0000000000 0.0000000000 0.0000000000 -0.0012550574 -0.1176810010 -0.3453215591 0.1085581625 0.3318998477 0.0237996073 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.0083670494 0.2060773289 -0.0599122059 -0.6498064078 0.3762761989 0.1189980365 0.0000000000
0.0000000000 0.0000000000 0.0000000000 -0.0418352472 -0.1626926281 0.8057158724 -0.7086167800 -0.3388138540 0.4462426370 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.1394508241 -0.3253852561 -0.1549453601 1.6630801981 -2.4378069984 1.1156065925 0.0000000000
2.200000000000001 5
0.0000000000 0.0000000000 0.0000000000 0.0000198330 0.0291788144 0.2354337726 0.4844379078 0.2393087702 0.0116209020 0.0000000000
0.0000000000 0.0000000000 0.0000000000 -0.0002479126 -0.0800881577 -0.3415646506 -0.0330808344 0.3968770453 0.0581045100 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.0024791258 0.1683326392 0.0947645822 -0.7527245592 0.2547301720 0.2324180401 0.0000000000
0.0000000000 0.0000000000 0.0000000000 -0.0185934432 -0.2082465639 0.7242146130 -0.2928467305 -0.9017819956 0.6972541203 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.0929672160 -0.1301541025 -0.6600672339 2.4946202971 -3.1918744174 1.3945082406 0.0000000000
2.400000000000001 5
0.0000000000 0.0000000000 0.0000000000 0.0000006198 0.0162440997 0.1699310142 0.4625541827 0.3223535608 0.0289165229 0.0000000000
0.0000000000 0.0000000000 0.0000000000 -0.0000154945 -0.0506950229 -0.3091759055 -0.1858793404 0.4252802514 0.1204855120 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.0003098907 0.1253817854 0.2230386810 -0.7558578987 0.0055091684 0.4016183733 0.0000000000
0.0000000000 0.0000000000 0.0000000000 -0.0046483608 -0.2147542691 0.5416889788 0.2892313388 -1.6155636210 1.0040459332 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.0464836080 0.0650770512 -1.1651891077 3.3261603961 -3.9459418364 1.6734098887 0.0000000000
2.600000000000001 5
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0083333333 0.1131944444 0.4108796296 0.4050925926 0.0625000000 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.0000000000 -0.0297619048 -0.2554563492 -0.3265542328 0.3885582011 0.2232142857 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0850340136 0.3047052154 -0.6259448224 -0.4015495087 0.6377551020 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.0000000000 -0.1822157434 0.2581389699 1.0376174279 -2.4801587302 1.3666180758 0.0000000000
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.2603082049 -1.6703109815 4.1577004952 -4.7000092554 1.9523115369 0.0000000000
2.800000000000001 5
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0038555364 0.0684357708 0.3346819777 0.4712322247 0.1217349914 0.0000594990
0.0000000000 0.0000000000 0.0000000000 0.0000000000 -0.0160647349 -0.1914380913 -0.4261617183 0.2543583031 0.3778187660 0.0014874755
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0535491164 0.3257571250 -0.3495567323 -0.9519842923 0.8924852740 0.0297495091
0.0000000000 0.0000000000 0.0000000000 0.0000000000 -0.1338727911 -0.0334681978 1.6548164455 -2.8262033676 0.8924852740 0.4462426370
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.2231213185 -1.2457606949 2.0142896809 1.2395628805 -6.6936395549 4.4624263700
3.000000000000001 5
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0015494536 0.0365412807 0.2447706285 0.4994577601 0.2157769084 0.0019039686
0.0000000000 0.0000000000 0.0000000000 0.0000000000 -0.0077472680 -0.1284755277 -0.4610054866 0.0110699852 0.5623586898 0.0237996073
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0309890720 0.2969786068 0.0074029450 -1.4528365606 0.8794698637 0.2379960731
0.0000000000 0.0000000000 0.0000000000 0.0000000000 -0.0929672160 -0.2401653081 1.8433333003 -1.9843335779 -1.3108377462 1.7849705480
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.1859344321 -0.8212104084 -0.1291211334 7.1791350165 -15.3395906468 8.9248527399
3.200000000000002 5
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0005077250 0.0164164402 0.1551381810 0.4705270511 0.3429523413 0.0144582614
0.0000000000 0.0000000000 0.0000000000 0.0000000000 -0.0031732810 -0.0748365430 -0.4235448634 -0.3076319611 0.6887011364 0.1204855120
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0158664049 0.2353516723 0.3591977769 -1.6665234283 0.2528708276 0.8032367466
0.0000000000 0.0000000000 0.0000000000 0.0000000000 -0.0594990183 -0.3619523611 1.6031679922 0.0454506390 -5.2433509847 4.0161837330
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.1487475457 -0.3966601218 -2.2725319477 13.1187071524 -23.9855417386 13.3872791099
3.400000000000002 5
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0001204855 0.0056527786 0.0795706402 0.3766845660 0.4770445350 0.0609269947
0.0000000000 0.0000000000 0.0000000000 0.0000000000 -0.0010040459 -0.0353926191 -0.3233864610 -0.6205561671 0.5995455762 0.3807937169
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0066936396 0.1578583328 0.6200913310 -1.3554620099 -1.3331498780 1.9039685845
0.0000000000 0.0000000000 0.0000000000 0.0000000000 -0.0334681978 -0.3988293568 0.9343205212 3.2631492830 -10.9050544416 7.1398821919
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.1115606592 0.0278901648 -4.4159427619 19.0582792884 -32.6314928304 17.8497054799
3.600000000000002 5
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000158664 0.0012071690 0.0282179603 0.2311647043 0.5534598679 0.1859344321
0.0000000000 0.0000000000 0.0000000000 0.0000000000 -0.0001983301 -0.0116188361 -0.1872841783 -0.7989946870 0.0684238710 0.9296721604
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0019833006 0.0814806000 0.7043471746 -0.2820694199 -4.2244302969 3.7186886416
0.0000000000 0.0000000000 0.0000000000 0.0000000000 -0.0148747546 -0.3507962952 -0.1632091126 7.6687623543 -18.2959481169 11.1560659249
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0743737728 0.4524404514 -6.5593535762 24.9978514243 -41.2774439222 22.3121318498
3.800000000000002 5
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000004958 0.0000811087 0.0041645870 0.0776951127 0.4553943297 0.4626643660
0.0000000000 0.0000000000 0.0000000000 0.0000000000 -0.0000123956 -0.0015938713 -0.0591392007 -0.6667229979 -1.2002997263 1.9277681918
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0002479126 0.0232004852 0.5262288751 1.7912372270 -8.7668084727 6.4258939727
0.0000000000 0.0000000000 0.0000000000 0.0000000000 -0.0037186886 -0.2178531763 -1.6894209093 13.2622898528 -27.4160320105 16.0647349319
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0371868864 0.8769907380 -8.7027643905 30.9374235603 -49.9233950140 26.7745582198
4.000000000000000 5
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 1.0000000000
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 -3.5714285714 3.5714285714
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 5.1020408163 -15.3061224490 10.2040816327
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 -3.6443148688 20.0437317784 -38.2653061224 21.8658892128
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 1.3015410246 -10.8461752048 36.8769956962 -58.5693461058 31.2369845898

3 3 0.0000000000 1.0000000000 2.0000000000 3.0000000000 4.0000000000
0.000000000000000 4
1.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-3.0000000000 3.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000
6.0000000000 -9.0000000000 3.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000
-6.0000000000 10.5000000000 -5.5000000000 1.0000000000 0.0000000000 0.0000000000 0.0000000000
0.235294117647059 4
0.4471809485 0.4795440668 0.0711038741 0.0021711107 0.0000000000 0.0000000000 0.0000000000
-1.7543252595 1.1730103806 0.5536332180 0.0276816609 0.0000000000 0.0000000000 0.0000000000
4.5882352941 -6.5294117647 1.7058823529 0.2352941176 0.0000000000 0.0000000000 0.0000000000
-6.0000000000 10.5000000000 -5.5000000000 1.0000000000 0.0000000000 0.0000000000 0.0000000000
0.470588235294118 4
0.1483818441 0.5975982088 0.2366510618 0.0173688853 0.0000000000 0.0000000000 0.0000000000
-0.8408304498 -0.0726643599 0.8027681661 0.1107266436 0.0000000000 0.0000000000 0.0000000000
3.1764705882 -4.0588235294 0.4117647059 0.4705882353 0.0000000000 0.0000000000 0.0000000000
-6.0000000000 10.5000000000 -5.5000000000 1.0000000000 0.0000000000 0.0000000000 0.0000000000
0.705882352941176 4
0.0254427030 0.4909423977 0.4249949115 0.0586199878 0.0000000000 0.0000000000 0.0000000000
-0.2595155709 -0.7370242215 0.7474048443 0.2491349481 0.0000000000 0.0000000000 0.0000000000
1.7647058824 -1.5882352941 -0.8823529412 0.7058823529 0.0000000000 0.0000000000 0.0000000000
-6.0000000000 10.5000000000 -5.5000000000 1.0000000000 0.0000000000 0.0000000000 0.0000000000
0.941176470588235 4
0.0002035416 0.2963566049 0.5644887713 0.1389510822 0.0000000000 0.0000000000 0.0000000000
-0.0103806228 -0.8200692042 0.3875432526 0.4429065744 0.0000000000 0.0000000000 0.0000000000
0.3529411765 0.8823529412 -2.1764705882 0.9411764706 0.0000000000 0.0000000000 0.0000000000
-6.0000000000 10.5000000000 -5.5000000000 1.0000000000 0.0000000000 0.0000000000 0.0000000000
1.176470588235294 4
0.0000000000 0.1396295542 0.5917294253 0.2677250831 0.0009159373 0.0000000000 0.0000000000
0.0000000000 -0.5086505190 -0.1366782007 0.6297577855 0.0155709343 0.0000000000 0.0000000000
0.0000000000 1.2352941176 -1.8823529412 0.4705882353 0.1764705882 0.0000000000 0.0000000000
0.0000000000 -1.5000000000 3.5000000000 -3.0000000000 1.0000000000 0.0000000000 0.0000000000
1.411764705882353 4
0.0000000000 0.0508854061 0.5150620802 0.4224167176 0.0116357962 0.0000000000 0.0000000000
0.0000000000 -0.2595155709 -0.4826989619 0.6574394464 0.0847750865 0.0000000000 0.0000000000
0.0000000000 0.8823529412 -1.0588235294 -0.2352941176 0.4117647059 0.0000000000 0.0000000000
0.0000000000 -1.5000000000 3.5000000000 -3.0000000000 1.0000000000 0.0000000000 0.0000000000
1.647058823529412 4
0.0000000000 0.0109912477 0.3797747473 0.5640816880 0.0451523170 0.0000000000 0.0000000000
0.0000000000 -0.0934256055 -0.6349480969 0.5190311419 0.2093425606 0.0000000000 0.0000000000
0.0000000000 0.5294117647 -0.2352941176 -0.9411764706 0.6470588235 0.0000000000 0.0000000000
0.0000000000 -1.5000000000 3.5000000000 -3.0000000000 1.0000000000 0.0000000000 0.0000000000
1.882352941176471 4
0.0000000000 0.0004070832 0.2314607504 0.6536400027 0.1144921636 0.0000000000 0.0000000000
0.0000000000 -0.0103806228 -0.5934256055 0.2145328720 0.3892733564 0.0000000000 0.0000000000
0.0000000000 0.1764705882 0.5882352941 -1.6470588235 0.8823529412 0.0000000000 0.0000000000
0.0000000000 -1.5000000000 3.5000000000 -3.0000000000 1.0000000000 0.0000000000 0.0000000000
2.117647058823529 4
0.0000000000 0.0000000000 0.1144921636 0.6536400027 0.2314607504 0.0004070832 0.0000000000
0.0000000000 0.0000000000 -0.3892733564 -0.2145328720 0.5934256055 0.0103806228 0.0000000000
0.0000000000 0.0000000000 0.8823529412 -1.6470588235 0.5882352941 0.1764705882 0.0000000000
0.0000000000 0.0000000000 -1.0000000000 3.0000000000 -3.5000000000 1.5000000000 0.0000000000
2.352941176470588 4
0.0000000000 0.0000000000 0.0451523170 0.5640816880 0.3797747473 0.0109912477 0.0000000000
0.0000000000 0.0000000000 -0.2093425606 -0.5190311419 0.6349480969 0.0934256055 0.0000000000
0.0000000000 0.0000000000 0.6470588235 -0.9411764706 -0.2352941176 0.5294117647 0.0000000000
0.0000000000 0.0000000000 -1.0000000000 3.0000000000 -3.5000000000 1.5000000000 0.0000000000
2.588235294117647 4
0.0000000000 0.0000000000 0.0116357962 0.4224167176 0.5150620802 0.0508854061 0.0000000000
0.0000000000 0.0000000000 -0.0847750865 -0.6574394464 0.4826989619 0.2595155709 0.0000000000
0.0000000000 0.0000000000 0.4117647059 -0.2352941176 -1.0588235294 0.8823529412 0.0000000000
0.0000000000 0.0000000000 -1.0000000000 3.0000000000 -3.5000000000 1.5000000000 0.0000000000
2.823529411764706 4
0.0000000000 0.0000000000 0.0009159373 0.2677250831 0.5917294253 0.1396295542 0.0000000000
0.0000000000 0.0000000000 -0.0155709343 -0.6297577855 0.1366782007 0.5086505190 0.0000000000
0.0000000000 0.0000000000 0.1764705882 0.4705882353 -1.8823529412 1.2352941176 0.0000000000
0.0000000000 0.0000000000 -1.0000000000 3.0000000000 -3.5000000000 1.5000000000 0.0000000000
3.058823529411765 4
0.0000000000 0.0000000000 0.0000000000 0.1389510822 0.5644887713 0.2963566049 0.0002035416
0.0000000000 0.0000000000 0.0000000000 -0.4429065744 -0.3875432526 0.8200692042 0.0103806228
0.0000000000 0.0000000000 0.0000000000 0.9411764706 -2.1764705882 0.8823529412 0.3529411765
0.0000000000 0.0000000000 0.0000000000 -1.0000000000 5.5000000000 -10.5000000000 6.0000000000
3.294117647058824 4
0.0000000000 0.0000000000 0.0000000000 0.0586199878 0.4249949115 0.4909423977 0.0254427030
0.0000000000 0.0000000000 0.0000000000 -0.2491349481 -0.7474048443 0.7370242215 0.2595155709
0.0000000000 0.0000000000 0.0000000000 0.7058823529 -0.8823529412 -1.5882352941 1.7647058824
0.0000000000 0.0000000000 0.0000000000 -1.0000000000 5.5000000000 -10.5000000000 6.0000000000
3.529411764705883 4
0.0000000000 0.0000000000 0.0000000000 0.0173688853 0.2366510618 0.5975982088 0.1483818441
0.0000000000 0.0000000000 0.0000000000 -0.1107266436 -0.8027681661 0.0726643599 0.8408304498
0.0000000000 0.0000000000 0.0000000000 0.4705882353 0.4117647059 -4.0588235294 3.1764705882
0.0000000000 0.0000000000 0.0000000000 -1.0000000000 5.5000000000 -10.5000000000 6.0000000000
3.764705882352942 4
0.0000000000 0.0000000000 0.0000000000 0.0021711107 0.0711038741 0.4795440668 0.4471809485
0.0000000000 0.0000000000 0.0000000000 -0.0276816609 -0.5536332180 -1.1730103806 1.7543252595
0.0000000000 0.0000000000 0.0000000000 0.2352941176 1.7058823529 -6.5294117647 4.5882352941
0.0000000000 0.0000000000 0.0000000000 -1.0000000000 5.5000000000 -10.5000000000 6.0000000000
4.000000000000000 4
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 1.0000000000
0.0000000000 0.0000000000 0.0000000000 0.0000000000 0.0000000000 -3.0000000000 3.0000000000
0.0000000000 0.0000000000 0.0000000000 0.0000000000 3.0000000000 -9.0000000000 6.0000000000
0.0000000000 0.0000000000 0.0000000000 -1.0000000000 5.5000000000 -10.5000000000 6.0000000000

