package Net::Int::Stats;

our $VERSION = '2.07';

use strict;
use warnings;

############ Global Declarations ##############

# store ifconfig output
my @interface = `/sbin/ifconfig`;

# hash of hashes
# key1 - interface, key2 - type ex: rx_packets, values ex: 'packets:12345'
my %interfaces;

# tmp array to store string tokens
my @tmp;

# interface name
my $key1;

# value types
my @key2;

########## End Global Declarations ###########

# generate ifconfig values
sub data {

    # loop through each line of ifconfig output
    foreach (@interface){

        # skip if blank line
        next if /^$/;

        # get interface name if not white space
        if (!/^\s/){

            # extract values
            extract($_);

            # store first token of interface name
            $key1 = shift(@tmp);
        }

        # get RX, TX, collisions and txqueuelen values
        # look for 'RX' or 'TX' or 'collisions' text
        if (/RX packets/ || /TX packets/ || /collisions/){

            # key2 values
            @key2 = qw(rx_packets rx_errors rx_dropped rx_overruns rx_frame) if (/RX packets/);
            @key2 = qw(tx_packets tx_errors tx_dropped tx_overruns tx_carrier) if (/TX packets/);
            @key2 = qw(collisions txqueuelen) if (/collisions/);

            # extract values
            extract($_);

            # shift first token of 'RX' or 'TX'
            shift(@tmp) if (/RX packets/ || /TX packets/);

            # build values hash
            build();
        }
    }
}

# extract values
sub extract {

    # ifconfig output line
    my $line = shift;

    # remove spaces
    $line =~ s/^\s+//;

    # store tokens split on spaces
    @tmp = split (/\s/, $line);
}

# build values hash
sub build {

    # values type count
    my $i = 0;

    # loop through value types
    for (@key2){
	
        # build hash with interface name, value type, and value
        $interfaces{$key1}{$_} = $tmp[$i];

        # increment values type count
        $i++;
    }
}

# validate interface name
sub validate {

    # interface name
    my $int = shift;

    # terminate program if specified interface name is not in ifconfig output
    die "specified interface $int not listed in ifconfig output!\n" if !(grep(/$int/, keys %interfaces));
}

# create new Net::Int::Stats object
sub new {

    # class name
    my $class = shift;

    # allocate object memory
    my $self  = {};

    # assign object reference to class
    bless($self, $class);

    # initialize values reference
    $self->{VALUES} = '';

    # return object reference
    return $self;
}

# get specific ifconfig value for specific interface
sub value {

    # object reference
    my $self = shift;

    # interface name
    my $int  = shift;

    # value type
    my $type = shift;

    # generate value data
    data();

    # validate if supplied interface is present
    validate($int);

    # user specified value
    $self->{VALUES} = $interfaces{$int}{$type};

    # return value
    return $self->{VALUES};
}

1;

__END__

=head1 NAME

Net::Int::Stats - Reports specific ifconfig values for a network interface

=head1 SYNOPSIS

  use Net::Int::Stats;

  my $get = Net::Int::Stats->new();

  # get value for specific interface
  my $int     = 'eth0';
  my $stat    = 'rx_packets';
  my $packets = $get->value($int, $stat);

=head1 DESCRIPTION

This module provides various statistics generated from the ifconfig command for specific interfaces. 
RX values consist of packets, errors, dropped, overruns, and frame. TX values consist of packets, 
errors, dropped, overruns, and carrier. In addition, collisions and txqueuelen are reported. Values 
are in the format of type:n - ex 'packets:123456'.

=head1 METHODS

Use this one method to get specific values which requires two arguments: B<value()>.
Ex: $packets = $get->value($int, 'rx_packets');

The first argument is the interface and the second is the type value to extract.

RX values - rx_packets, rx_errors, rx_dropped, rx_overruns, rx_frame

TX values - tx_packets, tx_errors, tx_dropped, tx_overruns, tx_carrier

Miscellaneous values - collisions, txqueuelen

=head1 DEPENDENCIES

This module is platform dependent. It uses the linux version
of /sbin/ifconfig. Other platforms such as the windows equivalent
of ipconfig, mac osx, and other versions of unix are not supported. 
This is due to the fact that each platform generates and displays
different information in different formats of ifconfig results.
The linux version is used over the other platforms because of the 
amount of data the default command outputs.  

=head1 SEE ALSO

linux command /sbin/ifconfig

=head1 NOTES

ifconfig output contains more information than the values that are
extracted in this module. More values and/or support for other
operating systems can be added if there are any requests to do so.   

=head1 AUTHOR

Bruce Burch <bcb12001@yahoo.com>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2009 by Bruce Burch

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.


=cut
