#!/usr/local/bin/perl -w
### do not edit next line. Set by CS-RCS... http://www.componentsoftware.com/csrcs/uhome.htm
#REV=' @(#) $RCSfile: cp,v $ $Revision: 1.2 $ $Date: 2004/08/05 14:17:43 $ '; 
require 5;
use strict;
use Cwd;
use File::Basename;
use File::Find;
use Getopt::Std;

#######   S U B R O U T I N E S   ###############
sub insufficientArgs($);
sub printUsage();
sub checkArgs(@);
sub findCopy;     ## used by find()
sub copyFile($$); ## used by find()

$main::opt_f = undef;  ## used and set by getopts()
$main::opt_i = undef;  ## used and set by getopts()
$main::opt_p = undef;  ## used and set by getopts()
$main::opt_v = undef;  ## used and set by getopts() not standard, but may be helpful
                       ## especially if problems are found w/ routine.

#######   P R O C E S S   - O P T I O N S   ####################################
my $VERBOSE = 0;
my $PRESERVE = 0;
unless (getopts("fipv")) 
{
    &printUsage();
    exit 1;
}
if (defined $main::opt_f) { $main::opt_i = undef; } ## -f overrides -i
$VERBOSE  = 1 if (defined $main::opt_v);
$PRESERVE = 1 if (defined $main::opt_p);

#######
$cp::BINMODE = 1;     ## default cp is binary mode
$cp::CWD = cwd();     ## where user invoked command
$cp::EXIT_STATUS = 0; ## this means everything worked
#$cp::PATH;          ## used below ... mentioned here
#$cp::TARGET;        ## used below

#######   O S   S T U F F   ####################################################
$cp::USES_DRIVE_LETTER = 0;  ## the Windows 'C:\' type of thing...
if ($^O =~ /Win32/i) 
{ 
    $cp::USES_DRIVE_LETTER = 1;
}

## Do not use binary mode on some OS's ...
#if ( $^O =~ /????/i ) ## need some help here
#{
#    $cp::BINMODE = 0;
#}

## default is that OS is case sensitive (file a.txt ne A.txt)
if ($^O =~ /(Win32|VMS|OS2)/i) ## OS is not case sensitive.. normalize path
{
    $cp::CASE_SENSITIVE = 0;
    $cp::CWD = uc $cp::CWD;
}
else              
{
    $cp::CASE_SENSITIVE = 1;
}
$cp::CWD =~ s|\\|/|g; ## normalize for compares

#######   C H E C K   P A T H S   &    C L E A N   U P  I F  N E E D E D  ######
my @paths = checkArgs(@ARGV); ## checks and cleans up args
if ($#paths < 1)  ## need at least one source and a target
{
    insufficientArgs($#paths);
    &printUsage();
    exit 1;
}

#######   E S T A B L I S H   T H E   T A R G E T   ############################
my $target = pop @paths; ## last is target...

#######   M O V E   S O U R C E S   T O   T A R G E T   ########################
my $path;
foreach $path (@paths)
{
    my $current_target = $target;

    if (-d $target) 
    {
        my $base_path = basename($path);
        $current_target = "$target/$base_path";
    }

    if ($path eq $current_target) ## check again... compare may wk out after code above
    {
        print STDERR "cp: $path and target $current_target are identical ... skipping\n";
        $cp::EXIT_STATUS++;
        next;
    }

    ## copy it ...
    if (! -d $path) ## cp'ing a file
    {
        copyFile($path, $current_target);
    }
    else ## cp'ing a directory
    {
        print "cp $path $current_target\n" if $VERBOSE;
        $cp::TARGET = $current_target;
        $cp::PATH = $path;
        &File::Find::find(\&findCopy, $path); ## copy files/dirs top -> down
    }
}

exit $cp::EXIT_STATUS;
################################################################################
#######   E N D   O F   C P
################################################################################

################################################################################
###  This is called via find() to copy directory trees (top down)
sub findCopy
{
    $_ = uc $_ if (! $cp::CASE_SENSITIVE); 
    my $dir_tail = cwd(); ## find() has "cd'ed" us here...
    $dir_tail = uc $dir_tail if (! $cp::CASE_SENSITIVE);
    $dir_tail =~ s|^$cp::PATH||;
    $dir_tail = "$dir_tail/" unless $dir_tail eq "";
    if (-d $_)
    {
        $_ =~ s|\.$||;
        $_ =~ s|\/$|| unless $_ eq '/';
        mkdir "$cp::TARGET/dir_tail$_", 0777;  ## umask will modify
        if (! -d "$cp::TARGET/$dir_tail$_")
        {
            print STDERR "mv: Unable to create dir $cp::TARGET/$dir_tail/$_\n";
            $cp::EXIT_STATUS++;
        }
    }
    else
    {
        if ((defined $main::opt_i) && (-e "$cp::TARGET/$dir_tail/$_"))
        {
            my $path2show = "$cp::TARGET/$dir_tail/$_";
            $path2show =~ s|^$cp::CWD|.|;
            print STDERR "cp: overwrite $path2show (yes/no)? ";
            my $response = <STDIN>;
            return if ($response !~ /^y/i);
        }
        copyFile($_, "$cp::TARGET/$dir_tail/$_");
    }
}

################################################################################
###   This copies a single file
sub copyFile($$)
{           ## source, target
    my ($path, $target) = @_;
    $path = uc $path if (! $cp::CASE_SENSITIVE);
    $target = uc $target if (! $cp::CASE_SENSITIVE);
    if ((defined $main::opt_i) && (-e $target)) ## used if '-i' option was given
    {
        my $path2show = $target;
        $path2show =~ s|^$cp::CWD|.|;
        print STDERR "cp: overwrite $path2show (yes/no)? ";
        my $response = <STDIN>;
        return if ($response !~ /^y/i); ## not this one
    }
    print "cp $path $target\n" if $VERBOSE;

    open(PATH, "<$path") or die "Unable to read $path: $!";
    open(TARGET, ">$target") or die "Unable to create $target: $!";
    if ($cp::BINMODE) { binmode PATH; binmode TARGET; }
    my $buffer;
    while (read PATH, $buffer, 1024) { print TARGET $buffer; }
    close PATH;
    close TARGET;

    if ($PRESERVE)  ## preserve as many file attributes as possible...
    {
        my($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,$ctime,$blksize,$blocks) = stat $path;
        utime $atime, $mtime, ($target);
        chown $uid, $gid, ($target);
        my $oldMode = (07777 & $mode); ## from man -s 2 mknod
        chmod $oldMode, $target;
    }
}

################################################################################
## print Insufficient arguments error
sub insufficientArgs($)
{
    my $arg_num = ($_[0] + 1); ## num to display
    print STDERR "cp: Insufficient arguments ($arg_num)\n";
    $cp::EXIT_STATUS++;
}

################################################################################
sub printUsage()
{
print STDERR <<EOE
Usage: cp [-fivp] file1 file2
       cp [-fivp] file1... filex dir
       cp [-fivp] dir1... dirx dir
       cp [-fivp] dir1... dirx file1... filex dir
EOE

}

################################################################################
sub checkArgs(@)
{
    my $target = $_[$#_];
    if ($#_ > 1)  ### cp'ing > 1 thing target has to be an existing directory
    {
        if (! -e $target) ## has to be an existing directory... sorry it's over
        {
            print STDERR "cp: $target not found\n";
            print STDERR "    exiting...\n";
            $cp::EXIT_STATUS++;
            exit $cp::EXIT_STATUS
        }
        elsif (! -d $target) ## can only work if a directory...
        {
            print STDERR "cp: Target $target must be a directory when cp'ing > 1 thing\n";
            print STDERR "    exiting...\n";
            printUsage();
            $cp::EXIT_STATUS++;
            exit $cp::EXIT_STATUS
        }
    }

    my $paths = "";
    my $Cwd = cwd();
    $Cwd = uc $Cwd if (! $cp::CASE_SENSITIVE);
    $Cwd =~ s|[/\\]$|| unless $Cwd eq '/'; ## remove any possible trailing slash (DOS or UNIX style)
    my $path;
    my @paths = "";
    foreach $path (@_)  ## check all paths passed in
    {
        $path =~ s|\\|/|g;         ## normalize
        ## if OS is not case sensitive then let's just set to uc and make life easy
        $path = uc $path if (! $cp::CASE_SENSITIVE);

        if ((! -e $path) && ($path ne $target))
        {
            print STDERR "cp: can not access $path ... skipping\n";
            $cp::EXIT_STATUS++;
            ## continue on... ignore this one
        }
        elsif (((! -d $path) && (! -r $path)) && ($path ne $target))
        {
            print STDERR "cp: can not read $path ... skipping\n";
            $cp::EXIT_STATUS++;
            ## continue on... ignore this one
        }
        else
        {
            $path =~ s|^\.\/||;       ## no need for './' at the path start
            if ($path =~ /^\.\.\//)   ## change relative to full path
            {                         ## ../../abc to /aaa/bbb/ccc/abc
                my $path_front = $Cwd;
                $path_front = uc $path if (! $cp::CASE_SENSITIVE);
                while ($path =~ /^\.\.\//)
                {
                    $path =~ s|^...||;
                    $path_front =~ s|(.*)\/[^\/]+|$1/|;
                }
                $path = $path_front.$path;
            }
            elsif (($path !~ /\//) && ($path !~ /^[a-z]:/i) && $cp::USES_DRIVE_LETTER) ## it's right here...
            {
                $path = "$Cwd/$path";
            }
            elsif (($path !~ /\//) && (! $cp::USES_DRIVE_LETTER)) ## it's right here...
            {
                $path = "$Cwd/$path";
            }

            while ($path =~ /\/\.\./) ## remove '/..' from middle or end of path
            {
                $path =~ s|[^\/]+\/\.\.||;
            }
            $path =~ s|\/\.?$|| unless $path eq '/'; ## remove '/' or '/.' from end of path

            my $tmpPath;
            my $path_cnt = 0; ## number used for splicing
            foreach $tmpPath (@paths)
            {
                if ($path eq $tmpPath)
                {
                    print STDERR "cp: $path found more than once on command line. Ignoring 1st occurance\n";
                    $cp::EXIT_STATUS++;
                    splice @paths, $path_cnt, 1;
                }
                else
                {
                    $path_cnt++;
                }
            }
            push @paths, $path;
        }
    }
    splice @paths, 0, 1; ## the "my @paths" initialization gave this....

    return @paths; ## checked, cleaned, and ready to go...
}

__END__

=pod

=head1 NAME

cp - copy files and/or directories

=head1 SYNOPSIS

cp [ B<->B<fiv> ] source_file  target_file

cp [ B<->B<fiv> ] source...  target_dir

=head1 DESCRIPTION

The cp utility copies the source files/directories to the target. If the
target is a file you may only specify one file as the source. cp will not
copy a file onto itself.

=head2 OPTIONS

B<-f> Force copy if possible (DEFAULT)

B<-i> Prompt for confirmation whenever the copy would overwrite an existing target.

B<-p> Preserve source file attributes (like modification date) as much as possible onto the target.

B<-v> Verbose. Echo "cp source target" before copy is done.

Specifying both B<-f> and B<-i> options is not considered an
error. The B<-f> option will override the B<-i> option.

=head1 BUGS

B<cp> has no known bugs, but be aware that the current copy mode
is binary mode.

=head1 EXIT STATUS

0 = All sources were copied successfully.

=head1 REVISION HISTORY

$Revision: 1.2 $

$Log: cp,v $
Revision 1.2  2004/08/05 14:17:43  cwest
cleanup, new version number on website

Revision 1.1  2004/07/23 20:10:02  cwest
initial import

Revision 1.1  1999/03/27 13:03:57  schumacks
Initial revision

=head1 AUTHOR

This Perl implementation of B<cp> was written by Ken Schumack schumacks@att.net

=head1 COPYRIGHT and LICENSE

This program is copyright by Ken Schumack 1999.

This program is free and open software. You may use, modify, distribute
and sell this program (and any modified variants) in any way you wish,
provided you do not restrict others from doing the same.

=cut


