use strict;
use Test::More;
use Test::Requires
    'Digest::MD5'
;

use_ok "Algorithm::ConsistentHash::Ketama";

my $ketama = Algorithm::ConsistentHash::Ketama->new();
$ketama->add_bucket( "localhost:1001", 100 );
$ketama->add_bucket( "localhost:1002", 200 );
$ketama->add_bucket( "localhost:1003", 400 );


{ # 1. simple check
    my $keys = add_keys(300);
    my $hashed = distribute($ketama, $keys);
    check_consistency($hashed);
    is $hashed->{"localhost:1000"}, -1;
    is $hashed->{"localhost:1004"}, -1;
}

{ # 2. add more buckets
    $ketama->add_bucket( "localhost:1000",   5 );
    $ketama->add_bucket( "localhost:1004", 800 );
    my $keys = add_keys(300);
    my $hashed = distribute($ketama, $keys);
    check_consistency($hashed);
}

{ # 3. remove buckets
    $ketama->remove_bucket( "localhost:1000" );
    my $keys = add_keys(300);
    my $hashed = distribute($ketama, $keys);
    check_consistency($hashed);
}

done_testing;

sub distribute {
    my ($ketama, $keys) = @_;

    local $Test::Builder::Level = $Test::Builder::Level + 1;
    my %hashed;
    while ( my ($key, $expected) = each %$keys ) {
        my $got = $ketama->hash($key);
        ok $got;
        $hashed{$got}++;
    }

    foreach my $key qw( localhost:1000 localhost:1001 localhost:1002 localhost:1003 localhost:1004 ) {
        if (! defined $hashed{$key}) {
            $hashed{$key} = -1;
        }
    }

    note( explain( \%hashed ) );
    return \%hashed;
}

sub check_consistency {
    my ($hashed) = @_;

    local $Test::Builder::Level = $Test::Builder::Level + 1;
    if ($hashed->{"localhost:1000"} > 0) {
        ok( $hashed->{"localhost:1001"} > $hashed->{"localhost:1000"}, "1001 > 1000" );
    }
    ok( $hashed->{"localhost:1002"} > $hashed->{"localhost:1001"}, "1002 > 1001" );
    ok( $hashed->{"localhost:1003"} > $hashed->{"localhost:1002"}, "1003 > 1002" );
    if ($hashed->{"localhost:1004"} > 0) {
        ok( $hashed->{"localhost:1004"} > $hashed->{"localhost:1003"}, "1004 > 1003" );
    }
}

sub add_keys {
    my ($howmany) = @_;

    my %keys;
    while ($howmany-- > 0) {
        my $key = <DATA>;
        chomp $key;
        $keys{ $key } = undef;
    }
    return \%keys;
}
__DATA__
6dc1cda6ba8caea25fb32b14851fc86d
ba3272461caec8ec41556596f7df010f
13209f88100162609aeabfa4d7da85b4
13fa74d50ec3cd71d2975e30f8f85bd8
7e0c51bf844b69e2770e4d2f85f6aa73
b30fe2d51f2ce81331667b7c63523fc3
b1ae7436ecfa94a9ffbbcb9acd6cf35a
3ee5e07aa0655623ac15d94b9d629fa9
aff7ea8210d12aa9cf3a3d9c91eb3980
29b2e192fa6cc3fea4fdbf73576aa64d
ee8a3610a0f43d7f2da4d38e51dad493
02c95a61e82cf22ec3d68bf95fa2daa9
8a73b9ac46a808de13f98863f9f5c319
7f147f3a2eaaec0c7d8accc14db60c53
da5aed33f08e0488605b6438f828d33d
b8079b34bf4abc5722c6f0afe1286d2d
55316ba8b906f3342f7913bcb00b81a2
3543af97b66aced00534c1d49f142763
a83855bb9a66ec84f65b2c04122c2447
0073ee1a9b4ba9b4feed48f26b98a092
77ab95e67d16ae5b00e1d1d45d933918
f791b63c743c1277d5704a0c9a1c0586
df06e29e5862c70184e82f580eca7bc7
c4e6a34043256f7758ef87105668d68c
40e83e64838d64e38317beacb25cb055
f73d91c112483c16ac5253cd35f09291
47ad87c460f7116b154688bc1b6bd055
efe9a2a688db6e7da1acd9d08963a2a6
2629359154e0c7b1f005d375a3a461f6
ed623b5692894f792ebe235c5c653a27
7c80af1d1d42a6246b3d47f8fe36ffbc
f1c4606776dd42833a543782de625775
59fc482306f261c61f210b140175b8da
e9ca428d24620c8696bd3a6c0c5854b5
524205160e478686a83a3e93854488c0
a47f9b030da6676b6716033b70cb3b86
7fc030509391d7019f138e268c52087c
ea82dbd8ee048dea6d016ca10b3f1e94
e61d2d2fcddc7a4d217a3ce048549329
5c4bca5772d4748ff824837d7938519c
72a80a26115e283e392a60bc57e2fc34
a2b1812f16e3db35a890300fba3c057e
51df27c338c35740a02bba62d9a30447
0c1146bf126d2275318b906a73a84cb5
02be0fb20a2d9e0bbaf8171260d3c3fa
4925d79c95187e1ec4e37f446884e8dc
37688b5001728cceb9f9baa721513857
49e8d92a3f631b3739e05a075e4b5206
65f4920d7df2945e778abd65a4a2969d
4c9e8591eec75e411879b23d5c0907f2
ffca1dcd3be8ed9c4c6926a3f9a3f012
9a05fda96ca9513751642ad9b5ed47c4
4ed1d04b8469c08eb1db913c344746c3
4a43b66a824730d5304eadce434d4356
f7dd8bdcb8d790b237228eb17d8f2ca7
d2fbba1212026e8d29d96d01d7160071
e55d1d314345dfa50c623f62a0a262c5
e77a1edbe164b82d5c7ef291144b64cb
4c504f9d7f6b33fa92da6c97f6e16bf5
8e2abb9f39ac2d079295f24d4d59c5cf
bf7167360ba8529c778794ee14c0539b
ab64618f3b6765b71ad1aacc1369f83b
8f05c798c08dc316f608d65317c4e60c
38cfd0dd64d589835394c6776fd35018
3ec0274e025483b2349bc4e526b40da7
d65b2b20ea8e7bee4e1a8b8bb7a75911
d332363113f9bc9d4f0992c55305d631
36fd406b77e26c98301a5811a34ce861
831f0c0e394b7d1d20460b15b9094bb4
6c3b3f8d463d0511b9cd55cbab552ad4
e59def80c2474c2604b5f07eee4cee0e
783e1ec2ff006229f6466454b9892021
ec9c7026c5793901e750c4ff2fd85ea6
60d768f77bb3bfa5c8d7ef5b4e6c8543
797c53c4c66b99fb6ddf9f91f5de846e
6e07e0b8ced814eb2c2cec5470da98ad
ba3b8187e6b983169a06a877de5f23ea
2606ec152d13ebb08e9c6875509e2016
2b8c0e3dbebc726e78635772a9602424
10370de71576153802b212828f12915a
e003890d1d2fa1e632a92da067867785
5d0b233ab867d57ec357ac0cc42afe91
bb498350e9cece014d01c32ca255e65d
f4c251b26631ada3f98d5810a725d4cb
cb26863b3e572285de3cd700622eefb8
13ca0e33e9ec08074fc8d6a685ba8a2b
73444e3f7ede827b9db22505ac645bc8
a4c5e6bd846a57661edb0147d4638e1d
133815acd388afea57ad2811fe26eef9
6bc0d0669cf08ccb2170c77aa4dc2e0c
5a68ebb5334056e5593d391618b97682
a6bc335bea6f56f8a8eba3934da9f32a
1f6cf877d409266c41caac2762fa7bc4
ddcf57637cd2ab61d35577d3dd88eda4
266d6afbc167b3ef8ee3bab6124c310b
ec7162a672d24dbeb039e224e411704e
d81f47ebd255cd116630c763d60b6035
90e32abc3df3e1c17dcf3710e96daf1f
3927210d541c829bbca1adc9f8c8ecea
59237fcd9a0287b0116b38326b93167c
4b16c935ab5e616c1d3cbe96711dfbe1
e11130fb371f2b8ad36f503d388cbd4f
d0d4dfde112b8435652343abfc13fda7
c14a667be99bb8fd5f5ae63e88e7d5db
f640b51ad384276354e2333c3067ad51
e2d52819869b6410f0c51fb232905a0e
3bd266b8ae815103ec9fdd19ed1b48ed
377b50dff669bc3b8701f343ff8f2801
7c6405d5f2311fd4bb104395451b6159
e94443c12693fbcce83b57e13c233ad5
8373b5e6deaf6197ee69a614950bf8f9
50532062b858d79ebc3d23cf546de4db
9edc7c01f50323a0ebbd8af5a207aa3e
815037aa8da5dacfa1b8b735356e6ced
e63c3c22a3d39a2b4aa57a27c1631582
9879ad9df1c712cc22ee62e794489eb5
769cf71c382d4c93fb515e019f71b48f
647a96170e8332a4d1298803582be9ce
eb3840b165f2cb8c5dff56b37cf8bd6c
6b8943fdb6c27b9788b4a7de30b6046e
135b7fd997f2d9a11297bcc5b0eaaf09
c10921e85049e98eb1199a65e3cf8aa2
510a91521d06be71f8f96e3c9d7be445
113264429316df34de72f3f17be81eef
ca9bece9402974e05d8232b86dff16ce
194104ef50d5de04c0f32cafecdec6d0
ac6ec03a6cdc76727449c3b2d9f25150
8d924c59f3daa290a7117064a8eed51f
aa653d2c1623f5e846a7794f964a368a
01e0be0dfda075f626a9654c2e903739
651ff19a8e452816715fb0343ea0df76
57ba251f2edd3adfe4b0ee0a6dc4a49b
60e89f1e9a559eb006840bdbf0d8828f
4582b9e44b3c2b1a42699abdb107a86a
8cc5b5f57fc2790c8f17231c4680600a
24ac4ff4c779152950596b8fb2ff66b4
f79072b0941241c8fff042f9af04470e
7779a97f53fca9845ce571806602b2c8
ad2836814ca1692f9b0ecaafc55e4ab2
4a152be330efff1f3309a9d88cc55841
6268a424c8c94959d318e00023c2d213
82cc62ed465f5a6a2f6544c8fc760d43
e88a7f76176b245597a33056d669d1fd
f9c730d83bd64b913c2f30c6aa1b9b86
e21e73f04796820d0317576caf260ea5
86b7758c9feb3264b2f7b29ca19d0b94
002aec2d164d058261cd5c8953bbf211
b769cd21e2011249380fbed12db81f19
aa648bbee11d89dd4da110502c2557d2
d961c1e07cfc7ff1d059ed0c85c52a1f
8140805e06a30fffc4b91658740c3970
54d011f2744c2ac7f18ae99a1ddbac14
fa01965f100b83da74aaae430ccce16c
75dff34a19d653ab1867942394734ffd
9ecd70fe6e1d9a38b6b1c18c0825f83b
df92c72d4819f3e40cee04de14fe01b7
382a004feb8b4e9152a09f9656fecdfa
ae7b59913e2222e351e1572dc814f589
cddfd7304e2fb20943f5cdc063e45b1c
5239739544e08338f43aa21fcbb98dbf
d7f61c9ea7ac3850e2d6e94c0b05b6b2
9402eb9699a5ce29e96bab2f29fe5984
d07ba80ac437febace8fd1e9059a55cd
26edbdcfca40cbabeae2b46f93488482
0efc7daf54d17e4d98c8b2cc0167727d
e1526f04a5f606a0e37efce2a6e8a241
9675c57750a6b3c83d323310a2decedc
ecedbeddbbb114a90629d220904cc4e2
35e4a7a05a32f90273a29d970fa30801
b067b851f4dea1ef4191e69976efa686
a3d87e6c027a436ebb7c7bf924dadc6f
b2898e6aad55bd154cda55020108c977
5e0833421d577b11e6364e6f876f2d19
120cbd9f0e32a6abcb44bd6e2055f3d1
37b3ea297d95739c06966cf60f404634
f665c8f0522be81bf25ac870cebd563e
0d6bdc6a17398fd4c75da65c162547b0
87f0e04953af8f4d2100b530924d99fa
2518efa0e7b2f510166acf414af72483
dfd20660f76d0cc839a2c2668180b3da
727159f2f069fcbe4ec9ede2cbb8af2e
8d26ad38abdec1701df0d49735dc5161
cee80d2e3b9919922f8953f6db578f87
c9035d4900ccebe64ff8cce8d81fae55
0c314844d0d4bc7f5659e1994e677403
80203de6ff26c6b1cb38081bdeccd579
679d57bfebde69e03661d3f467671ea8
62388c7e2d5719b0721a0bd9efeee5c3
825e3eecd3357ee8ad64e2d3cd11967e
2953d43e400cc77ae84b09b4fc20bfa3
6c18df2317713dededc49e83c1fd8d06
f16773d527fdb55b17ce8bc7b734c8e9
089a365b327dfa3890b109257fda6be3
c2b60613bab2f23ad2de3279cc0d71f8
791877b3492a1f14dbfd90fbcc5a5ebb
673ce870626e844b6adc912e24cff85d
8ecd49ed14ccc961f8c66239fa152f70
bf2f352e210c7c1b3904c1aeb972d63f
771123ba38ddf05469ec7408256c0a54
25d3e51b8f98895eca3d28b6b576ec21
68a554d0202b9b13659ef6c38c5c94e4
5a2ab1d6f262734c3e6ea6723867ebd2
b81845f0000044bfff1ec9cf2c4fcf67
7791116d0974efaa97507fd12239965d
e18a421940488d8ed707bce9e4a7a48d
5793694570d844d60cf50f551f125b23
7774e56765072eddb78ad897f2f5370a
a228be28e06c0661b529e5b5dbbf8fda
7b75d0538dde24f28dddafe3f4eb67c3
6625b1cd6bc73e0295ac78471f75c0ef
60f769690688fd1de1c3792e7437ecbc
0d4bbb4b9cfc230c782a0cfd7eaf7846
0d4f6dd2ccbaa94d0492ea6713e7b575
f788f9e5e5d3d6a8c8b40c9a9632341a
3662e03464eb47d0c03b09426aa0a0f6
03bb76c7be739097bcf58a8079768acf
eda1beb9a102e927c49624f5e51ef34c
05582fa48864632c450ec9d40177e1fe
d15a4cf02ccc70a5d8e0314d495b40c4
773d37179110c8a2cc2677cc0c84cd37
c625d7326d1fde145a979d5ecb93eb91
44de564cc03ef60cf4423d8abe1a9c3b
369c5e27bc989847a2bc5e6494db7e98
8f49460dd6a1e7928a21bce168133fb8
60a00c5cbb7a534b0c7803e7d9e08c53
5991020124eb2da0a27a22591dbd84e4
9e575d03dacb0179b88c15b106c298ae
3ee9ec953b0dab55574d28ffa4dcacc0
d383d59b77c0b8fd1d664faec365eb20
a6d203e296c593ed9cacbec3ef3d3219
2c1b42c0e624bb06f5590248e25fa6c9
2c33fe5c71f853952c9fdcf5425659d2
b4d01193788b100614d58eec73c1ef87
54fede7a4156bc139fff8745685f8c71
beae1ad66112c6c29227f53e29959f5d
4b2c34647599d2a4bf34274e53d2edb0
17c6473d1a3537ef9ecb48544d094864
ddd816b7abe7dffc460df56b3183d33e
c53e3c088a4287758dd54c7bd8b15e69
0f1e5157d11a3537a4870474fc242198
2c5d5b5b76da4eda258bb1ec8d5fe357
bdfdd78b5c1cb7b024c11c9712af42da
617e83b485cac7962886efad65906320
cb349d893e015e006af4323f206cb107
60af5a04f87878beff512a2c9abfd6a1
62e15ead07052faf09326e8650c4a63e
0060726ba074af8f2d9c104870145030
ad75f37b45868ce061c0180fb4072d24
e726d533f12c0e8adf2768eaa55faa83
9b4b9fc2c35e00fe342ce5c8801cec2d
91ef9f9a403512a0a446407ea5301933
5d487ef840dd9c9f77ec1f91c4fd003b
03790b82d83f0467469b8faf3e63887e
ec83033fd57002a2a8edd88d44810a1c
de6296f5ad2b9a62a7c3afb6d3e92b7e
dae6bacb459203ab0b6275c69d3d23d2
9c80d783d62cb5d3a5eef9655e92e02e
74f757162dc6b260716b50cf0ef86748
a2dd7c23e89c190485e14bf4ad0f5c54
9b1768e7a5fdae16f2b3b433aa410faf
b9e16d6843b20b4170f79bf1bc1a4992
ef4077f596cd69afb122d0e74e189f9b
1331b6d8492e5fa3ef27185b8637472f
7d34f30a794c47e9e9064b3be83458bc
bd47aa6816e3ff5b20608894a6d7d713
d8eaa9e01c99654b28ee22461c8a32bf
8c42118265297286a9ce1f66ba51e8ea
78243d265b62c24be422dca7d75b8cb4
c4d4bafc775e89432db32a6bca25aebf
165a7ed5ae9b53ded5512550e8bbcb52
e20d620e70558b57317f2d1429ebdbc7
27eaad840d7444ab3cc28897eeff4852
61db3f4f849d6ac021752c5906eecad0
e5f3c71ae2cae84cae7c016524a23b92
19147d5dd853bfd465d5a8c4c9c34fba
d93dad41499e351d8e932bdd41e36cb6
dab6daec1db636bc3b82d994dfe075b8
4591f9f42de6bc847257342777e22193
e84c4984952cb3d59eea935f4deeabb6
9e6f7ece67c81d3b78390075c30ac2f5
a6853b188f6b455299a3a51b7a5bcf3c
e84900c838d0bf3513c050918dd9dcea
d7a2bf90ddba66a7a88e7c2e645ef8ec
8cf41cc8cbd75ebdc15922bbcdca7c5d
34d299aed93729045473b6023137291f
a9ee39996ce94aa9a1154a825e32ba71
688970631dde11598d11c75f68fa9d91
89e2d197746613ed47077aa3f4c4fa54
7bab1daa79cc98f245a8254a38cf5074
fd83465cfc41b26d914583804de6a7bd
d027d3677c1e969de2376ab28593f40b
a54e6822b937890ff17ecd07e341e238
65412f036eddefd925d16b2591839528
2e9d3de8c8c2a84f5ed92ba93807f809
932f3eca3e30c0e6cb77b9e6ba69349c
b99a676d249849dbb388db14d3833d85
d79d5ba2448e1b6b6dbbe30fb9fa58d6
b9eda8e16c276dbff7054bf8e61ae1fd
cb98c6110ec9d2b53bfbfcf63abae9b5
d37eb9ccece9428526f2c7c503b84f05
30a522f721bff308d367c121e824b41c
d034d2b86d761734591fda87575e4030
5d56f1fb4f6810c61dd9a62d2942ef1f
aec0b38679df80e0c37e7157e21b4af7
ab44c7a9b4a6940a316e91d9cdbfd925
05df81da369bad9e0d590b70995b189a
765b9bda086f7d0da0530e3994c63d2b
00d6bc3be6ac161853816f40c1baecf3
d02a8c083ce0dd98c3444bde74e7e9ad
58ab38e63ae742a6979df9b6784e0d1b
ef90d362fc4efa35569aec62ce6aa24b
93c9039345503f778f366a5bc069ce95
b22669a6b86c33deb2eb6bc543557c73
c3c2d9c9a86b0a16adf485c40443eefb
3bddcf641c5be077542b1b1a54f4a688
e27d318fc90d1ffa4cb1fa28c8ef9c7b
6f61112743ca5f9e562f11f989163897
c30321f795122bcd048940f2234bd465
e98dd9090bdbbf813c5ad935f6572d16
794a276f67e45e1d92e9d65417088484
d714f0d536b3d37fe67cddc62bb25f1e
bd989be0d384ce924bbc35c381c4d8da
bae233c3f3843444554c8435f90c4edf
a63833d7f04590e0b631472d8a81b500
319b22fa63a7420bb9aa40a2b6c4b0b8
e1d851291640333a2dc27ce950d8d74c
b653486ff605cd55a9e81013ef4f8484
ccb9ece6c0ad77764d811a36ada81e67
e5e2c35d31ba20de12cb7f59ca14a0e5
a0bdc3408fd62cbeaf1fe419e31defb0
502413181eac3604d7bfe8b2fac442a4
38bc56952ec01c6727100b1982e94104
1eec0cfd7e69a91febdec8d1210f24d6
18f24ffb9a18907686eeca4252a0f38e
85462ee8fd7a9abb7728738dea2a8b3e
a74aa90b827af1e93b33b5452d62b30f
d8aae86677d0d33ce1991b94fcc97c70
19248b7b5bf9953597e247354223d21c
b1c972e20d27797ca184fe08c789b275
1155e7cc34a5b39373eb1648e1244587
c8d3939fb837389a9afd50a1b38abbb9
72f087f0d1d9b744c15b6bac2e8ad4ad
be14e2124af10119d811d34d93404d65
861386f19fbfc2ac81ef7b01df014e8b
1dd0e36c4edeab912e9a873af51949ad
7f358f049d4b156db831427acf646b1c
7331afdc3321b3e49ef7de95abb38b67
4a7440ead8c544154ea21f8288246315
50bcc015c055c3b9f6dee32e0210be11
00b7943d583a7c4bc62f7dbb40b7bc07
9971fd2db74725a22a91dfbedb9737b1
a918151bbaf9a217d6a13c0e8fa1bb62
384b025f3805c091481a92e57a953339
9e9f73fdcfc509aa414adfc8f33d1bc5
8fc1958ca6c22fe4805da5bbef6e4c7b
c13b2c793f7141b46a703a1ab4d71ff7
32c2a27aa99ad6bd4d60f62777754304
93d632b72aa81cc18c005ab25dfe56e0
f0593d0998f721785a36e9bbc6a66af0
1e2d862669eaeb20906628e8558d4b8e
ebd8a92bc503e9f2e757334b9ff66f9b
4fe94b0f155988552d05e9d6b780751d
54803e5af07b8cbc5398171cb932fe8c
5a0d6c53b99775214dc81e4825a6f795
f4bec6518f1b4e4bcaa527d90d63577a
81c3382e31c2fdb02a67777371b1d2b8
12e80df7310ed6b7a8b5b152c4d9037e
0518557bd2eedb6dd8f5ae5881e95f95
69d523f717e0695b64c6c4315166a8e9
69befe58fb637eb5f43f376486232fd7
86c125dd41fb435dbd6fec683d42d9f5
94118eadc9a89de7737f1483134be188
d0b3459cbbb959267b0a43aa01f80e60
79bcd8765127537c41f5608730836d90
e4884ebe20f97b83287bc796580e4966
54e4f4ce8e7fc2816604547d968c63ce
7fd066e3ce562402c2b80c511ab6d6e2
d6fe4d8f9a15531c23c22ce178e15c89
eaa6e588180219fff32ba66b2b7407b7
69a119bea3989372ed9e9194cca59f90
7778cb0ebbdfc545a5bacb2608b2d660
42390a1e65bdfbda6840404c110e03db
687d5773fcfc54fb1e829c841c255870
309e8c85d142e0681f779d7c09face16
f88ea24219e3c5d41c16c354d4b030bf
61c4023d47d9f09a0be83a5ed8a01ab6
990a1b3d1914351708207e4c47c11324
8857ddbfdc4fadc8edc7556afc18932f
f1bfe55eb0ab230ae8d0b9b2f6d98671
4c533b29db0e6b0279aed4951b564f0c
7ea878224c3732e12941ec5e3d8e9508
26cd7d0c0713fa0719f4bb06cc5f0a7b
8f42c4fc4e3fdbef15da20fea29b3918
5744527b93ca691cb9814c9c5f0993e1
e62a30401c686efd997b8e88d0a7c620
846a4d881d0bb22c72ff763760b9660c
33e063c1fd61c92cf227b44c7b6aeeae
605fd03c3864de0a024b92c119e3de6b
3a3c6adac015d8de52285d5934350f83
bf91970543fa59bf1941d49de2cb1fc8
5260163aadb2daec8c07a7d193484513
6392bce21434c0bd6c98a9bfe339221b
4438ef2ba3b0d8b0018ddbf3bc44e037
26d4e6ad2a5d585431346b417264269c
d2bc952ebe449c0b35d1f00587147be5
57c0a5f65098189b46a09d99e45cc871
e79f05dd3bd0da17f61d1a9c4b095580
fb1fd9999bd55d9eb6c25c67ec59ce69
834f6264799940371d19bd50d3106ffc
feb41a07dc0380fc2be14bf20de15802
a513269c6891015be0d02aebc9fc3c6b
fdbafc6ddd2488e2453f719213059d4f
65381f8e82138a894c35ae985ef07d4f
e5786c5f8efd776cfa507371ddb4dba6
56160c6b2667d854e4fa1fe5b4517de0
0c15649db90c06d6d7b6d04a857ba874
e316cb2539df96845033579735e33205
0e2387f10d59eee2ac3b31489f01cd14
81b0583ea0884e749aceb1149994bb6d
c39deca5eded7b06d69cb60298f23a15
8c58b3192a73fb2df1d82a27ec3b343f
a7ed9ef5456bd0f8fa7a0ec1028bb819
64eb92c7f266fde5c9cba4cfd5ff6ae1
5210d18e92596323f651dae8e4d019af
f8f71dc19f7b53367a427a2a52975670
7fd365934b1cadb99b3841146f2997fa
6ca41ef7b11827e682bb2fd452d0bfbb
89f713ab8a18f751d5784ec7b51fad11
95d2fc458d0497687a47ff2a379d0106
2f3935c7f6fc9c5165f773dc1a46e38c
5e1e735e9b1a4b903cbe635659b182c9
849ee96e144db23b412dfa695d5f814b
378ee06dfe4b4517892b9f448b470de4
37fee7729bdc646a538b36dfd50e976b
e055cc3708159a433a703e9edba27e42
5c3384a5e36e1510438d73c9b0518dac
bc918d7cddffef67bee07ca750d21708
47bb315cb385c95744213d2b5aab9d3d
d468abdf7afa7170f1a244c2c255f40d
8d2bd566312ea2080d4221a22a945509
9e3367d4d2dcef365306fdc7cf60d8b9
995be8e5c8e2ee6531ce9d9aa087130a
c8687d794efed1da154970a94bd90141
5ac114ce41512916a03fde0fd739a52b
74d015bf2a9858575a9fe19b154702fb
3bae3e1fdcb2e19ed16d5271beef4dd1
ea3a679ed736ea8b34e069214e2b0dcc
4f2a5dfaa22e9f419d0fee37f922de5c
efb75465362f85919e0aeb4aac6da9d9
ceb1e24e8ecb29f0d5ab3ea8ff79a1d8
17badcbde57013d0533845be0e68aba0
e39c45e54d54eff2873237737c1dee58
eaa89846dbb7f5f1896133cba156c473
047e76f3454788fe47983886c43ebe86
167d7b2dab321df4b5d19f288b36716f
b443ce9650dc4e45fb1fcc8a2b8f4579
b1ba65eeb3189b74ba1e59ebd6bf26e9
b0165af03a95847bffd6817bf8abaed1
45452065355f80c6777833b42243ef83
8495bfbb563ec963b6937d91f70bb9a0
3a8b41a56ed608baa2e3ecd2b0037a1f
6337f20598049993ae8fb377563c2ad5
c94fb4ef2176093bf99e4608fc308d6d
78335cfd3cafffc679725decc63fb054
b5f80725ca8959e8775e866c2f7591fd
5991d82a38bed684bb2a137959052495
8db74ea9113893bd10f3f54c1fa6a609
73dfd8cb8be8d62e25d6ca8a88d431b2
d445c05c62925d001c8a631e8c0a6e94
9e316d1186455f4f2621648f09a72d17
c7aba7987ea5b9a232340e295087df5b
a66acd5d284247524d089e11ea62cc88
ce2e35645fcba46f1940510ef843845e
fd152f7146bba6d56378672e259833ee
6ff30eb294aa37abdd8c1c38fa8ced58
f7f335d55a246cbcf3eccfc53c4f2869
6e16e2bf817fb4b1247b30599bfcf34e
eb5eb0a5bb072ba4d78e68fcccd96e8a
83bb67c815719a7fc587a8fe4578f686
8c18fc5943a17c08ed9eddbec1ac6987
2c340c66b13eecd74742184a23cc3364
8e9c301c9ad198a32d43d98a98a3526b
e1bb6d77e6e0e1cdf32844b48a653b45
4a59fd20f606705194000cf0c6d6ae3a
388e5e8016040b87ab298762ce12edf3
97508639b37ec1c09d539edd793c8b87
6b965e3767eefd1f5605b2e916bc4204
fec47fae77c7fcfa9923099ac9818f9d
903eb38d647721add29e9a85caa47d4e
0ddb8e0e891c44c2beec07da29d7a33c
41738be3a6d0322237a4ba7c5c3283d4
72f8269f9ef8b8a5bd82b3a02c6f4452
64f1af706d4710dee22d526ffc45da00
11f8b587ec82f48f3dc361f904a8ff73
c033fbad00c45d1eaeebcf92b990b77d
42d82f593f2879d122e5a8a8e5271787
2e9d2b38b965a7777fd1925501943586
1ab91d1a4cf72dcaa2d68e6f4a04eed7
338a04ce3b9a9a6345ef6f8412462d87
272aa105e102d10ab3856fb0ca97cf8f
cfb7959139325d45723d7a74a65fe7ac
10e70f6ab0a37542bf126c241072fcce
bc500032964132d24d93a3a8db49e9fb
4901e804d67fcc91c9f383bb9636395c
d4a485e477938488bff9600bff69915d
961674cb0dfdd7d2dfe3417743b08463
3b11b1512d030f8f1f0c9188093243a5
8b4ddc3d2aedff731fbfd49a612a7044
cc43ee346f4080cb208b130c3693d796
aa3a670339d6a33cfac5fa2af90432dc
aa7b3ef2113f5dee2f9f4d216d3c12f3
6dc88cb1e36d749a0ffc58f6048c7479
55328b7f1172709bddb6870710ff0118
64880906748a5aba1f59c34ddf3ce16d
5dc74fd9ae9b8880369fa8c979a866bd
e069738a540d354afbff07031de88515
8ea484ee6bf506d2a8dea360f9946006
db37d96dc8c5ecd7cffe5351861735d1
15b46b0c2dce76343d2a7b8e616df6b1
bf0b22daff921e122f0bf72a6479395a
e64062dee1953bcb1e4fb1b43e9d6ccf
665945013af2726df31d04ca8dfd4ec9
56e1adf88bc074f3b5d252b7d0d298ac
bcd60f71e861c40a6decf01465a27813
598020318b372e995ebdda44322e66b7
019ab9a82b57723d84c08f3564e59ad6
e2b549a6252b85e502d93e93d5481647
f5713ab7d4c1409d276986c689380e5b
e5f06f92020e35f85321e1ad605cf71f
7d4172a6cb366c35a904fab03ba42dd8
b7f760549a21e8c88aa4815dd32d682b
e9066b206c2ee8fdd18ebd22e5a6bc92
6fc7bc13aef09077cec33e3de2ae083e
f4d66592a3568f4aad539c2abe8f8007
7d91e9f414f2ebdddfff6cedbfb02cc3
add5d4b51f6b55cc00a4fedb9866467c
dceceadd7af4121c61a1fffa63dc64ce
bfea3767fb326ba49a11133373705f1d
393344e0d19a2a9103d4c9f2683957f5
3b0f9bafec437f029c983439604aaeea
94aa98cd82b64cb1ea911010f6f1a11f
00a32e703d21736e427f54a3e3a69311
a2d8795c51129fe6476706b2425e0179
51e73e1fb5547b83bbcaa1bac24314ee
fdfa03fda56e525538ea8a89be42e3b4
0303649b7a7a5ab2e6191e8da1984f70
eb23f1b2de22c24af3d110f03d8afe69
437773aea533e050305b2a9d847ff1ae
91320af2b01cefd8804cdb8705afced8
135f8847f59f2c4cbe8f9221a0ecf0ed
997479593c1c33fc6033bea162bf05a4
e7429a0fd7661a14695d8b6244efc798
81385881e10f2364fa5f4824671ff1f3
3db966b7c0c949389d613cc8a3965028
d3ed19574bb1d556df7f60eba44da0c1
738fe5cae8c014d2f9d3e34be8017587
b196d08debed8b8be97e2a9991c41b62
ef6ca895c92a08bc15a125f71e69a958
9cf5fca760b25be0a36f0ad4f001fa08
384fa35dc8ae1ee11045537e785746cb
d7a57b48e1ef2161706e7aaf200952bb
3084f84888128450b42991de322dcd33
39287fe32783329a7b65dc1786a2bbee
831f5f104bdbda885e555862226f9678
859ce824dfeaa0c83011e37b5b5171fa
8d12d89a139a4e5edff02047f9ef3fa7
eb5daebeec77e5f71f059a70666ee174
aab658b6eaef65c03ec348a94e020a01
8de4d0c2d13ca143733bef3a8659792a
de652475323ec81eb502486cd1fe4ceb
d328663c7693c951dec3e117a9eac667
383066ce7908750899bf3438b6b46d12
1ae756d1fd1dad8d73aa7d9bc413f407
8a7738c89c210da777a61fd02c3ae6c2
29b2af25799edff824e5f6ad04a16c15
4131d7672ccc781e045ca02aa2dfb8e8
d36167e2c54d3a4e24157385a30d8dcf
b646103a2824753ef04fb8c5e7ff352d
0b33c2c14f53aa8f4a21725512fdda4f
b86f3f70c928b4dd56e748146a2eaa23
6792ebe4eafb29b2ca599582bcc1078a
3c95e05860c49bf8433c5755f31d17ca
249ea9c5ab33ac096bbb605b107576b6
7e7d6128fda17e7851fbd29259210db6
76232ae8b4aded575fd172c46e057ac7
997d75d73711e745388e03c2d11755d9
534dbf1a30ac597c3c9a0d4be255046e
f004e9ec172a2c3318a0f838d3d95067
ce1162c189d631cba69bdd327c8ddba4
457bbef05dc827592e8b1a761e00d8b0
143eb0767254ae15783ff24c2839f288
d59ee83d018a6db0549de804a3a33d21
a35d41848d9974453454f50217685074
300881c87988acab403cc5f91f79cf73
7f943e589f1ae08c55bf9f490fe7aeef
d562dcaa563076f04c33246ee9d3b5c3
24f0b5ed631e9ffc8ef9f043643c4b24
36552f1428f79d01985d3d0ce9e1688f
14915e7f199dbf37d652c695f7b981a7
e5a17cb6b668c0d16b5b84ad07c866d0
a0953f3c5091c6c2fc17d666d0bb61a1
adbd8e749de1d9362762bc3afb3420f7
7d1133445aa0ed76c585ba323b674802
b930479f21bfa497e6575ab78085a37f
85ee408b12f708d69f6f8ccbd29d4b8a
5510b7ebd8c84f504cfdc1089abce2cc
0840ffd956c82410a9c27b8aee1c96c7
07c6d933979377cc79a9e23237e8ec8e
a071e3377f08f9977da0d8f263bf572f
d66f8111d49c858155a47e1cce171f41
0d5f8924e4a9513aad185f553beff73d
ecb912852454acf6bd8a0e6bf59f8b29
b130053d09ea120545333f952e040d16
895d3f5f0fd290eba62b1150b2363e58
83c9f11423832e88d3cc36752a485628
5de6ca0da5b2005bab47e4ef713c0600
5dc49767277dbafcc3acea1762098b31
7ae0439e969a5a628d64973b391ab4f8
652f65205b6cc083167025bd7376fb9e
a23c930f3562290d9a3cde406a401942
d05065687d9913311b00d001ebaa93fd
526e09e13acd742d8570d0014109fdb6
534ef987093b3d7a56af1908c5993321
357a75aa0428bec8080e293f86bb3064
ea2a7bb683002d861f8298ab6ddb472f
3dfe27ba8f40926d934684977945109e
eaec3d3e1087d1a16f58265653497c0b
b50ab7aed8bec37646869eabf6fedc78
786a311c66e687630f978a5d43a4b9c0
d04be631613592b78bddc2185dff057c
e96609bdb300ed185690cdec03f451b4
86f2999218da6224ab9768a5fdb7fecf
b93d93163f0cb6435931735757d6eeff
0838e8ac16e7952d38b5803839125d0e
daac45ce5326ace60451d2b0a1689ffd
fa2fe7da918d6098093e71085cfecc4e
db35c82e3b861728abcacbe1d30fb9ca
cedb2864a0d31014d3717cce2bfa246b
e910da2e72c70c48cdcce7cd999d145f
6f192de4e8682a6dc7696c840aa4cbeb
b49e8cc7ddc16a781d541a4901cdc773
e04d9f073375918e2a4147196c3afc4a
b9edf909715df8e8b982dd3a73529c22
18873d8f5937d6a73aaf35eeb1bb8498
3a1b5c8857cab3b4165387ab7af55d92
37bdaecad5ecca81e6f3f58bb5c22010
3ed0a2315db28c6567e00b00c99c5a43
1b8fd03c2fb0b70aa83468cf930107de
fb6cbc0678c0e6c45ad406a277abb60c
ac6d1efd6de66c7de496bd4066867074
ac3c0a28fecb9f2f35a954cb140f963c
8630ff3268c6d3040d1154d69c1a325d
959b0e1626bb9df479be7df935225524
9ae849f7ce4957b5f18a8b1db13a8a78
a9483e09e0d28e712ed90111257c2f4a
a6e5951df2b83822e69cf1d3116ae09d
3c5e06e53f31856a329d4bc0b1f65322
6793ff83d5d61e394733d253a08aca57
86b587bc7e8ad19b7f1cc9c9ef0d0b4f
a2bc839c31b1f2998bf0139229a481e2
299079034503c6b50a9a8eceed1549bf
2377a603e6229ad9fab5bd3e4341c171
4f89f055e40e99ab974b60fc1da49d05
e05f1a1fd8523efbe03bd375ea600883
1be27dc8f6bcdce5fc4607f8045193a4
a67de39c376e597794477540a0347b39
d0348e5de6fcb512b2cfd30cf3c0d7cf
30806d32d3c476619ce8646a6ce6e5d4
e3ae3506baa12cbf5cef6607dbce57dd
8165b7722aaaa2748b996d2eefae1bdf
92ec81881e6742b8ab1614dc9d7c69e1
272587b7f532da0f0da287fd62ac607e
8045053960491c2b6b85c3d6651cd0e3
5fd70d908ef97ccead04cde7d9f08865
e5fd95e08158ca63f0213a595d2c724a
2be88d6f756f5e4a8daaec3b53d474e9
4f26b0f5388ca932db5cc501108cacb6
d0596c4c3dc67644bcd9f6ba0ef1e928
3fdac5be3f9c5b9806ef2dfdc6724d7b
0c8f218c31f65c870b15a96046ac63a0
1d17f6b5a97db3727959ac717f710fcd
7ef42d0722003f8bf12f8852726c85b7
ed8f03bab2eb1488ee0ed95f2b8be058
cca6e9131ec90d2c89b814eb05d4dfb5
ae78afced11f36f5791d9c9d862aa396
2d361ce8fcfe7f2cc149f27c75c54c77
433f20b66b3aabe10390325838225b9d
c87aa101861f22784debd2213f7b4983
a782bc4987f106d96c66ecec90acb415
8e18e47d4a4cd45e65a269792e69dcf5
ba78451e81ecdbdd5cf5fece28cf0d17
36c3648811c94c82ff7fde5ff49a5b43
e7268b67a09e9fa0a003b225c5eb90cc
b92b4bf3312a3bfd8846c5c850ccda81
5cfd3c4e97bb222665453a814fa985a9
637c524e589174df32efeee48f2379f2
36bbc6e67b131ffe66dabd71ecaca09a
5e104c75ac3daa798cbf6d1282cf6ad7
345f5be52c4a41e25c5fd319d92bc529
79e08c44d925d807614397ddecf042f5
ffed7b440f817823b1ea3cb18cea5528
3b2c6d13f7749fcf55a6ba4940c1a6e0
539e3fd9e5010db3fbc3b8f0f66c0d51
8c841810f37cb1bd8f1c9904ed2639b1
a92d2c3f122e5d20c4631e87573dfb6c
50338a95695dd3997a90bde6ef7115fe
00ec7cc5bff563f9dfb640f34d1ca26e
67723dcf7bb0b2f7af0e8f751bfcc9a6
cb416c57b266b526e8111f060fb636b2
c4cf12730d6ddaf7b39559b5a5e7b1a7
c02d69ebed8bd71a94850e3cff9d46bc
a713bcb01066f9cf42d26ba14a80f441
5bb4270a831c30922e2620c05c2910f4
d7954aebc2b13f3492f7c5bc29728534
baae1cf831b1d84ef7dea26364fcdb92
a05318159f1a19a94be6e48966da6be5
a9652b271227955eb0e6f3191c5d3e90
e2a06701b626ffeb2e98ad6ad0784524
7538195e972718b0fde0ada2324e6af7
31ffedea0b3cf4802599a6d2d9a1b9e3
441069a010b132560284e486163790db
2f40e94a655e67c9230dde06aaca7a7f
3a9a341aa1e0eddfc9fa20d090045848
defe65292219122821fc25e05eaa182a
7623dabbaefb657eeb048dc9f14007a6
1ed4456918e8805cd4b5c6b3075f7f35
2a0ab45e5c2c600ad430afa45b151f8b
b74c519e5e366e3474b72371d5ba916d
3b9aa87d40d923169c43f0b19f222d9e
1daec5b40881874effca597450384532
ab6fac7d6daea52638551f0a904c55cc
4a0a68696ecc49c67c3091a5ff55185f
a889edf014db30bffbd828d1994dd73e
0c81201539ab8362e9633b81de4768ec
d45b1ac0911f96e34e6c9204e5368526
9fc66cf95507089911bb657761a45d79
930bf0b24b6a33db2d981dfc40ae8656
da4b7dee573ab2b498e3eb1de473d875
e7d6081a86a654dc16853284d35930f0
4ac34eba006db53e952fa3a296704f0f
bce1b7fd9870b2c1afa7e8484976576e
fa6a2ac561ee1b4dd5eb7b0439a63685
d279aeeef9f2b87bf22b9e1c4dbe012f
e462bae0f76062aac89886fe41f35ef2
74309bb044b1fda48b6814daf2caa39c
f9f01bc77f7ceeea914fe660cbf964cb
302a77954bac334699b474b80222ec99
809eedfe42e97ab26070db5e3b14aad9
c440bbdda1946399d85b945aa65dbd0c
c8874b2e564c33675dfb9aacafa64de5
6786028055cdf8f73c412b9939d59ead
a73da1a33b978a16f71c6046ee38d092
7cffbe24f0a993973fe523810fd2936f
a67b127cddcd7a8a7f791afe73041e51
6c7d79524358983a9bd076499bf81347
3634df73c4719b8e7266b697ab69ae74
59e03fcfcb2f177a97f0dd3140945373
ff78c29fd6c1d4c2bd388db023808632
121f1c231e137d5660c25229954131fb
c09e6aed5d4ab6044b9be45aa83ec9e8
0339ade3bb496c042ddf3ef6da186770
356d058d3ce3aa34d488593e5687d440
14dff43726eb5f68b7bc685f73e55a4b
9729ad5bb78fe7d2b2ec4ff2c6b3e34e
50de5c6f32a146956120a837ea539618
2d4de278b3ae2e6a6a0dfe8d13c18d1e
fe6ca13ebf8c9f5112a6c243c07cfe45
ca1d6c9de815e89abae0fc031bef8a2d
778b12a891e93a259ebaaccf7c261abb
a9526d590dd806f5402881b3cc186902
be516138865a6533474f72241ff2be31
52f7f52b6f740cefca498e70927ee869
ade735a9bf3b79076a14e6021b738940
d7fecd191fd05e522a2e23a5d03d4ed9
2ef64b010b18f8e2ef544275277d821a
f80de71aed2ecee5dfd086b812987372
7ad85d29fa429dd0eb4f4ea094237e7b
6fdec49b0275a82b6fec8f8947ad2e14
6d6a3e95129f82e177fa73db0c62ffe8
698190ee2d7cfbb50e0edaf3715e9284
47a7a1175a3a5049561dd7017e4418cd
d5ca4438922a3d2c1b309076427002c8
41698680d8ceedb948c775c44ed6bf9f
8b1ecf5abe9f9433d4509ce33e21cdd2
758fd87c2aee4a54398e394557ee1207
74d11445066ba5b1da27d7cd3ebc1268
935c24b40b5795f5a94654f832649684
67a895971ed5af4b137e8a3f9d80cc73
57710f2e23a4522d9b38983e5196abf0
7b7c644e5205c5f00d2238f1dcc1341f
1ec85ab6e0aed0b071064f55b0c05a3c
9aae35a05b4d3d63b13f3a1f4f9fc0c4
16998be17b1197951623e4ae1c836d09
15e2e52481b05891807ef2d48701a047
dee599f75e34f46cf834d79b5322dc6b
4b7d1bb2b0d6c4e91286e052aa54b430
1926d9c6f1db3c7cb9ad646c355e6d84
e806ed83adaf7cded9275007a39810a5
d7b318853088b3492dcc8cd02e6d2609
cf2455e206f2701134115d3db38cd792
9cf54dee16826cc8b50e7862b5fd5c1b
d2d351a71bbed8f2718c2171b57b1fc3
21f6165afd55b441156f694bdfa14ada
43c4d146f5291b3f36f051050f4b1a84
0376710c128da019c8b121ac5f2003c2
84f1775660dd65aa557e83be7095c58c
747a88cda1275e04970c44cbf9598d30
49d8c5aee2d8ae18311ed2c335b05bb7
5f44ed39775faa6c4358b2deae34510e
86356cd2d3cf6a1150495199857cc42e
a1e1a2b5f4cfdf441a007315d628e3e3
6db0a7760fe1ba60f33d051a56cd7b25
856d7316135cfc806da6a134144ce82f
4a66fbe80e326cef5e3eb9d2676ca4e6
0e5e7fc4ae02f15999f3d27149289c47
35a9a2256ad7d2b6f6d2556ea7cee93b
400dc66b32654d2cebe95846a9ee8580
c1c39a5eee725ce0afc53ddc3bfb66ad
8c1fc0660534365507b2c2817607dfb8
10ad2c6c45f5c132564a6b9898ab995c
13a2d52725789a23a1c778f0dc6931e3
e15a247468a9ef0b3d8517ed3ada6c48
3df7bc08f76d8edaddb852dc13a5cce9
304d0bceed6a9a50f803c3390f067d61
054bfa711e81afa63a0e0e99143247ea
7877d83d4c1f5e305334f24d1c9197c8
277a7a19872a7356145d3801a7c3b4b6
0afb1282968487a7a4690f29e2b40efd
83292a2059f484d757fc7891ca1acb7a
4cad304e38025719209c2c97e62bff0c
82cb8d705b4c4c19957f5d1da4c1e3f4
1ff62b4db1fbe0638ffe461f9136f491
b5e96d21cd7f46b5f4e72cd92e1381c2
af1dbb54ccb1aec096c82d06217ee54a
df2a8f0a205032ade5cbf601cdf185b0
21c3c692508cdae62f70525408c51d4d
d4afb0fe9c3bc046b5e55e151fd4df49
93e0af163326c4cd0fe00dc8c10ba46a
1c12d430cd4cdb7da540af97198b1d52
7f8bcb09bf282970397e967936930e06
bfd66e448300978af2b2cd9a8e053f44
5f304b74aa452b51c2d2bb0d9a1bed3a
64603c0cccdc93939780b8aaffe8003c
04a6274950f7daeb4f6f296250dec31c
e637ea9767697282b7fe1510dbc1c2b7
59055585a1e0829aa63ea337e1d0ff2f
39df7e14239d8b86827e498aa83e8a2c
a75ab16a6f988bc8e48e55a6450bd619
9f6cce4075bcc944f3244f30ae4e34e8
124edc05118c7b6ed27baadaa8815aac
2ad451143e6f0a5152ecd0a494aee439
744fce003db9b2f6357ec3203e757703
cb25b048a8d1e91c0513bb7e8daa62c4
0c3625f0492177f703261975fbf26a65
8aabc596699cf6ddbee03cc977c717df
cf6680a23d700cfe482e87c1a25f9baa
66ca739f5088a3fb9a28b18f24742c08
eabc723eb05476f042592a9203808163
a9f6938b8721835ae7fad9ef6161f497
af50d1d251ddb25c11462d87cff6c2b0
d2654fd5796afaa66db88b258ee98cb1
dd50e0b15f8b3aa3d56ef90fbb2a5310
59c32076f508835c5c67d6c3702d92d9
0949f8375b2668a6f9acfdcaf6e91c36
c9661a85bdbc886b92b81d192f6f63b7
fd807df84403ddcd81c4117bffb71999
894361e398e09ca0e8d09dd2d6269b1e
a1d1d183f510a8b6a2d9bf307ae3ca20
ccb7efd3be55f2d837e273d170499c76
5ebfdac03cfcdb6342c0c0672fd44ae5
c6570ff73a9ceb2d62958c22370197ee
1e05641e3097e0be33895afb23384b7b
cc61adc64470ee07e6e162e40d9689c2
fe1f862aa3461e3e1870e316b33ef546
58597655313b2821449681613043212a
de91752f11f43e89d1de98ea2523a266
0214110862118965531df018333c2f4d
049e0cb18038be2a5ec80eccd3d92f0a
5913181175cab7e215caaf13fb94bd3b
45da3aa5112e2ab69e7889d11e349f00
4436af6801d3c1812a8823c5eb0c3f58
f64a61eb3b45743fd2c6575af220ca4d
57b557a9c324ba81e51c200a0458e333
aaeaa18b52e256bb188c26f6c1a318a5
de23c5e306fa4207bf2f30886ec51b5d
6c32423f3f724717f8b9e60713510f5f
708bd3acfef280a723406ec70b01a76c
fde377a9467d9fccf8ca2c42d97ed5b4
5eff66d9640c6e00d5322556fc0bb27c
647f5f0458278264166a823811018e19
839bdcdc96e541939ca7aac4bf3ce177
b75a54865eb5e2df826bcf54a989895e
544a666e8227e966bec157d52b32253c
748e8aac31faf2febf4b3d3f69d52416
3c39d86824d1d6908508296b1b0c2426
3dc3b95e3910b31a848b33e185fae806
0fe138e1b20dce0f1633465fa0964c85
5318d8427e9ab8a6d55386839321a479
d3151404d32a5ba11fc7bed5f3479a02
2d89180deec8b05d97fd801830ab3b47
1510b73f80fa48fb1269c8b86183df4f
e61b6917fe484fc0d859c25c0bb266ea
0efe2b6dc46765fd266e74d5b3e14056
3ebcd7570560133fffb1bd9136ce6b39
17fd4f7ffae7c61a69a9587595bb1802
c9938f19d5f8aced634cc2a045d81a72
56e5db1621b6dbb9c3fdff07b2c0b6ea
32c65890bb1878f7e4a76d2eb65f074e
0eef126bb8679e8ce36041c4a84d8f8f
a3c8b6cbfb13cdce5ecee8f19a31c7b5
205a8d6f26b76628f82bc7b544220b66
b35590b39426fa357be5e82f8e4a5d2f
3e92545b2ea5e39bf64296dddb1189b7
ba02ee687542a63f259dad7f24535f40
4441d38bcadc11e153d14d9a9b7465d9
9e14a0ccc2653c4afa13ada8eb7d15e8
1db4b70fefd7b17f0b956af985f8451f
04b026130cad510c1b3f88f555966542
001ff60a3e27f6a9b6d78b78d01e862e
fd090ad61ced34606d21e574b9214a04
af147701f1fc05bb8385dd93a3709b8d
9668894274ed245e261c8326dfa264ff
98ac482101de1bcbafc7a5c512d73c9d
c88b7661e5a3d5123ada2e12b108689f
1fe507ca40a3dbc4b0f7305c9346dd6e
653846c7aa210b12007f45cf184de431
12595fb475f53e10a1bb3c60e51660a7
e05283943d1c7b8e3e023d67a5ddf0d2
1c18782e463cda4d57a87a3ecb6357a5
f74eb5895b530f0a06acf05645ce3944
615448bedec291248c3734120fb37b03
41ae00a6a2e4749cda6d03797dfb5286
c064411c93a2ae9404eb297764be1c06
555b3567ebe3bde7c0cfcd9a025e40f7
732880b7776d29366d7d55ac9e83d121
5f6ac101a947c0b19c01bd35a3e30639
90666cdc50e8a920b351cfd7f0d56e3e
9c5adc010fd3f4bb75018935dd576b3e
e58a683e1ccfd1e8a9f662f69f543b75
9c6b91c525d8c3def27c2efad62ba59d
bceb8fcb56161b215126514f9a34ca0b
d4749d4b4a37215190c9032b78724563
ca9cf76606fd107a7d9496b9fc2dc212
5fd18de6bbccf47750c4a366b9fa3585
4d9adb0d3bdd6c26f1b1b6a4076a39f5
ba30a3f728a44b4501199bf047a51d2d
b268468302c925e9f28e223d57102f1c
2782c0993a2b79ac7ca24ce8b341e0b7
cdd862c803a9abd701d59682388288be
07483b36d5a7ab229db8bc509f820e79
e4f249b7bf8794ddd41a7ee3640a70f2
3ade7f88ba6550b94ebf7e25939da0af
ebc080c377bba1f5e8299b0d1d0e1607
af19a7bb632a0c2276e7a9f8d11cec9c
e8098eb63ed8842eb945b29531a71fc0
ae2f13fba12b08c9c0f62b712576e175
59fdc0c8d7bb6856142c2181de46328c
1ed3d891c7779d90e4df14f59025cedf
27f9912fb6f714642ed340aa05d6ba82
5c9bf6e78b8b63c1c7c170f2eb763e91
20e4d1753b054e81653e28ff8825f7cd
b9afc8ff94c61bb58039012d19f11b0b
48ba396008e7f30be60226856c3b7ddb
3778c7414a282deecbebc562918147f8
11df6a383c74e9084c565ab8f4029af4
47b45563789c4a4cdbeff2a9acad135b
755e9047376a10c7f9629f2b12a048c9
082a4fd0eb1415ab29c290109bc3f2d0
6182d1f47e35956421edd292a40fcfc9
e1ba034ff5af4879408bbbf4fa2c808f
f161541860b55b20f98271923772aad7
bef7dae3389e34a6f0e928b442de58d8
e022dcbfa05f2736a00a9d55193a82fd
581cc77f53ed2f6fd179be6d6dbe0132
77bc07364c35abc00a9c91f6b66c6760
0f373871ffd3699de550dca1b9723910
95bb98308e13e58a6a89df9da2012cc9
b82f2a7725a13c0351ffb05be8db3f0f
0747c5ad22e0155534fbec8ae3dd60c7
d3a47d3b9e5be5fca3a58635303c6979
4881b7bb0199d87a8e626e87f92d2669
b7aeb22da49ecc095477b7fae58c51bd
a2b1aab292869ab03e213ac899f60306
ffc18988681af0a8d76b0beac0e21190
72a037ee3459b5cc4a3088edeed88f26
cd43733021983c8704901efc017a5bef
e915fee02f264d7aa6295c3cec305522
de7db13ac8ce3337076f82acc4d91dcf
275d509562d4b6a9bbdcfc600e126528
5581d0ae383fd48fc1d1d2503193013a
024ffeebc5b113220819151db9e54380
6c00c29768aad2bdbcb697754b0d5c86
693ee828286a0052956cb513bd524247
0ae2aa136786378bb4190af7284029c4
404c6df0c7b5d56f9a8062b79dc14e14
220950e3acea44e19310c079dda65ba8
ce462a55ae4976e70cc380e0a8be2538
4c967786c2736466712f73d55ec26003
1604df88b457dcdc0753895f21535ae2
53e6cad330e60444a291287596c5fbe9
